<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TrainingRequest extends Model
{
    use HasFactory;

    protected $fillable = [
    'request_code',
    'user_id',
    'title',
    'description',
    'training_date',
    'vendor',
    'cost',
    'purposes',
    'expected_result',
    'status',
    'hrd_notes',
    'approved_by_hrd',
    'approved_at_hrd',
    'approved_by_director',
    'approved_at_director',
];

    protected $casts = [
        'training_date'        => 'date',
        'cost'                 => 'integer',
        'purposes'             => 'array',   // supaya otomatis jadi array di PHP
        'approved_at_hrd'      => 'datetime',
        'approved_at_director' => 'datetime',
    ];

    /**
     * ⬇⬇⬇ BOOTED ADA DI SINI (DI DALAM CLASS, DI LUAR FUNCTION LAIN) ⬇⬇⬇
     */
    protected static function booted()
    {
        static::creating(function (TrainingRequest $model) {

            if (!$model->request_code) {
                $year = now()->format('Y');

                // dapatkan request_code terakhir tahun ini
                $last = static::whereYear('created_at', $year)
                    ->orderBy('id', 'desc')
                    ->value('request_code');

                $next = 1;

                if ($last) {
                    // Format TR-2025-0007 → ambil 0007
                    $parts = explode('-', $last);
                    $lastSeq = (int) end($parts);
                    $next = $lastSeq + 1;
                }

                // Format TR-2025-0001
                $model->request_code = 'TR-' . $year . '-' . str_pad($next, 4, '0', STR_PAD_LEFT);
            }
        });
    }
    /**
     * ⬆⬆⬆ SAMPAI DI SINI ⬆⬆⬆
     */

    public function user()
    {
        return $this->belongsTo(User::class);
    }
    // app/Models/TrainingRequest.php

public function hrdApprover()
{
    return $this->belongsTo(User::class, 'approved_by_hrd');
}

public function directorApprover()
{
    return $this->belongsTo(User::class, 'approved_by_director');
}
}
