/**
 * @license React
 * eslint-plugin-react-hooks.production.js
 *
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

'use strict';

var core$1 = require('@babel/core');
var BabelParser = require('@babel/parser');
var v4 = require('zod/v4');
var v4$1 = require('zod-validation-error/v4');
var crypto = require('crypto');
var HermesParser = require('hermes-parser');
var util = require('util');

const SETTINGS_KEY = 'react-hooks';
const SETTINGS_ADDITIONAL_EFFECT_HOOKS_KEY = 'additionalEffectHooks';
function getAdditionalEffectHooksFromSettings(settings) {
    var _a;
    const additionalHooks = (_a = settings[SETTINGS_KEY]) === null || _a === void 0 ? void 0 : _a[SETTINGS_ADDITIONAL_EFFECT_HOOKS_KEY];
    if (additionalHooks != null && typeof additionalHooks === 'string') {
        return new RegExp(additionalHooks);
    }
    return undefined;
}

const rule$1 = {
    meta: {
        type: 'suggestion',
        docs: {
            description: 'verifies the list of dependencies for Hooks like useEffect and similar',
            recommended: true,
            url: 'https://github.com/facebook/react/issues/14920',
        },
        fixable: 'code',
        hasSuggestions: true,
        schema: [
            {
                type: 'object',
                additionalProperties: false,
                enableDangerousAutofixThisMayCauseInfiniteLoops: false,
                properties: {
                    additionalHooks: {
                        type: 'string',
                    },
                    enableDangerousAutofixThisMayCauseInfiniteLoops: {
                        type: 'boolean',
                    },
                    experimental_autoDependenciesHooks: {
                        type: 'array',
                        items: {
                            type: 'string',
                        },
                    },
                    requireExplicitEffectDeps: {
                        type: 'boolean',
                    },
                },
            },
        ],
    },
    create(context) {
        const rawOptions = context.options && context.options[0];
        const settings = context.settings || {};
        const additionalHooks = rawOptions && rawOptions.additionalHooks
            ? new RegExp(rawOptions.additionalHooks)
            : getAdditionalEffectHooksFromSettings(settings);
        const enableDangerousAutofixThisMayCauseInfiniteLoops = (rawOptions &&
            rawOptions.enableDangerousAutofixThisMayCauseInfiniteLoops) ||
            false;
        const experimental_autoDependenciesHooks = rawOptions && Array.isArray(rawOptions.experimental_autoDependenciesHooks)
            ? rawOptions.experimental_autoDependenciesHooks
            : [];
        const requireExplicitEffectDeps = (rawOptions && rawOptions.requireExplicitEffectDeps) || false;
        const options = {
            additionalHooks,
            experimental_autoDependenciesHooks,
            enableDangerousAutofixThisMayCauseInfiniteLoops,
            requireExplicitEffectDeps,
        };
        function reportProblem(problem) {
            if (enableDangerousAutofixThisMayCauseInfiniteLoops) {
                if (Array.isArray(problem.suggest) &&
                    problem.suggest.length > 0 &&
                    problem.suggest[0]) {
                    problem.fix = problem.suggest[0].fix;
                }
            }
            context.report(problem);
        }
        const getSourceCode = typeof context.getSourceCode === 'function'
            ? () => {
                return context.getSourceCode();
            }
            : () => {
                return context.sourceCode;
            };
        const getScope = typeof context.getScope === 'function'
            ? () => {
                return context.getScope();
            }
            : (node) => {
                return context.sourceCode.getScope(node);
            };
        const scopeManager = getSourceCode().scopeManager;
        const setStateCallSites = new WeakMap();
        const stateVariables = new WeakSet();
        const stableKnownValueCache = new WeakMap();
        const functionWithoutCapturedValueCache = new WeakMap();
        const useEffectEventVariables = new WeakSet();
        function memoizeWithWeakMap(fn, map) {
            return function (arg) {
                if (map.has(arg)) {
                    return map.get(arg);
                }
                const result = fn(arg);
                map.set(arg, result);
                return result;
            };
        }
        function visitFunctionWithDependencies(node, declaredDependenciesNode, reactiveHook, reactiveHookName, isEffect, isAutoDepsHook) {
            if (isEffect && node.async) {
                reportProblem({
                    node: node,
                    message: `Effect callbacks are synchronous to prevent race conditions. ` +
                        `Put the async function inside:\n\n` +
                        'useEffect(() => {\n' +
                        '  async function fetchData() {\n' +
                        '    // You can await here\n' +
                        '    const response = await MyAPI.getData(someId);\n' +
                        '    // ...\n' +
                        '  }\n' +
                        '  fetchData();\n' +
                        `}, [someId]); // Or [] if effect doesn't need props or state\n\n` +
                        'Learn more about data fetching with Hooks: https://react.dev/link/hooks-data-fetching',
                });
            }
            const scope = scopeManager.acquire(node);
            if (!scope) {
                throw new Error('Unable to acquire scope for the current node. This is a bug in eslint-plugin-react-hooks, please file an issue.');
            }
            const pureScopes = new Set();
            let componentScope = null;
            {
                let currentScope = scope.upper;
                while (currentScope) {
                    pureScopes.add(currentScope);
                    if (currentScope.type === 'function' ||
                        currentScope.type === 'hook' ||
                        currentScope.type === 'component') {
                        break;
                    }
                    currentScope = currentScope.upper;
                }
                if (!currentScope) {
                    return;
                }
                componentScope = currentScope;
            }
            const isArray = Array.isArray;
            function isStableKnownHookValue(resolved) {
                if (!isArray(resolved.defs)) {
                    return false;
                }
                const def = resolved.defs[0];
                if (def == null) {
                    return false;
                }
                const defNode = def.node;
                if (defNode.type !== 'VariableDeclarator') {
                    return false;
                }
                let init = defNode.init;
                if (init == null) {
                    return false;
                }
                while (init.type === 'TSAsExpression' || init.type === 'AsExpression') {
                    init = init.expression;
                }
                let declaration = defNode.parent;
                if (declaration == null && componentScope != null) {
                    fastFindReferenceWithParent(componentScope.block, def.node.id);
                    declaration = def.node.parent;
                    if (declaration == null) {
                        return false;
                    }
                }
                if (declaration != null &&
                    'kind' in declaration &&
                    declaration.kind === 'const' &&
                    init.type === 'Literal' &&
                    (typeof init.value === 'string' ||
                        typeof init.value === 'number' ||
                        init.value === null)) {
                    return true;
                }
                if (init.type !== 'CallExpression') {
                    return false;
                }
                let callee = init.callee;
                if (callee.type === 'MemberExpression' &&
                    'name' in callee.object &&
                    callee.object.name === 'React' &&
                    callee.property != null &&
                    !callee.computed) {
                    callee = callee.property;
                }
                if (callee.type !== 'Identifier') {
                    return false;
                }
                const definitionNode = def.node;
                const id = definitionNode.id;
                const { name } = callee;
                if (name === 'useRef' && id.type === 'Identifier') {
                    return true;
                }
                else if (isUseEffectEventIdentifier$1(callee) &&
                    id.type === 'Identifier') {
                    for (const ref of resolved.references) {
                        if (ref !== id) {
                            useEffectEventVariables.add(ref.identifier);
                        }
                    }
                    return true;
                }
                else if (name === 'useState' ||
                    name === 'useReducer' ||
                    name === 'useActionState') {
                    if (id.type === 'ArrayPattern' &&
                        id.elements.length === 2 &&
                        isArray(resolved.identifiers)) {
                        if (id.elements[1] === resolved.identifiers[0]) {
                            if (name === 'useState') {
                                const references = resolved.references;
                                let writeCount = 0;
                                for (const reference of references) {
                                    if (reference.isWrite()) {
                                        writeCount++;
                                    }
                                    if (writeCount > 1) {
                                        return false;
                                    }
                                    setStateCallSites.set(reference.identifier, id.elements[0]);
                                }
                            }
                            return true;
                        }
                        else if (id.elements[0] === resolved.identifiers[0]) {
                            if (name === 'useState') {
                                const references = resolved.references;
                                for (const reference of references) {
                                    stateVariables.add(reference.identifier);
                                }
                            }
                            return false;
                        }
                    }
                }
                else if (name === 'useTransition') {
                    if (id.type === 'ArrayPattern' &&
                        id.elements.length === 2 &&
                        Array.isArray(resolved.identifiers)) {
                        if (id.elements[1] === resolved.identifiers[0]) {
                            return true;
                        }
                    }
                }
                return false;
            }
            function isFunctionWithoutCapturedValues(resolved) {
                if (!isArray(resolved.defs)) {
                    return false;
                }
                const def = resolved.defs[0];
                if (def == null) {
                    return false;
                }
                if (def.node == null || def.node.id == null) {
                    return false;
                }
                const fnNode = def.node;
                const childScopes = (componentScope === null || componentScope === void 0 ? void 0 : componentScope.childScopes) || [];
                let fnScope = null;
                for (const childScope of childScopes) {
                    const childScopeBlock = childScope.block;
                    if ((fnNode.type === 'FunctionDeclaration' &&
                        childScopeBlock === fnNode) ||
                        (fnNode.type === 'VariableDeclarator' &&
                            childScopeBlock.parent === fnNode)) {
                        fnScope = childScope;
                        break;
                    }
                }
                if (fnScope == null) {
                    return false;
                }
                for (const ref of fnScope.through) {
                    if (ref.resolved == null) {
                        continue;
                    }
                    if (pureScopes.has(ref.resolved.scope) &&
                        !memoizedIsStableKnownHookValue(ref.resolved)) {
                        return false;
                    }
                }
                return true;
            }
            const memoizedIsStableKnownHookValue = memoizeWithWeakMap(isStableKnownHookValue, stableKnownValueCache);
            const memoizedIsFunctionWithoutCapturedValues = memoizeWithWeakMap(isFunctionWithoutCapturedValues, functionWithoutCapturedValueCache);
            const currentRefsInEffectCleanup = new Map();
            function isInsideEffectCleanup(reference) {
                let curScope = reference.from;
                let isInReturnedFunction = false;
                while (curScope != null && curScope.block !== node) {
                    if (curScope.type === 'function') {
                        isInReturnedFunction =
                            curScope.block.parent != null &&
                                curScope.block.parent.type === 'ReturnStatement';
                    }
                    curScope = curScope.upper;
                }
                return isInReturnedFunction;
            }
            const dependencies = new Map();
            const optionalChains = new Map();
            gatherDependenciesRecursively(scope);
            function gatherDependenciesRecursively(currentScope) {
                var _a, _b, _c, _d, _e;
                for (const reference of currentScope.references) {
                    if (!reference.resolved) {
                        continue;
                    }
                    if (!pureScopes.has(reference.resolved.scope)) {
                        continue;
                    }
                    const referenceNode = fastFindReferenceWithParent(node, reference.identifier);
                    if (referenceNode == null) {
                        continue;
                    }
                    const dependencyNode = getDependency(referenceNode);
                    const dependency = analyzePropertyChain(dependencyNode, optionalChains);
                    if (isEffect &&
                        dependencyNode.type === 'Identifier' &&
                        (((_a = dependencyNode.parent) === null || _a === void 0 ? void 0 : _a.type) === 'MemberExpression' ||
                            ((_b = dependencyNode.parent) === null || _b === void 0 ? void 0 : _b.type) === 'OptionalMemberExpression') &&
                        !dependencyNode.parent.computed &&
                        dependencyNode.parent.property.type === 'Identifier' &&
                        dependencyNode.parent.property.name === 'current' &&
                        isInsideEffectCleanup(reference)) {
                        currentRefsInEffectCleanup.set(dependency, {
                            reference,
                            dependencyNode,
                        });
                    }
                    if (((_c = dependencyNode.parent) === null || _c === void 0 ? void 0 : _c.type) === 'TSTypeQuery' ||
                        ((_d = dependencyNode.parent) === null || _d === void 0 ? void 0 : _d.type) === 'TSTypeReference') {
                        continue;
                    }
                    const def = reference.resolved.defs[0];
                    if (def == null) {
                        continue;
                    }
                    if (def.node != null && def.node.init === node.parent) {
                        continue;
                    }
                    if (def.type === 'TypeParameter') {
                        continue;
                    }
                    if (!dependencies.has(dependency)) {
                        const resolved = reference.resolved;
                        const isStable = memoizedIsStableKnownHookValue(resolved) ||
                            memoizedIsFunctionWithoutCapturedValues(resolved);
                        dependencies.set(dependency, {
                            isStable,
                            references: [reference],
                        });
                    }
                    else {
                        (_e = dependencies.get(dependency)) === null || _e === void 0 ? void 0 : _e.references.push(reference);
                    }
                }
                for (const childScope of currentScope.childScopes) {
                    gatherDependenciesRecursively(childScope);
                }
            }
            currentRefsInEffectCleanup.forEach(({ reference, dependencyNode }, dependency) => {
                var _a, _b;
                const references = ((_a = reference.resolved) === null || _a === void 0 ? void 0 : _a.references) || [];
                let foundCurrentAssignment = false;
                for (const ref of references) {
                    const { identifier } = ref;
                    const { parent } = identifier;
                    if (parent != null &&
                        parent.type === 'MemberExpression' &&
                        !parent.computed &&
                        parent.property.type === 'Identifier' &&
                        parent.property.name === 'current' &&
                        ((_b = parent.parent) === null || _b === void 0 ? void 0 : _b.type) === 'AssignmentExpression' &&
                        parent.parent.left === parent) {
                        foundCurrentAssignment = true;
                        break;
                    }
                }
                if (foundCurrentAssignment) {
                    return;
                }
                reportProblem({
                    node: dependencyNode.parent.property,
                    message: `The ref value '${dependency}.current' will likely have ` +
                        `changed by the time this effect cleanup function runs. If ` +
                        `this ref points to a node rendered by React, copy ` +
                        `'${dependency}.current' to a variable inside the effect, and ` +
                        `use that variable in the cleanup function.`,
                });
            });
            const staleAssignments = new Set();
            function reportStaleAssignment(writeExpr, key) {
                if (staleAssignments.has(key)) {
                    return;
                }
                staleAssignments.add(key);
                reportProblem({
                    node: writeExpr,
                    message: `Assignments to the '${key}' variable from inside React Hook ` +
                        `${getSourceCode().getText(reactiveHook)} will be lost after each ` +
                        `render. To preserve the value over time, store it in a useRef ` +
                        `Hook and keep the mutable value in the '.current' property. ` +
                        `Otherwise, you can move this variable directly inside ` +
                        `${getSourceCode().getText(reactiveHook)}.`,
                });
            }
            const stableDependencies = new Set();
            dependencies.forEach(({ isStable, references }, key) => {
                if (isStable) {
                    stableDependencies.add(key);
                }
                references.forEach(reference => {
                    if (reference.writeExpr) {
                        reportStaleAssignment(reference.writeExpr, key);
                    }
                });
            });
            if (staleAssignments.size > 0) {
                return;
            }
            if (!declaredDependenciesNode) {
                if (isAutoDepsHook) {
                    return;
                }
                let setStateInsideEffectWithoutDeps = null;
                dependencies.forEach(({ references }, key) => {
                    if (setStateInsideEffectWithoutDeps) {
                        return;
                    }
                    references.forEach(reference => {
                        if (setStateInsideEffectWithoutDeps) {
                            return;
                        }
                        const id = reference.identifier;
                        const isSetState = setStateCallSites.has(id);
                        if (!isSetState) {
                            return;
                        }
                        let fnScope = reference.from;
                        while (fnScope != null && fnScope.type !== 'function') {
                            fnScope = fnScope.upper;
                        }
                        const isDirectlyInsideEffect = (fnScope === null || fnScope === void 0 ? void 0 : fnScope.block) === node;
                        if (isDirectlyInsideEffect) {
                            setStateInsideEffectWithoutDeps = key;
                        }
                    });
                });
                if (setStateInsideEffectWithoutDeps) {
                    const { suggestedDependencies } = collectRecommendations({
                        dependencies,
                        declaredDependencies: [],
                        stableDependencies,
                        externalDependencies: new Set(),
                        isEffect: true,
                    });
                    reportProblem({
                        node: reactiveHook,
                        message: `React Hook ${reactiveHookName} contains a call to '${setStateInsideEffectWithoutDeps}'. ` +
                            `Without a list of dependencies, this can lead to an infinite chain of updates. ` +
                            `To fix this, pass [` +
                            suggestedDependencies.join(', ') +
                            `] as a second argument to the ${reactiveHookName} Hook.`,
                        suggest: [
                            {
                                desc: `Add dependencies array: [${suggestedDependencies.join(', ')}]`,
                                fix(fixer) {
                                    return fixer.insertTextAfter(node, `, [${suggestedDependencies.join(', ')}]`);
                                },
                            },
                        ],
                    });
                }
                return;
            }
            if (isAutoDepsHook &&
                declaredDependenciesNode.type === 'Literal' &&
                declaredDependenciesNode.value === null) {
                return;
            }
            const declaredDependencies = [];
            const externalDependencies = new Set();
            const isArrayExpression = declaredDependenciesNode.type === 'ArrayExpression';
            const isTSAsArrayExpression = declaredDependenciesNode.type === 'TSAsExpression' &&
                declaredDependenciesNode.expression.type === 'ArrayExpression';
            if (!isArrayExpression && !isTSAsArrayExpression) {
                reportProblem({
                    node: declaredDependenciesNode,
                    message: `React Hook ${getSourceCode().getText(reactiveHook)} was passed a ` +
                        'dependency list that is not an array literal. This means we ' +
                        "can't statically verify whether you've passed the correct " +
                        'dependencies.',
                });
            }
            else {
                const arrayExpression = isTSAsArrayExpression
                    ? declaredDependenciesNode.expression
                    : declaredDependenciesNode;
                arrayExpression.elements.forEach(declaredDependencyNode => {
                    if (declaredDependencyNode === null) {
                        return;
                    }
                    if (declaredDependencyNode.type === 'SpreadElement') {
                        reportProblem({
                            node: declaredDependencyNode,
                            message: `React Hook ${getSourceCode().getText(reactiveHook)} has a spread ` +
                                "element in its dependency array. This means we can't " +
                                "statically verify whether you've passed the " +
                                'correct dependencies.',
                        });
                        return;
                    }
                    if (useEffectEventVariables.has(declaredDependencyNode)) {
                        reportProblem({
                            node: declaredDependencyNode,
                            message: 'Functions returned from `useEffectEvent` must not be included in the dependency array. ' +
                                `Remove \`${getSourceCode().getText(declaredDependencyNode)}\` from the list.`,
                            suggest: [
                                {
                                    desc: `Remove the dependency \`${getSourceCode().getText(declaredDependencyNode)}\``,
                                    fix(fixer) {
                                        return fixer.removeRange(declaredDependencyNode.range);
                                    },
                                },
                            ],
                        });
                    }
                    let declaredDependency;
                    try {
                        declaredDependency = analyzePropertyChain(declaredDependencyNode, null);
                    }
                    catch (error) {
                        if (error instanceof Error &&
                            /Unsupported node type/.test(error.message)) {
                            if (declaredDependencyNode.type === 'Literal') {
                                if (declaredDependencyNode.value &&
                                    dependencies.has(declaredDependencyNode.value)) {
                                    reportProblem({
                                        node: declaredDependencyNode,
                                        message: `The ${declaredDependencyNode.raw} literal is not a valid dependency ` +
                                            `because it never changes. ` +
                                            `Did you mean to include ${declaredDependencyNode.value} in the array instead?`,
                                    });
                                }
                                else {
                                    reportProblem({
                                        node: declaredDependencyNode,
                                        message: `The ${declaredDependencyNode.raw} literal is not a valid dependency ` +
                                            'because it never changes. You can safely remove it.',
                                    });
                                }
                            }
                            else {
                                reportProblem({
                                    node: declaredDependencyNode,
                                    message: `React Hook ${getSourceCode().getText(reactiveHook)} has a ` +
                                        `complex expression in the dependency array. ` +
                                        'Extract it to a separate variable so it can be statically checked.',
                                });
                            }
                            return;
                        }
                        else {
                            throw error;
                        }
                    }
                    let maybeID = declaredDependencyNode;
                    while (maybeID.type === 'MemberExpression' ||
                        maybeID.type === 'OptionalMemberExpression' ||
                        maybeID.type === 'ChainExpression') {
                        maybeID = maybeID.object || maybeID.expression.object;
                    }
                    const isDeclaredInComponent = !componentScope.through.some(ref => ref.identifier === maybeID);
                    declaredDependencies.push({
                        key: declaredDependency,
                        node: declaredDependencyNode,
                    });
                    if (!isDeclaredInComponent) {
                        externalDependencies.add(declaredDependency);
                    }
                });
            }
            const { suggestedDependencies, unnecessaryDependencies, missingDependencies, duplicateDependencies, } = collectRecommendations({
                dependencies,
                declaredDependencies,
                stableDependencies,
                externalDependencies,
                isEffect,
            });
            let suggestedDeps = suggestedDependencies;
            const problemCount = duplicateDependencies.size +
                missingDependencies.size +
                unnecessaryDependencies.size;
            if (problemCount === 0) {
                const constructions = scanForConstructions({
                    declaredDependencies,
                    declaredDependenciesNode,
                    componentScope,
                    scope,
                });
                constructions.forEach(({ construction, isUsedOutsideOfHook, depType }) => {
                    var _a;
                    const wrapperHook = depType === 'function' ? 'useCallback' : 'useMemo';
                    const constructionType = depType === 'function' ? 'definition' : 'initialization';
                    const defaultAdvice = `wrap the ${constructionType} of '${construction.name.name}' in its own ${wrapperHook}() Hook.`;
                    const advice = isUsedOutsideOfHook
                        ? `To fix this, ${defaultAdvice}`
                        : `Move it inside the ${reactiveHookName} callback. Alternatively, ${defaultAdvice}`;
                    const causation = depType === 'conditional' || depType === 'logical expression'
                        ? 'could make'
                        : 'makes';
                    const message = `The '${construction.name.name}' ${depType} ${causation} the dependencies of ` +
                        `${reactiveHookName} Hook (at line ${(_a = declaredDependenciesNode.loc) === null || _a === void 0 ? void 0 : _a.start.line}) ` +
                        `change on every render. ${advice}`;
                    let suggest;
                    if (isUsedOutsideOfHook &&
                        construction.type === 'Variable' &&
                        depType === 'function') {
                        suggest = [
                            {
                                desc: `Wrap the ${constructionType} of '${construction.name.name}' in its own ${wrapperHook}() Hook.`,
                                fix(fixer) {
                                    const [before, after] = wrapperHook === 'useMemo'
                                        ? [`useMemo(() => { return `, '; })']
                                        : ['useCallback(', ')'];
                                    return [
                                        fixer.insertTextBefore(construction.node.init, before),
                                        fixer.insertTextAfter(construction.node.init, after),
                                    ];
                                },
                            },
                        ];
                    }
                    reportProblem({
                        node: construction.node,
                        message,
                        suggest,
                    });
                });
                return;
            }
            if (!isEffect && missingDependencies.size > 0) {
                suggestedDeps = collectRecommendations({
                    dependencies,
                    declaredDependencies: [],
                    stableDependencies,
                    externalDependencies,
                    isEffect,
                }).suggestedDependencies;
            }
            function areDeclaredDepsAlphabetized() {
                if (declaredDependencies.length === 0) {
                    return true;
                }
                const declaredDepKeys = declaredDependencies.map(dep => dep.key);
                const sortedDeclaredDepKeys = declaredDepKeys.slice().sort();
                return declaredDepKeys.join(',') === sortedDeclaredDepKeys.join(',');
            }
            if (areDeclaredDepsAlphabetized()) {
                suggestedDeps.sort();
            }
            function formatDependency(path) {
                const members = path.split('.');
                let finalPath = '';
                for (let i = 0; i < members.length; i++) {
                    if (i !== 0) {
                        const pathSoFar = members.slice(0, i + 1).join('.');
                        const isOptional = optionalChains.get(pathSoFar) === true;
                        finalPath += isOptional ? '?.' : '.';
                    }
                    finalPath += members[i];
                }
                return finalPath;
            }
            function getWarningMessage(deps, singlePrefix, label, fixVerb) {
                if (deps.size === 0) {
                    return null;
                }
                return ((deps.size > 1 ? '' : singlePrefix + ' ') +
                    label +
                    ' ' +
                    (deps.size > 1 ? 'dependencies' : 'dependency') +
                    ': ' +
                    joinEnglish(Array.from(deps)
                        .sort()
                        .map(name => "'" + formatDependency(name) + "'")) +
                    `. Either ${fixVerb} ${deps.size > 1 ? 'them' : 'it'} or remove the dependency array.`);
            }
            let extraWarning = '';
            if (unnecessaryDependencies.size > 0) {
                let badRef = null;
                Array.from(unnecessaryDependencies.keys()).forEach(key => {
                    if (badRef !== null) {
                        return;
                    }
                    if (key.endsWith('.current')) {
                        badRef = key;
                    }
                });
                if (badRef !== null) {
                    extraWarning =
                        ` Mutable values like '${badRef}' aren't valid dependencies ` +
                            "because mutating them doesn't re-render the component.";
                }
                else if (externalDependencies.size > 0) {
                    const dep = Array.from(externalDependencies)[0];
                    if (!scope.set.has(dep)) {
                        extraWarning =
                            ` Outer scope values like '${dep}' aren't valid dependencies ` +
                                `because mutating them doesn't re-render the component.`;
                    }
                }
            }
            if (!extraWarning && missingDependencies.has('props')) {
                const propDep = dependencies.get('props');
                if (propDep == null) {
                    return;
                }
                const refs = propDep.references;
                if (!Array.isArray(refs)) {
                    return;
                }
                let isPropsOnlyUsedInMembers = true;
                for (const ref of refs) {
                    const id = fastFindReferenceWithParent(componentScope.block, ref.identifier);
                    if (!id) {
                        isPropsOnlyUsedInMembers = false;
                        break;
                    }
                    const parent = id.parent;
                    if (parent == null) {
                        isPropsOnlyUsedInMembers = false;
                        break;
                    }
                    if (parent.type !== 'MemberExpression' &&
                        parent.type !== 'OptionalMemberExpression') {
                        isPropsOnlyUsedInMembers = false;
                        break;
                    }
                }
                if (isPropsOnlyUsedInMembers) {
                    extraWarning =
                        ` However, 'props' will change when *any* prop changes, so the ` +
                            `preferred fix is to destructure the 'props' object outside of ` +
                            `the ${reactiveHookName} call and refer to those specific props ` +
                            `inside ${getSourceCode().getText(reactiveHook)}.`;
                }
            }
            if (!extraWarning && missingDependencies.size > 0) {
                let missingCallbackDep = null;
                missingDependencies.forEach(missingDep => {
                    var _a;
                    if (missingCallbackDep) {
                        return;
                    }
                    const topScopeRef = componentScope.set.get(missingDep);
                    const usedDep = dependencies.get(missingDep);
                    if (!(usedDep === null || usedDep === void 0 ? void 0 : usedDep.references) ||
                        ((_a = usedDep === null || usedDep === void 0 ? void 0 : usedDep.references[0]) === null || _a === void 0 ? void 0 : _a.resolved) !== topScopeRef) {
                        return;
                    }
                    const def = topScopeRef === null || topScopeRef === void 0 ? void 0 : topScopeRef.defs[0];
                    if (def == null || def.name == null || def.type !== 'Parameter') {
                        return;
                    }
                    let isFunctionCall = false;
                    let id;
                    for (const reference of usedDep.references) {
                        id = reference.identifier;
                        if (id != null &&
                            id.parent != null &&
                            (id.parent.type === 'CallExpression' ||
                                id.parent.type === 'OptionalCallExpression') &&
                            id.parent.callee === id) {
                            isFunctionCall = true;
                            break;
                        }
                    }
                    if (!isFunctionCall) {
                        return;
                    }
                    missingCallbackDep = missingDep;
                });
                if (missingCallbackDep !== null) {
                    extraWarning =
                        ` If '${missingCallbackDep}' changes too often, ` +
                            `find the parent component that defines it ` +
                            `and wrap that definition in useCallback.`;
                }
            }
            if (!extraWarning && missingDependencies.size > 0) {
                let setStateRecommendation = null;
                for (const missingDep of missingDependencies) {
                    if (setStateRecommendation !== null) {
                        break;
                    }
                    const usedDep = dependencies.get(missingDep);
                    const references = usedDep.references;
                    let id;
                    let maybeCall;
                    for (const reference of references) {
                        id = reference.identifier;
                        maybeCall = id.parent;
                        while (maybeCall != null && maybeCall !== componentScope.block) {
                            if (maybeCall.type === 'CallExpression') {
                                const correspondingStateVariable = setStateCallSites.get(maybeCall.callee);
                                if (correspondingStateVariable != null) {
                                    if ('name' in correspondingStateVariable &&
                                        correspondingStateVariable.name === missingDep) {
                                        setStateRecommendation = {
                                            missingDep,
                                            setter: 'name' in maybeCall.callee ? maybeCall.callee.name : '',
                                            form: 'updater',
                                        };
                                    }
                                    else if (stateVariables.has(id)) {
                                        setStateRecommendation = {
                                            missingDep,
                                            setter: 'name' in maybeCall.callee ? maybeCall.callee.name : '',
                                            form: 'reducer',
                                        };
                                    }
                                    else {
                                        const resolved = reference.resolved;
                                        if (resolved != null) {
                                            const def = resolved.defs[0];
                                            if (def != null && def.type === 'Parameter') {
                                                setStateRecommendation = {
                                                    missingDep,
                                                    setter: 'name' in maybeCall.callee
                                                        ? maybeCall.callee.name
                                                        : '',
                                                    form: 'inlineReducer',
                                                };
                                            }
                                        }
                                    }
                                    break;
                                }
                            }
                            maybeCall = maybeCall.parent;
                        }
                        if (setStateRecommendation !== null) {
                            break;
                        }
                    }
                }
                if (setStateRecommendation !== null) {
                    switch (setStateRecommendation.form) {
                        case 'reducer':
                            extraWarning =
                                ` You can also replace multiple useState variables with useReducer ` +
                                    `if '${setStateRecommendation.setter}' needs the ` +
                                    `current value of '${setStateRecommendation.missingDep}'.`;
                            break;
                        case 'inlineReducer':
                            extraWarning =
                                ` If '${setStateRecommendation.setter}' needs the ` +
                                    `current value of '${setStateRecommendation.missingDep}', ` +
                                    `you can also switch to useReducer instead of useState and ` +
                                    `read '${setStateRecommendation.missingDep}' in the reducer.`;
                            break;
                        case 'updater':
                            extraWarning =
                                ` You can also do a functional update '${setStateRecommendation.setter}(${setStateRecommendation.missingDep.slice(0, 1)} => ...)' if you only need '${setStateRecommendation.missingDep}'` + ` in the '${setStateRecommendation.setter}' call.`;
                            break;
                        default:
                            throw new Error('Unknown case.');
                    }
                }
            }
            reportProblem({
                node: declaredDependenciesNode,
                message: `React Hook ${getSourceCode().getText(reactiveHook)} has ` +
                    (getWarningMessage(missingDependencies, 'a', 'missing', 'include') ||
                        getWarningMessage(unnecessaryDependencies, 'an', 'unnecessary', 'exclude') ||
                        getWarningMessage(duplicateDependencies, 'a', 'duplicate', 'omit')) +
                    extraWarning,
                suggest: [
                    {
                        desc: `Update the dependencies array to be: [${suggestedDeps
                            .map(formatDependency)
                            .join(', ')}]`,
                        fix(fixer) {
                            return fixer.replaceText(declaredDependenciesNode, `[${suggestedDeps.map(formatDependency).join(', ')}]`);
                        },
                    },
                ],
            });
        }
        function visitCallExpression(node) {
            const callbackIndex = getReactiveHookCallbackIndex(node.callee, options);
            if (callbackIndex === -1) {
                return;
            }
            let callback = node.arguments[callbackIndex];
            const reactiveHook = node.callee;
            const nodeWithoutNamespace = getNodeWithoutReactNamespace$1(reactiveHook);
            const reactiveHookName = 'name' in nodeWithoutNamespace ? nodeWithoutNamespace.name : '';
            const maybeNode = node.arguments[callbackIndex + 1];
            const declaredDependenciesNode = maybeNode &&
                !(maybeNode.type === 'Identifier' && maybeNode.name === 'undefined')
                ? maybeNode
                : undefined;
            const isEffect = /Effect($|[^a-z])/g.test(reactiveHookName);
            if (!callback) {
                reportProblem({
                    node: reactiveHook,
                    message: `React Hook ${reactiveHookName} requires an effect callback. ` +
                        `Did you forget to pass a callback to the hook?`,
                });
                return;
            }
            if (!maybeNode && isEffect && options.requireExplicitEffectDeps) {
                reportProblem({
                    node: reactiveHook,
                    message: `React Hook ${reactiveHookName} always requires dependencies. ` +
                        `Please add a dependency array or an explicit \`undefined\``,
                });
            }
            const isAutoDepsHook = options.experimental_autoDependenciesHooks.includes(reactiveHookName);
            if ((!declaredDependenciesNode ||
                (isAutoDepsHook &&
                    declaredDependenciesNode.type === 'Literal' &&
                    declaredDependenciesNode.value === null)) &&
                !isEffect) {
                if (reactiveHookName === 'useMemo' ||
                    reactiveHookName === 'useCallback') {
                    reportProblem({
                        node: reactiveHook,
                        message: `React Hook ${reactiveHookName} does nothing when called with ` +
                            `only one argument. Did you forget to pass an array of ` +
                            `dependencies?`,
                    });
                }
                return;
            }
            while (callback.type === 'TSAsExpression' ||
                callback.type === 'AsExpression') {
                callback = callback.expression;
            }
            switch (callback.type) {
                case 'FunctionExpression':
                case 'ArrowFunctionExpression':
                    visitFunctionWithDependencies(callback, declaredDependenciesNode, reactiveHook, reactiveHookName, isEffect, isAutoDepsHook);
                    return;
                case 'Identifier':
                    if (!declaredDependenciesNode ||
                        (isAutoDepsHook &&
                            declaredDependenciesNode.type === 'Literal' &&
                            declaredDependenciesNode.value === null)) {
                        return;
                    }
                    if ('elements' in declaredDependenciesNode &&
                        declaredDependenciesNode.elements &&
                        declaredDependenciesNode.elements.some(el => el && el.type === 'Identifier' && el.name === callback.name)) {
                        return;
                    }
                    const variable = getScope(callback).set.get(callback.name);
                    if (variable == null || variable.defs == null) {
                        return;
                    }
                    const def = variable.defs[0];
                    if (!def || !def.node) {
                        break;
                    }
                    if (def.type === 'Parameter') {
                        reportProblem({
                            node: reactiveHook,
                            message: getUnknownDependenciesMessage(reactiveHookName),
                        });
                        return;
                    }
                    if (def.type !== 'Variable' && def.type !== 'FunctionName') {
                        break;
                    }
                    switch (def.node.type) {
                        case 'FunctionDeclaration':
                            visitFunctionWithDependencies(def.node, declaredDependenciesNode, reactiveHook, reactiveHookName, isEffect, isAutoDepsHook);
                            return;
                        case 'VariableDeclarator':
                            const init = def.node.init;
                            if (!init) {
                                break;
                            }
                            switch (init.type) {
                                case 'ArrowFunctionExpression':
                                case 'FunctionExpression':
                                    visitFunctionWithDependencies(init, declaredDependenciesNode, reactiveHook, reactiveHookName, isEffect, isAutoDepsHook);
                                    return;
                            }
                            break;
                    }
                    break;
                default:
                    reportProblem({
                        node: reactiveHook,
                        message: getUnknownDependenciesMessage(reactiveHookName),
                    });
                    return;
            }
            reportProblem({
                node: reactiveHook,
                message: `React Hook ${reactiveHookName} has a missing dependency: '${callback.name}'. ` +
                    `Either include it or remove the dependency array.`,
                suggest: [
                    {
                        desc: `Update the dependencies array to be: [${callback.name}]`,
                        fix(fixer) {
                            return fixer.replaceText(declaredDependenciesNode, `[${callback.name}]`);
                        },
                    },
                ],
            });
        }
        return {
            CallExpression: visitCallExpression,
        };
    },
};
function collectRecommendations({ dependencies, declaredDependencies, stableDependencies, externalDependencies, isEffect, }) {
    const depTree = createDepTree();
    function createDepTree() {
        return {
            isUsed: false,
            isSatisfiedRecursively: false,
            isSubtreeUsed: false,
            children: new Map(),
        };
    }
    dependencies.forEach((_, key) => {
        const node = getOrCreateNodeByPath(depTree, key);
        node.isUsed = true;
        markAllParentsByPath(depTree, key, parent => {
            parent.isSubtreeUsed = true;
        });
    });
    declaredDependencies.forEach(({ key }) => {
        const node = getOrCreateNodeByPath(depTree, key);
        node.isSatisfiedRecursively = true;
    });
    stableDependencies.forEach(key => {
        const node = getOrCreateNodeByPath(depTree, key);
        node.isSatisfiedRecursively = true;
    });
    function getOrCreateNodeByPath(rootNode, path) {
        const keys = path.split('.');
        let node = rootNode;
        for (const key of keys) {
            let child = node.children.get(key);
            if (!child) {
                child = createDepTree();
                node.children.set(key, child);
            }
            node = child;
        }
        return node;
    }
    function markAllParentsByPath(rootNode, path, fn) {
        const keys = path.split('.');
        let node = rootNode;
        for (const key of keys) {
            const child = node.children.get(key);
            if (!child) {
                return;
            }
            fn(child);
            node = child;
        }
    }
    const missingDependencies = new Set();
    const satisfyingDependencies = new Set();
    scanTreeRecursively(depTree, missingDependencies, satisfyingDependencies, key => key);
    function scanTreeRecursively(node, missingPaths, satisfyingPaths, keyToPath) {
        node.children.forEach((child, key) => {
            const path = keyToPath(key);
            if (child.isSatisfiedRecursively) {
                if (child.isSubtreeUsed) {
                    satisfyingPaths.add(path);
                }
                return;
            }
            if (child.isUsed) {
                missingPaths.add(path);
                return;
            }
            scanTreeRecursively(child, missingPaths, satisfyingPaths, childKey => path + '.' + childKey);
        });
    }
    const suggestedDependencies = [];
    const unnecessaryDependencies = new Set();
    const duplicateDependencies = new Set();
    declaredDependencies.forEach(({ key }) => {
        if (satisfyingDependencies.has(key)) {
            if (suggestedDependencies.indexOf(key) === -1) {
                suggestedDependencies.push(key);
            }
            else {
                duplicateDependencies.add(key);
            }
        }
        else {
            if (isEffect &&
                !key.endsWith('.current') &&
                !externalDependencies.has(key)) {
                if (suggestedDependencies.indexOf(key) === -1) {
                    suggestedDependencies.push(key);
                }
            }
            else {
                unnecessaryDependencies.add(key);
            }
        }
    });
    missingDependencies.forEach(key => {
        suggestedDependencies.push(key);
    });
    return {
        suggestedDependencies,
        unnecessaryDependencies,
        duplicateDependencies,
        missingDependencies,
    };
}
function getConstructionExpressionType(node) {
    switch (node.type) {
        case 'ObjectExpression':
            return 'object';
        case 'ArrayExpression':
            return 'array';
        case 'ArrowFunctionExpression':
        case 'FunctionExpression':
            return 'function';
        case 'ClassExpression':
            return 'class';
        case 'ConditionalExpression':
            if (getConstructionExpressionType(node.consequent) != null ||
                getConstructionExpressionType(node.alternate) != null) {
                return 'conditional';
            }
            return null;
        case 'LogicalExpression':
            if (getConstructionExpressionType(node.left) != null ||
                getConstructionExpressionType(node.right) != null) {
                return 'logical expression';
            }
            return null;
        case 'JSXFragment':
            return 'JSX fragment';
        case 'JSXElement':
            return 'JSX element';
        case 'AssignmentExpression':
            if (getConstructionExpressionType(node.right) != null) {
                return 'assignment expression';
            }
            return null;
        case 'NewExpression':
            return 'object construction';
        case 'Literal':
            if (node.value instanceof RegExp) {
                return 'regular expression';
            }
            return null;
        case 'TypeCastExpression':
        case 'AsExpression':
        case 'TSAsExpression':
            return getConstructionExpressionType(node.expression);
    }
    return null;
}
function scanForConstructions({ declaredDependencies, declaredDependenciesNode, componentScope, scope, }) {
    const constructions = declaredDependencies
        .map(({ key }) => {
        const ref = componentScope.variables.find(v => v.name === key);
        if (ref == null) {
            return null;
        }
        const node = ref.defs[0];
        if (node == null) {
            return null;
        }
        if (node.type === 'Variable' &&
            node.node.type === 'VariableDeclarator' &&
            node.node.id.type === 'Identifier' &&
            node.node.init != null) {
            const constantExpressionType = getConstructionExpressionType(node.node.init);
            if (constantExpressionType) {
                return [ref, constantExpressionType];
            }
        }
        if (node.type === 'FunctionName' &&
            node.node.type === 'FunctionDeclaration') {
            return [ref, 'function'];
        }
        if (node.type === 'ClassName' && node.node.type === 'ClassDeclaration') {
            return [ref, 'class'];
        }
        return null;
    })
        .filter(Boolean);
    function isUsedOutsideOfHook(ref) {
        let foundWriteExpr = false;
        for (const reference of ref.references) {
            if (reference.writeExpr) {
                if (foundWriteExpr) {
                    return true;
                }
                else {
                    foundWriteExpr = true;
                    continue;
                }
            }
            let currentScope = reference.from;
            while (currentScope !== scope && currentScope != null) {
                currentScope = currentScope.upper;
            }
            if (currentScope !== scope) {
                if (!isAncestorNodeOf(declaredDependenciesNode, reference.identifier)) {
                    return true;
                }
            }
        }
        return false;
    }
    return constructions.map(([ref, depType]) => ({
        construction: ref.defs[0],
        depType,
        isUsedOutsideOfHook: isUsedOutsideOfHook(ref),
    }));
}
function getDependency(node) {
    if (node.parent &&
        (node.parent.type === 'MemberExpression' ||
            node.parent.type === 'OptionalMemberExpression') &&
        node.parent.object === node &&
        'name' in node.parent.property &&
        node.parent.property.name !== 'current' &&
        !node.parent.computed &&
        !(node.parent.parent != null &&
            (node.parent.parent.type === 'CallExpression' ||
                node.parent.parent.type === 'OptionalCallExpression') &&
            node.parent.parent.callee === node.parent)) {
        return getDependency(node.parent);
    }
    else if (node.type === 'MemberExpression' &&
        node.parent &&
        node.parent.type === 'AssignmentExpression' &&
        node.parent.left === node) {
        return node.object;
    }
    else {
        return node;
    }
}
function markNode(node, optionalChains, result) {
    if (optionalChains) {
        if ('optional' in node && node.optional) {
            if (!optionalChains.has(result)) {
                optionalChains.set(result, true);
            }
        }
        else {
            optionalChains.set(result, false);
        }
    }
}
function analyzePropertyChain(node, optionalChains) {
    if (node.type === 'Identifier' || node.type === 'JSXIdentifier') {
        const result = node.name;
        if (optionalChains) {
            optionalChains.set(result, false);
        }
        return result;
    }
    else if (node.type === 'MemberExpression' && !node.computed) {
        const object = analyzePropertyChain(node.object, optionalChains);
        const property = analyzePropertyChain(node.property, null);
        const result = `${object}.${property}`;
        markNode(node, optionalChains, result);
        return result;
    }
    else if (node.type === 'OptionalMemberExpression' && !node.computed) {
        const object = analyzePropertyChain(node.object, optionalChains);
        const property = analyzePropertyChain(node.property, null);
        const result = `${object}.${property}`;
        markNode(node, optionalChains, result);
        return result;
    }
    else if (node.type === 'ChainExpression' &&
        (!('computed' in node) || !node.computed)) {
        const expression = node.expression;
        if (expression.type === 'CallExpression') {
            throw new Error(`Unsupported node type: ${expression.type}`);
        }
        const object = analyzePropertyChain(expression.object, optionalChains);
        const property = analyzePropertyChain(expression.property, null);
        const result = `${object}.${property}`;
        markNode(expression, optionalChains, result);
        return result;
    }
    else {
        throw new Error(`Unsupported node type: ${node.type}`);
    }
}
function getNodeWithoutReactNamespace$1(node) {
    if (node.type === 'MemberExpression' &&
        node.object.type === 'Identifier' &&
        node.object.name === 'React' &&
        node.property.type === 'Identifier' &&
        !node.computed) {
        return node.property;
    }
    return node;
}
function getReactiveHookCallbackIndex(calleeNode, options) {
    const node = getNodeWithoutReactNamespace$1(calleeNode);
    if (node.type !== 'Identifier') {
        return -1;
    }
    switch (node.name) {
        case 'useEffect':
        case 'useLayoutEffect':
        case 'useCallback':
        case 'useMemo':
            return 0;
        case 'useImperativeHandle':
            return 1;
        default:
            if (node === calleeNode && options && options.additionalHooks) {
                let name;
                try {
                    name = analyzePropertyChain(node, null);
                }
                catch (error) {
                    if (error instanceof Error &&
                        /Unsupported node type/.test(error.message)) {
                        return 0;
                    }
                    else {
                        throw error;
                    }
                }
                return options.additionalHooks.test(name) ? 0 : -1;
            }
            else {
                return -1;
            }
    }
}
function fastFindReferenceWithParent(start, target) {
    const queue = [start];
    let item;
    while (queue.length) {
        item = queue.shift();
        if (isSameIdentifier(item, target)) {
            return item;
        }
        if (!isAncestorNodeOf(item, target)) {
            continue;
        }
        for (const [key, value] of Object.entries(item)) {
            if (key === 'parent') {
                continue;
            }
            if (isNodeLike(value)) {
                value.parent = item;
                queue.push(value);
            }
            else if (Array.isArray(value)) {
                value.forEach(val => {
                    if (isNodeLike(val)) {
                        val.parent = item;
                        queue.push(val);
                    }
                });
            }
        }
    }
    return null;
}
function joinEnglish(arr) {
    let s = '';
    for (let i = 0; i < arr.length; i++) {
        s += arr[i];
        if (i === 0 && arr.length === 2) {
            s += ' and ';
        }
        else if (i === arr.length - 2 && arr.length > 2) {
            s += ', and ';
        }
        else if (i < arr.length - 1) {
            s += ', ';
        }
    }
    return s;
}
function isNodeLike(val) {
    return (typeof val === 'object' &&
        val !== null &&
        !Array.isArray(val) &&
        'type' in val &&
        typeof val.type === 'string');
}
function isSameIdentifier(a, b) {
    return ((a.type === 'Identifier' || a.type === 'JSXIdentifier') &&
        a.type === b.type &&
        a.name === b.name &&
        !!a.range &&
        !!b.range &&
        a.range[0] === b.range[0] &&
        a.range[1] === b.range[1]);
}
function isAncestorNodeOf(a, b) {
    return (!!a.range &&
        !!b.range &&
        a.range[0] <= b.range[0] &&
        a.range[1] >= b.range[1]);
}
function isUseEffectEventIdentifier$1(node) {
    return node.type === 'Identifier' && node.name === 'useEffectEvent';
}
function getUnknownDependenciesMessage(reactiveHookName) {
    return (`React Hook ${reactiveHookName} received a function whose dependencies ` +
        `are unknown. Pass an inline function instead.`);
}

var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

function getDefaultExportFromCjs (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
}

var invariant_1;
var hasRequiredInvariant;

function requireInvariant () {
	if (hasRequiredInvariant) return invariant_1;
	hasRequiredInvariant = 1;

	var invariant = function(condition, format, a, b, c, d, e, f) {

	  if (!condition) {
	    var error;
	    if (format === undefined) {
	      error = new Error(
	        'Minified exception occurred; use the non-minified dev environment ' +
	        'for the full error message and additional helpful warnings.'
	      );
	    } else {
	      var args = [a, b, c, d, e, f];
	      var argIndex = 0;
	      error = new Error(
	        format.replace(/%s/g, function() { return args[argIndex++]; })
	      );
	      error.name = 'Invariant Violation';
	    }

	    error.framesToPop = 1; // we don't care about invariant's own frame
	    throw error;
	  }
	};

	invariant_1 = invariant;
	return invariant_1;
}

var invariantExports = requireInvariant();
var invariant = /*@__PURE__*/getDefaultExportFromCjs(invariantExports);

var lib$3 = {};

var isReactComponent = {};

var buildMatchMemberExpression = {};

var matchesPattern = {};

var generated$3 = {};

var shallowEqual = {};

var hasRequiredShallowEqual;

function requireShallowEqual () {
	if (hasRequiredShallowEqual) return shallowEqual;
	hasRequiredShallowEqual = 1;

	Object.defineProperty(shallowEqual, "__esModule", {
	  value: true
	});
	shallowEqual.default = shallowEqual$1;
	function shallowEqual$1(actual, expected) {
	  const keys = Object.keys(expected);
	  for (const key of keys) {
	    if (actual[key] !== expected[key]) {
	      return false;
	    }
	  }
	  return true;
	}

	
	return shallowEqual;
}

var deprecationWarning = {};

var hasRequiredDeprecationWarning;

function requireDeprecationWarning () {
	if (hasRequiredDeprecationWarning) return deprecationWarning;
	hasRequiredDeprecationWarning = 1;

	Object.defineProperty(deprecationWarning, "__esModule", {
	  value: true
	});
	deprecationWarning.default = deprecationWarning$1;
	const warnings = new Set();
	function deprecationWarning$1(oldName, newName, prefix = "") {
	  if (warnings.has(oldName)) return;
	  warnings.add(oldName);
	  const {
	    internal,
	    trace
	  } = captureShortStackTrace(1, 2);
	  if (internal) {
	    return;
	  }
	  console.warn(`${prefix}\`${oldName}\` has been deprecated, please migrate to \`${newName}\`\n${trace}`);
	}
	function captureShortStackTrace(skip, length) {
	  const {
	    stackTraceLimit,
	    prepareStackTrace
	  } = Error;
	  let stackTrace;
	  Error.stackTraceLimit = 1 + skip + length;
	  Error.prepareStackTrace = function (err, stack) {
	    stackTrace = stack;
	  };
	  Error.stackTraceLimit = stackTraceLimit;
	  Error.prepareStackTrace = prepareStackTrace;
	  if (!stackTrace) return {
	    internal: false,
	    trace: ""
	  };
	  const shortStackTrace = stackTrace.slice(1 + skip, 1 + skip + length);
	  return {
	    internal: /[\\/]@babel[\\/]/.test(shortStackTrace[1].getFileName()),
	    trace: shortStackTrace.map(frame => `    at ${frame}`).join("\n")
	  };
	}

	
	return deprecationWarning;
}

var hasRequiredGenerated$3;

function requireGenerated$3 () {
	if (hasRequiredGenerated$3) return generated$3;
	hasRequiredGenerated$3 = 1;

	Object.defineProperty(generated$3, "__esModule", {
	  value: true
	});
	generated$3.isAccessor = isAccessor;
	generated$3.isAnyTypeAnnotation = isAnyTypeAnnotation;
	generated$3.isArgumentPlaceholder = isArgumentPlaceholder;
	generated$3.isArrayExpression = isArrayExpression;
	generated$3.isArrayPattern = isArrayPattern;
	generated$3.isArrayTypeAnnotation = isArrayTypeAnnotation;
	generated$3.isArrowFunctionExpression = isArrowFunctionExpression;
	generated$3.isAssignmentExpression = isAssignmentExpression;
	generated$3.isAssignmentPattern = isAssignmentPattern;
	generated$3.isAwaitExpression = isAwaitExpression;
	generated$3.isBigIntLiteral = isBigIntLiteral;
	generated$3.isBinary = isBinary;
	generated$3.isBinaryExpression = isBinaryExpression;
	generated$3.isBindExpression = isBindExpression;
	generated$3.isBlock = isBlock;
	generated$3.isBlockParent = isBlockParent;
	generated$3.isBlockStatement = isBlockStatement;
	generated$3.isBooleanLiteral = isBooleanLiteral;
	generated$3.isBooleanLiteralTypeAnnotation = isBooleanLiteralTypeAnnotation;
	generated$3.isBooleanTypeAnnotation = isBooleanTypeAnnotation;
	generated$3.isBreakStatement = isBreakStatement;
	generated$3.isCallExpression = isCallExpression;
	generated$3.isCatchClause = isCatchClause;
	generated$3.isClass = isClass;
	generated$3.isClassAccessorProperty = isClassAccessorProperty;
	generated$3.isClassBody = isClassBody;
	generated$3.isClassDeclaration = isClassDeclaration;
	generated$3.isClassExpression = isClassExpression;
	generated$3.isClassImplements = isClassImplements;
	generated$3.isClassMethod = isClassMethod;
	generated$3.isClassPrivateMethod = isClassPrivateMethod;
	generated$3.isClassPrivateProperty = isClassPrivateProperty;
	generated$3.isClassProperty = isClassProperty;
	generated$3.isCompletionStatement = isCompletionStatement;
	generated$3.isConditional = isConditional;
	generated$3.isConditionalExpression = isConditionalExpression;
	generated$3.isContinueStatement = isContinueStatement;
	generated$3.isDebuggerStatement = isDebuggerStatement;
	generated$3.isDecimalLiteral = isDecimalLiteral;
	generated$3.isDeclaration = isDeclaration;
	generated$3.isDeclareClass = isDeclareClass;
	generated$3.isDeclareExportAllDeclaration = isDeclareExportAllDeclaration;
	generated$3.isDeclareExportDeclaration = isDeclareExportDeclaration;
	generated$3.isDeclareFunction = isDeclareFunction;
	generated$3.isDeclareInterface = isDeclareInterface;
	generated$3.isDeclareModule = isDeclareModule;
	generated$3.isDeclareModuleExports = isDeclareModuleExports;
	generated$3.isDeclareOpaqueType = isDeclareOpaqueType;
	generated$3.isDeclareTypeAlias = isDeclareTypeAlias;
	generated$3.isDeclareVariable = isDeclareVariable;
	generated$3.isDeclaredPredicate = isDeclaredPredicate;
	generated$3.isDecorator = isDecorator;
	generated$3.isDirective = isDirective;
	generated$3.isDirectiveLiteral = isDirectiveLiteral;
	generated$3.isDoExpression = isDoExpression;
	generated$3.isDoWhileStatement = isDoWhileStatement;
	generated$3.isEmptyStatement = isEmptyStatement;
	generated$3.isEmptyTypeAnnotation = isEmptyTypeAnnotation;
	generated$3.isEnumBody = isEnumBody;
	generated$3.isEnumBooleanBody = isEnumBooleanBody;
	generated$3.isEnumBooleanMember = isEnumBooleanMember;
	generated$3.isEnumDeclaration = isEnumDeclaration;
	generated$3.isEnumDefaultedMember = isEnumDefaultedMember;
	generated$3.isEnumMember = isEnumMember;
	generated$3.isEnumNumberBody = isEnumNumberBody;
	generated$3.isEnumNumberMember = isEnumNumberMember;
	generated$3.isEnumStringBody = isEnumStringBody;
	generated$3.isEnumStringMember = isEnumStringMember;
	generated$3.isEnumSymbolBody = isEnumSymbolBody;
	generated$3.isExistsTypeAnnotation = isExistsTypeAnnotation;
	generated$3.isExportAllDeclaration = isExportAllDeclaration;
	generated$3.isExportDeclaration = isExportDeclaration;
	generated$3.isExportDefaultDeclaration = isExportDefaultDeclaration;
	generated$3.isExportDefaultSpecifier = isExportDefaultSpecifier;
	generated$3.isExportNamedDeclaration = isExportNamedDeclaration;
	generated$3.isExportNamespaceSpecifier = isExportNamespaceSpecifier;
	generated$3.isExportSpecifier = isExportSpecifier;
	generated$3.isExpression = isExpression;
	generated$3.isExpressionStatement = isExpressionStatement;
	generated$3.isExpressionWrapper = isExpressionWrapper;
	generated$3.isFile = isFile;
	generated$3.isFlow = isFlow;
	generated$3.isFlowBaseAnnotation = isFlowBaseAnnotation;
	generated$3.isFlowDeclaration = isFlowDeclaration;
	generated$3.isFlowPredicate = isFlowPredicate;
	generated$3.isFlowType = isFlowType;
	generated$3.isFor = isFor;
	generated$3.isForInStatement = isForInStatement;
	generated$3.isForOfStatement = isForOfStatement;
	generated$3.isForStatement = isForStatement;
	generated$3.isForXStatement = isForXStatement;
	generated$3.isFunction = isFunction;
	generated$3.isFunctionDeclaration = isFunctionDeclaration;
	generated$3.isFunctionExpression = isFunctionExpression;
	generated$3.isFunctionParent = isFunctionParent;
	generated$3.isFunctionTypeAnnotation = isFunctionTypeAnnotation;
	generated$3.isFunctionTypeParam = isFunctionTypeParam;
	generated$3.isGenericTypeAnnotation = isGenericTypeAnnotation;
	generated$3.isIdentifier = isIdentifier;
	generated$3.isIfStatement = isIfStatement;
	generated$3.isImmutable = isImmutable;
	generated$3.isImport = isImport;
	generated$3.isImportAttribute = isImportAttribute;
	generated$3.isImportDeclaration = isImportDeclaration;
	generated$3.isImportDefaultSpecifier = isImportDefaultSpecifier;
	generated$3.isImportExpression = isImportExpression;
	generated$3.isImportNamespaceSpecifier = isImportNamespaceSpecifier;
	generated$3.isImportOrExportDeclaration = isImportOrExportDeclaration;
	generated$3.isImportSpecifier = isImportSpecifier;
	generated$3.isIndexedAccessType = isIndexedAccessType;
	generated$3.isInferredPredicate = isInferredPredicate;
	generated$3.isInterfaceDeclaration = isInterfaceDeclaration;
	generated$3.isInterfaceExtends = isInterfaceExtends;
	generated$3.isInterfaceTypeAnnotation = isInterfaceTypeAnnotation;
	generated$3.isInterpreterDirective = isInterpreterDirective;
	generated$3.isIntersectionTypeAnnotation = isIntersectionTypeAnnotation;
	generated$3.isJSX = isJSX;
	generated$3.isJSXAttribute = isJSXAttribute;
	generated$3.isJSXClosingElement = isJSXClosingElement;
	generated$3.isJSXClosingFragment = isJSXClosingFragment;
	generated$3.isJSXElement = isJSXElement;
	generated$3.isJSXEmptyExpression = isJSXEmptyExpression;
	generated$3.isJSXExpressionContainer = isJSXExpressionContainer;
	generated$3.isJSXFragment = isJSXFragment;
	generated$3.isJSXIdentifier = isJSXIdentifier;
	generated$3.isJSXMemberExpression = isJSXMemberExpression;
	generated$3.isJSXNamespacedName = isJSXNamespacedName;
	generated$3.isJSXOpeningElement = isJSXOpeningElement;
	generated$3.isJSXOpeningFragment = isJSXOpeningFragment;
	generated$3.isJSXSpreadAttribute = isJSXSpreadAttribute;
	generated$3.isJSXSpreadChild = isJSXSpreadChild;
	generated$3.isJSXText = isJSXText;
	generated$3.isLVal = isLVal;
	generated$3.isLabeledStatement = isLabeledStatement;
	generated$3.isLiteral = isLiteral;
	generated$3.isLogicalExpression = isLogicalExpression;
	generated$3.isLoop = isLoop;
	generated$3.isMemberExpression = isMemberExpression;
	generated$3.isMetaProperty = isMetaProperty;
	generated$3.isMethod = isMethod;
	generated$3.isMiscellaneous = isMiscellaneous;
	generated$3.isMixedTypeAnnotation = isMixedTypeAnnotation;
	generated$3.isModuleDeclaration = isModuleDeclaration;
	generated$3.isModuleExpression = isModuleExpression;
	generated$3.isModuleSpecifier = isModuleSpecifier;
	generated$3.isNewExpression = isNewExpression;
	generated$3.isNoop = isNoop;
	generated$3.isNullLiteral = isNullLiteral;
	generated$3.isNullLiteralTypeAnnotation = isNullLiteralTypeAnnotation;
	generated$3.isNullableTypeAnnotation = isNullableTypeAnnotation;
	generated$3.isNumberLiteral = isNumberLiteral;
	generated$3.isNumberLiteralTypeAnnotation = isNumberLiteralTypeAnnotation;
	generated$3.isNumberTypeAnnotation = isNumberTypeAnnotation;
	generated$3.isNumericLiteral = isNumericLiteral;
	generated$3.isObjectExpression = isObjectExpression;
	generated$3.isObjectMember = isObjectMember;
	generated$3.isObjectMethod = isObjectMethod;
	generated$3.isObjectPattern = isObjectPattern;
	generated$3.isObjectProperty = isObjectProperty;
	generated$3.isObjectTypeAnnotation = isObjectTypeAnnotation;
	generated$3.isObjectTypeCallProperty = isObjectTypeCallProperty;
	generated$3.isObjectTypeIndexer = isObjectTypeIndexer;
	generated$3.isObjectTypeInternalSlot = isObjectTypeInternalSlot;
	generated$3.isObjectTypeProperty = isObjectTypeProperty;
	generated$3.isObjectTypeSpreadProperty = isObjectTypeSpreadProperty;
	generated$3.isOpaqueType = isOpaqueType;
	generated$3.isOptionalCallExpression = isOptionalCallExpression;
	generated$3.isOptionalIndexedAccessType = isOptionalIndexedAccessType;
	generated$3.isOptionalMemberExpression = isOptionalMemberExpression;
	generated$3.isParenthesizedExpression = isParenthesizedExpression;
	generated$3.isPattern = isPattern;
	generated$3.isPatternLike = isPatternLike;
	generated$3.isPipelineBareFunction = isPipelineBareFunction;
	generated$3.isPipelinePrimaryTopicReference = isPipelinePrimaryTopicReference;
	generated$3.isPipelineTopicExpression = isPipelineTopicExpression;
	generated$3.isPlaceholder = isPlaceholder;
	generated$3.isPrivate = isPrivate;
	generated$3.isPrivateName = isPrivateName;
	generated$3.isProgram = isProgram;
	generated$3.isProperty = isProperty;
	generated$3.isPureish = isPureish;
	generated$3.isQualifiedTypeIdentifier = isQualifiedTypeIdentifier;
	generated$3.isRecordExpression = isRecordExpression;
	generated$3.isRegExpLiteral = isRegExpLiteral;
	generated$3.isRegexLiteral = isRegexLiteral;
	generated$3.isRestElement = isRestElement;
	generated$3.isRestProperty = isRestProperty;
	generated$3.isReturnStatement = isReturnStatement;
	generated$3.isScopable = isScopable;
	generated$3.isSequenceExpression = isSequenceExpression;
	generated$3.isSpreadElement = isSpreadElement;
	generated$3.isSpreadProperty = isSpreadProperty;
	generated$3.isStandardized = isStandardized;
	generated$3.isStatement = isStatement;
	generated$3.isStaticBlock = isStaticBlock;
	generated$3.isStringLiteral = isStringLiteral;
	generated$3.isStringLiteralTypeAnnotation = isStringLiteralTypeAnnotation;
	generated$3.isStringTypeAnnotation = isStringTypeAnnotation;
	generated$3.isSuper = isSuper;
	generated$3.isSwitchCase = isSwitchCase;
	generated$3.isSwitchStatement = isSwitchStatement;
	generated$3.isSymbolTypeAnnotation = isSymbolTypeAnnotation;
	generated$3.isTSAnyKeyword = isTSAnyKeyword;
	generated$3.isTSArrayType = isTSArrayType;
	generated$3.isTSAsExpression = isTSAsExpression;
	generated$3.isTSBaseType = isTSBaseType;
	generated$3.isTSBigIntKeyword = isTSBigIntKeyword;
	generated$3.isTSBooleanKeyword = isTSBooleanKeyword;
	generated$3.isTSCallSignatureDeclaration = isTSCallSignatureDeclaration;
	generated$3.isTSConditionalType = isTSConditionalType;
	generated$3.isTSConstructSignatureDeclaration = isTSConstructSignatureDeclaration;
	generated$3.isTSConstructorType = isTSConstructorType;
	generated$3.isTSDeclareFunction = isTSDeclareFunction;
	generated$3.isTSDeclareMethod = isTSDeclareMethod;
	generated$3.isTSEntityName = isTSEntityName;
	generated$3.isTSEnumDeclaration = isTSEnumDeclaration;
	generated$3.isTSEnumMember = isTSEnumMember;
	generated$3.isTSExportAssignment = isTSExportAssignment;
	generated$3.isTSExpressionWithTypeArguments = isTSExpressionWithTypeArguments;
	generated$3.isTSExternalModuleReference = isTSExternalModuleReference;
	generated$3.isTSFunctionType = isTSFunctionType;
	generated$3.isTSImportEqualsDeclaration = isTSImportEqualsDeclaration;
	generated$3.isTSImportType = isTSImportType;
	generated$3.isTSIndexSignature = isTSIndexSignature;
	generated$3.isTSIndexedAccessType = isTSIndexedAccessType;
	generated$3.isTSInferType = isTSInferType;
	generated$3.isTSInstantiationExpression = isTSInstantiationExpression;
	generated$3.isTSInterfaceBody = isTSInterfaceBody;
	generated$3.isTSInterfaceDeclaration = isTSInterfaceDeclaration;
	generated$3.isTSIntersectionType = isTSIntersectionType;
	generated$3.isTSIntrinsicKeyword = isTSIntrinsicKeyword;
	generated$3.isTSLiteralType = isTSLiteralType;
	generated$3.isTSMappedType = isTSMappedType;
	generated$3.isTSMethodSignature = isTSMethodSignature;
	generated$3.isTSModuleBlock = isTSModuleBlock;
	generated$3.isTSModuleDeclaration = isTSModuleDeclaration;
	generated$3.isTSNamedTupleMember = isTSNamedTupleMember;
	generated$3.isTSNamespaceExportDeclaration = isTSNamespaceExportDeclaration;
	generated$3.isTSNeverKeyword = isTSNeverKeyword;
	generated$3.isTSNonNullExpression = isTSNonNullExpression;
	generated$3.isTSNullKeyword = isTSNullKeyword;
	generated$3.isTSNumberKeyword = isTSNumberKeyword;
	generated$3.isTSObjectKeyword = isTSObjectKeyword;
	generated$3.isTSOptionalType = isTSOptionalType;
	generated$3.isTSParameterProperty = isTSParameterProperty;
	generated$3.isTSParenthesizedType = isTSParenthesizedType;
	generated$3.isTSPropertySignature = isTSPropertySignature;
	generated$3.isTSQualifiedName = isTSQualifiedName;
	generated$3.isTSRestType = isTSRestType;
	generated$3.isTSSatisfiesExpression = isTSSatisfiesExpression;
	generated$3.isTSStringKeyword = isTSStringKeyword;
	generated$3.isTSSymbolKeyword = isTSSymbolKeyword;
	generated$3.isTSThisType = isTSThisType;
	generated$3.isTSTupleType = isTSTupleType;
	generated$3.isTSType = isTSType;
	generated$3.isTSTypeAliasDeclaration = isTSTypeAliasDeclaration;
	generated$3.isTSTypeAnnotation = isTSTypeAnnotation;
	generated$3.isTSTypeAssertion = isTSTypeAssertion;
	generated$3.isTSTypeElement = isTSTypeElement;
	generated$3.isTSTypeLiteral = isTSTypeLiteral;
	generated$3.isTSTypeOperator = isTSTypeOperator;
	generated$3.isTSTypeParameter = isTSTypeParameter;
	generated$3.isTSTypeParameterDeclaration = isTSTypeParameterDeclaration;
	generated$3.isTSTypeParameterInstantiation = isTSTypeParameterInstantiation;
	generated$3.isTSTypePredicate = isTSTypePredicate;
	generated$3.isTSTypeQuery = isTSTypeQuery;
	generated$3.isTSTypeReference = isTSTypeReference;
	generated$3.isTSUndefinedKeyword = isTSUndefinedKeyword;
	generated$3.isTSUnionType = isTSUnionType;
	generated$3.isTSUnknownKeyword = isTSUnknownKeyword;
	generated$3.isTSVoidKeyword = isTSVoidKeyword;
	generated$3.isTaggedTemplateExpression = isTaggedTemplateExpression;
	generated$3.isTemplateElement = isTemplateElement;
	generated$3.isTemplateLiteral = isTemplateLiteral;
	generated$3.isTerminatorless = isTerminatorless;
	generated$3.isThisExpression = isThisExpression;
	generated$3.isThisTypeAnnotation = isThisTypeAnnotation;
	generated$3.isThrowStatement = isThrowStatement;
	generated$3.isTopicReference = isTopicReference;
	generated$3.isTryStatement = isTryStatement;
	generated$3.isTupleExpression = isTupleExpression;
	generated$3.isTupleTypeAnnotation = isTupleTypeAnnotation;
	generated$3.isTypeAlias = isTypeAlias;
	generated$3.isTypeAnnotation = isTypeAnnotation;
	generated$3.isTypeCastExpression = isTypeCastExpression;
	generated$3.isTypeParameter = isTypeParameter;
	generated$3.isTypeParameterDeclaration = isTypeParameterDeclaration;
	generated$3.isTypeParameterInstantiation = isTypeParameterInstantiation;
	generated$3.isTypeScript = isTypeScript;
	generated$3.isTypeofTypeAnnotation = isTypeofTypeAnnotation;
	generated$3.isUnaryExpression = isUnaryExpression;
	generated$3.isUnaryLike = isUnaryLike;
	generated$3.isUnionTypeAnnotation = isUnionTypeAnnotation;
	generated$3.isUpdateExpression = isUpdateExpression;
	generated$3.isUserWhitespacable = isUserWhitespacable;
	generated$3.isV8IntrinsicIdentifier = isV8IntrinsicIdentifier;
	generated$3.isVariableDeclaration = isVariableDeclaration;
	generated$3.isVariableDeclarator = isVariableDeclarator;
	generated$3.isVariance = isVariance;
	generated$3.isVoidTypeAnnotation = isVoidTypeAnnotation;
	generated$3.isWhile = isWhile;
	generated$3.isWhileStatement = isWhileStatement;
	generated$3.isWithStatement = isWithStatement;
	generated$3.isYieldExpression = isYieldExpression;
	var _shallowEqual = requireShallowEqual();
	var _deprecationWarning = requireDeprecationWarning();
	function isArrayExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ArrayExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isAssignmentExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "AssignmentExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBinaryExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BinaryExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isInterpreterDirective(node, opts) {
	  if (!node) return false;
	  if (node.type !== "InterpreterDirective") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDirective(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Directive") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDirectiveLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DirectiveLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBlockStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BlockStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBreakStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BreakStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isCallExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "CallExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isCatchClause(node, opts) {
	  if (!node) return false;
	  if (node.type !== "CatchClause") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isConditionalExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ConditionalExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isContinueStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ContinueStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDebuggerStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DebuggerStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDoWhileStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DoWhileStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEmptyStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EmptyStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExpressionStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExpressionStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFile(node, opts) {
	  if (!node) return false;
	  if (node.type !== "File") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isForInStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ForInStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isForStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ForStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunctionDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "FunctionDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunctionExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "FunctionExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isIdentifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Identifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isIfStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "IfStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isLabeledStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "LabeledStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStringLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "StringLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNumericLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NumericLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNullLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NullLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBooleanLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BooleanLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isRegExpLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "RegExpLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isLogicalExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "LogicalExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isMemberExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "MemberExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNewExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NewExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isProgram(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Program") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectMethod(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectMethod") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isRestElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "RestElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isReturnStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ReturnStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSequenceExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "SequenceExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isParenthesizedExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ParenthesizedExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSwitchCase(node, opts) {
	  if (!node) return false;
	  if (node.type !== "SwitchCase") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSwitchStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "SwitchStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isThisExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ThisExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isThrowStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ThrowStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTryStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TryStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isUnaryExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "UnaryExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isUpdateExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "UpdateExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isVariableDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "VariableDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isVariableDeclarator(node, opts) {
	  if (!node) return false;
	  if (node.type !== "VariableDeclarator") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isWhileStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "WhileStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isWithStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "WithStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isAssignmentPattern(node, opts) {
	  if (!node) return false;
	  if (node.type !== "AssignmentPattern") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isArrayPattern(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ArrayPattern") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isArrowFunctionExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ArrowFunctionExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportAllDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportAllDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportDefaultDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportDefaultDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportNamedDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportNamedDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isForOfStatement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ForOfStatement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportDefaultSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportDefaultSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportNamespaceSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportNamespaceSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isMetaProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "MetaProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassMethod(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassMethod") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectPattern(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectPattern") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSpreadElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "SpreadElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSuper(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Super") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTaggedTemplateExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TaggedTemplateExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTemplateElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TemplateElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTemplateLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TemplateLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isYieldExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "YieldExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isAwaitExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "AwaitExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImport(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Import") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBigIntLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BigIntLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportNamespaceSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportNamespaceSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isOptionalMemberExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "OptionalMemberExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isOptionalCallExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "OptionalCallExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassAccessorProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassAccessorProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassPrivateProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassPrivateProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassPrivateMethod(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassPrivateMethod") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPrivateName(node, opts) {
	  if (!node) return false;
	  if (node.type !== "PrivateName") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStaticBlock(node, opts) {
	  if (!node) return false;
	  if (node.type !== "StaticBlock") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isAnyTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "AnyTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isArrayTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ArrayTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBooleanTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BooleanTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBooleanLiteralTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BooleanLiteralTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNullLiteralTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NullLiteralTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClassImplements(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ClassImplements") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareClass(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareClass") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareFunction(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareFunction") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareInterface(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareInterface") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareModule(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareModule") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareModuleExports(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareModuleExports") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareTypeAlias(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareTypeAlias") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareOpaqueType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareOpaqueType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareVariable(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareVariable") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareExportDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareExportDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclareExportAllDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclareExportAllDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclaredPredicate(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DeclaredPredicate") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExistsTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExistsTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunctionTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "FunctionTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunctionTypeParam(node, opts) {
	  if (!node) return false;
	  if (node.type !== "FunctionTypeParam") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isGenericTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "GenericTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isInferredPredicate(node, opts) {
	  if (!node) return false;
	  if (node.type !== "InferredPredicate") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isInterfaceExtends(node, opts) {
	  if (!node) return false;
	  if (node.type !== "InterfaceExtends") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isInterfaceDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "InterfaceDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isInterfaceTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "InterfaceTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isIntersectionTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "IntersectionTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isMixedTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "MixedTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEmptyTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EmptyTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNullableTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NullableTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNumberLiteralTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NumberLiteralTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNumberTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "NumberTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeInternalSlot(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeInternalSlot") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeCallProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeCallProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeIndexer(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeIndexer") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectTypeSpreadProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ObjectTypeSpreadProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isOpaqueType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "OpaqueType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isQualifiedTypeIdentifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "QualifiedTypeIdentifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStringLiteralTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "StringLiteralTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStringTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "StringTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSymbolTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "SymbolTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isThisTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ThisTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTupleTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TupleTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeofTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeofTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeAlias(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeAlias") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeCastExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeCastExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeParameter(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeParameter") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeParameterDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeParameterDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeParameterInstantiation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TypeParameterInstantiation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isUnionTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "UnionTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isVariance(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Variance") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isVoidTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "VoidTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumBooleanBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumBooleanBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumNumberBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumNumberBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumStringBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumStringBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumSymbolBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumSymbolBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumBooleanMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumBooleanMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumNumberMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumNumberMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumStringMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumStringMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumDefaultedMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "EnumDefaultedMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isIndexedAccessType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "IndexedAccessType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isOptionalIndexedAccessType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "OptionalIndexedAccessType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXAttribute(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXAttribute") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXClosingElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXClosingElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXEmptyExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXEmptyExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXExpressionContainer(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXExpressionContainer") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXSpreadChild(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXSpreadChild") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXIdentifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXIdentifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXMemberExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXMemberExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXNamespacedName(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXNamespacedName") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXOpeningElement(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXOpeningElement") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXSpreadAttribute(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXSpreadAttribute") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXText(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXText") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXFragment(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXFragment") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXOpeningFragment(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXOpeningFragment") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSXClosingFragment(node, opts) {
	  if (!node) return false;
	  if (node.type !== "JSXClosingFragment") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNoop(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Noop") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPlaceholder(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Placeholder") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isV8IntrinsicIdentifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "V8IntrinsicIdentifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isArgumentPlaceholder(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ArgumentPlaceholder") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBindExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "BindExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportAttribute(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ImportAttribute") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDecorator(node, opts) {
	  if (!node) return false;
	  if (node.type !== "Decorator") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDoExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DoExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportDefaultSpecifier(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ExportDefaultSpecifier") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isRecordExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "RecordExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTupleExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TupleExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDecimalLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "DecimalLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isModuleExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "ModuleExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTopicReference(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TopicReference") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPipelineTopicExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "PipelineTopicExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPipelineBareFunction(node, opts) {
	  if (!node) return false;
	  if (node.type !== "PipelineBareFunction") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPipelinePrimaryTopicReference(node, opts) {
	  if (!node) return false;
	  if (node.type !== "PipelinePrimaryTopicReference") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSParameterProperty(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSParameterProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSDeclareFunction(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSDeclareFunction") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSDeclareMethod(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSDeclareMethod") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSQualifiedName(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSQualifiedName") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSCallSignatureDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSCallSignatureDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSConstructSignatureDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSConstructSignatureDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSPropertySignature(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSPropertySignature") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSMethodSignature(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSMethodSignature") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSIndexSignature(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSIndexSignature") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSAnyKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSAnyKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSBooleanKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSBooleanKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSBigIntKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSBigIntKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSIntrinsicKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSIntrinsicKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNeverKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNeverKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNullKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNullKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNumberKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNumberKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSObjectKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSObjectKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSStringKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSStringKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSSymbolKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSSymbolKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSUndefinedKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSUndefinedKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSUnknownKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSUnknownKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSVoidKeyword(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSVoidKeyword") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSThisType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSThisType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSFunctionType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSFunctionType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSConstructorType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSConstructorType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeReference(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeReference") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypePredicate(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypePredicate") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeQuery(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeQuery") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeLiteral(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSArrayType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSArrayType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTupleType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTupleType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSOptionalType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSOptionalType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSRestType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSRestType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNamedTupleMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNamedTupleMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSUnionType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSUnionType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSIntersectionType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSIntersectionType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSConditionalType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSConditionalType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSInferType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSInferType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSParenthesizedType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSParenthesizedType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeOperator(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeOperator") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSIndexedAccessType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSIndexedAccessType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSMappedType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSMappedType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSLiteralType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSLiteralType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSExpressionWithTypeArguments(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSExpressionWithTypeArguments") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSInterfaceDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSInterfaceDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSInterfaceBody(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSInterfaceBody") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeAliasDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeAliasDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSInstantiationExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSInstantiationExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSAsExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSAsExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSSatisfiesExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSSatisfiesExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeAssertion(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeAssertion") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSEnumDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSEnumDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSEnumMember(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSEnumMember") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSModuleDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSModuleDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSModuleBlock(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSModuleBlock") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSImportType(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSImportType") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSImportEqualsDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSImportEqualsDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSExternalModuleReference(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSExternalModuleReference") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNonNullExpression(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNonNullExpression") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSExportAssignment(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSExportAssignment") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSNamespaceExportDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSNamespaceExportDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeAnnotation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeAnnotation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeParameterInstantiation(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeParameterInstantiation") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeParameterDeclaration(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeParameterDeclaration") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeParameter(node, opts) {
	  if (!node) return false;
	  if (node.type !== "TSTypeParameter") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStandardized(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ArrayExpression":
	    case "AssignmentExpression":
	    case "BinaryExpression":
	    case "InterpreterDirective":
	    case "Directive":
	    case "DirectiveLiteral":
	    case "BlockStatement":
	    case "BreakStatement":
	    case "CallExpression":
	    case "CatchClause":
	    case "ConditionalExpression":
	    case "ContinueStatement":
	    case "DebuggerStatement":
	    case "DoWhileStatement":
	    case "EmptyStatement":
	    case "ExpressionStatement":
	    case "File":
	    case "ForInStatement":
	    case "ForStatement":
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "Identifier":
	    case "IfStatement":
	    case "LabeledStatement":
	    case "StringLiteral":
	    case "NumericLiteral":
	    case "NullLiteral":
	    case "BooleanLiteral":
	    case "RegExpLiteral":
	    case "LogicalExpression":
	    case "MemberExpression":
	    case "NewExpression":
	    case "Program":
	    case "ObjectExpression":
	    case "ObjectMethod":
	    case "ObjectProperty":
	    case "RestElement":
	    case "ReturnStatement":
	    case "SequenceExpression":
	    case "ParenthesizedExpression":
	    case "SwitchCase":
	    case "SwitchStatement":
	    case "ThisExpression":
	    case "ThrowStatement":
	    case "TryStatement":
	    case "UnaryExpression":
	    case "UpdateExpression":
	    case "VariableDeclaration":
	    case "VariableDeclarator":
	    case "WhileStatement":
	    case "WithStatement":
	    case "AssignmentPattern":
	    case "ArrayPattern":
	    case "ArrowFunctionExpression":
	    case "ClassBody":
	    case "ClassExpression":
	    case "ClassDeclaration":
	    case "ExportAllDeclaration":
	    case "ExportDefaultDeclaration":
	    case "ExportNamedDeclaration":
	    case "ExportSpecifier":
	    case "ForOfStatement":
	    case "ImportDeclaration":
	    case "ImportDefaultSpecifier":
	    case "ImportNamespaceSpecifier":
	    case "ImportSpecifier":
	    case "ImportExpression":
	    case "MetaProperty":
	    case "ClassMethod":
	    case "ObjectPattern":
	    case "SpreadElement":
	    case "Super":
	    case "TaggedTemplateExpression":
	    case "TemplateElement":
	    case "TemplateLiteral":
	    case "YieldExpression":
	    case "AwaitExpression":
	    case "Import":
	    case "BigIntLiteral":
	    case "ExportNamespaceSpecifier":
	    case "OptionalMemberExpression":
	    case "OptionalCallExpression":
	    case "ClassProperty":
	    case "ClassAccessorProperty":
	    case "ClassPrivateProperty":
	    case "ClassPrivateMethod":
	    case "PrivateName":
	    case "StaticBlock":
	      break;
	    case "Placeholder":
	      switch (node.expectedNode) {
	        case "Identifier":
	        case "StringLiteral":
	        case "BlockStatement":
	        case "ClassBody":
	          break;
	        default:
	          return false;
	      }
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExpression(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ArrayExpression":
	    case "AssignmentExpression":
	    case "BinaryExpression":
	    case "CallExpression":
	    case "ConditionalExpression":
	    case "FunctionExpression":
	    case "Identifier":
	    case "StringLiteral":
	    case "NumericLiteral":
	    case "NullLiteral":
	    case "BooleanLiteral":
	    case "RegExpLiteral":
	    case "LogicalExpression":
	    case "MemberExpression":
	    case "NewExpression":
	    case "ObjectExpression":
	    case "SequenceExpression":
	    case "ParenthesizedExpression":
	    case "ThisExpression":
	    case "UnaryExpression":
	    case "UpdateExpression":
	    case "ArrowFunctionExpression":
	    case "ClassExpression":
	    case "ImportExpression":
	    case "MetaProperty":
	    case "Super":
	    case "TaggedTemplateExpression":
	    case "TemplateLiteral":
	    case "YieldExpression":
	    case "AwaitExpression":
	    case "Import":
	    case "BigIntLiteral":
	    case "OptionalMemberExpression":
	    case "OptionalCallExpression":
	    case "TypeCastExpression":
	    case "JSXElement":
	    case "JSXFragment":
	    case "BindExpression":
	    case "DoExpression":
	    case "RecordExpression":
	    case "TupleExpression":
	    case "DecimalLiteral":
	    case "ModuleExpression":
	    case "TopicReference":
	    case "PipelineTopicExpression":
	    case "PipelineBareFunction":
	    case "PipelinePrimaryTopicReference":
	    case "TSInstantiationExpression":
	    case "TSAsExpression":
	    case "TSSatisfiesExpression":
	    case "TSTypeAssertion":
	    case "TSNonNullExpression":
	      break;
	    case "Placeholder":
	      switch (node.expectedNode) {
	        case "Expression":
	        case "Identifier":
	        case "StringLiteral":
	          break;
	        default:
	          return false;
	      }
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBinary(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BinaryExpression":
	    case "LogicalExpression":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isScopable(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BlockStatement":
	    case "CatchClause":
	    case "DoWhileStatement":
	    case "ForInStatement":
	    case "ForStatement":
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "Program":
	    case "ObjectMethod":
	    case "SwitchStatement":
	    case "WhileStatement":
	    case "ArrowFunctionExpression":
	    case "ClassExpression":
	    case "ClassDeclaration":
	    case "ForOfStatement":
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	    case "StaticBlock":
	    case "TSModuleBlock":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "BlockStatement") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBlockParent(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BlockStatement":
	    case "CatchClause":
	    case "DoWhileStatement":
	    case "ForInStatement":
	    case "ForStatement":
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "Program":
	    case "ObjectMethod":
	    case "SwitchStatement":
	    case "WhileStatement":
	    case "ArrowFunctionExpression":
	    case "ForOfStatement":
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	    case "StaticBlock":
	    case "TSModuleBlock":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "BlockStatement") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isBlock(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BlockStatement":
	    case "Program":
	    case "TSModuleBlock":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "BlockStatement") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isStatement(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BlockStatement":
	    case "BreakStatement":
	    case "ContinueStatement":
	    case "DebuggerStatement":
	    case "DoWhileStatement":
	    case "EmptyStatement":
	    case "ExpressionStatement":
	    case "ForInStatement":
	    case "ForStatement":
	    case "FunctionDeclaration":
	    case "IfStatement":
	    case "LabeledStatement":
	    case "ReturnStatement":
	    case "SwitchStatement":
	    case "ThrowStatement":
	    case "TryStatement":
	    case "VariableDeclaration":
	    case "WhileStatement":
	    case "WithStatement":
	    case "ClassDeclaration":
	    case "ExportAllDeclaration":
	    case "ExportDefaultDeclaration":
	    case "ExportNamedDeclaration":
	    case "ForOfStatement":
	    case "ImportDeclaration":
	    case "DeclareClass":
	    case "DeclareFunction":
	    case "DeclareInterface":
	    case "DeclareModule":
	    case "DeclareModuleExports":
	    case "DeclareTypeAlias":
	    case "DeclareOpaqueType":
	    case "DeclareVariable":
	    case "DeclareExportDeclaration":
	    case "DeclareExportAllDeclaration":
	    case "InterfaceDeclaration":
	    case "OpaqueType":
	    case "TypeAlias":
	    case "EnumDeclaration":
	    case "TSDeclareFunction":
	    case "TSInterfaceDeclaration":
	    case "TSTypeAliasDeclaration":
	    case "TSEnumDeclaration":
	    case "TSModuleDeclaration":
	    case "TSImportEqualsDeclaration":
	    case "TSExportAssignment":
	    case "TSNamespaceExportDeclaration":
	      break;
	    case "Placeholder":
	      switch (node.expectedNode) {
	        case "Statement":
	        case "Declaration":
	        case "BlockStatement":
	          break;
	        default:
	          return false;
	      }
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTerminatorless(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BreakStatement":
	    case "ContinueStatement":
	    case "ReturnStatement":
	    case "ThrowStatement":
	    case "YieldExpression":
	    case "AwaitExpression":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isCompletionStatement(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "BreakStatement":
	    case "ContinueStatement":
	    case "ReturnStatement":
	    case "ThrowStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isConditional(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ConditionalExpression":
	    case "IfStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isLoop(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "DoWhileStatement":
	    case "ForInStatement":
	    case "ForStatement":
	    case "WhileStatement":
	    case "ForOfStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isWhile(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "DoWhileStatement":
	    case "WhileStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExpressionWrapper(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ExpressionStatement":
	    case "ParenthesizedExpression":
	    case "TypeCastExpression":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFor(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ForInStatement":
	    case "ForStatement":
	    case "ForOfStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isForXStatement(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ForInStatement":
	    case "ForOfStatement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunction(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "ObjectMethod":
	    case "ArrowFunctionExpression":
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFunctionParent(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "ObjectMethod":
	    case "ArrowFunctionExpression":
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	    case "StaticBlock":
	    case "TSModuleBlock":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPureish(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	    case "StringLiteral":
	    case "NumericLiteral":
	    case "NullLiteral":
	    case "BooleanLiteral":
	    case "RegExpLiteral":
	    case "ArrowFunctionExpression":
	    case "BigIntLiteral":
	    case "DecimalLiteral":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "StringLiteral") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isDeclaration(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "FunctionDeclaration":
	    case "VariableDeclaration":
	    case "ClassDeclaration":
	    case "ExportAllDeclaration":
	    case "ExportDefaultDeclaration":
	    case "ExportNamedDeclaration":
	    case "ImportDeclaration":
	    case "DeclareClass":
	    case "DeclareFunction":
	    case "DeclareInterface":
	    case "DeclareModule":
	    case "DeclareModuleExports":
	    case "DeclareTypeAlias":
	    case "DeclareOpaqueType":
	    case "DeclareVariable":
	    case "DeclareExportDeclaration":
	    case "DeclareExportAllDeclaration":
	    case "InterfaceDeclaration":
	    case "OpaqueType":
	    case "TypeAlias":
	    case "EnumDeclaration":
	    case "TSDeclareFunction":
	    case "TSInterfaceDeclaration":
	    case "TSTypeAliasDeclaration":
	    case "TSEnumDeclaration":
	    case "TSModuleDeclaration":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "Declaration") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPatternLike(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "Identifier":
	    case "RestElement":
	    case "AssignmentPattern":
	    case "ArrayPattern":
	    case "ObjectPattern":
	    case "TSAsExpression":
	    case "TSSatisfiesExpression":
	    case "TSTypeAssertion":
	    case "TSNonNullExpression":
	      break;
	    case "Placeholder":
	      switch (node.expectedNode) {
	        case "Pattern":
	        case "Identifier":
	          break;
	        default:
	          return false;
	      }
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isLVal(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "Identifier":
	    case "MemberExpression":
	    case "RestElement":
	    case "AssignmentPattern":
	    case "ArrayPattern":
	    case "ObjectPattern":
	    case "TSParameterProperty":
	    case "TSAsExpression":
	    case "TSSatisfiesExpression":
	    case "TSTypeAssertion":
	    case "TSNonNullExpression":
	      break;
	    case "Placeholder":
	      switch (node.expectedNode) {
	        case "Pattern":
	        case "Identifier":
	          break;
	        default:
	          return false;
	      }
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSEntityName(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "Identifier":
	    case "TSQualifiedName":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "Identifier") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isLiteral(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "StringLiteral":
	    case "NumericLiteral":
	    case "NullLiteral":
	    case "BooleanLiteral":
	    case "RegExpLiteral":
	    case "TemplateLiteral":
	    case "BigIntLiteral":
	    case "DecimalLiteral":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "StringLiteral") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImmutable(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "StringLiteral":
	    case "NumericLiteral":
	    case "NullLiteral":
	    case "BooleanLiteral":
	    case "BigIntLiteral":
	    case "JSXAttribute":
	    case "JSXClosingElement":
	    case "JSXElement":
	    case "JSXExpressionContainer":
	    case "JSXSpreadChild":
	    case "JSXOpeningElement":
	    case "JSXText":
	    case "JSXFragment":
	    case "JSXOpeningFragment":
	    case "JSXClosingFragment":
	    case "DecimalLiteral":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "StringLiteral") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isUserWhitespacable(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ObjectMethod":
	    case "ObjectProperty":
	    case "ObjectTypeInternalSlot":
	    case "ObjectTypeCallProperty":
	    case "ObjectTypeIndexer":
	    case "ObjectTypeProperty":
	    case "ObjectTypeSpreadProperty":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isMethod(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ObjectMethod":
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isObjectMember(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ObjectMethod":
	    case "ObjectProperty":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isProperty(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ObjectProperty":
	    case "ClassProperty":
	    case "ClassAccessorProperty":
	    case "ClassPrivateProperty":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isUnaryLike(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "UnaryExpression":
	    case "SpreadElement":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPattern(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "AssignmentPattern":
	    case "ArrayPattern":
	    case "ObjectPattern":
	      break;
	    case "Placeholder":
	      if (node.expectedNode === "Pattern") break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isClass(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ClassExpression":
	    case "ClassDeclaration":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isImportOrExportDeclaration(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ExportAllDeclaration":
	    case "ExportDefaultDeclaration":
	    case "ExportNamedDeclaration":
	    case "ImportDeclaration":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isExportDeclaration(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ExportAllDeclaration":
	    case "ExportDefaultDeclaration":
	    case "ExportNamedDeclaration":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isModuleSpecifier(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ExportSpecifier":
	    case "ImportDefaultSpecifier":
	    case "ImportNamespaceSpecifier":
	    case "ImportSpecifier":
	    case "ExportNamespaceSpecifier":
	    case "ExportDefaultSpecifier":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isAccessor(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ClassAccessorProperty":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isPrivate(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "ClassPrivateProperty":
	    case "ClassPrivateMethod":
	    case "PrivateName":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFlow(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "AnyTypeAnnotation":
	    case "ArrayTypeAnnotation":
	    case "BooleanTypeAnnotation":
	    case "BooleanLiteralTypeAnnotation":
	    case "NullLiteralTypeAnnotation":
	    case "ClassImplements":
	    case "DeclareClass":
	    case "DeclareFunction":
	    case "DeclareInterface":
	    case "DeclareModule":
	    case "DeclareModuleExports":
	    case "DeclareTypeAlias":
	    case "DeclareOpaqueType":
	    case "DeclareVariable":
	    case "DeclareExportDeclaration":
	    case "DeclareExportAllDeclaration":
	    case "DeclaredPredicate":
	    case "ExistsTypeAnnotation":
	    case "FunctionTypeAnnotation":
	    case "FunctionTypeParam":
	    case "GenericTypeAnnotation":
	    case "InferredPredicate":
	    case "InterfaceExtends":
	    case "InterfaceDeclaration":
	    case "InterfaceTypeAnnotation":
	    case "IntersectionTypeAnnotation":
	    case "MixedTypeAnnotation":
	    case "EmptyTypeAnnotation":
	    case "NullableTypeAnnotation":
	    case "NumberLiteralTypeAnnotation":
	    case "NumberTypeAnnotation":
	    case "ObjectTypeAnnotation":
	    case "ObjectTypeInternalSlot":
	    case "ObjectTypeCallProperty":
	    case "ObjectTypeIndexer":
	    case "ObjectTypeProperty":
	    case "ObjectTypeSpreadProperty":
	    case "OpaqueType":
	    case "QualifiedTypeIdentifier":
	    case "StringLiteralTypeAnnotation":
	    case "StringTypeAnnotation":
	    case "SymbolTypeAnnotation":
	    case "ThisTypeAnnotation":
	    case "TupleTypeAnnotation":
	    case "TypeofTypeAnnotation":
	    case "TypeAlias":
	    case "TypeAnnotation":
	    case "TypeCastExpression":
	    case "TypeParameter":
	    case "TypeParameterDeclaration":
	    case "TypeParameterInstantiation":
	    case "UnionTypeAnnotation":
	    case "Variance":
	    case "VoidTypeAnnotation":
	    case "EnumDeclaration":
	    case "EnumBooleanBody":
	    case "EnumNumberBody":
	    case "EnumStringBody":
	    case "EnumSymbolBody":
	    case "EnumBooleanMember":
	    case "EnumNumberMember":
	    case "EnumStringMember":
	    case "EnumDefaultedMember":
	    case "IndexedAccessType":
	    case "OptionalIndexedAccessType":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFlowType(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "AnyTypeAnnotation":
	    case "ArrayTypeAnnotation":
	    case "BooleanTypeAnnotation":
	    case "BooleanLiteralTypeAnnotation":
	    case "NullLiteralTypeAnnotation":
	    case "ExistsTypeAnnotation":
	    case "FunctionTypeAnnotation":
	    case "GenericTypeAnnotation":
	    case "InterfaceTypeAnnotation":
	    case "IntersectionTypeAnnotation":
	    case "MixedTypeAnnotation":
	    case "EmptyTypeAnnotation":
	    case "NullableTypeAnnotation":
	    case "NumberLiteralTypeAnnotation":
	    case "NumberTypeAnnotation":
	    case "ObjectTypeAnnotation":
	    case "StringLiteralTypeAnnotation":
	    case "StringTypeAnnotation":
	    case "SymbolTypeAnnotation":
	    case "ThisTypeAnnotation":
	    case "TupleTypeAnnotation":
	    case "TypeofTypeAnnotation":
	    case "UnionTypeAnnotation":
	    case "VoidTypeAnnotation":
	    case "IndexedAccessType":
	    case "OptionalIndexedAccessType":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFlowBaseAnnotation(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "AnyTypeAnnotation":
	    case "BooleanTypeAnnotation":
	    case "NullLiteralTypeAnnotation":
	    case "MixedTypeAnnotation":
	    case "EmptyTypeAnnotation":
	    case "NumberTypeAnnotation":
	    case "StringTypeAnnotation":
	    case "SymbolTypeAnnotation":
	    case "ThisTypeAnnotation":
	    case "VoidTypeAnnotation":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFlowDeclaration(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "DeclareClass":
	    case "DeclareFunction":
	    case "DeclareInterface":
	    case "DeclareModule":
	    case "DeclareModuleExports":
	    case "DeclareTypeAlias":
	    case "DeclareOpaqueType":
	    case "DeclareVariable":
	    case "DeclareExportDeclaration":
	    case "DeclareExportAllDeclaration":
	    case "InterfaceDeclaration":
	    case "OpaqueType":
	    case "TypeAlias":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isFlowPredicate(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "DeclaredPredicate":
	    case "InferredPredicate":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumBody(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "EnumBooleanBody":
	    case "EnumNumberBody":
	    case "EnumStringBody":
	    case "EnumSymbolBody":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isEnumMember(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "EnumBooleanMember":
	    case "EnumNumberMember":
	    case "EnumStringMember":
	    case "EnumDefaultedMember":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isJSX(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "JSXAttribute":
	    case "JSXClosingElement":
	    case "JSXElement":
	    case "JSXEmptyExpression":
	    case "JSXExpressionContainer":
	    case "JSXSpreadChild":
	    case "JSXIdentifier":
	    case "JSXMemberExpression":
	    case "JSXNamespacedName":
	    case "JSXOpeningElement":
	    case "JSXSpreadAttribute":
	    case "JSXText":
	    case "JSXFragment":
	    case "JSXOpeningFragment":
	    case "JSXClosingFragment":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isMiscellaneous(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "Noop":
	    case "Placeholder":
	    case "V8IntrinsicIdentifier":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTypeScript(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "TSParameterProperty":
	    case "TSDeclareFunction":
	    case "TSDeclareMethod":
	    case "TSQualifiedName":
	    case "TSCallSignatureDeclaration":
	    case "TSConstructSignatureDeclaration":
	    case "TSPropertySignature":
	    case "TSMethodSignature":
	    case "TSIndexSignature":
	    case "TSAnyKeyword":
	    case "TSBooleanKeyword":
	    case "TSBigIntKeyword":
	    case "TSIntrinsicKeyword":
	    case "TSNeverKeyword":
	    case "TSNullKeyword":
	    case "TSNumberKeyword":
	    case "TSObjectKeyword":
	    case "TSStringKeyword":
	    case "TSSymbolKeyword":
	    case "TSUndefinedKeyword":
	    case "TSUnknownKeyword":
	    case "TSVoidKeyword":
	    case "TSThisType":
	    case "TSFunctionType":
	    case "TSConstructorType":
	    case "TSTypeReference":
	    case "TSTypePredicate":
	    case "TSTypeQuery":
	    case "TSTypeLiteral":
	    case "TSArrayType":
	    case "TSTupleType":
	    case "TSOptionalType":
	    case "TSRestType":
	    case "TSNamedTupleMember":
	    case "TSUnionType":
	    case "TSIntersectionType":
	    case "TSConditionalType":
	    case "TSInferType":
	    case "TSParenthesizedType":
	    case "TSTypeOperator":
	    case "TSIndexedAccessType":
	    case "TSMappedType":
	    case "TSLiteralType":
	    case "TSExpressionWithTypeArguments":
	    case "TSInterfaceDeclaration":
	    case "TSInterfaceBody":
	    case "TSTypeAliasDeclaration":
	    case "TSInstantiationExpression":
	    case "TSAsExpression":
	    case "TSSatisfiesExpression":
	    case "TSTypeAssertion":
	    case "TSEnumDeclaration":
	    case "TSEnumMember":
	    case "TSModuleDeclaration":
	    case "TSModuleBlock":
	    case "TSImportType":
	    case "TSImportEqualsDeclaration":
	    case "TSExternalModuleReference":
	    case "TSNonNullExpression":
	    case "TSExportAssignment":
	    case "TSNamespaceExportDeclaration":
	    case "TSTypeAnnotation":
	    case "TSTypeParameterInstantiation":
	    case "TSTypeParameterDeclaration":
	    case "TSTypeParameter":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSTypeElement(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "TSCallSignatureDeclaration":
	    case "TSConstructSignatureDeclaration":
	    case "TSPropertySignature":
	    case "TSMethodSignature":
	    case "TSIndexSignature":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSType(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "TSAnyKeyword":
	    case "TSBooleanKeyword":
	    case "TSBigIntKeyword":
	    case "TSIntrinsicKeyword":
	    case "TSNeverKeyword":
	    case "TSNullKeyword":
	    case "TSNumberKeyword":
	    case "TSObjectKeyword":
	    case "TSStringKeyword":
	    case "TSSymbolKeyword":
	    case "TSUndefinedKeyword":
	    case "TSUnknownKeyword":
	    case "TSVoidKeyword":
	    case "TSThisType":
	    case "TSFunctionType":
	    case "TSConstructorType":
	    case "TSTypeReference":
	    case "TSTypePredicate":
	    case "TSTypeQuery":
	    case "TSTypeLiteral":
	    case "TSArrayType":
	    case "TSTupleType":
	    case "TSOptionalType":
	    case "TSRestType":
	    case "TSUnionType":
	    case "TSIntersectionType":
	    case "TSConditionalType":
	    case "TSInferType":
	    case "TSParenthesizedType":
	    case "TSTypeOperator":
	    case "TSIndexedAccessType":
	    case "TSMappedType":
	    case "TSLiteralType":
	    case "TSExpressionWithTypeArguments":
	    case "TSImportType":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isTSBaseType(node, opts) {
	  if (!node) return false;
	  switch (node.type) {
	    case "TSAnyKeyword":
	    case "TSBooleanKeyword":
	    case "TSBigIntKeyword":
	    case "TSIntrinsicKeyword":
	    case "TSNeverKeyword":
	    case "TSNullKeyword":
	    case "TSNumberKeyword":
	    case "TSObjectKeyword":
	    case "TSStringKeyword":
	    case "TSSymbolKeyword":
	    case "TSUndefinedKeyword":
	    case "TSUnknownKeyword":
	    case "TSVoidKeyword":
	    case "TSThisType":
	    case "TSLiteralType":
	      break;
	    default:
	      return false;
	  }
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isNumberLiteral(node, opts) {
	  (0, _deprecationWarning.default)("isNumberLiteral", "isNumericLiteral");
	  if (!node) return false;
	  if (node.type !== "NumberLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isRegexLiteral(node, opts) {
	  (0, _deprecationWarning.default)("isRegexLiteral", "isRegExpLiteral");
	  if (!node) return false;
	  if (node.type !== "RegexLiteral") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isRestProperty(node, opts) {
	  (0, _deprecationWarning.default)("isRestProperty", "isRestElement");
	  if (!node) return false;
	  if (node.type !== "RestProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isSpreadProperty(node, opts) {
	  (0, _deprecationWarning.default)("isSpreadProperty", "isSpreadElement");
	  if (!node) return false;
	  if (node.type !== "SpreadProperty") return false;
	  return opts == null || (0, _shallowEqual.default)(node, opts);
	}
	function isModuleDeclaration(node, opts) {
	  (0, _deprecationWarning.default)("isModuleDeclaration", "isImportOrExportDeclaration");
	  return isImportOrExportDeclaration(node, opts);
	}

	
	return generated$3;
}

var hasRequiredMatchesPattern;

function requireMatchesPattern () {
	if (hasRequiredMatchesPattern) return matchesPattern;
	hasRequiredMatchesPattern = 1;

	Object.defineProperty(matchesPattern, "__esModule", {
	  value: true
	});
	matchesPattern.default = matchesPattern$1;
	var _index = requireGenerated$3();
	function matchesPattern$1(member, match, allowPartial) {
	  if (!(0, _index.isMemberExpression)(member)) return false;
	  const parts = Array.isArray(match) ? match : match.split(".");
	  const nodes = [];
	  let node;
	  for (node = member; (0, _index.isMemberExpression)(node); node = node.object) {
	    nodes.push(node.property);
	  }
	  nodes.push(node);
	  if (nodes.length < parts.length) return false;
	  if (!allowPartial && nodes.length > parts.length) return false;
	  for (let i = 0, j = nodes.length - 1; i < parts.length; i++, j--) {
	    const node = nodes[j];
	    let value;
	    if ((0, _index.isIdentifier)(node)) {
	      value = node.name;
	    } else if ((0, _index.isStringLiteral)(node)) {
	      value = node.value;
	    } else if ((0, _index.isThisExpression)(node)) {
	      value = "this";
	    } else {
	      return false;
	    }
	    if (parts[i] !== value) return false;
	  }
	  return true;
	}

	
	return matchesPattern;
}

var hasRequiredBuildMatchMemberExpression;

function requireBuildMatchMemberExpression () {
	if (hasRequiredBuildMatchMemberExpression) return buildMatchMemberExpression;
	hasRequiredBuildMatchMemberExpression = 1;

	Object.defineProperty(buildMatchMemberExpression, "__esModule", {
	  value: true
	});
	buildMatchMemberExpression.default = buildMatchMemberExpression$1;
	var _matchesPattern = requireMatchesPattern();
	function buildMatchMemberExpression$1(match, allowPartial) {
	  const parts = match.split(".");
	  return member => (0, _matchesPattern.default)(member, parts, allowPartial);
	}

	
	return buildMatchMemberExpression;
}

var hasRequiredIsReactComponent;

function requireIsReactComponent () {
	if (hasRequiredIsReactComponent) return isReactComponent;
	hasRequiredIsReactComponent = 1;

	Object.defineProperty(isReactComponent, "__esModule", {
	  value: true
	});
	isReactComponent.default = void 0;
	var _buildMatchMemberExpression = requireBuildMatchMemberExpression();
	const isReactComponent$1 = (0, _buildMatchMemberExpression.default)("React.Component");
	isReactComponent.default = isReactComponent$1;

	
	return isReactComponent;
}

var isCompatTag = {};

var hasRequiredIsCompatTag;

function requireIsCompatTag () {
	if (hasRequiredIsCompatTag) return isCompatTag;
	hasRequiredIsCompatTag = 1;

	Object.defineProperty(isCompatTag, "__esModule", {
	  value: true
	});
	isCompatTag.default = isCompatTag$1;
	function isCompatTag$1(tagName) {
	  return !!tagName && /^[a-z]/.test(tagName);
	}

	
	return isCompatTag;
}

var buildChildren = {};

var cleanJSXElementLiteralChild = {};

var generated$2 = {};

var validate$1 = {};

var definitions = {};

var core = {};

var is = {};

var isType = {};

var hasRequiredIsType;

function requireIsType () {
	if (hasRequiredIsType) return isType;
	hasRequiredIsType = 1;

	Object.defineProperty(isType, "__esModule", {
	  value: true
	});
	isType.default = isType$1;
	var _index = requireDefinitions();
	function isType$1(nodeType, targetType) {
	  if (nodeType === targetType) return true;
	  if (nodeType == null) return false;
	  if (_index.ALIAS_KEYS[targetType]) return false;
	  const aliases = _index.FLIPPED_ALIAS_KEYS[targetType];
	  if (aliases) {
	    if (aliases[0] === nodeType) return true;
	    for (const alias of aliases) {
	      if (nodeType === alias) return true;
	    }
	  }
	  return false;
	}

	
	return isType;
}

var isPlaceholderType = {};

var hasRequiredIsPlaceholderType;

function requireIsPlaceholderType () {
	if (hasRequiredIsPlaceholderType) return isPlaceholderType;
	hasRequiredIsPlaceholderType = 1;

	Object.defineProperty(isPlaceholderType, "__esModule", {
	  value: true
	});
	isPlaceholderType.default = isPlaceholderType$1;
	var _index = requireDefinitions();
	function isPlaceholderType$1(placeholderType, targetType) {
	  if (placeholderType === targetType) return true;
	  const aliases = _index.PLACEHOLDERS_ALIAS[placeholderType];
	  if (aliases) {
	    for (const alias of aliases) {
	      if (targetType === alias) return true;
	    }
	  }
	  return false;
	}

	
	return isPlaceholderType;
}

var hasRequiredIs;

function requireIs () {
	if (hasRequiredIs) return is;
	hasRequiredIs = 1;

	Object.defineProperty(is, "__esModule", {
	  value: true
	});
	is.default = is$1;
	var _shallowEqual = requireShallowEqual();
	var _isType = requireIsType();
	var _isPlaceholderType = requireIsPlaceholderType();
	var _index = requireDefinitions();
	function is$1(type, node, opts) {
	  if (!node) return false;
	  const matches = (0, _isType.default)(node.type, type);
	  if (!matches) {
	    if (!opts && node.type === "Placeholder" && type in _index.FLIPPED_ALIAS_KEYS) {
	      return (0, _isPlaceholderType.default)(node.expectedNode, type);
	    }
	    return false;
	  }
	  if (opts === undefined) {
	    return true;
	  } else {
	    return (0, _shallowEqual.default)(node, opts);
	  }
	}

	
	return is;
}

var isValidIdentifier = {};

var lib$2 = {};

var identifier = {};

var hasRequiredIdentifier;

function requireIdentifier () {
	if (hasRequiredIdentifier) return identifier;
	hasRequiredIdentifier = 1;

	Object.defineProperty(identifier, "__esModule", {
	  value: true
	});
	identifier.isIdentifierChar = isIdentifierChar;
	identifier.isIdentifierName = isIdentifierName;
	identifier.isIdentifierStart = isIdentifierStart;
	let nonASCIIidentifierStartChars = "\xaa\xb5\xba\xc0-\xd6\xd8-\xf6\xf8-\u02c1\u02c6-\u02d1\u02e0-\u02e4\u02ec\u02ee\u0370-\u0374\u0376\u0377\u037a-\u037d\u037f\u0386\u0388-\u038a\u038c\u038e-\u03a1\u03a3-\u03f5\u03f7-\u0481\u048a-\u052f\u0531-\u0556\u0559\u0560-\u0588\u05d0-\u05ea\u05ef-\u05f2\u0620-\u064a\u066e\u066f\u0671-\u06d3\u06d5\u06e5\u06e6\u06ee\u06ef\u06fa-\u06fc\u06ff\u0710\u0712-\u072f\u074d-\u07a5\u07b1\u07ca-\u07ea\u07f4\u07f5\u07fa\u0800-\u0815\u081a\u0824\u0828\u0840-\u0858\u0860-\u086a\u0870-\u0887\u0889-\u088e\u08a0-\u08c9\u0904-\u0939\u093d\u0950\u0958-\u0961\u0971-\u0980\u0985-\u098c\u098f\u0990\u0993-\u09a8\u09aa-\u09b0\u09b2\u09b6-\u09b9\u09bd\u09ce\u09dc\u09dd\u09df-\u09e1\u09f0\u09f1\u09fc\u0a05-\u0a0a\u0a0f\u0a10\u0a13-\u0a28\u0a2a-\u0a30\u0a32\u0a33\u0a35\u0a36\u0a38\u0a39\u0a59-\u0a5c\u0a5e\u0a72-\u0a74\u0a85-\u0a8d\u0a8f-\u0a91\u0a93-\u0aa8\u0aaa-\u0ab0\u0ab2\u0ab3\u0ab5-\u0ab9\u0abd\u0ad0\u0ae0\u0ae1\u0af9\u0b05-\u0b0c\u0b0f\u0b10\u0b13-\u0b28\u0b2a-\u0b30\u0b32\u0b33\u0b35-\u0b39\u0b3d\u0b5c\u0b5d\u0b5f-\u0b61\u0b71\u0b83\u0b85-\u0b8a\u0b8e-\u0b90\u0b92-\u0b95\u0b99\u0b9a\u0b9c\u0b9e\u0b9f\u0ba3\u0ba4\u0ba8-\u0baa\u0bae-\u0bb9\u0bd0\u0c05-\u0c0c\u0c0e-\u0c10\u0c12-\u0c28\u0c2a-\u0c39\u0c3d\u0c58-\u0c5a\u0c5d\u0c60\u0c61\u0c80\u0c85-\u0c8c\u0c8e-\u0c90\u0c92-\u0ca8\u0caa-\u0cb3\u0cb5-\u0cb9\u0cbd\u0cdd\u0cde\u0ce0\u0ce1\u0cf1\u0cf2\u0d04-\u0d0c\u0d0e-\u0d10\u0d12-\u0d3a\u0d3d\u0d4e\u0d54-\u0d56\u0d5f-\u0d61\u0d7a-\u0d7f\u0d85-\u0d96\u0d9a-\u0db1\u0db3-\u0dbb\u0dbd\u0dc0-\u0dc6\u0e01-\u0e30\u0e32\u0e33\u0e40-\u0e46\u0e81\u0e82\u0e84\u0e86-\u0e8a\u0e8c-\u0ea3\u0ea5\u0ea7-\u0eb0\u0eb2\u0eb3\u0ebd\u0ec0-\u0ec4\u0ec6\u0edc-\u0edf\u0f00\u0f40-\u0f47\u0f49-\u0f6c\u0f88-\u0f8c\u1000-\u102a\u103f\u1050-\u1055\u105a-\u105d\u1061\u1065\u1066\u106e-\u1070\u1075-\u1081\u108e\u10a0-\u10c5\u10c7\u10cd\u10d0-\u10fa\u10fc-\u1248\u124a-\u124d\u1250-\u1256\u1258\u125a-\u125d\u1260-\u1288\u128a-\u128d\u1290-\u12b0\u12b2-\u12b5\u12b8-\u12be\u12c0\u12c2-\u12c5\u12c8-\u12d6\u12d8-\u1310\u1312-\u1315\u1318-\u135a\u1380-\u138f\u13a0-\u13f5\u13f8-\u13fd\u1401-\u166c\u166f-\u167f\u1681-\u169a\u16a0-\u16ea\u16ee-\u16f8\u1700-\u1711\u171f-\u1731\u1740-\u1751\u1760-\u176c\u176e-\u1770\u1780-\u17b3\u17d7\u17dc\u1820-\u1878\u1880-\u18a8\u18aa\u18b0-\u18f5\u1900-\u191e\u1950-\u196d\u1970-\u1974\u1980-\u19ab\u19b0-\u19c9\u1a00-\u1a16\u1a20-\u1a54\u1aa7\u1b05-\u1b33\u1b45-\u1b4c\u1b83-\u1ba0\u1bae\u1baf\u1bba-\u1be5\u1c00-\u1c23\u1c4d-\u1c4f\u1c5a-\u1c7d\u1c80-\u1c8a\u1c90-\u1cba\u1cbd-\u1cbf\u1ce9-\u1cec\u1cee-\u1cf3\u1cf5\u1cf6\u1cfa\u1d00-\u1dbf\u1e00-\u1f15\u1f18-\u1f1d\u1f20-\u1f45\u1f48-\u1f4d\u1f50-\u1f57\u1f59\u1f5b\u1f5d\u1f5f-\u1f7d\u1f80-\u1fb4\u1fb6-\u1fbc\u1fbe\u1fc2-\u1fc4\u1fc6-\u1fcc\u1fd0-\u1fd3\u1fd6-\u1fdb\u1fe0-\u1fec\u1ff2-\u1ff4\u1ff6-\u1ffc\u2071\u207f\u2090-\u209c\u2102\u2107\u210a-\u2113\u2115\u2118-\u211d\u2124\u2126\u2128\u212a-\u2139\u213c-\u213f\u2145-\u2149\u214e\u2160-\u2188\u2c00-\u2ce4\u2ceb-\u2cee\u2cf2\u2cf3\u2d00-\u2d25\u2d27\u2d2d\u2d30-\u2d67\u2d6f\u2d80-\u2d96\u2da0-\u2da6\u2da8-\u2dae\u2db0-\u2db6\u2db8-\u2dbe\u2dc0-\u2dc6\u2dc8-\u2dce\u2dd0-\u2dd6\u2dd8-\u2dde\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303c\u3041-\u3096\u309b-\u309f\u30a1-\u30fa\u30fc-\u30ff\u3105-\u312f\u3131-\u318e\u31a0-\u31bf\u31f0-\u31ff\u3400-\u4dbf\u4e00-\ua48c\ua4d0-\ua4fd\ua500-\ua60c\ua610-\ua61f\ua62a\ua62b\ua640-\ua66e\ua67f-\ua69d\ua6a0-\ua6ef\ua717-\ua71f\ua722-\ua788\ua78b-\ua7cd\ua7d0\ua7d1\ua7d3\ua7d5-\ua7dc\ua7f2-\ua801\ua803-\ua805\ua807-\ua80a\ua80c-\ua822\ua840-\ua873\ua882-\ua8b3\ua8f2-\ua8f7\ua8fb\ua8fd\ua8fe\ua90a-\ua925\ua930-\ua946\ua960-\ua97c\ua984-\ua9b2\ua9cf\ua9e0-\ua9e4\ua9e6-\ua9ef\ua9fa-\ua9fe\uaa00-\uaa28\uaa40-\uaa42\uaa44-\uaa4b\uaa60-\uaa76\uaa7a\uaa7e-\uaaaf\uaab1\uaab5\uaab6\uaab9-\uaabd\uaac0\uaac2\uaadb-\uaadd\uaae0-\uaaea\uaaf2-\uaaf4\uab01-\uab06\uab09-\uab0e\uab11-\uab16\uab20-\uab26\uab28-\uab2e\uab30-\uab5a\uab5c-\uab69\uab70-\uabe2\uac00-\ud7a3\ud7b0-\ud7c6\ud7cb-\ud7fb\uf900-\ufa6d\ufa70-\ufad9\ufb00-\ufb06\ufb13-\ufb17\ufb1d\ufb1f-\ufb28\ufb2a-\ufb36\ufb38-\ufb3c\ufb3e\ufb40\ufb41\ufb43\ufb44\ufb46-\ufbb1\ufbd3-\ufd3d\ufd50-\ufd8f\ufd92-\ufdc7\ufdf0-\ufdfb\ufe70-\ufe74\ufe76-\ufefc\uff21-\uff3a\uff41-\uff5a\uff66-\uffbe\uffc2-\uffc7\uffca-\uffcf\uffd2-\uffd7\uffda-\uffdc";
	let nonASCIIidentifierChars = "\xb7\u0300-\u036f\u0387\u0483-\u0487\u0591-\u05bd\u05bf\u05c1\u05c2\u05c4\u05c5\u05c7\u0610-\u061a\u064b-\u0669\u0670\u06d6-\u06dc\u06df-\u06e4\u06e7\u06e8\u06ea-\u06ed\u06f0-\u06f9\u0711\u0730-\u074a\u07a6-\u07b0\u07c0-\u07c9\u07eb-\u07f3\u07fd\u0816-\u0819\u081b-\u0823\u0825-\u0827\u0829-\u082d\u0859-\u085b\u0897-\u089f\u08ca-\u08e1\u08e3-\u0903\u093a-\u093c\u093e-\u094f\u0951-\u0957\u0962\u0963\u0966-\u096f\u0981-\u0983\u09bc\u09be-\u09c4\u09c7\u09c8\u09cb-\u09cd\u09d7\u09e2\u09e3\u09e6-\u09ef\u09fe\u0a01-\u0a03\u0a3c\u0a3e-\u0a42\u0a47\u0a48\u0a4b-\u0a4d\u0a51\u0a66-\u0a71\u0a75\u0a81-\u0a83\u0abc\u0abe-\u0ac5\u0ac7-\u0ac9\u0acb-\u0acd\u0ae2\u0ae3\u0ae6-\u0aef\u0afa-\u0aff\u0b01-\u0b03\u0b3c\u0b3e-\u0b44\u0b47\u0b48\u0b4b-\u0b4d\u0b55-\u0b57\u0b62\u0b63\u0b66-\u0b6f\u0b82\u0bbe-\u0bc2\u0bc6-\u0bc8\u0bca-\u0bcd\u0bd7\u0be6-\u0bef\u0c00-\u0c04\u0c3c\u0c3e-\u0c44\u0c46-\u0c48\u0c4a-\u0c4d\u0c55\u0c56\u0c62\u0c63\u0c66-\u0c6f\u0c81-\u0c83\u0cbc\u0cbe-\u0cc4\u0cc6-\u0cc8\u0cca-\u0ccd\u0cd5\u0cd6\u0ce2\u0ce3\u0ce6-\u0cef\u0cf3\u0d00-\u0d03\u0d3b\u0d3c\u0d3e-\u0d44\u0d46-\u0d48\u0d4a-\u0d4d\u0d57\u0d62\u0d63\u0d66-\u0d6f\u0d81-\u0d83\u0dca\u0dcf-\u0dd4\u0dd6\u0dd8-\u0ddf\u0de6-\u0def\u0df2\u0df3\u0e31\u0e34-\u0e3a\u0e47-\u0e4e\u0e50-\u0e59\u0eb1\u0eb4-\u0ebc\u0ec8-\u0ece\u0ed0-\u0ed9\u0f18\u0f19\u0f20-\u0f29\u0f35\u0f37\u0f39\u0f3e\u0f3f\u0f71-\u0f84\u0f86\u0f87\u0f8d-\u0f97\u0f99-\u0fbc\u0fc6\u102b-\u103e\u1040-\u1049\u1056-\u1059\u105e-\u1060\u1062-\u1064\u1067-\u106d\u1071-\u1074\u1082-\u108d\u108f-\u109d\u135d-\u135f\u1369-\u1371\u1712-\u1715\u1732-\u1734\u1752\u1753\u1772\u1773\u17b4-\u17d3\u17dd\u17e0-\u17e9\u180b-\u180d\u180f-\u1819\u18a9\u1920-\u192b\u1930-\u193b\u1946-\u194f\u19d0-\u19da\u1a17-\u1a1b\u1a55-\u1a5e\u1a60-\u1a7c\u1a7f-\u1a89\u1a90-\u1a99\u1ab0-\u1abd\u1abf-\u1ace\u1b00-\u1b04\u1b34-\u1b44\u1b50-\u1b59\u1b6b-\u1b73\u1b80-\u1b82\u1ba1-\u1bad\u1bb0-\u1bb9\u1be6-\u1bf3\u1c24-\u1c37\u1c40-\u1c49\u1c50-\u1c59\u1cd0-\u1cd2\u1cd4-\u1ce8\u1ced\u1cf4\u1cf7-\u1cf9\u1dc0-\u1dff\u200c\u200d\u203f\u2040\u2054\u20d0-\u20dc\u20e1\u20e5-\u20f0\u2cef-\u2cf1\u2d7f\u2de0-\u2dff\u302a-\u302f\u3099\u309a\u30fb\ua620-\ua629\ua66f\ua674-\ua67d\ua69e\ua69f\ua6f0\ua6f1\ua802\ua806\ua80b\ua823-\ua827\ua82c\ua880\ua881\ua8b4-\ua8c5\ua8d0-\ua8d9\ua8e0-\ua8f1\ua8ff-\ua909\ua926-\ua92d\ua947-\ua953\ua980-\ua983\ua9b3-\ua9c0\ua9d0-\ua9d9\ua9e5\ua9f0-\ua9f9\uaa29-\uaa36\uaa43\uaa4c\uaa4d\uaa50-\uaa59\uaa7b-\uaa7d\uaab0\uaab2-\uaab4\uaab7\uaab8\uaabe\uaabf\uaac1\uaaeb-\uaaef\uaaf5\uaaf6\uabe3-\uabea\uabec\uabed\uabf0-\uabf9\ufb1e\ufe00-\ufe0f\ufe20-\ufe2f\ufe33\ufe34\ufe4d-\ufe4f\uff10-\uff19\uff3f\uff65";
	const nonASCIIidentifierStart = new RegExp("[" + nonASCIIidentifierStartChars + "]");
	const nonASCIIidentifier = new RegExp("[" + nonASCIIidentifierStartChars + nonASCIIidentifierChars + "]");
	nonASCIIidentifierStartChars = nonASCIIidentifierChars = null;
	const astralIdentifierStartCodes = [0, 11, 2, 25, 2, 18, 2, 1, 2, 14, 3, 13, 35, 122, 70, 52, 268, 28, 4, 48, 48, 31, 14, 29, 6, 37, 11, 29, 3, 35, 5, 7, 2, 4, 43, 157, 19, 35, 5, 35, 5, 39, 9, 51, 13, 10, 2, 14, 2, 6, 2, 1, 2, 10, 2, 14, 2, 6, 2, 1, 4, 51, 13, 310, 10, 21, 11, 7, 25, 5, 2, 41, 2, 8, 70, 5, 3, 0, 2, 43, 2, 1, 4, 0, 3, 22, 11, 22, 10, 30, 66, 18, 2, 1, 11, 21, 11, 25, 71, 55, 7, 1, 65, 0, 16, 3, 2, 2, 2, 28, 43, 28, 4, 28, 36, 7, 2, 27, 28, 53, 11, 21, 11, 18, 14, 17, 111, 72, 56, 50, 14, 50, 14, 35, 39, 27, 10, 22, 251, 41, 7, 1, 17, 2, 60, 28, 11, 0, 9, 21, 43, 17, 47, 20, 28, 22, 13, 52, 58, 1, 3, 0, 14, 44, 33, 24, 27, 35, 30, 0, 3, 0, 9, 34, 4, 0, 13, 47, 15, 3, 22, 0, 2, 0, 36, 17, 2, 24, 20, 1, 64, 6, 2, 0, 2, 3, 2, 14, 2, 9, 8, 46, 39, 7, 3, 1, 3, 21, 2, 6, 2, 1, 2, 4, 4, 0, 19, 0, 13, 4, 31, 9, 2, 0, 3, 0, 2, 37, 2, 0, 26, 0, 2, 0, 45, 52, 19, 3, 21, 2, 31, 47, 21, 1, 2, 0, 185, 46, 42, 3, 37, 47, 21, 0, 60, 42, 14, 0, 72, 26, 38, 6, 186, 43, 117, 63, 32, 7, 3, 0, 3, 7, 2, 1, 2, 23, 16, 0, 2, 0, 95, 7, 3, 38, 17, 0, 2, 0, 29, 0, 11, 39, 8, 0, 22, 0, 12, 45, 20, 0, 19, 72, 200, 32, 32, 8, 2, 36, 18, 0, 50, 29, 113, 6, 2, 1, 2, 37, 22, 0, 26, 5, 2, 1, 2, 31, 15, 0, 328, 18, 16, 0, 2, 12, 2, 33, 125, 0, 80, 921, 103, 110, 18, 195, 2637, 96, 16, 1071, 18, 5, 26, 3994, 6, 582, 6842, 29, 1763, 568, 8, 30, 18, 78, 18, 29, 19, 47, 17, 3, 32, 20, 6, 18, 433, 44, 212, 63, 129, 74, 6, 0, 67, 12, 65, 1, 2, 0, 29, 6135, 9, 1237, 42, 9, 8936, 3, 2, 6, 2, 1, 2, 290, 16, 0, 30, 2, 3, 0, 15, 3, 9, 395, 2309, 106, 6, 12, 4, 8, 8, 9, 5991, 84, 2, 70, 2, 1, 3, 0, 3, 1, 3, 3, 2, 11, 2, 0, 2, 6, 2, 64, 2, 3, 3, 7, 2, 6, 2, 27, 2, 3, 2, 4, 2, 0, 4, 6, 2, 339, 3, 24, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 30, 2, 24, 2, 7, 1845, 30, 7, 5, 262, 61, 147, 44, 11, 6, 17, 0, 322, 29, 19, 43, 485, 27, 229, 29, 3, 0, 496, 6, 2, 3, 2, 1, 2, 14, 2, 196, 60, 67, 8, 0, 1205, 3, 2, 26, 2, 1, 2, 0, 3, 0, 2, 9, 2, 3, 2, 0, 2, 0, 7, 0, 5, 0, 2, 0, 2, 0, 2, 2, 2, 1, 2, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 1, 2, 0, 3, 3, 2, 6, 2, 3, 2, 3, 2, 0, 2, 9, 2, 16, 6, 2, 2, 4, 2, 16, 4421, 42719, 33, 4153, 7, 221, 3, 5761, 15, 7472, 16, 621, 2467, 541, 1507, 4938, 6, 4191];
	const astralIdentifierCodes = [509, 0, 227, 0, 150, 4, 294, 9, 1368, 2, 2, 1, 6, 3, 41, 2, 5, 0, 166, 1, 574, 3, 9, 9, 7, 9, 32, 4, 318, 1, 80, 3, 71, 10, 50, 3, 123, 2, 54, 14, 32, 10, 3, 1, 11, 3, 46, 10, 8, 0, 46, 9, 7, 2, 37, 13, 2, 9, 6, 1, 45, 0, 13, 2, 49, 13, 9, 3, 2, 11, 83, 11, 7, 0, 3, 0, 158, 11, 6, 9, 7, 3, 56, 1, 2, 6, 3, 1, 3, 2, 10, 0, 11, 1, 3, 6, 4, 4, 68, 8, 2, 0, 3, 0, 2, 3, 2, 4, 2, 0, 15, 1, 83, 17, 10, 9, 5, 0, 82, 19, 13, 9, 214, 6, 3, 8, 28, 1, 83, 16, 16, 9, 82, 12, 9, 9, 7, 19, 58, 14, 5, 9, 243, 14, 166, 9, 71, 5, 2, 1, 3, 3, 2, 0, 2, 1, 13, 9, 120, 6, 3, 6, 4, 0, 29, 9, 41, 6, 2, 3, 9, 0, 10, 10, 47, 15, 343, 9, 54, 7, 2, 7, 17, 9, 57, 21, 2, 13, 123, 5, 4, 0, 2, 1, 2, 6, 2, 0, 9, 9, 49, 4, 2, 1, 2, 4, 9, 9, 330, 3, 10, 1, 2, 0, 49, 6, 4, 4, 14, 10, 5350, 0, 7, 14, 11465, 27, 2343, 9, 87, 9, 39, 4, 60, 6, 26, 9, 535, 9, 470, 0, 2, 54, 8, 3, 82, 0, 12, 1, 19628, 1, 4178, 9, 519, 45, 3, 22, 543, 4, 4, 5, 9, 7, 3, 6, 31, 3, 149, 2, 1418, 49, 513, 54, 5, 49, 9, 0, 15, 0, 23, 4, 2, 14, 1361, 6, 2, 16, 3, 6, 2, 1, 2, 4, 101, 0, 161, 6, 10, 9, 357, 0, 62, 13, 499, 13, 245, 1, 2, 9, 726, 6, 110, 6, 6, 9, 4759, 9, 787719, 239];
	function isInAstralSet(code, set) {
	  let pos = 0x10000;
	  for (let i = 0, length = set.length; i < length; i += 2) {
	    pos += set[i];
	    if (pos > code) return false;
	    pos += set[i + 1];
	    if (pos >= code) return true;
	  }
	  return false;
	}
	function isIdentifierStart(code) {
	  if (code < 65) return code === 36;
	  if (code <= 90) return true;
	  if (code < 97) return code === 95;
	  if (code <= 122) return true;
	  if (code <= 0xffff) {
	    return code >= 0xaa && nonASCIIidentifierStart.test(String.fromCharCode(code));
	  }
	  return isInAstralSet(code, astralIdentifierStartCodes);
	}
	function isIdentifierChar(code) {
	  if (code < 48) return code === 36;
	  if (code < 58) return true;
	  if (code < 65) return false;
	  if (code <= 90) return true;
	  if (code < 97) return code === 95;
	  if (code <= 122) return true;
	  if (code <= 0xffff) {
	    return code >= 0xaa && nonASCIIidentifier.test(String.fromCharCode(code));
	  }
	  return isInAstralSet(code, astralIdentifierStartCodes) || isInAstralSet(code, astralIdentifierCodes);
	}
	function isIdentifierName(name) {
	  let isFirst = true;
	  for (let i = 0; i < name.length; i++) {
	    let cp = name.charCodeAt(i);
	    if ((cp & 0xfc00) === 0xd800 && i + 1 < name.length) {
	      const trail = name.charCodeAt(++i);
	      if ((trail & 0xfc00) === 0xdc00) {
	        cp = 0x10000 + ((cp & 0x3ff) << 10) + (trail & 0x3ff);
	      }
	    }
	    if (isFirst) {
	      isFirst = false;
	      if (!isIdentifierStart(cp)) {
	        return false;
	      }
	    } else if (!isIdentifierChar(cp)) {
	      return false;
	    }
	  }
	  return !isFirst;
	}

	
	return identifier;
}

var keyword = {};

var hasRequiredKeyword;

function requireKeyword () {
	if (hasRequiredKeyword) return keyword;
	hasRequiredKeyword = 1;

	Object.defineProperty(keyword, "__esModule", {
	  value: true
	});
	keyword.isKeyword = isKeyword;
	keyword.isReservedWord = isReservedWord;
	keyword.isStrictBindOnlyReservedWord = isStrictBindOnlyReservedWord;
	keyword.isStrictBindReservedWord = isStrictBindReservedWord;
	keyword.isStrictReservedWord = isStrictReservedWord;
	const reservedWords = {
	  keyword: ["break", "case", "catch", "continue", "debugger", "default", "do", "else", "finally", "for", "function", "if", "return", "switch", "throw", "try", "var", "const", "while", "with", "new", "this", "super", "class", "extends", "export", "import", "null", "true", "false", "in", "instanceof", "typeof", "void", "delete"],
	  strict: ["implements", "interface", "let", "package", "private", "protected", "public", "static", "yield"],
	  strictBind: ["eval", "arguments"]
	};
	const keywords = new Set(reservedWords.keyword);
	const reservedWordsStrictSet = new Set(reservedWords.strict);
	const reservedWordsStrictBindSet = new Set(reservedWords.strictBind);
	function isReservedWord(word, inModule) {
	  return inModule && word === "await" || word === "enum";
	}
	function isStrictReservedWord(word, inModule) {
	  return isReservedWord(word, inModule) || reservedWordsStrictSet.has(word);
	}
	function isStrictBindOnlyReservedWord(word) {
	  return reservedWordsStrictBindSet.has(word);
	}
	function isStrictBindReservedWord(word, inModule) {
	  return isStrictReservedWord(word, inModule) || isStrictBindOnlyReservedWord(word);
	}
	function isKeyword(word) {
	  return keywords.has(word);
	}

	
	return keyword;
}

var hasRequiredLib$3;

function requireLib$3 () {
	if (hasRequiredLib$3) return lib$2;
	hasRequiredLib$3 = 1;
	(function (exports) {

		Object.defineProperty(exports, "__esModule", {
		  value: true
		});
		Object.defineProperty(exports, "isIdentifierChar", {
		  enumerable: true,
		  get: function () {
		    return _identifier.isIdentifierChar;
		  }
		});
		Object.defineProperty(exports, "isIdentifierName", {
		  enumerable: true,
		  get: function () {
		    return _identifier.isIdentifierName;
		  }
		});
		Object.defineProperty(exports, "isIdentifierStart", {
		  enumerable: true,
		  get: function () {
		    return _identifier.isIdentifierStart;
		  }
		});
		Object.defineProperty(exports, "isKeyword", {
		  enumerable: true,
		  get: function () {
		    return _keyword.isKeyword;
		  }
		});
		Object.defineProperty(exports, "isReservedWord", {
		  enumerable: true,
		  get: function () {
		    return _keyword.isReservedWord;
		  }
		});
		Object.defineProperty(exports, "isStrictBindOnlyReservedWord", {
		  enumerable: true,
		  get: function () {
		    return _keyword.isStrictBindOnlyReservedWord;
		  }
		});
		Object.defineProperty(exports, "isStrictBindReservedWord", {
		  enumerable: true,
		  get: function () {
		    return _keyword.isStrictBindReservedWord;
		  }
		});
		Object.defineProperty(exports, "isStrictReservedWord", {
		  enumerable: true,
		  get: function () {
		    return _keyword.isStrictReservedWord;
		  }
		});
		var _identifier = requireIdentifier();
		var _keyword = requireKeyword();

		
} (lib$2));
	return lib$2;
}

var hasRequiredIsValidIdentifier;

function requireIsValidIdentifier () {
	if (hasRequiredIsValidIdentifier) return isValidIdentifier;
	hasRequiredIsValidIdentifier = 1;

	Object.defineProperty(isValidIdentifier, "__esModule", {
	  value: true
	});
	isValidIdentifier.default = isValidIdentifier$1;
	var _helperValidatorIdentifier = requireLib$3();
	function isValidIdentifier$1(name, reserved = true) {
	  if (typeof name !== "string") return false;
	  if (reserved) {
	    if ((0, _helperValidatorIdentifier.isKeyword)(name) || (0, _helperValidatorIdentifier.isStrictReservedWord)(name, true)) {
	      return false;
	    }
	  }
	  return (0, _helperValidatorIdentifier.isIdentifierName)(name);
	}

	
	return isValidIdentifier;
}

var lib$1 = {};

var hasRequiredLib$2;

function requireLib$2 () {
	if (hasRequiredLib$2) return lib$1;
	hasRequiredLib$2 = 1;

	Object.defineProperty(lib$1, "__esModule", {
	  value: true
	});
	lib$1.readCodePoint = readCodePoint;
	lib$1.readInt = readInt;
	lib$1.readStringContents = readStringContents;
	var _isDigit = function isDigit(code) {
	  return code >= 48 && code <= 57;
	};
	const forbiddenNumericSeparatorSiblings = {
	  decBinOct: new Set([46, 66, 69, 79, 95, 98, 101, 111]),
	  hex: new Set([46, 88, 95, 120])
	};
	const isAllowedNumericSeparatorSibling = {
	  bin: ch => ch === 48 || ch === 49,
	  oct: ch => ch >= 48 && ch <= 55,
	  dec: ch => ch >= 48 && ch <= 57,
	  hex: ch => ch >= 48 && ch <= 57 || ch >= 65 && ch <= 70 || ch >= 97 && ch <= 102
	};
	function readStringContents(type, input, pos, lineStart, curLine, errors) {
	  const initialPos = pos;
	  const initialLineStart = lineStart;
	  const initialCurLine = curLine;
	  let out = "";
	  let firstInvalidLoc = null;
	  let chunkStart = pos;
	  const {
	    length
	  } = input;
	  for (;;) {
	    if (pos >= length) {
	      errors.unterminated(initialPos, initialLineStart, initialCurLine);
	      out += input.slice(chunkStart, pos);
	      break;
	    }
	    const ch = input.charCodeAt(pos);
	    if (isStringEnd(type, ch, input, pos)) {
	      out += input.slice(chunkStart, pos);
	      break;
	    }
	    if (ch === 92) {
	      out += input.slice(chunkStart, pos);
	      const res = readEscapedChar(input, pos, lineStart, curLine, type === "template", errors);
	      if (res.ch === null && !firstInvalidLoc) {
	        firstInvalidLoc = {
	          pos,
	          lineStart,
	          curLine
	        };
	      } else {
	        out += res.ch;
	      }
	      ({
	        pos,
	        lineStart,
	        curLine
	      } = res);
	      chunkStart = pos;
	    } else if (ch === 8232 || ch === 8233) {
	      ++pos;
	      ++curLine;
	      lineStart = pos;
	    } else if (ch === 10 || ch === 13) {
	      if (type === "template") {
	        out += input.slice(chunkStart, pos) + "\n";
	        ++pos;
	        if (ch === 13 && input.charCodeAt(pos) === 10) {
	          ++pos;
	        }
	        ++curLine;
	        chunkStart = lineStart = pos;
	      } else {
	        errors.unterminated(initialPos, initialLineStart, initialCurLine);
	      }
	    } else {
	      ++pos;
	    }
	  }
	  return {
	    pos,
	    str: out,
	    firstInvalidLoc,
	    lineStart,
	    curLine,
	    containsInvalid: !!firstInvalidLoc
	  };
	}
	function isStringEnd(type, ch, input, pos) {
	  if (type === "template") {
	    return ch === 96 || ch === 36 && input.charCodeAt(pos + 1) === 123;
	  }
	  return ch === (type === "double" ? 34 : 39);
	}
	function readEscapedChar(input, pos, lineStart, curLine, inTemplate, errors) {
	  const throwOnInvalid = !inTemplate;
	  pos++;
	  const res = ch => ({
	    pos,
	    ch,
	    lineStart,
	    curLine
	  });
	  const ch = input.charCodeAt(pos++);
	  switch (ch) {
	    case 110:
	      return res("\n");
	    case 114:
	      return res("\r");
	    case 120:
	      {
	        let code;
	        ({
	          code,
	          pos
	        } = readHexChar(input, pos, lineStart, curLine, 2, false, throwOnInvalid, errors));
	        return res(code === null ? null : String.fromCharCode(code));
	      }
	    case 117:
	      {
	        let code;
	        ({
	          code,
	          pos
	        } = readCodePoint(input, pos, lineStart, curLine, throwOnInvalid, errors));
	        return res(code === null ? null : String.fromCodePoint(code));
	      }
	    case 116:
	      return res("\t");
	    case 98:
	      return res("\b");
	    case 118:
	      return res("\u000b");
	    case 102:
	      return res("\f");
	    case 13:
	      if (input.charCodeAt(pos) === 10) {
	        ++pos;
	      }
	    case 10:
	      lineStart = pos;
	      ++curLine;
	    case 8232:
	    case 8233:
	      return res("");
	    case 56:
	    case 57:
	      if (inTemplate) {
	        return res(null);
	      } else {
	        errors.strictNumericEscape(pos - 1, lineStart, curLine);
	      }
	    default:
	      if (ch >= 48 && ch <= 55) {
	        const startPos = pos - 1;
	        const match = /^[0-7]+/.exec(input.slice(startPos, pos + 2));
	        let octalStr = match[0];
	        let octal = parseInt(octalStr, 8);
	        if (octal > 255) {
	          octalStr = octalStr.slice(0, -1);
	          octal = parseInt(octalStr, 8);
	        }
	        pos += octalStr.length - 1;
	        const next = input.charCodeAt(pos);
	        if (octalStr !== "0" || next === 56 || next === 57) {
	          if (inTemplate) {
	            return res(null);
	          } else {
	            errors.strictNumericEscape(startPos, lineStart, curLine);
	          }
	        }
	        return res(String.fromCharCode(octal));
	      }
	      return res(String.fromCharCode(ch));
	  }
	}
	function readHexChar(input, pos, lineStart, curLine, len, forceLen, throwOnInvalid, errors) {
	  const initialPos = pos;
	  let n;
	  ({
	    n,
	    pos
	  } = readInt(input, pos, lineStart, curLine, 16, len, forceLen, false, errors, !throwOnInvalid));
	  if (n === null) {
	    if (throwOnInvalid) {
	      errors.invalidEscapeSequence(initialPos, lineStart, curLine);
	    } else {
	      pos = initialPos - 1;
	    }
	  }
	  return {
	    code: n,
	    pos
	  };
	}
	function readInt(input, pos, lineStart, curLine, radix, len, forceLen, allowNumSeparator, errors, bailOnError) {
	  const start = pos;
	  const forbiddenSiblings = radix === 16 ? forbiddenNumericSeparatorSiblings.hex : forbiddenNumericSeparatorSiblings.decBinOct;
	  const isAllowedSibling = radix === 16 ? isAllowedNumericSeparatorSibling.hex : radix === 10 ? isAllowedNumericSeparatorSibling.dec : radix === 8 ? isAllowedNumericSeparatorSibling.oct : isAllowedNumericSeparatorSibling.bin;
	  let invalid = false;
	  let total = 0;
	  for (let i = 0, e = len == null ? Infinity : len; i < e; ++i) {
	    const code = input.charCodeAt(pos);
	    let val;
	    if (code === 95 && allowNumSeparator !== "bail") {
	      const prev = input.charCodeAt(pos - 1);
	      const next = input.charCodeAt(pos + 1);
	      if (!allowNumSeparator) {
	        if (bailOnError) return {
	          n: null,
	          pos
	        };
	        errors.numericSeparatorInEscapeSequence(pos, lineStart, curLine);
	      } else if (Number.isNaN(next) || !isAllowedSibling(next) || forbiddenSiblings.has(prev) || forbiddenSiblings.has(next)) {
	        if (bailOnError) return {
	          n: null,
	          pos
	        };
	        errors.unexpectedNumericSeparator(pos, lineStart, curLine);
	      }
	      ++pos;
	      continue;
	    }
	    if (code >= 97) {
	      val = code - 97 + 10;
	    } else if (code >= 65) {
	      val = code - 65 + 10;
	    } else if (_isDigit(code)) {
	      val = code - 48;
	    } else {
	      val = Infinity;
	    }
	    if (val >= radix) {
	      if (val <= 9 && bailOnError) {
	        return {
	          n: null,
	          pos
	        };
	      } else if (val <= 9 && errors.invalidDigit(pos, lineStart, curLine, radix)) {
	        val = 0;
	      } else if (forceLen) {
	        val = 0;
	        invalid = true;
	      } else {
	        break;
	      }
	    }
	    ++pos;
	    total = total * radix + val;
	  }
	  if (pos === start || len != null && pos - start !== len || invalid) {
	    return {
	      n: null,
	      pos
	    };
	  }
	  return {
	    n: total,
	    pos
	  };
	}
	function readCodePoint(input, pos, lineStart, curLine, throwOnInvalid, errors) {
	  const ch = input.charCodeAt(pos);
	  let code;
	  if (ch === 123) {
	    ++pos;
	    ({
	      code,
	      pos
	    } = readHexChar(input, pos, lineStart, curLine, input.indexOf("}", pos) - pos, true, throwOnInvalid, errors));
	    ++pos;
	    if (code !== null && code > 0x10ffff) {
	      if (throwOnInvalid) {
	        errors.invalidCodePoint(pos, lineStart, curLine);
	      } else {
	        return {
	          code: null,
	          pos
	        };
	      }
	    }
	  } else {
	    ({
	      code,
	      pos
	    } = readHexChar(input, pos, lineStart, curLine, 4, false, throwOnInvalid, errors));
	  }
	  return {
	    code,
	    pos
	  };
	}

	
	return lib$1;
}

var constants = {};

var hasRequiredConstants;

function requireConstants () {
	if (hasRequiredConstants) return constants;
	hasRequiredConstants = 1;

	Object.defineProperty(constants, "__esModule", {
	  value: true
	});
	constants.UPDATE_OPERATORS = constants.UNARY_OPERATORS = constants.STRING_UNARY_OPERATORS = constants.STATEMENT_OR_BLOCK_KEYS = constants.NUMBER_UNARY_OPERATORS = constants.NUMBER_BINARY_OPERATORS = constants.NOT_LOCAL_BINDING = constants.LOGICAL_OPERATORS = constants.INHERIT_KEYS = constants.FOR_INIT_KEYS = constants.FLATTENABLE_KEYS = constants.EQUALITY_BINARY_OPERATORS = constants.COMPARISON_BINARY_OPERATORS = constants.COMMENT_KEYS = constants.BOOLEAN_UNARY_OPERATORS = constants.BOOLEAN_NUMBER_BINARY_OPERATORS = constants.BOOLEAN_BINARY_OPERATORS = constants.BLOCK_SCOPED_SYMBOL = constants.BINARY_OPERATORS = constants.ASSIGNMENT_OPERATORS = void 0;
	constants.STATEMENT_OR_BLOCK_KEYS = ["consequent", "body", "alternate"];
	constants.FLATTENABLE_KEYS = ["body", "expressions"];
	constants.FOR_INIT_KEYS = ["left", "init"];
	constants.COMMENT_KEYS = ["leadingComments", "trailingComments", "innerComments"];
	const LOGICAL_OPERATORS = constants.LOGICAL_OPERATORS = ["||", "&&", "??"];
	constants.UPDATE_OPERATORS = ["++", "--"];
	const BOOLEAN_NUMBER_BINARY_OPERATORS = constants.BOOLEAN_NUMBER_BINARY_OPERATORS = [">", "<", ">=", "<="];
	const EQUALITY_BINARY_OPERATORS = constants.EQUALITY_BINARY_OPERATORS = ["==", "===", "!=", "!=="];
	const COMPARISON_BINARY_OPERATORS = constants.COMPARISON_BINARY_OPERATORS = [...EQUALITY_BINARY_OPERATORS, "in", "instanceof"];
	const BOOLEAN_BINARY_OPERATORS = constants.BOOLEAN_BINARY_OPERATORS = [...COMPARISON_BINARY_OPERATORS, ...BOOLEAN_NUMBER_BINARY_OPERATORS];
	const NUMBER_BINARY_OPERATORS = constants.NUMBER_BINARY_OPERATORS = ["-", "/", "%", "*", "**", "&", "|", ">>", ">>>", "<<", "^"];
	constants.BINARY_OPERATORS = ["+", ...NUMBER_BINARY_OPERATORS, ...BOOLEAN_BINARY_OPERATORS, "|>"];
	constants.ASSIGNMENT_OPERATORS = ["=", "+=", ...NUMBER_BINARY_OPERATORS.map(op => op + "="), ...LOGICAL_OPERATORS.map(op => op + "=")];
	const BOOLEAN_UNARY_OPERATORS = constants.BOOLEAN_UNARY_OPERATORS = ["delete", "!"];
	const NUMBER_UNARY_OPERATORS = constants.NUMBER_UNARY_OPERATORS = ["+", "-", "~"];
	const STRING_UNARY_OPERATORS = constants.STRING_UNARY_OPERATORS = ["typeof"];
	constants.UNARY_OPERATORS = ["void", "throw", ...BOOLEAN_UNARY_OPERATORS, ...NUMBER_UNARY_OPERATORS, ...STRING_UNARY_OPERATORS];
	constants.INHERIT_KEYS = {
	  optional: ["typeAnnotation", "typeParameters", "returnType"],
	  force: ["start", "loc", "end"]
	};
	constants.BLOCK_SCOPED_SYMBOL = Symbol.for("var used to be block scoped");
	constants.NOT_LOCAL_BINDING = Symbol.for("should not be considered a local binding");

	
	return constants;
}

var utils = {};

var hasRequiredUtils;

function requireUtils () {
	if (hasRequiredUtils) return utils;
	hasRequiredUtils = 1;

	Object.defineProperty(utils, "__esModule", {
	  value: true
	});
	utils.VISITOR_KEYS = utils.NODE_PARENT_VALIDATIONS = utils.NODE_FIELDS = utils.FLIPPED_ALIAS_KEYS = utils.DEPRECATED_KEYS = utils.BUILDER_KEYS = utils.ALIAS_KEYS = void 0;
	utils.arrayOf = arrayOf;
	utils.arrayOfType = arrayOfType;
	utils.assertEach = assertEach;
	utils.assertNodeOrValueType = assertNodeOrValueType;
	utils.assertNodeType = assertNodeType;
	utils.assertOneOf = assertOneOf;
	utils.assertOptionalChainStart = assertOptionalChainStart;
	utils.assertShape = assertShape;
	utils.assertValueType = assertValueType;
	utils.chain = chain;
	utils.default = defineType;
	utils.defineAliasedType = defineAliasedType;
	utils.validate = validate;
	utils.validateArrayOfType = validateArrayOfType;
	utils.validateOptional = validateOptional;
	utils.validateOptionalType = validateOptionalType;
	utils.validateType = validateType;
	var _is = requireIs();
	var _validate = requireValidate();
	const VISITOR_KEYS = utils.VISITOR_KEYS = {};
	const ALIAS_KEYS = utils.ALIAS_KEYS = {};
	const FLIPPED_ALIAS_KEYS = utils.FLIPPED_ALIAS_KEYS = {};
	const NODE_FIELDS = utils.NODE_FIELDS = {};
	const BUILDER_KEYS = utils.BUILDER_KEYS = {};
	const DEPRECATED_KEYS = utils.DEPRECATED_KEYS = {};
	const NODE_PARENT_VALIDATIONS = utils.NODE_PARENT_VALIDATIONS = {};
	function getType(val) {
	  if (Array.isArray(val)) {
	    return "array";
	  } else if (val === null) {
	    return "null";
	  } else {
	    return typeof val;
	  }
	}
	function validate(validate) {
	  return {
	    validate
	  };
	}
	function validateType(...typeNames) {
	  return validate(assertNodeType(...typeNames));
	}
	function validateOptional(validate) {
	  return {
	    validate,
	    optional: true
	  };
	}
	function validateOptionalType(...typeNames) {
	  return {
	    validate: assertNodeType(...typeNames),
	    optional: true
	  };
	}
	function arrayOf(elementType) {
	  return chain(assertValueType("array"), assertEach(elementType));
	}
	function arrayOfType(...typeNames) {
	  return arrayOf(assertNodeType(...typeNames));
	}
	function validateArrayOfType(...typeNames) {
	  return validate(arrayOfType(...typeNames));
	}
	function assertEach(callback) {
	  const childValidator = process.env.BABEL_TYPES_8_BREAKING ? _validate.validateChild : () => {};
	  function validator(node, key, val) {
	    if (!Array.isArray(val)) return;
	    for (let i = 0; i < val.length; i++) {
	      const subkey = `${key}[${i}]`;
	      const v = val[i];
	      callback(node, subkey, v);
	      childValidator(node, subkey, v);
	    }
	  }
	  validator.each = callback;
	  return validator;
	}
	function assertOneOf(...values) {
	  function validate(node, key, val) {
	    if (!values.includes(val)) {
	      throw new TypeError(`Property ${key} expected value to be one of ${JSON.stringify(values)} but got ${JSON.stringify(val)}`);
	    }
	  }
	  validate.oneOf = values;
	  return validate;
	}
	function assertNodeType(...types) {
	  function validate(node, key, val) {
	    for (const type of types) {
	      if ((0, _is.default)(type, val)) {
	        (0, _validate.validateChild)(node, key, val);
	        return;
	      }
	    }
	    throw new TypeError(`Property ${key} of ${node.type} expected node to be of a type ${JSON.stringify(types)} but instead got ${JSON.stringify(val == null ? void 0 : val.type)}`);
	  }
	  validate.oneOfNodeTypes = types;
	  return validate;
	}
	function assertNodeOrValueType(...types) {
	  function validate(node, key, val) {
	    for (const type of types) {
	      if (getType(val) === type || (0, _is.default)(type, val)) {
	        (0, _validate.validateChild)(node, key, val);
	        return;
	      }
	    }
	    throw new TypeError(`Property ${key} of ${node.type} expected node to be of a type ${JSON.stringify(types)} but instead got ${JSON.stringify(val == null ? void 0 : val.type)}`);
	  }
	  validate.oneOfNodeOrValueTypes = types;
	  return validate;
	}
	function assertValueType(type) {
	  function validate(node, key, val) {
	    const valid = getType(val) === type;
	    if (!valid) {
	      throw new TypeError(`Property ${key} expected type of ${type} but got ${getType(val)}`);
	    }
	  }
	  validate.type = type;
	  return validate;
	}
	function assertShape(shape) {
	  function validate(node, key, val) {
	    const errors = [];
	    for (const property of Object.keys(shape)) {
	      try {
	        (0, _validate.validateField)(node, property, val[property], shape[property]);
	      } catch (error) {
	        if (error instanceof TypeError) {
	          errors.push(error.message);
	          continue;
	        }
	        throw error;
	      }
	    }
	    if (errors.length) {
	      throw new TypeError(`Property ${key} of ${node.type} expected to have the following:\n${errors.join("\n")}`);
	    }
	  }
	  validate.shapeOf = shape;
	  return validate;
	}
	function assertOptionalChainStart() {
	  function validate(node) {
	    var _current;
	    let current = node;
	    while (node) {
	      const {
	        type
	      } = current;
	      if (type === "OptionalCallExpression") {
	        if (current.optional) return;
	        current = current.callee;
	        continue;
	      }
	      if (type === "OptionalMemberExpression") {
	        if (current.optional) return;
	        current = current.object;
	        continue;
	      }
	      break;
	    }
	    throw new TypeError(`Non-optional ${node.type} must chain from an optional OptionalMemberExpression or OptionalCallExpression. Found chain from ${(_current = current) == null ? void 0 : _current.type}`);
	  }
	  return validate;
	}
	function chain(...fns) {
	  function validate(...args) {
	    for (const fn of fns) {
	      fn(...args);
	    }
	  }
	  validate.chainOf = fns;
	  if (fns.length >= 2 && "type" in fns[0] && fns[0].type === "array" && !("each" in fns[1])) {
	    throw new Error(`An assertValueType("array") validator can only be followed by an assertEach(...) validator.`);
	  }
	  return validate;
	}
	const validTypeOpts = new Set(["aliases", "builder", "deprecatedAlias", "fields", "inherits", "visitor", "validate"]);
	const validFieldKeys = new Set(["default", "optional", "deprecated", "validate"]);
	const store = {};
	function defineAliasedType(...aliases) {
	  return (type, opts = {}) => {
	    let defined = opts.aliases;
	    if (!defined) {
	      var _store$opts$inherits$, _defined;
	      if (opts.inherits) defined = (_store$opts$inherits$ = store[opts.inherits].aliases) == null ? void 0 : _store$opts$inherits$.slice();
	      (_defined = defined) != null ? _defined : defined = [];
	      opts.aliases = defined;
	    }
	    const additional = aliases.filter(a => !defined.includes(a));
	    defined.unshift(...additional);
	    defineType(type, opts);
	  };
	}
	function defineType(type, opts = {}) {
	  const inherits = opts.inherits && store[opts.inherits] || {};
	  let fields = opts.fields;
	  if (!fields) {
	    fields = {};
	    if (inherits.fields) {
	      const keys = Object.getOwnPropertyNames(inherits.fields);
	      for (const key of keys) {
	        const field = inherits.fields[key];
	        const def = field.default;
	        if (Array.isArray(def) ? def.length > 0 : def && typeof def === "object") {
	          throw new Error("field defaults can only be primitives or empty arrays currently");
	        }
	        fields[key] = {
	          default: Array.isArray(def) ? [] : def,
	          optional: field.optional,
	          deprecated: field.deprecated,
	          validate: field.validate
	        };
	      }
	    }
	  }
	  const visitor = opts.visitor || inherits.visitor || [];
	  const aliases = opts.aliases || inherits.aliases || [];
	  const builder = opts.builder || inherits.builder || opts.visitor || [];
	  for (const k of Object.keys(opts)) {
	    if (!validTypeOpts.has(k)) {
	      throw new Error(`Unknown type option "${k}" on ${type}`);
	    }
	  }
	  if (opts.deprecatedAlias) {
	    DEPRECATED_KEYS[opts.deprecatedAlias] = type;
	  }
	  for (const key of visitor.concat(builder)) {
	    fields[key] = fields[key] || {};
	  }
	  for (const key of Object.keys(fields)) {
	    const field = fields[key];
	    if (field.default !== undefined && !builder.includes(key)) {
	      field.optional = true;
	    }
	    if (field.default === undefined) {
	      field.default = null;
	    } else if (!field.validate && field.default != null) {
	      field.validate = assertValueType(getType(field.default));
	    }
	    for (const k of Object.keys(field)) {
	      if (!validFieldKeys.has(k)) {
	        throw new Error(`Unknown field key "${k}" on ${type}.${key}`);
	      }
	    }
	  }
	  VISITOR_KEYS[type] = opts.visitor = visitor;
	  BUILDER_KEYS[type] = opts.builder = builder;
	  NODE_FIELDS[type] = opts.fields = fields;
	  ALIAS_KEYS[type] = opts.aliases = aliases;
	  aliases.forEach(alias => {
	    FLIPPED_ALIAS_KEYS[alias] = FLIPPED_ALIAS_KEYS[alias] || [];
	    FLIPPED_ALIAS_KEYS[alias].push(type);
	  });
	  if (opts.validate) {
	    NODE_PARENT_VALIDATIONS[type] = opts.validate;
	  }
	  store[type] = opts;
	}

	
	return utils;
}

var hasRequiredCore;

function requireCore () {
	if (hasRequiredCore) return core;
	hasRequiredCore = 1;

	Object.defineProperty(core, "__esModule", {
	  value: true
	});
	core.patternLikeCommon = core.importAttributes = core.functionTypeAnnotationCommon = core.functionDeclarationCommon = core.functionCommon = core.classMethodOrPropertyCommon = core.classMethodOrDeclareMethodCommon = void 0;
	var _is = requireIs();
	var _isValidIdentifier = requireIsValidIdentifier();
	var _helperValidatorIdentifier = requireLib$3();
	var _helperStringParser = requireLib$2();
	var _index = requireConstants();
	var _utils = requireUtils();
	const defineType = (0, _utils.defineAliasedType)("Standardized");
	defineType("ArrayExpression", {
	  fields: {
	    elements: {
	      validate: (0, _utils.arrayOf)((0, _utils.assertNodeOrValueType)("null", "Expression", "SpreadElement")),
	      default: !process.env.BABEL_TYPES_8_BREAKING ? [] : undefined
	    }
	  },
	  visitor: ["elements"],
	  aliases: ["Expression"]
	});
	defineType("AssignmentExpression", {
	  fields: {
	    operator: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertValueType)("string") : Object.assign(function () {
	        const identifier = (0, _utils.assertOneOf)(..._index.ASSIGNMENT_OPERATORS);
	        const pattern = (0, _utils.assertOneOf)("=");
	        return function (node, key, val) {
	          const validator = (0, _is.default)("Pattern", node.left) ? pattern : identifier;
	          validator(node, key, val);
	        };
	      }(), {
	        type: "string"
	      })
	    },
	    left: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertNodeType)("LVal", "OptionalMemberExpression") : (0, _utils.assertNodeType)("Identifier", "MemberExpression", "OptionalMemberExpression", "ArrayPattern", "ObjectPattern", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression")
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  builder: ["operator", "left", "right"],
	  visitor: ["left", "right"],
	  aliases: ["Expression"]
	});
	defineType("BinaryExpression", {
	  builder: ["operator", "left", "right"],
	  fields: {
	    operator: {
	      validate: (0, _utils.assertOneOf)(..._index.BINARY_OPERATORS)
	    },
	    left: {
	      validate: function () {
	        const expression = (0, _utils.assertNodeType)("Expression");
	        const inOp = (0, _utils.assertNodeType)("Expression", "PrivateName");
	        const validator = Object.assign(function (node, key, val) {
	          const validator = node.operator === "in" ? inOp : expression;
	          validator(node, key, val);
	        }, {
	          oneOfNodeTypes: ["Expression", "PrivateName"]
	        });
	        return validator;
	      }()
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  visitor: ["left", "right"],
	  aliases: ["Binary", "Expression"]
	});
	defineType("InterpreterDirective", {
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  }
	});
	defineType("Directive", {
	  visitor: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertNodeType)("DirectiveLiteral")
	    }
	  }
	});
	defineType("DirectiveLiteral", {
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  }
	});
	defineType("BlockStatement", {
	  builder: ["body", "directives"],
	  visitor: ["directives", "body"],
	  fields: {
	    directives: {
	      validate: (0, _utils.arrayOfType)("Directive"),
	      default: []
	    },
	    body: (0, _utils.validateArrayOfType)("Statement")
	  },
	  aliases: ["Scopable", "BlockParent", "Block", "Statement"]
	});
	defineType("BreakStatement", {
	  visitor: ["label"],
	  fields: {
	    label: {
	      validate: (0, _utils.assertNodeType)("Identifier"),
	      optional: true
	    }
	  },
	  aliases: ["Statement", "Terminatorless", "CompletionStatement"]
	});
	defineType("CallExpression", {
	  visitor: ["callee", "arguments", "typeParameters", "typeArguments"],
	  builder: ["callee", "arguments"],
	  aliases: ["Expression"],
	  fields: Object.assign({
	    callee: {
	      validate: (0, _utils.assertNodeType)("Expression", "Super", "V8IntrinsicIdentifier")
	    },
	    arguments: (0, _utils.validateArrayOfType)("Expression", "SpreadElement", "ArgumentPlaceholder")
	  }, !process.env.BABEL_TYPES_8_BREAKING ? {
	    optional: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    }
	  } : {}, {
	    typeArguments: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation"),
	      optional: true
	    },
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TSTypeParameterInstantiation"),
	      optional: true
	    }
	  })
	});
	defineType("CatchClause", {
	  visitor: ["param", "body"],
	  fields: {
	    param: {
	      validate: (0, _utils.assertNodeType)("Identifier", "ArrayPattern", "ObjectPattern"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    }
	  },
	  aliases: ["Scopable", "BlockParent"]
	});
	defineType("ConditionalExpression", {
	  visitor: ["test", "consequent", "alternate"],
	  fields: {
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    consequent: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    alternate: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  aliases: ["Expression", "Conditional"]
	});
	defineType("ContinueStatement", {
	  visitor: ["label"],
	  fields: {
	    label: {
	      validate: (0, _utils.assertNodeType)("Identifier"),
	      optional: true
	    }
	  },
	  aliases: ["Statement", "Terminatorless", "CompletionStatement"]
	});
	defineType("DebuggerStatement", {
	  aliases: ["Statement"]
	});
	defineType("DoWhileStatement", {
	  builder: ["test", "body"],
	  visitor: ["body", "test"],
	  fields: {
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  },
	  aliases: ["Statement", "BlockParent", "Loop", "While", "Scopable"]
	});
	defineType("EmptyStatement", {
	  aliases: ["Statement"]
	});
	defineType("ExpressionStatement", {
	  visitor: ["expression"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  aliases: ["Statement", "ExpressionWrapper"]
	});
	defineType("File", {
	  builder: ["program", "comments", "tokens"],
	  visitor: ["program"],
	  fields: {
	    program: {
	      validate: (0, _utils.assertNodeType)("Program")
	    },
	    comments: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? Object.assign(() => {}, {
	        each: {
	          oneOfNodeTypes: ["CommentBlock", "CommentLine"]
	        }
	      }) : (0, _utils.assertEach)((0, _utils.assertNodeType)("CommentBlock", "CommentLine")),
	      optional: true
	    },
	    tokens: {
	      validate: (0, _utils.assertEach)(Object.assign(() => {}, {
	        type: "any"
	      })),
	      optional: true
	    }
	  }
	});
	defineType("ForInStatement", {
	  visitor: ["left", "right", "body"],
	  aliases: ["Scopable", "Statement", "For", "BlockParent", "Loop", "ForXStatement"],
	  fields: {
	    left: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertNodeType)("VariableDeclaration", "LVal") : (0, _utils.assertNodeType)("VariableDeclaration", "Identifier", "MemberExpression", "ArrayPattern", "ObjectPattern", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression")
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	defineType("ForStatement", {
	  visitor: ["init", "test", "update", "body"],
	  aliases: ["Scopable", "Statement", "For", "BlockParent", "Loop"],
	  fields: {
	    init: {
	      validate: (0, _utils.assertNodeType)("VariableDeclaration", "Expression"),
	      optional: true
	    },
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    update: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	const functionCommon = () => ({
	  params: (0, _utils.validateArrayOfType)("Identifier", "Pattern", "RestElement"),
	  generator: {
	    default: false
	  },
	  async: {
	    default: false
	  }
	});
	core.functionCommon = functionCommon;
	const functionTypeAnnotationCommon = () => ({
	  returnType: {
	    validate: (0, _utils.assertNodeType)("TypeAnnotation", "TSTypeAnnotation", "Noop"),
	    optional: true
	  },
	  typeParameters: {
	    validate: (0, _utils.assertNodeType)("TypeParameterDeclaration", "TSTypeParameterDeclaration", "Noop"),
	    optional: true
	  }
	});
	core.functionTypeAnnotationCommon = functionTypeAnnotationCommon;
	const functionDeclarationCommon = () => Object.assign({}, functionCommon(), {
	  declare: {
	    validate: (0, _utils.assertValueType)("boolean"),
	    optional: true
	  },
	  id: {
	    validate: (0, _utils.assertNodeType)("Identifier"),
	    optional: true
	  }
	});
	core.functionDeclarationCommon = functionDeclarationCommon;
	defineType("FunctionDeclaration", {
	  builder: ["id", "params", "body", "generator", "async"],
	  visitor: ["id", "typeParameters", "params", "returnType", "body"],
	  fields: Object.assign({}, functionDeclarationCommon(), functionTypeAnnotationCommon(), {
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    },
	    predicate: {
	      validate: (0, _utils.assertNodeType)("DeclaredPredicate", "InferredPredicate"),
	      optional: true
	    }
	  }),
	  aliases: ["Scopable", "Function", "BlockParent", "FunctionParent", "Statement", "Pureish", "Declaration"],
	  validate: !process.env.BABEL_TYPES_8_BREAKING ? undefined : function () {
	    const identifier = (0, _utils.assertNodeType)("Identifier");
	    return function (parent, key, node) {
	      if (!(0, _is.default)("ExportDefaultDeclaration", parent)) {
	        identifier(node, "id", node.id);
	      }
	    };
	  }()
	});
	defineType("FunctionExpression", {
	  inherits: "FunctionDeclaration",
	  aliases: ["Scopable", "Function", "BlockParent", "FunctionParent", "Expression", "Pureish"],
	  fields: Object.assign({}, functionCommon(), functionTypeAnnotationCommon(), {
	    id: {
	      validate: (0, _utils.assertNodeType)("Identifier"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    },
	    predicate: {
	      validate: (0, _utils.assertNodeType)("DeclaredPredicate", "InferredPredicate"),
	      optional: true
	    }
	  })
	});
	const patternLikeCommon = () => ({
	  typeAnnotation: {
	    validate: (0, _utils.assertNodeType)("TypeAnnotation", "TSTypeAnnotation", "Noop"),
	    optional: true
	  },
	  optional: {
	    validate: (0, _utils.assertValueType)("boolean"),
	    optional: true
	  },
	  decorators: {
	    validate: (0, _utils.arrayOfType)("Decorator"),
	    optional: true
	  }
	});
	core.patternLikeCommon = patternLikeCommon;
	defineType("Identifier", {
	  builder: ["name"],
	  visitor: ["typeAnnotation", "decorators"],
	  aliases: ["Expression", "PatternLike", "LVal", "TSEntityName"],
	  fields: Object.assign({}, patternLikeCommon(), {
	    name: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertValueType)("string"), Object.assign(function (node, key, val) {
	        if (!(0, _isValidIdentifier.default)(val, false)) {
	          throw new TypeError(`"${val}" is not a valid identifier name`);
	        }
	      }, {
	        type: "string"
	      })) : (0, _utils.assertValueType)("string")
	    }
	  }),
	  validate: process.env.BABEL_TYPES_8_BREAKING ? function (parent, key, node) {
	    const match = /\.(\w+)$/.exec(key);
	    if (!match) return;
	    const [, parentKey] = match;
	    const nonComp = {
	      computed: false
	    };
	    if (parentKey === "property") {
	      if ((0, _is.default)("MemberExpression", parent, nonComp)) return;
	      if ((0, _is.default)("OptionalMemberExpression", parent, nonComp)) return;
	    } else if (parentKey === "key") {
	      if ((0, _is.default)("Property", parent, nonComp)) return;
	      if ((0, _is.default)("Method", parent, nonComp)) return;
	    } else if (parentKey === "exported") {
	      if ((0, _is.default)("ExportSpecifier", parent)) return;
	    } else if (parentKey === "imported") {
	      if ((0, _is.default)("ImportSpecifier", parent, {
	        imported: node
	      })) return;
	    } else if (parentKey === "meta") {
	      if ((0, _is.default)("MetaProperty", parent, {
	        meta: node
	      })) return;
	    }
	    if (((0, _helperValidatorIdentifier.isKeyword)(node.name) || (0, _helperValidatorIdentifier.isReservedWord)(node.name, false)) && node.name !== "this") {
	      throw new TypeError(`"${node.name}" is not a valid identifier`);
	    }
	  } : undefined
	});
	defineType("IfStatement", {
	  visitor: ["test", "consequent", "alternate"],
	  aliases: ["Statement", "Conditional"],
	  fields: {
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    consequent: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    },
	    alternate: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	defineType("LabeledStatement", {
	  visitor: ["label", "body"],
	  aliases: ["Statement"],
	  fields: {
	    label: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	defineType("StringLiteral", {
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  },
	  aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	});
	defineType("NumericLiteral", {
	  builder: ["value"],
	  deprecatedAlias: "NumberLiteral",
	  fields: {
	    value: {
	      validate: (0, _utils.chain)((0, _utils.assertValueType)("number"), Object.assign(function (node, key, val) {
	      }, {
	        type: "number"
	      }))
	    }
	  },
	  aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	});
	defineType("NullLiteral", {
	  aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	});
	defineType("BooleanLiteral", {
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("boolean")
	    }
	  },
	  aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	});
	defineType("RegExpLiteral", {
	  builder: ["pattern", "flags"],
	  deprecatedAlias: "RegexLiteral",
	  aliases: ["Expression", "Pureish", "Literal"],
	  fields: {
	    pattern: {
	      validate: (0, _utils.assertValueType)("string")
	    },
	    flags: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertValueType)("string"), Object.assign(function (node, key, val) {
	        const invalid = /[^gimsuy]/.exec(val);
	        if (invalid) {
	          throw new TypeError(`"${invalid[0]}" is not a valid RegExp flag`);
	        }
	      }, {
	        type: "string"
	      })) : (0, _utils.assertValueType)("string"),
	      default: ""
	    }
	  }
	});
	defineType("LogicalExpression", {
	  builder: ["operator", "left", "right"],
	  visitor: ["left", "right"],
	  aliases: ["Binary", "Expression"],
	  fields: {
	    operator: {
	      validate: (0, _utils.assertOneOf)(..._index.LOGICAL_OPERATORS)
	    },
	    left: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("MemberExpression", {
	  builder: ["object", "property", "computed", ...(!process.env.BABEL_TYPES_8_BREAKING ? ["optional"] : [])],
	  visitor: ["object", "property"],
	  aliases: ["Expression", "LVal"],
	  fields: Object.assign({
	    object: {
	      validate: (0, _utils.assertNodeType)("Expression", "Super")
	    },
	    property: {
	      validate: function () {
	        const normal = (0, _utils.assertNodeType)("Identifier", "PrivateName");
	        const computed = (0, _utils.assertNodeType)("Expression");
	        const validator = function (node, key, val) {
	          const validator = node.computed ? computed : normal;
	          validator(node, key, val);
	        };
	        validator.oneOfNodeTypes = ["Expression", "Identifier", "PrivateName"];
	        return validator;
	      }()
	    },
	    computed: {
	      default: false
	    }
	  }, !process.env.BABEL_TYPES_8_BREAKING ? {
	    optional: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    }
	  } : {})
	});
	defineType("NewExpression", {
	  inherits: "CallExpression"
	});
	defineType("Program", {
	  visitor: ["directives", "body"],
	  builder: ["body", "directives", "sourceType", "interpreter"],
	  fields: {
	    sourceType: {
	      validate: (0, _utils.assertOneOf)("script", "module"),
	      default: "script"
	    },
	    interpreter: {
	      validate: (0, _utils.assertNodeType)("InterpreterDirective"),
	      default: null,
	      optional: true
	    },
	    directives: {
	      validate: (0, _utils.arrayOfType)("Directive"),
	      default: []
	    },
	    body: (0, _utils.validateArrayOfType)("Statement")
	  },
	  aliases: ["Scopable", "BlockParent", "Block"]
	});
	defineType("ObjectExpression", {
	  visitor: ["properties"],
	  aliases: ["Expression"],
	  fields: {
	    properties: (0, _utils.validateArrayOfType)("ObjectMethod", "ObjectProperty", "SpreadElement")
	  }
	});
	defineType("ObjectMethod", {
	  builder: ["kind", "key", "params", "body", "computed", "generator", "async"],
	  visitor: ["decorators", "key", "typeParameters", "params", "returnType", "body"],
	  fields: Object.assign({}, functionCommon(), functionTypeAnnotationCommon(), {
	    kind: Object.assign({
	      validate: (0, _utils.assertOneOf)("method", "get", "set")
	    }, !process.env.BABEL_TYPES_8_BREAKING ? {
	      default: "method"
	    } : {}),
	    computed: {
	      default: false
	    },
	    key: {
	      validate: function () {
	        const normal = (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral");
	        const computed = (0, _utils.assertNodeType)("Expression");
	        const validator = function (node, key, val) {
	          const validator = node.computed ? computed : normal;
	          validator(node, key, val);
	        };
	        validator.oneOfNodeTypes = ["Expression", "Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral"];
	        return validator;
	      }()
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    }
	  }),
	  aliases: ["UserWhitespacable", "Function", "Scopable", "BlockParent", "FunctionParent", "Method", "ObjectMember"]
	});
	defineType("ObjectProperty", {
	  builder: ["key", "value", "computed", "shorthand", ...(!process.env.BABEL_TYPES_8_BREAKING ? ["decorators"] : [])],
	  fields: {
	    computed: {
	      default: false
	    },
	    key: {
	      validate: function () {
	        const normal = (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral", "DecimalLiteral", "PrivateName");
	        const computed = (0, _utils.assertNodeType)("Expression");
	        const validator = Object.assign(function (node, key, val) {
	          const validator = node.computed ? computed : normal;
	          validator(node, key, val);
	        }, {
	          oneOfNodeTypes: ["Expression", "Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral", "DecimalLiteral", "PrivateName"]
	        });
	        return validator;
	      }()
	    },
	    value: {
	      validate: (0, _utils.assertNodeType)("Expression", "PatternLike")
	    },
	    shorthand: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertValueType)("boolean"), Object.assign(function (node, key, shorthand) {
	        if (!shorthand) return;
	        if (node.computed) {
	          throw new TypeError("Property shorthand of ObjectProperty cannot be true if computed is true");
	        }
	        if (!(0, _is.default)("Identifier", node.key)) {
	          throw new TypeError("Property shorthand of ObjectProperty cannot be true if key is not an Identifier");
	        }
	      }, {
	        type: "boolean"
	      })) : (0, _utils.assertValueType)("boolean"),
	      default: false
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    }
	  },
	  visitor: ["key", "value", "decorators"],
	  aliases: ["UserWhitespacable", "Property", "ObjectMember"],
	  validate: !process.env.BABEL_TYPES_8_BREAKING ? undefined : function () {
	    const pattern = (0, _utils.assertNodeType)("Identifier", "Pattern", "TSAsExpression", "TSSatisfiesExpression", "TSNonNullExpression", "TSTypeAssertion");
	    const expression = (0, _utils.assertNodeType)("Expression");
	    return function (parent, key, node) {
	      const validator = (0, _is.default)("ObjectPattern", parent) ? pattern : expression;
	      validator(node, "value", node.value);
	    };
	  }()
	});
	defineType("RestElement", {
	  visitor: ["argument", "typeAnnotation"],
	  builder: ["argument"],
	  aliases: ["LVal", "PatternLike"],
	  deprecatedAlias: "RestProperty",
	  fields: Object.assign({}, patternLikeCommon(), {
	    argument: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertNodeType)("LVal") : (0, _utils.assertNodeType)("Identifier", "ArrayPattern", "ObjectPattern", "MemberExpression", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression")
	    }
	  }),
	  validate: process.env.BABEL_TYPES_8_BREAKING ? function (parent, key) {
	    const match = /(\w+)\[(\d+)\]/.exec(key);
	    if (!match) throw new Error("Internal Babel error: malformed key.");
	    const [, listKey, index] = match;
	    if (parent[listKey].length > +index + 1) {
	      throw new TypeError(`RestElement must be last element of ${listKey}`);
	    }
	  } : undefined
	});
	defineType("ReturnStatement", {
	  visitor: ["argument"],
	  aliases: ["Statement", "Terminatorless", "CompletionStatement"],
	  fields: {
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    }
	  }
	});
	defineType("SequenceExpression", {
	  visitor: ["expressions"],
	  fields: {
	    expressions: (0, _utils.validateArrayOfType)("Expression")
	  },
	  aliases: ["Expression"]
	});
	defineType("ParenthesizedExpression", {
	  visitor: ["expression"],
	  aliases: ["Expression", "ExpressionWrapper"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("SwitchCase", {
	  visitor: ["test", "consequent"],
	  fields: {
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    consequent: (0, _utils.validateArrayOfType)("Statement")
	  }
	});
	defineType("SwitchStatement", {
	  visitor: ["discriminant", "cases"],
	  aliases: ["Statement", "BlockParent", "Scopable"],
	  fields: {
	    discriminant: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    cases: (0, _utils.validateArrayOfType)("SwitchCase")
	  }
	});
	defineType("ThisExpression", {
	  aliases: ["Expression"]
	});
	defineType("ThrowStatement", {
	  visitor: ["argument"],
	  aliases: ["Statement", "Terminatorless", "CompletionStatement"],
	  fields: {
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("TryStatement", {
	  visitor: ["block", "handler", "finalizer"],
	  aliases: ["Statement"],
	  fields: {
	    block: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertNodeType)("BlockStatement"), Object.assign(function (node) {
	        if (!node.handler && !node.finalizer) {
	          throw new TypeError("TryStatement expects either a handler or finalizer, or both");
	        }
	      }, {
	        oneOfNodeTypes: ["BlockStatement"]
	      })) : (0, _utils.assertNodeType)("BlockStatement")
	    },
	    handler: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("CatchClause")
	    },
	    finalizer: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    }
	  }
	});
	defineType("UnaryExpression", {
	  builder: ["operator", "argument", "prefix"],
	  fields: {
	    prefix: {
	      default: true
	    },
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    operator: {
	      validate: (0, _utils.assertOneOf)(..._index.UNARY_OPERATORS)
	    }
	  },
	  visitor: ["argument"],
	  aliases: ["UnaryLike", "Expression"]
	});
	defineType("UpdateExpression", {
	  builder: ["operator", "argument", "prefix"],
	  fields: {
	    prefix: {
	      default: false
	    },
	    argument: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertNodeType)("Expression") : (0, _utils.assertNodeType)("Identifier", "MemberExpression")
	    },
	    operator: {
	      validate: (0, _utils.assertOneOf)(..._index.UPDATE_OPERATORS)
	    }
	  },
	  visitor: ["argument"],
	  aliases: ["Expression"]
	});
	defineType("VariableDeclaration", {
	  builder: ["kind", "declarations"],
	  visitor: ["declarations"],
	  aliases: ["Statement", "Declaration"],
	  fields: {
	    declare: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    kind: {
	      validate: (0, _utils.assertOneOf)("var", "let", "const", "using", "await using")
	    },
	    declarations: (0, _utils.validateArrayOfType)("VariableDeclarator")
	  },
	  validate: process.env.BABEL_TYPES_8_BREAKING ? (() => {
	    const withoutInit = (0, _utils.assertNodeType)("Identifier");
	    return function (parent, key, node) {
	      if ((0, _is.default)("ForXStatement", parent, {
	        left: node
	      })) {
	        if (node.declarations.length !== 1) {
	          throw new TypeError(`Exactly one VariableDeclarator is required in the VariableDeclaration of a ${parent.type}`);
	        }
	      } else {
	        node.declarations.forEach(decl => {
	          if (!decl.init) withoutInit(decl, "id", decl.id);
	        });
	      }
	    };
	  })() : undefined
	});
	defineType("VariableDeclarator", {
	  visitor: ["id", "init"],
	  fields: {
	    id: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertNodeType)("LVal") : (0, _utils.assertNodeType)("Identifier", "ArrayPattern", "ObjectPattern")
	    },
	    definite: {
	      optional: true,
	      validate: (0, _utils.assertValueType)("boolean")
	    },
	    init: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("WhileStatement", {
	  visitor: ["test", "body"],
	  aliases: ["Statement", "BlockParent", "Loop", "While", "Scopable"],
	  fields: {
	    test: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	defineType("WithStatement", {
	  visitor: ["object", "body"],
	  aliases: ["Statement"],
	  fields: {
	    object: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    }
	  }
	});
	defineType("AssignmentPattern", {
	  visitor: ["left", "right", "decorators"],
	  builder: ["left", "right"],
	  aliases: ["Pattern", "PatternLike", "LVal"],
	  fields: Object.assign({}, patternLikeCommon(), {
	    left: {
	      validate: (0, _utils.assertNodeType)("Identifier", "ObjectPattern", "ArrayPattern", "MemberExpression", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression")
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    }
	  })
	});
	defineType("ArrayPattern", {
	  visitor: ["elements", "typeAnnotation"],
	  builder: ["elements"],
	  aliases: ["Pattern", "PatternLike", "LVal"],
	  fields: Object.assign({}, patternLikeCommon(), {
	    elements: {
	      validate: (0, _utils.chain)((0, _utils.assertValueType)("array"), (0, _utils.assertEach)((0, _utils.assertNodeOrValueType)("null", "PatternLike", "LVal")))
	    }
	  })
	});
	defineType("ArrowFunctionExpression", {
	  builder: ["params", "body", "async"],
	  visitor: ["typeParameters", "params", "returnType", "body"],
	  aliases: ["Scopable", "Function", "BlockParent", "FunctionParent", "Expression", "Pureish"],
	  fields: Object.assign({}, functionCommon(), functionTypeAnnotationCommon(), {
	    expression: {
	      validate: (0, _utils.assertValueType)("boolean")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement", "Expression")
	    },
	    predicate: {
	      validate: (0, _utils.assertNodeType)("DeclaredPredicate", "InferredPredicate"),
	      optional: true
	    }
	  })
	});
	defineType("ClassBody", {
	  visitor: ["body"],
	  fields: {
	    body: (0, _utils.validateArrayOfType)("ClassMethod", "ClassPrivateMethod", "ClassProperty", "ClassPrivateProperty", "ClassAccessorProperty", "TSDeclareMethod", "TSIndexSignature", "StaticBlock")
	  }
	});
	defineType("ClassExpression", {
	  builder: ["id", "superClass", "body", "decorators"],
	  visitor: ["decorators", "id", "typeParameters", "superClass", "superTypeParameters", "mixins", "implements", "body"],
	  aliases: ["Scopable", "Class", "Expression"],
	  fields: {
	    id: {
	      validate: (0, _utils.assertNodeType)("Identifier"),
	      optional: true
	    },
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterDeclaration", "TSTypeParameterDeclaration", "Noop"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("ClassBody")
	    },
	    superClass: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    superTypeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation", "TSTypeParameterInstantiation"),
	      optional: true
	    },
	    implements: {
	      validate: (0, _utils.arrayOfType)("TSExpressionWithTypeArguments", "ClassImplements"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    mixins: {
	      validate: (0, _utils.assertNodeType)("InterfaceExtends"),
	      optional: true
	    }
	  }
	});
	defineType("ClassDeclaration", {
	  inherits: "ClassExpression",
	  aliases: ["Scopable", "Class", "Statement", "Declaration"],
	  fields: {
	    id: {
	      validate: (0, _utils.assertNodeType)("Identifier"),
	      optional: true
	    },
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterDeclaration", "TSTypeParameterDeclaration", "Noop"),
	      optional: true
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("ClassBody")
	    },
	    superClass: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    superTypeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation", "TSTypeParameterInstantiation"),
	      optional: true
	    },
	    implements: {
	      validate: (0, _utils.arrayOfType)("TSExpressionWithTypeArguments", "ClassImplements"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    mixins: {
	      validate: (0, _utils.assertNodeType)("InterfaceExtends"),
	      optional: true
	    },
	    declare: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    abstract: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    }
	  },
	  validate: !process.env.BABEL_TYPES_8_BREAKING ? undefined : function () {
	    const identifier = (0, _utils.assertNodeType)("Identifier");
	    return function (parent, key, node) {
	      if (!(0, _is.default)("ExportDefaultDeclaration", parent)) {
	        identifier(node, "id", node.id);
	      }
	    };
	  }()
	});
	const importAttributes = core.importAttributes = {
	  attributes: {
	    optional: true,
	    validate: (0, _utils.arrayOfType)("ImportAttribute")
	  },
	  assertions: {
	    deprecated: true,
	    optional: true,
	    validate: (0, _utils.arrayOfType)("ImportAttribute")
	  }
	};
	defineType("ExportAllDeclaration", {
	  builder: ["source"],
	  visitor: ["source", "attributes", "assertions"],
	  aliases: ["Statement", "Declaration", "ImportOrExportDeclaration", "ExportDeclaration"],
	  fields: Object.assign({
	    source: {
	      validate: (0, _utils.assertNodeType)("StringLiteral")
	    },
	    exportKind: (0, _utils.validateOptional)((0, _utils.assertOneOf)("type", "value"))
	  }, importAttributes)
	});
	defineType("ExportDefaultDeclaration", {
	  visitor: ["declaration"],
	  aliases: ["Statement", "Declaration", "ImportOrExportDeclaration", "ExportDeclaration"],
	  fields: {
	    declaration: (0, _utils.validateType)("TSDeclareFunction", "FunctionDeclaration", "ClassDeclaration", "Expression"),
	    exportKind: (0, _utils.validateOptional)((0, _utils.assertOneOf)("value"))
	  }
	});
	defineType("ExportNamedDeclaration", {
	  builder: ["declaration", "specifiers", "source"],
	  visitor: process.env ? ["declaration", "specifiers", "source", "attributes"] : ["declaration", "specifiers", "source", "attributes", "assertions"],
	  aliases: ["Statement", "Declaration", "ImportOrExportDeclaration", "ExportDeclaration"],
	  fields: Object.assign({
	    declaration: {
	      optional: true,
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertNodeType)("Declaration"), Object.assign(function (node, key, val) {
	        if (val && node.specifiers.length) {
	          throw new TypeError("Only declaration or specifiers is allowed on ExportNamedDeclaration");
	        }
	        if (val && node.source) {
	          throw new TypeError("Cannot export a declaration from a source");
	        }
	      }, {
	        oneOfNodeTypes: ["Declaration"]
	      })) : (0, _utils.assertNodeType)("Declaration")
	    }
	  }, importAttributes, {
	    specifiers: {
	      default: [],
	      validate: (0, _utils.arrayOf)(function () {
	        const sourced = (0, _utils.assertNodeType)("ExportSpecifier", "ExportDefaultSpecifier", "ExportNamespaceSpecifier");
	        const sourceless = (0, _utils.assertNodeType)("ExportSpecifier");
	        if (!process.env.BABEL_TYPES_8_BREAKING) return sourced;
	        return Object.assign(function (node, key, val) {
	          const validator = node.source ? sourced : sourceless;
	          validator(node, key, val);
	        }, {
	          oneOfNodeTypes: ["ExportSpecifier", "ExportDefaultSpecifier", "ExportNamespaceSpecifier"]
	        });
	      }())
	    },
	    source: {
	      validate: (0, _utils.assertNodeType)("StringLiteral"),
	      optional: true
	    },
	    exportKind: (0, _utils.validateOptional)((0, _utils.assertOneOf)("type", "value"))
	  })
	});
	defineType("ExportSpecifier", {
	  visitor: ["local", "exported"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    local: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    },
	    exported: {
	      validate: (0, _utils.assertNodeType)("Identifier", "StringLiteral")
	    },
	    exportKind: {
	      validate: (0, _utils.assertOneOf)("type", "value"),
	      optional: true
	    }
	  }
	});
	defineType("ForOfStatement", {
	  visitor: ["left", "right", "body"],
	  builder: ["left", "right", "body", "await"],
	  aliases: ["Scopable", "Statement", "For", "BlockParent", "Loop", "ForXStatement"],
	  fields: {
	    left: {
	      validate: function () {
	        if (!process.env.BABEL_TYPES_8_BREAKING) {
	          return (0, _utils.assertNodeType)("VariableDeclaration", "LVal");
	        }
	        const declaration = (0, _utils.assertNodeType)("VariableDeclaration");
	        const lval = (0, _utils.assertNodeType)("Identifier", "MemberExpression", "ArrayPattern", "ObjectPattern", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression");
	        return Object.assign(function (node, key, val) {
	          if ((0, _is.default)("VariableDeclaration", val)) {
	            declaration(node, key, val);
	          } else {
	            lval(node, key, val);
	          }
	        }, {
	          oneOfNodeTypes: ["VariableDeclaration", "Identifier", "MemberExpression", "ArrayPattern", "ObjectPattern", "TSAsExpression", "TSSatisfiesExpression", "TSTypeAssertion", "TSNonNullExpression"]
	        });
	      }()
	    },
	    right: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("Statement")
	    },
	    await: {
	      default: false
	    }
	  }
	});
	defineType("ImportDeclaration", {
	  builder: ["specifiers", "source"],
	  visitor: ["specifiers", "source", "attributes", "assertions"],
	  aliases: ["Statement", "Declaration", "ImportOrExportDeclaration"],
	  fields: Object.assign({}, importAttributes, {
	    module: {
	      optional: true,
	      validate: (0, _utils.assertValueType)("boolean")
	    },
	    phase: {
	      default: null,
	      validate: (0, _utils.assertOneOf)("source", "defer")
	    },
	    specifiers: (0, _utils.validateArrayOfType)("ImportSpecifier", "ImportDefaultSpecifier", "ImportNamespaceSpecifier"),
	    source: {
	      validate: (0, _utils.assertNodeType)("StringLiteral")
	    },
	    importKind: {
	      validate: (0, _utils.assertOneOf)("type", "typeof", "value"),
	      optional: true
	    }
	  })
	});
	defineType("ImportDefaultSpecifier", {
	  visitor: ["local"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    local: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	defineType("ImportNamespaceSpecifier", {
	  visitor: ["local"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    local: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	defineType("ImportSpecifier", {
	  visitor: ["imported", "local"],
	  builder: ["local", "imported"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    local: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    },
	    imported: {
	      validate: (0, _utils.assertNodeType)("Identifier", "StringLiteral")
	    },
	    importKind: {
	      validate: (0, _utils.assertOneOf)("type", "typeof", "value"),
	      optional: true
	    }
	  }
	});
	defineType("ImportExpression", {
	  visitor: ["source", "options"],
	  aliases: ["Expression"],
	  fields: {
	    phase: {
	      default: null,
	      validate: (0, _utils.assertOneOf)("source", "defer")
	    },
	    source: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    options: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    }
	  }
	});
	defineType("MetaProperty", {
	  visitor: ["meta", "property"],
	  aliases: ["Expression"],
	  fields: {
	    meta: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertNodeType)("Identifier"), Object.assign(function (node, key, val) {
	        let property;
	        switch (val.name) {
	          case "function":
	            property = "sent";
	            break;
	          case "new":
	            property = "target";
	            break;
	          case "import":
	            property = "meta";
	            break;
	        }
	        if (!(0, _is.default)("Identifier", node.property, {
	          name: property
	        })) {
	          throw new TypeError("Unrecognised MetaProperty");
	        }
	      }, {
	        oneOfNodeTypes: ["Identifier"]
	      })) : (0, _utils.assertNodeType)("Identifier")
	    },
	    property: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	const classMethodOrPropertyCommon = () => ({
	  abstract: {
	    validate: (0, _utils.assertValueType)("boolean"),
	    optional: true
	  },
	  accessibility: {
	    validate: (0, _utils.assertOneOf)("public", "private", "protected"),
	    optional: true
	  },
	  static: {
	    default: false
	  },
	  override: {
	    default: false
	  },
	  computed: {
	    default: false
	  },
	  optional: {
	    validate: (0, _utils.assertValueType)("boolean"),
	    optional: true
	  },
	  key: {
	    validate: (0, _utils.chain)(function () {
	      const normal = (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral");
	      const computed = (0, _utils.assertNodeType)("Expression");
	      return function (node, key, val) {
	        const validator = node.computed ? computed : normal;
	        validator(node, key, val);
	      };
	    }(), (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral", "Expression"))
	  }
	});
	core.classMethodOrPropertyCommon = classMethodOrPropertyCommon;
	const classMethodOrDeclareMethodCommon = () => Object.assign({}, functionCommon(), classMethodOrPropertyCommon(), {
	  params: (0, _utils.validateArrayOfType)("Identifier", "Pattern", "RestElement", "TSParameterProperty"),
	  kind: {
	    validate: (0, _utils.assertOneOf)("get", "set", "method", "constructor"),
	    default: "method"
	  },
	  access: {
	    validate: (0, _utils.chain)((0, _utils.assertValueType)("string"), (0, _utils.assertOneOf)("public", "private", "protected")),
	    optional: true
	  },
	  decorators: {
	    validate: (0, _utils.arrayOfType)("Decorator"),
	    optional: true
	  }
	});
	core.classMethodOrDeclareMethodCommon = classMethodOrDeclareMethodCommon;
	defineType("ClassMethod", {
	  aliases: ["Function", "Scopable", "BlockParent", "FunctionParent", "Method"],
	  builder: ["kind", "key", "params", "body", "computed", "static", "generator", "async"],
	  visitor: ["decorators", "key", "typeParameters", "params", "returnType", "body"],
	  fields: Object.assign({}, classMethodOrDeclareMethodCommon(), functionTypeAnnotationCommon(), {
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    }
	  })
	});
	defineType("ObjectPattern", {
	  visitor: ["properties", "typeAnnotation", "decorators"],
	  builder: ["properties"],
	  aliases: ["Pattern", "PatternLike", "LVal"],
	  fields: Object.assign({}, patternLikeCommon(), {
	    properties: (0, _utils.validateArrayOfType)("RestElement", "ObjectProperty")
	  })
	});
	defineType("SpreadElement", {
	  visitor: ["argument"],
	  aliases: ["UnaryLike"],
	  deprecatedAlias: "SpreadProperty",
	  fields: {
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("Super", {
	  aliases: ["Expression"]
	});
	defineType("TaggedTemplateExpression", {
	  visitor: ["tag", "typeParameters", "quasi"],
	  builder: ["tag", "quasi"],
	  aliases: ["Expression"],
	  fields: {
	    tag: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    quasi: {
	      validate: (0, _utils.assertNodeType)("TemplateLiteral")
	    },
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation", "TSTypeParameterInstantiation"),
	      optional: true
	    }
	  }
	});
	defineType("TemplateElement", {
	  builder: ["value", "tail"],
	  fields: {
	    value: {
	      validate: (0, _utils.chain)((0, _utils.assertShape)({
	        raw: {
	          validate: (0, _utils.assertValueType)("string")
	        },
	        cooked: {
	          validate: (0, _utils.assertValueType)("string"),
	          optional: true
	        }
	      }), function templateElementCookedValidator(node) {
	        const raw = node.value.raw;
	        let unterminatedCalled = false;
	        const error = () => {
	          throw new Error("Internal @babel/types error.");
	        };
	        const {
	          str,
	          firstInvalidLoc
	        } = (0, _helperStringParser.readStringContents)("template", raw, 0, 0, 0, {
	          unterminated() {
	            unterminatedCalled = true;
	          },
	          strictNumericEscape: error,
	          invalidEscapeSequence: error,
	          numericSeparatorInEscapeSequence: error,
	          unexpectedNumericSeparator: error,
	          invalidDigit: error,
	          invalidCodePoint: error
	        });
	        if (!unterminatedCalled) throw new Error("Invalid raw");
	        node.value.cooked = firstInvalidLoc ? null : str;
	      })
	    },
	    tail: {
	      default: false
	    }
	  }
	});
	defineType("TemplateLiteral", {
	  visitor: ["quasis", "expressions"],
	  aliases: ["Expression", "Literal"],
	  fields: {
	    quasis: (0, _utils.validateArrayOfType)("TemplateElement"),
	    expressions: {
	      validate: (0, _utils.chain)((0, _utils.assertValueType)("array"), (0, _utils.assertEach)((0, _utils.assertNodeType)("Expression", "TSType")), function (node, key, val) {
	        if (node.quasis.length !== val.length + 1) {
	          throw new TypeError(`Number of ${node.type} quasis should be exactly one more than the number of expressions.\nExpected ${val.length + 1} quasis but got ${node.quasis.length}`);
	        }
	      })
	    }
	  }
	});
	defineType("YieldExpression", {
	  builder: ["argument", "delegate"],
	  visitor: ["argument"],
	  aliases: ["Expression", "Terminatorless"],
	  fields: {
	    delegate: {
	      validate: process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.chain)((0, _utils.assertValueType)("boolean"), Object.assign(function (node, key, val) {
	        if (val && !node.argument) {
	          throw new TypeError("Property delegate of YieldExpression cannot be true if there is no argument");
	        }
	      }, {
	        type: "boolean"
	      })) : (0, _utils.assertValueType)("boolean"),
	      default: false
	    },
	    argument: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("AwaitExpression", {
	  builder: ["argument"],
	  visitor: ["argument"],
	  aliases: ["Expression", "Terminatorless"],
	  fields: {
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("Import", {
	  aliases: ["Expression"]
	});
	defineType("BigIntLiteral", {
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  },
	  aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	});
	defineType("ExportNamespaceSpecifier", {
	  visitor: ["exported"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    exported: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	defineType("OptionalMemberExpression", {
	  builder: ["object", "property", "computed", "optional"],
	  visitor: ["object", "property"],
	  aliases: ["Expression"],
	  fields: {
	    object: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    property: {
	      validate: function () {
	        const normal = (0, _utils.assertNodeType)("Identifier");
	        const computed = (0, _utils.assertNodeType)("Expression");
	        const validator = Object.assign(function (node, key, val) {
	          const validator = node.computed ? computed : normal;
	          validator(node, key, val);
	        }, {
	          oneOfNodeTypes: ["Expression", "Identifier"]
	        });
	        return validator;
	      }()
	    },
	    computed: {
	      default: false
	    },
	    optional: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertValueType)("boolean") : (0, _utils.chain)((0, _utils.assertValueType)("boolean"), (0, _utils.assertOptionalChainStart)())
	    }
	  }
	});
	defineType("OptionalCallExpression", {
	  visitor: ["callee", "arguments", "typeParameters", "typeArguments"],
	  builder: ["callee", "arguments", "optional"],
	  aliases: ["Expression"],
	  fields: {
	    callee: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    arguments: (0, _utils.validateArrayOfType)("Expression", "SpreadElement", "ArgumentPlaceholder"),
	    optional: {
	      validate: !process.env.BABEL_TYPES_8_BREAKING ? (0, _utils.assertValueType)("boolean") : (0, _utils.chain)((0, _utils.assertValueType)("boolean"), (0, _utils.assertOptionalChainStart)())
	    },
	    typeArguments: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation"),
	      optional: true
	    },
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TSTypeParameterInstantiation"),
	      optional: true
	    }
	  }
	});
	defineType("ClassProperty", {
	  visitor: ["decorators", "key", "typeAnnotation", "value"],
	  builder: ["key", "value", "typeAnnotation", "decorators", "computed", "static"],
	  aliases: ["Property"],
	  fields: Object.assign({}, classMethodOrPropertyCommon(), {
	    value: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    definite: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    typeAnnotation: {
	      validate: (0, _utils.assertNodeType)("TypeAnnotation", "TSTypeAnnotation", "Noop"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    readonly: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    declare: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    variance: {
	      validate: (0, _utils.assertNodeType)("Variance"),
	      optional: true
	    }
	  })
	});
	defineType("ClassAccessorProperty", {
	  visitor: ["decorators", "key", "typeAnnotation", "value"],
	  builder: ["key", "value", "typeAnnotation", "decorators", "computed", "static"],
	  aliases: ["Property", "Accessor"],
	  fields: Object.assign({}, classMethodOrPropertyCommon(), {
	    key: {
	      validate: (0, _utils.chain)(function () {
	        const normal = (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral", "PrivateName");
	        const computed = (0, _utils.assertNodeType)("Expression");
	        return function (node, key, val) {
	          const validator = node.computed ? computed : normal;
	          validator(node, key, val);
	        };
	      }(), (0, _utils.assertNodeType)("Identifier", "StringLiteral", "NumericLiteral", "BigIntLiteral", "Expression", "PrivateName"))
	    },
	    value: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    definite: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    typeAnnotation: {
	      validate: (0, _utils.assertNodeType)("TypeAnnotation", "TSTypeAnnotation", "Noop"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    readonly: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    declare: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    variance: {
	      validate: (0, _utils.assertNodeType)("Variance"),
	      optional: true
	    }
	  })
	});
	defineType("ClassPrivateProperty", {
	  visitor: ["decorators", "key", "typeAnnotation", "value"],
	  builder: ["key", "value", "decorators", "static"],
	  aliases: ["Property", "Private"],
	  fields: {
	    key: {
	      validate: (0, _utils.assertNodeType)("PrivateName")
	    },
	    value: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    },
	    typeAnnotation: {
	      validate: (0, _utils.assertNodeType)("TypeAnnotation", "TSTypeAnnotation", "Noop"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    },
	    static: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      default: false
	    },
	    readonly: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    definite: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    variance: {
	      validate: (0, _utils.assertNodeType)("Variance"),
	      optional: true
	    }
	  }
	});
	defineType("ClassPrivateMethod", {
	  builder: ["kind", "key", "params", "body", "static"],
	  visitor: ["decorators", "key", "typeParameters", "params", "returnType", "body"],
	  aliases: ["Function", "Scopable", "BlockParent", "FunctionParent", "Method", "Private"],
	  fields: Object.assign({}, classMethodOrDeclareMethodCommon(), functionTypeAnnotationCommon(), {
	    kind: {
	      validate: (0, _utils.assertOneOf)("get", "set", "method"),
	      default: "method"
	    },
	    key: {
	      validate: (0, _utils.assertNodeType)("PrivateName")
	    },
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    }
	  })
	});
	defineType("PrivateName", {
	  visitor: ["id"],
	  aliases: ["Private"],
	  fields: {
	    id: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	defineType("StaticBlock", {
	  visitor: ["body"],
	  fields: {
	    body: (0, _utils.validateArrayOfType)("Statement")
	  },
	  aliases: ["Scopable", "BlockParent", "FunctionParent"]
	});

	
	return core;
}

var flow = {};

var hasRequiredFlow;

function requireFlow () {
	if (hasRequiredFlow) return flow;
	hasRequiredFlow = 1;

	var _core = requireCore();
	var _utils = requireUtils();
	const defineType = (0, _utils.defineAliasedType)("Flow");
	const defineInterfaceishType = name => {
	  const isDeclareClass = name === "DeclareClass";
	  defineType(name, {
	    builder: ["id", "typeParameters", "extends", "body"],
	    visitor: ["id", "typeParameters", "extends", ...(isDeclareClass ? ["mixins", "implements"] : []), "body"],
	    aliases: ["FlowDeclaration", "Statement", "Declaration"],
	    fields: Object.assign({
	      id: (0, _utils.validateType)("Identifier"),
	      typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	      extends: (0, _utils.validateOptional)((0, _utils.arrayOfType)("InterfaceExtends"))
	    }, isDeclareClass ? {
	      mixins: (0, _utils.validateOptional)((0, _utils.arrayOfType)("InterfaceExtends")),
	      implements: (0, _utils.validateOptional)((0, _utils.arrayOfType)("ClassImplements"))
	    } : {}, {
	      body: (0, _utils.validateType)("ObjectTypeAnnotation")
	    })
	  });
	};
	defineType("AnyTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("ArrayTypeAnnotation", {
	  visitor: ["elementType"],
	  aliases: ["FlowType"],
	  fields: {
	    elementType: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("BooleanTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("BooleanLiteralTypeAnnotation", {
	  builder: ["value"],
	  aliases: ["FlowType"],
	  fields: {
	    value: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("NullLiteralTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("ClassImplements", {
	  visitor: ["id", "typeParameters"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterInstantiation")
	  }
	});
	defineInterfaceishType("DeclareClass");
	defineType("DeclareFunction", {
	  visitor: ["id"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    predicate: (0, _utils.validateOptionalType)("DeclaredPredicate")
	  }
	});
	defineInterfaceishType("DeclareInterface");
	defineType("DeclareModule", {
	  builder: ["id", "body", "kind"],
	  visitor: ["id", "body"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier", "StringLiteral"),
	    body: (0, _utils.validateType)("BlockStatement"),
	    kind: (0, _utils.validateOptional)((0, _utils.assertOneOf)("CommonJS", "ES"))
	  }
	});
	defineType("DeclareModuleExports", {
	  visitor: ["typeAnnotation"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("TypeAnnotation")
	  }
	});
	defineType("DeclareTypeAlias", {
	  visitor: ["id", "typeParameters", "right"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	    right: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("DeclareOpaqueType", {
	  visitor: ["id", "typeParameters", "supertype"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	    supertype: (0, _utils.validateOptionalType)("FlowType"),
	    impltype: (0, _utils.validateOptionalType)("FlowType")
	  }
	});
	defineType("DeclareVariable", {
	  visitor: ["id"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier")
	  }
	});
	defineType("DeclareExportDeclaration", {
	  visitor: ["declaration", "specifiers", "source", "attributes"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: Object.assign({
	    declaration: (0, _utils.validateOptionalType)("Flow"),
	    specifiers: (0, _utils.validateOptional)((0, _utils.arrayOfType)("ExportSpecifier", "ExportNamespaceSpecifier")),
	    source: (0, _utils.validateOptionalType)("StringLiteral"),
	    default: (0, _utils.validateOptional)((0, _utils.assertValueType)("boolean"))
	  }, _core.importAttributes)
	});
	defineType("DeclareExportAllDeclaration", {
	  visitor: ["source", "attributes"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: Object.assign({
	    source: (0, _utils.validateType)("StringLiteral"),
	    exportKind: (0, _utils.validateOptional)((0, _utils.assertOneOf)("type", "value"))
	  }, _core.importAttributes)
	});
	defineType("DeclaredPredicate", {
	  visitor: ["value"],
	  aliases: ["FlowPredicate"],
	  fields: {
	    value: (0, _utils.validateType)("Flow")
	  }
	});
	defineType("ExistsTypeAnnotation", {
	  aliases: ["FlowType"]
	});
	defineType("FunctionTypeAnnotation", {
	  visitor: ["typeParameters", "params", "rest", "returnType"],
	  aliases: ["FlowType"],
	  fields: {
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	    params: (0, _utils.validateArrayOfType)("FunctionTypeParam"),
	    rest: (0, _utils.validateOptionalType)("FunctionTypeParam"),
	    this: (0, _utils.validateOptionalType)("FunctionTypeParam"),
	    returnType: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("FunctionTypeParam", {
	  visitor: ["name", "typeAnnotation"],
	  fields: {
	    name: (0, _utils.validateOptionalType)("Identifier"),
	    typeAnnotation: (0, _utils.validateType)("FlowType"),
	    optional: (0, _utils.validateOptional)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("GenericTypeAnnotation", {
	  visitor: ["id", "typeParameters"],
	  aliases: ["FlowType"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier", "QualifiedTypeIdentifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterInstantiation")
	  }
	});
	defineType("InferredPredicate", {
	  aliases: ["FlowPredicate"]
	});
	defineType("InterfaceExtends", {
	  visitor: ["id", "typeParameters"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier", "QualifiedTypeIdentifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterInstantiation")
	  }
	});
	defineInterfaceishType("InterfaceDeclaration");
	defineType("InterfaceTypeAnnotation", {
	  visitor: ["extends", "body"],
	  aliases: ["FlowType"],
	  fields: {
	    extends: (0, _utils.validateOptional)((0, _utils.arrayOfType)("InterfaceExtends")),
	    body: (0, _utils.validateType)("ObjectTypeAnnotation")
	  }
	});
	defineType("IntersectionTypeAnnotation", {
	  visitor: ["types"],
	  aliases: ["FlowType"],
	  fields: {
	    types: (0, _utils.validate)((0, _utils.arrayOfType)("FlowType"))
	  }
	});
	defineType("MixedTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("EmptyTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("NullableTypeAnnotation", {
	  visitor: ["typeAnnotation"],
	  aliases: ["FlowType"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("NumberLiteralTypeAnnotation", {
	  builder: ["value"],
	  aliases: ["FlowType"],
	  fields: {
	    value: (0, _utils.validate)((0, _utils.assertValueType)("number"))
	  }
	});
	defineType("NumberTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("ObjectTypeAnnotation", {
	  visitor: ["properties", "indexers", "callProperties", "internalSlots"],
	  aliases: ["FlowType"],
	  builder: ["properties", "indexers", "callProperties", "internalSlots", "exact"],
	  fields: {
	    properties: (0, _utils.validate)((0, _utils.arrayOfType)("ObjectTypeProperty", "ObjectTypeSpreadProperty")),
	    indexers: {
	      validate: (0, _utils.arrayOfType)("ObjectTypeIndexer"),
	      optional: true,
	      default: []
	    },
	    callProperties: {
	      validate: (0, _utils.arrayOfType)("ObjectTypeCallProperty"),
	      optional: true,
	      default: []
	    },
	    internalSlots: {
	      validate: (0, _utils.arrayOfType)("ObjectTypeInternalSlot"),
	      optional: true,
	      default: []
	    },
	    exact: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      default: false
	    },
	    inexact: (0, _utils.validateOptional)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("ObjectTypeInternalSlot", {
	  visitor: ["id", "value"],
	  builder: ["id", "value", "optional", "static", "method"],
	  aliases: ["UserWhitespacable"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    value: (0, _utils.validateType)("FlowType"),
	    optional: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    static: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    method: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("ObjectTypeCallProperty", {
	  visitor: ["value"],
	  aliases: ["UserWhitespacable"],
	  fields: {
	    value: (0, _utils.validateType)("FlowType"),
	    static: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("ObjectTypeIndexer", {
	  visitor: ["variance", "id", "key", "value"],
	  builder: ["id", "key", "value", "variance"],
	  aliases: ["UserWhitespacable"],
	  fields: {
	    id: (0, _utils.validateOptionalType)("Identifier"),
	    key: (0, _utils.validateType)("FlowType"),
	    value: (0, _utils.validateType)("FlowType"),
	    static: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    variance: (0, _utils.validateOptionalType)("Variance")
	  }
	});
	defineType("ObjectTypeProperty", {
	  visitor: ["key", "value", "variance"],
	  aliases: ["UserWhitespacable"],
	  fields: {
	    key: (0, _utils.validateType)("Identifier", "StringLiteral"),
	    value: (0, _utils.validateType)("FlowType"),
	    kind: (0, _utils.validate)((0, _utils.assertOneOf)("init", "get", "set")),
	    static: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    proto: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    optional: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    variance: (0, _utils.validateOptionalType)("Variance"),
	    method: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("ObjectTypeSpreadProperty", {
	  visitor: ["argument"],
	  aliases: ["UserWhitespacable"],
	  fields: {
	    argument: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("OpaqueType", {
	  visitor: ["id", "typeParameters", "supertype", "impltype"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	    supertype: (0, _utils.validateOptionalType)("FlowType"),
	    impltype: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("QualifiedTypeIdentifier", {
	  visitor: ["qualification", "id"],
	  builder: ["id", "qualification"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    qualification: (0, _utils.validateType)("Identifier", "QualifiedTypeIdentifier")
	  }
	});
	defineType("StringLiteralTypeAnnotation", {
	  builder: ["value"],
	  aliases: ["FlowType"],
	  fields: {
	    value: (0, _utils.validate)((0, _utils.assertValueType)("string"))
	  }
	});
	defineType("StringTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("SymbolTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("ThisTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("TupleTypeAnnotation", {
	  visitor: ["types"],
	  aliases: ["FlowType"],
	  fields: {
	    types: (0, _utils.validate)((0, _utils.arrayOfType)("FlowType"))
	  }
	});
	defineType("TypeofTypeAnnotation", {
	  visitor: ["argument"],
	  aliases: ["FlowType"],
	  fields: {
	    argument: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("TypeAlias", {
	  visitor: ["id", "typeParameters", "right"],
	  aliases: ["FlowDeclaration", "Statement", "Declaration"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TypeParameterDeclaration"),
	    right: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("TypeAnnotation", {
	  visitor: ["typeAnnotation"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("TypeCastExpression", {
	  visitor: ["expression", "typeAnnotation"],
	  aliases: ["ExpressionWrapper", "Expression"],
	  fields: {
	    expression: (0, _utils.validateType)("Expression"),
	    typeAnnotation: (0, _utils.validateType)("TypeAnnotation")
	  }
	});
	defineType("TypeParameter", {
	  visitor: ["bound", "default", "variance"],
	  fields: {
	    name: (0, _utils.validate)((0, _utils.assertValueType)("string")),
	    bound: (0, _utils.validateOptionalType)("TypeAnnotation"),
	    default: (0, _utils.validateOptionalType)("FlowType"),
	    variance: (0, _utils.validateOptionalType)("Variance")
	  }
	});
	defineType("TypeParameterDeclaration", {
	  visitor: ["params"],
	  fields: {
	    params: (0, _utils.validate)((0, _utils.arrayOfType)("TypeParameter"))
	  }
	});
	defineType("TypeParameterInstantiation", {
	  visitor: ["params"],
	  fields: {
	    params: (0, _utils.validate)((0, _utils.arrayOfType)("FlowType"))
	  }
	});
	defineType("UnionTypeAnnotation", {
	  visitor: ["types"],
	  aliases: ["FlowType"],
	  fields: {
	    types: (0, _utils.validate)((0, _utils.arrayOfType)("FlowType"))
	  }
	});
	defineType("Variance", {
	  builder: ["kind"],
	  fields: {
	    kind: (0, _utils.validate)((0, _utils.assertOneOf)("minus", "plus"))
	  }
	});
	defineType("VoidTypeAnnotation", {
	  aliases: ["FlowType", "FlowBaseAnnotation"]
	});
	defineType("EnumDeclaration", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "body"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    body: (0, _utils.validateType)("EnumBooleanBody", "EnumNumberBody", "EnumStringBody", "EnumSymbolBody")
	  }
	});
	defineType("EnumBooleanBody", {
	  aliases: ["EnumBody"],
	  visitor: ["members"],
	  fields: {
	    explicitType: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    members: (0, _utils.validateArrayOfType)("EnumBooleanMember"),
	    hasUnknownMembers: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("EnumNumberBody", {
	  aliases: ["EnumBody"],
	  visitor: ["members"],
	  fields: {
	    explicitType: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    members: (0, _utils.validateArrayOfType)("EnumNumberMember"),
	    hasUnknownMembers: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("EnumStringBody", {
	  aliases: ["EnumBody"],
	  visitor: ["members"],
	  fields: {
	    explicitType: (0, _utils.validate)((0, _utils.assertValueType)("boolean")),
	    members: (0, _utils.validateArrayOfType)("EnumStringMember", "EnumDefaultedMember"),
	    hasUnknownMembers: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("EnumSymbolBody", {
	  aliases: ["EnumBody"],
	  visitor: ["members"],
	  fields: {
	    members: (0, _utils.validateArrayOfType)("EnumDefaultedMember"),
	    hasUnknownMembers: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});
	defineType("EnumBooleanMember", {
	  aliases: ["EnumMember"],
	  visitor: ["id"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    init: (0, _utils.validateType)("BooleanLiteral")
	  }
	});
	defineType("EnumNumberMember", {
	  aliases: ["EnumMember"],
	  visitor: ["id", "init"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    init: (0, _utils.validateType)("NumericLiteral")
	  }
	});
	defineType("EnumStringMember", {
	  aliases: ["EnumMember"],
	  visitor: ["id", "init"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier"),
	    init: (0, _utils.validateType)("StringLiteral")
	  }
	});
	defineType("EnumDefaultedMember", {
	  aliases: ["EnumMember"],
	  visitor: ["id"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier")
	  }
	});
	defineType("IndexedAccessType", {
	  visitor: ["objectType", "indexType"],
	  aliases: ["FlowType"],
	  fields: {
	    objectType: (0, _utils.validateType)("FlowType"),
	    indexType: (0, _utils.validateType)("FlowType")
	  }
	});
	defineType("OptionalIndexedAccessType", {
	  visitor: ["objectType", "indexType"],
	  aliases: ["FlowType"],
	  fields: {
	    objectType: (0, _utils.validateType)("FlowType"),
	    indexType: (0, _utils.validateType)("FlowType"),
	    optional: (0, _utils.validate)((0, _utils.assertValueType)("boolean"))
	  }
	});

	
	return flow;
}

var jsx = {};

var hasRequiredJsx;

function requireJsx () {
	if (hasRequiredJsx) return jsx;
	hasRequiredJsx = 1;

	var _utils = requireUtils();
	const defineType = (0, _utils.defineAliasedType)("JSX");
	defineType("JSXAttribute", {
	  visitor: ["name", "value"],
	  aliases: ["Immutable"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier", "JSXNamespacedName")
	    },
	    value: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("JSXElement", "JSXFragment", "StringLiteral", "JSXExpressionContainer")
	    }
	  }
	});
	defineType("JSXClosingElement", {
	  visitor: ["name"],
	  aliases: ["Immutable"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier", "JSXMemberExpression", "JSXNamespacedName")
	    }
	  }
	});
	defineType("JSXElement", {
	  builder: ["openingElement", "closingElement", "children", "selfClosing"],
	  visitor: ["openingElement", "children", "closingElement"],
	  aliases: ["Immutable", "Expression"],
	  fields: Object.assign({
	    openingElement: {
	      validate: (0, _utils.assertNodeType)("JSXOpeningElement")
	    },
	    closingElement: {
	      optional: true,
	      validate: (0, _utils.assertNodeType)("JSXClosingElement")
	    },
	    children: (0, _utils.validateArrayOfType)("JSXText", "JSXExpressionContainer", "JSXSpreadChild", "JSXElement", "JSXFragment")
	  }, {
	    selfClosing: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    }
	  })
	});
	defineType("JSXEmptyExpression", {});
	defineType("JSXExpressionContainer", {
	  visitor: ["expression"],
	  aliases: ["Immutable"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression", "JSXEmptyExpression")
	    }
	  }
	});
	defineType("JSXSpreadChild", {
	  visitor: ["expression"],
	  aliases: ["Immutable"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("JSXIdentifier", {
	  builder: ["name"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  }
	});
	defineType("JSXMemberExpression", {
	  visitor: ["object", "property"],
	  fields: {
	    object: {
	      validate: (0, _utils.assertNodeType)("JSXMemberExpression", "JSXIdentifier")
	    },
	    property: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier")
	    }
	  }
	});
	defineType("JSXNamespacedName", {
	  visitor: ["namespace", "name"],
	  fields: {
	    namespace: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier")
	    },
	    name: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier")
	    }
	  }
	});
	defineType("JSXOpeningElement", {
	  builder: ["name", "attributes", "selfClosing"],
	  visitor: ["name", "attributes"],
	  aliases: ["Immutable"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertNodeType)("JSXIdentifier", "JSXMemberExpression", "JSXNamespacedName")
	    },
	    selfClosing: {
	      default: false
	    },
	    attributes: (0, _utils.validateArrayOfType)("JSXAttribute", "JSXSpreadAttribute"),
	    typeParameters: {
	      validate: (0, _utils.assertNodeType)("TypeParameterInstantiation", "TSTypeParameterInstantiation"),
	      optional: true
	    }
	  }
	});
	defineType("JSXSpreadAttribute", {
	  visitor: ["argument"],
	  fields: {
	    argument: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	defineType("JSXText", {
	  aliases: ["Immutable"],
	  builder: ["value"],
	  fields: {
	    value: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  }
	});
	defineType("JSXFragment", {
	  builder: ["openingFragment", "closingFragment", "children"],
	  visitor: ["openingFragment", "children", "closingFragment"],
	  aliases: ["Immutable", "Expression"],
	  fields: {
	    openingFragment: {
	      validate: (0, _utils.assertNodeType)("JSXOpeningFragment")
	    },
	    closingFragment: {
	      validate: (0, _utils.assertNodeType)("JSXClosingFragment")
	    },
	    children: (0, _utils.validateArrayOfType)("JSXText", "JSXExpressionContainer", "JSXSpreadChild", "JSXElement", "JSXFragment")
	  }
	});
	defineType("JSXOpeningFragment", {
	  aliases: ["Immutable"]
	});
	defineType("JSXClosingFragment", {
	  aliases: ["Immutable"]
	});

	
	return jsx;
}

var misc = {};

var placeholders = {};

var hasRequiredPlaceholders;

function requirePlaceholders () {
	if (hasRequiredPlaceholders) return placeholders;
	hasRequiredPlaceholders = 1;

	Object.defineProperty(placeholders, "__esModule", {
	  value: true
	});
	placeholders.PLACEHOLDERS_FLIPPED_ALIAS = placeholders.PLACEHOLDERS_ALIAS = placeholders.PLACEHOLDERS = void 0;
	var _utils = requireUtils();
	const PLACEHOLDERS = placeholders.PLACEHOLDERS = ["Identifier", "StringLiteral", "Expression", "Statement", "Declaration", "BlockStatement", "ClassBody", "Pattern"];
	const PLACEHOLDERS_ALIAS = placeholders.PLACEHOLDERS_ALIAS = {
	  Declaration: ["Statement"],
	  Pattern: ["PatternLike", "LVal"]
	};
	for (const type of PLACEHOLDERS) {
	  const alias = _utils.ALIAS_KEYS[type];
	  if (alias != null && alias.length) PLACEHOLDERS_ALIAS[type] = alias;
	}
	const PLACEHOLDERS_FLIPPED_ALIAS = placeholders.PLACEHOLDERS_FLIPPED_ALIAS = {};
	Object.keys(PLACEHOLDERS_ALIAS).forEach(type => {
	  PLACEHOLDERS_ALIAS[type].forEach(alias => {
	    if (!hasOwnProperty.call(PLACEHOLDERS_FLIPPED_ALIAS, alias)) {
	      PLACEHOLDERS_FLIPPED_ALIAS[alias] = [];
	    }
	    PLACEHOLDERS_FLIPPED_ALIAS[alias].push(type);
	  });
	});

	
	return placeholders;
}

var hasRequiredMisc;

function requireMisc () {
	if (hasRequiredMisc) return misc;
	hasRequiredMisc = 1;

	var _utils = requireUtils();
	var _placeholders = requirePlaceholders();
	var _core = requireCore();
	const defineType = (0, _utils.defineAliasedType)("Miscellaneous");
	{
	  defineType("Noop", {
	    visitor: []
	  });
	}
	defineType("Placeholder", {
	  visitor: [],
	  builder: ["expectedNode", "name"],
	  fields: Object.assign({
	    name: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    },
	    expectedNode: {
	      validate: (0, _utils.assertOneOf)(..._placeholders.PLACEHOLDERS)
	    }
	  }, (0, _core.patternLikeCommon)())
	});
	defineType("V8IntrinsicIdentifier", {
	  builder: ["name"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertValueType)("string")
	    }
	  }
	});

	
	return misc;
}

var experimental = {};

var hasRequiredExperimental;

function requireExperimental () {
	if (hasRequiredExperimental) return experimental;
	hasRequiredExperimental = 1;

	var _utils = requireUtils();
	(0, _utils.default)("ArgumentPlaceholder", {});
	(0, _utils.default)("BindExpression", {
	  visitor: ["object", "callee"],
	  aliases: ["Expression"],
	  fields: !process.env.BABEL_TYPES_8_BREAKING ? {
	    object: {
	      validate: Object.assign(() => {}, {
	        oneOfNodeTypes: ["Expression"]
	      })
	    },
	    callee: {
	      validate: Object.assign(() => {}, {
	        oneOfNodeTypes: ["Expression"]
	      })
	    }
	  } : {
	    object: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    },
	    callee: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	(0, _utils.default)("ImportAttribute", {
	  visitor: ["key", "value"],
	  fields: {
	    key: {
	      validate: (0, _utils.assertNodeType)("Identifier", "StringLiteral")
	    },
	    value: {
	      validate: (0, _utils.assertNodeType)("StringLiteral")
	    }
	  }
	});
	(0, _utils.default)("Decorator", {
	  visitor: ["expression"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  }
	});
	(0, _utils.default)("DoExpression", {
	  visitor: ["body"],
	  builder: ["body", "async"],
	  aliases: ["Expression"],
	  fields: {
	    body: {
	      validate: (0, _utils.assertNodeType)("BlockStatement")
	    },
	    async: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      default: false
	    }
	  }
	});
	(0, _utils.default)("ExportDefaultSpecifier", {
	  visitor: ["exported"],
	  aliases: ["ModuleSpecifier"],
	  fields: {
	    exported: {
	      validate: (0, _utils.assertNodeType)("Identifier")
	    }
	  }
	});
	(0, _utils.default)("RecordExpression", {
	  visitor: ["properties"],
	  aliases: ["Expression"],
	  fields: {
	    properties: (0, _utils.validateArrayOfType)("ObjectProperty", "SpreadElement")
	  }
	});
	(0, _utils.default)("TupleExpression", {
	  fields: {
	    elements: {
	      validate: (0, _utils.arrayOfType)("Expression", "SpreadElement"),
	      default: []
	    }
	  },
	  visitor: ["elements"],
	  aliases: ["Expression"]
	});
	{
	  (0, _utils.default)("DecimalLiteral", {
	    builder: ["value"],
	    fields: {
	      value: {
	        validate: (0, _utils.assertValueType)("string")
	      }
	    },
	    aliases: ["Expression", "Pureish", "Literal", "Immutable"]
	  });
	}
	(0, _utils.default)("ModuleExpression", {
	  visitor: ["body"],
	  fields: {
	    body: {
	      validate: (0, _utils.assertNodeType)("Program")
	    }
	  },
	  aliases: ["Expression"]
	});
	(0, _utils.default)("TopicReference", {
	  aliases: ["Expression"]
	});
	(0, _utils.default)("PipelineTopicExpression", {
	  builder: ["expression"],
	  visitor: ["expression"],
	  fields: {
	    expression: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  aliases: ["Expression"]
	});
	(0, _utils.default)("PipelineBareFunction", {
	  builder: ["callee"],
	  visitor: ["callee"],
	  fields: {
	    callee: {
	      validate: (0, _utils.assertNodeType)("Expression")
	    }
	  },
	  aliases: ["Expression"]
	});
	(0, _utils.default)("PipelinePrimaryTopicReference", {
	  aliases: ["Expression"]
	});

	
	return experimental;
}

var typescript = {};

var hasRequiredTypescript;

function requireTypescript () {
	if (hasRequiredTypescript) return typescript;
	hasRequiredTypescript = 1;

	var _utils = requireUtils();
	var _core = requireCore();
	var _is = requireIs();
	const defineType = (0, _utils.defineAliasedType)("TypeScript");
	const bool = (0, _utils.assertValueType)("boolean");
	const tSFunctionTypeAnnotationCommon = () => ({
	  returnType: {
	    validate: (0, _utils.assertNodeType)("TSTypeAnnotation", "Noop"),
	    optional: true
	  },
	  typeParameters: {
	    validate: (0, _utils.assertNodeType)("TSTypeParameterDeclaration", "Noop"),
	    optional: true
	  }
	});
	defineType("TSParameterProperty", {
	  aliases: ["LVal"],
	  visitor: ["parameter"],
	  fields: {
	    accessibility: {
	      validate: (0, _utils.assertOneOf)("public", "private", "protected"),
	      optional: true
	    },
	    readonly: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    parameter: {
	      validate: (0, _utils.assertNodeType)("Identifier", "AssignmentPattern")
	    },
	    override: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    decorators: {
	      validate: (0, _utils.arrayOfType)("Decorator"),
	      optional: true
	    }
	  }
	});
	defineType("TSDeclareFunction", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "typeParameters", "params", "returnType"],
	  fields: Object.assign({}, (0, _core.functionDeclarationCommon)(), tSFunctionTypeAnnotationCommon())
	});
	defineType("TSDeclareMethod", {
	  visitor: ["decorators", "key", "typeParameters", "params", "returnType"],
	  fields: Object.assign({}, (0, _core.classMethodOrDeclareMethodCommon)(), tSFunctionTypeAnnotationCommon())
	});
	defineType("TSQualifiedName", {
	  aliases: ["TSEntityName"],
	  visitor: ["left", "right"],
	  fields: {
	    left: (0, _utils.validateType)("TSEntityName"),
	    right: (0, _utils.validateType)("Identifier")
	  }
	});
	const signatureDeclarationCommon = () => ({
	  typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterDeclaration"),
	  ["parameters"]: (0, _utils.validateArrayOfType)("ArrayPattern", "Identifier", "ObjectPattern", "RestElement"),
	  ["typeAnnotation"]: (0, _utils.validateOptionalType)("TSTypeAnnotation")
	});
	const callConstructSignatureDeclaration = {
	  aliases: ["TSTypeElement"],
	  visitor: ["typeParameters", "parameters", "typeAnnotation"],
	  fields: signatureDeclarationCommon()
	};
	defineType("TSCallSignatureDeclaration", callConstructSignatureDeclaration);
	defineType("TSConstructSignatureDeclaration", callConstructSignatureDeclaration);
	const namedTypeElementCommon = () => ({
	  key: (0, _utils.validateType)("Expression"),
	  computed: {
	    default: false
	  },
	  optional: (0, _utils.validateOptional)(bool)
	});
	defineType("TSPropertySignature", {
	  aliases: ["TSTypeElement"],
	  visitor: ["key", "typeAnnotation"],
	  fields: Object.assign({}, namedTypeElementCommon(), {
	    readonly: (0, _utils.validateOptional)(bool),
	    typeAnnotation: (0, _utils.validateOptionalType)("TSTypeAnnotation"),
	    kind: {
	      validate: (0, _utils.assertOneOf)("get", "set")
	    }
	  })
	});
	defineType("TSMethodSignature", {
	  aliases: ["TSTypeElement"],
	  visitor: ["key", "typeParameters", "parameters", "typeAnnotation"],
	  fields: Object.assign({}, signatureDeclarationCommon(), namedTypeElementCommon(), {
	    kind: {
	      validate: (0, _utils.assertOneOf)("method", "get", "set")
	    }
	  })
	});
	defineType("TSIndexSignature", {
	  aliases: ["TSTypeElement"],
	  visitor: ["parameters", "typeAnnotation"],
	  fields: {
	    readonly: (0, _utils.validateOptional)(bool),
	    static: (0, _utils.validateOptional)(bool),
	    parameters: (0, _utils.validateArrayOfType)("Identifier"),
	    typeAnnotation: (0, _utils.validateOptionalType)("TSTypeAnnotation")
	  }
	});
	const tsKeywordTypes = ["TSAnyKeyword", "TSBooleanKeyword", "TSBigIntKeyword", "TSIntrinsicKeyword", "TSNeverKeyword", "TSNullKeyword", "TSNumberKeyword", "TSObjectKeyword", "TSStringKeyword", "TSSymbolKeyword", "TSUndefinedKeyword", "TSUnknownKeyword", "TSVoidKeyword"];
	for (const type of tsKeywordTypes) {
	  defineType(type, {
	    aliases: ["TSType", "TSBaseType"],
	    visitor: [],
	    fields: {}
	  });
	}
	defineType("TSThisType", {
	  aliases: ["TSType", "TSBaseType"],
	  visitor: [],
	  fields: {}
	});
	const fnOrCtrBase = {
	  aliases: ["TSType"],
	  visitor: ["typeParameters", "parameters", "typeAnnotation"]
	};
	defineType("TSFunctionType", Object.assign({}, fnOrCtrBase, {
	  fields: signatureDeclarationCommon()
	}));
	defineType("TSConstructorType", Object.assign({}, fnOrCtrBase, {
	  fields: Object.assign({}, signatureDeclarationCommon(), {
	    abstract: (0, _utils.validateOptional)(bool)
	  })
	}));
	defineType("TSTypeReference", {
	  aliases: ["TSType"],
	  visitor: ["typeName", "typeParameters"],
	  fields: {
	    typeName: (0, _utils.validateType)("TSEntityName"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterInstantiation")
	  }
	});
	defineType("TSTypePredicate", {
	  aliases: ["TSType"],
	  visitor: ["parameterName", "typeAnnotation"],
	  builder: ["parameterName", "typeAnnotation", "asserts"],
	  fields: {
	    parameterName: (0, _utils.validateType)("Identifier", "TSThisType"),
	    typeAnnotation: (0, _utils.validateOptionalType)("TSTypeAnnotation"),
	    asserts: (0, _utils.validateOptional)(bool)
	  }
	});
	defineType("TSTypeQuery", {
	  aliases: ["TSType"],
	  visitor: ["exprName", "typeParameters"],
	  fields: {
	    exprName: (0, _utils.validateType)("TSEntityName", "TSImportType"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterInstantiation")
	  }
	});
	defineType("TSTypeLiteral", {
	  aliases: ["TSType"],
	  visitor: ["members"],
	  fields: {
	    members: (0, _utils.validateArrayOfType)("TSTypeElement")
	  }
	});
	defineType("TSArrayType", {
	  aliases: ["TSType"],
	  visitor: ["elementType"],
	  fields: {
	    elementType: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSTupleType", {
	  aliases: ["TSType"],
	  visitor: ["elementTypes"],
	  fields: {
	    elementTypes: (0, _utils.validateArrayOfType)("TSType", "TSNamedTupleMember")
	  }
	});
	defineType("TSOptionalType", {
	  aliases: ["TSType"],
	  visitor: ["typeAnnotation"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSRestType", {
	  aliases: ["TSType"],
	  visitor: ["typeAnnotation"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSNamedTupleMember", {
	  visitor: ["label", "elementType"],
	  builder: ["label", "elementType", "optional"],
	  fields: {
	    label: (0, _utils.validateType)("Identifier"),
	    optional: {
	      validate: bool,
	      default: false
	    },
	    elementType: (0, _utils.validateType)("TSType")
	  }
	});
	const unionOrIntersection = {
	  aliases: ["TSType"],
	  visitor: ["types"],
	  fields: {
	    types: (0, _utils.validateArrayOfType)("TSType")
	  }
	};
	defineType("TSUnionType", unionOrIntersection);
	defineType("TSIntersectionType", unionOrIntersection);
	defineType("TSConditionalType", {
	  aliases: ["TSType"],
	  visitor: ["checkType", "extendsType", "trueType", "falseType"],
	  fields: {
	    checkType: (0, _utils.validateType)("TSType"),
	    extendsType: (0, _utils.validateType)("TSType"),
	    trueType: (0, _utils.validateType)("TSType"),
	    falseType: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSInferType", {
	  aliases: ["TSType"],
	  visitor: ["typeParameter"],
	  fields: {
	    typeParameter: (0, _utils.validateType)("TSTypeParameter")
	  }
	});
	defineType("TSParenthesizedType", {
	  aliases: ["TSType"],
	  visitor: ["typeAnnotation"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSTypeOperator", {
	  aliases: ["TSType"],
	  visitor: ["typeAnnotation"],
	  fields: {
	    operator: (0, _utils.validate)((0, _utils.assertValueType)("string")),
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSIndexedAccessType", {
	  aliases: ["TSType"],
	  visitor: ["objectType", "indexType"],
	  fields: {
	    objectType: (0, _utils.validateType)("TSType"),
	    indexType: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSMappedType", {
	  aliases: ["TSType"],
	  visitor: ["typeParameter", "nameType", "typeAnnotation"],
	  builder: ["typeParameter", "typeAnnotation", "nameType"],
	  fields: Object.assign({}, {
	    typeParameter: (0, _utils.validateType)("TSTypeParameter")
	  }, {
	    readonly: (0, _utils.validateOptional)((0, _utils.assertOneOf)(true, false, "+", "-")),
	    optional: (0, _utils.validateOptional)((0, _utils.assertOneOf)(true, false, "+", "-")),
	    typeAnnotation: (0, _utils.validateOptionalType)("TSType"),
	    nameType: (0, _utils.validateOptionalType)("TSType")
	  })
	});
	defineType("TSLiteralType", {
	  aliases: ["TSType", "TSBaseType"],
	  visitor: ["literal"],
	  fields: {
	    literal: {
	      validate: function () {
	        const unaryExpression = (0, _utils.assertNodeType)("NumericLiteral", "BigIntLiteral");
	        const unaryOperator = (0, _utils.assertOneOf)("-");
	        const literal = (0, _utils.assertNodeType)("NumericLiteral", "StringLiteral", "BooleanLiteral", "BigIntLiteral", "TemplateLiteral");
	        function validator(parent, key, node) {
	          if ((0, _is.default)("UnaryExpression", node)) {
	            unaryOperator(node, "operator", node.operator);
	            unaryExpression(node, "argument", node.argument);
	          } else {
	            literal(parent, key, node);
	          }
	        }
	        validator.oneOfNodeTypes = ["NumericLiteral", "StringLiteral", "BooleanLiteral", "BigIntLiteral", "TemplateLiteral", "UnaryExpression"];
	        return validator;
	      }()
	    }
	  }
	});
	const expressionWithTypeArguments = {
	  aliases: ["TSType"],
	  visitor: ["expression", "typeParameters"],
	  fields: {
	    expression: (0, _utils.validateType)("TSEntityName"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterInstantiation")
	  }
	};
	{
	  defineType("TSExpressionWithTypeArguments", expressionWithTypeArguments);
	}
	defineType("TSInterfaceDeclaration", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "typeParameters", "extends", "body"],
	  fields: {
	    declare: (0, _utils.validateOptional)(bool),
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterDeclaration"),
	    extends: (0, _utils.validateOptional)((0, _utils.arrayOfType)("TSExpressionWithTypeArguments")),
	    body: (0, _utils.validateType)("TSInterfaceBody")
	  }
	});
	defineType("TSInterfaceBody", {
	  visitor: ["body"],
	  fields: {
	    body: (0, _utils.validateArrayOfType)("TSTypeElement")
	  }
	});
	defineType("TSTypeAliasDeclaration", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "typeParameters", "typeAnnotation"],
	  fields: {
	    declare: (0, _utils.validateOptional)(bool),
	    id: (0, _utils.validateType)("Identifier"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterDeclaration"),
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	});
	defineType("TSInstantiationExpression", {
	  aliases: ["Expression"],
	  visitor: ["expression", "typeParameters"],
	  fields: {
	    expression: (0, _utils.validateType)("Expression"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterInstantiation")
	  }
	});
	const TSTypeExpression = {
	  aliases: ["Expression", "LVal", "PatternLike"],
	  visitor: ["expression", "typeAnnotation"],
	  fields: {
	    expression: (0, _utils.validateType)("Expression"),
	    typeAnnotation: (0, _utils.validateType)("TSType")
	  }
	};
	defineType("TSAsExpression", TSTypeExpression);
	defineType("TSSatisfiesExpression", TSTypeExpression);
	defineType("TSTypeAssertion", {
	  aliases: ["Expression", "LVal", "PatternLike"],
	  visitor: ["typeAnnotation", "expression"],
	  fields: {
	    typeAnnotation: (0, _utils.validateType)("TSType"),
	    expression: (0, _utils.validateType)("Expression")
	  }
	});
	defineType("TSEnumDeclaration", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "members"],
	  fields: {
	    declare: (0, _utils.validateOptional)(bool),
	    const: (0, _utils.validateOptional)(bool),
	    id: (0, _utils.validateType)("Identifier"),
	    members: (0, _utils.validateArrayOfType)("TSEnumMember"),
	    initializer: (0, _utils.validateOptionalType)("Expression")
	  }
	});
	defineType("TSEnumMember", {
	  visitor: ["id", "initializer"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier", "StringLiteral"),
	    initializer: (0, _utils.validateOptionalType)("Expression")
	  }
	});
	defineType("TSModuleDeclaration", {
	  aliases: ["Statement", "Declaration"],
	  visitor: ["id", "body"],
	  fields: Object.assign({
	    kind: {
	      validate: (0, _utils.assertOneOf)("global", "module", "namespace")
	    },
	    declare: (0, _utils.validateOptional)(bool)
	  }, {
	    global: (0, _utils.validateOptional)(bool)
	  }, {
	    id: (0, _utils.validateType)("Identifier", "StringLiteral"),
	    body: (0, _utils.validateType)("TSModuleBlock", "TSModuleDeclaration")
	  })
	});
	defineType("TSModuleBlock", {
	  aliases: ["Scopable", "Block", "BlockParent", "FunctionParent"],
	  visitor: ["body"],
	  fields: {
	    body: (0, _utils.validateArrayOfType)("Statement")
	  }
	});
	defineType("TSImportType", {
	  aliases: ["TSType"],
	  visitor: ["argument", "qualifier", "typeParameters"],
	  fields: {
	    argument: (0, _utils.validateType)("StringLiteral"),
	    qualifier: (0, _utils.validateOptionalType)("TSEntityName"),
	    typeParameters: (0, _utils.validateOptionalType)("TSTypeParameterInstantiation"),
	    options: {
	      validate: (0, _utils.assertNodeType)("Expression"),
	      optional: true
	    }
	  }
	});
	defineType("TSImportEqualsDeclaration", {
	  aliases: ["Statement"],
	  visitor: ["id", "moduleReference"],
	  fields: {
	    isExport: (0, _utils.validate)(bool),
	    id: (0, _utils.validateType)("Identifier"),
	    moduleReference: (0, _utils.validateType)("TSEntityName", "TSExternalModuleReference"),
	    importKind: {
	      validate: (0, _utils.assertOneOf)("type", "value"),
	      optional: true
	    }
	  }
	});
	defineType("TSExternalModuleReference", {
	  visitor: ["expression"],
	  fields: {
	    expression: (0, _utils.validateType)("StringLiteral")
	  }
	});
	defineType("TSNonNullExpression", {
	  aliases: ["Expression", "LVal", "PatternLike"],
	  visitor: ["expression"],
	  fields: {
	    expression: (0, _utils.validateType)("Expression")
	  }
	});
	defineType("TSExportAssignment", {
	  aliases: ["Statement"],
	  visitor: ["expression"],
	  fields: {
	    expression: (0, _utils.validateType)("Expression")
	  }
	});
	defineType("TSNamespaceExportDeclaration", {
	  aliases: ["Statement"],
	  visitor: ["id"],
	  fields: {
	    id: (0, _utils.validateType)("Identifier")
	  }
	});
	defineType("TSTypeAnnotation", {
	  visitor: ["typeAnnotation"],
	  fields: {
	    typeAnnotation: {
	      validate: (0, _utils.assertNodeType)("TSType")
	    }
	  }
	});
	defineType("TSTypeParameterInstantiation", {
	  visitor: ["params"],
	  fields: {
	    params: (0, _utils.validateArrayOfType)("TSType")
	  }
	});
	defineType("TSTypeParameterDeclaration", {
	  visitor: ["params"],
	  fields: {
	    params: (0, _utils.validateArrayOfType)("TSTypeParameter")
	  }
	});
	defineType("TSTypeParameter", {
	  builder: ["constraint", "default", "name"],
	  visitor: ["constraint", "default"],
	  fields: {
	    name: {
	      validate: (0, _utils.assertValueType)("string")
	    },
	    in: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    out: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    const: {
	      validate: (0, _utils.assertValueType)("boolean"),
	      optional: true
	    },
	    constraint: {
	      validate: (0, _utils.assertNodeType)("TSType"),
	      optional: true
	    },
	    default: {
	      validate: (0, _utils.assertNodeType)("TSType"),
	      optional: true
	    }
	  }
	});

	
	return typescript;
}

var deprecatedAliases = {};

var hasRequiredDeprecatedAliases;

function requireDeprecatedAliases () {
	if (hasRequiredDeprecatedAliases) return deprecatedAliases;
	hasRequiredDeprecatedAliases = 1;

	Object.defineProperty(deprecatedAliases, "__esModule", {
	  value: true
	});
	deprecatedAliases.DEPRECATED_ALIASES = void 0;
	deprecatedAliases.DEPRECATED_ALIASES = {
	  ModuleDeclaration: "ImportOrExportDeclaration"
	};

	
	return deprecatedAliases;
}

var hasRequiredDefinitions;

function requireDefinitions () {
	if (hasRequiredDefinitions) return definitions;
	hasRequiredDefinitions = 1;
	(function (exports) {

		Object.defineProperty(exports, "__esModule", {
		  value: true
		});
		Object.defineProperty(exports, "ALIAS_KEYS", {
		  enumerable: true,
		  get: function () {
		    return _utils.ALIAS_KEYS;
		  }
		});
		Object.defineProperty(exports, "BUILDER_KEYS", {
		  enumerable: true,
		  get: function () {
		    return _utils.BUILDER_KEYS;
		  }
		});
		Object.defineProperty(exports, "DEPRECATED_ALIASES", {
		  enumerable: true,
		  get: function () {
		    return _deprecatedAliases.DEPRECATED_ALIASES;
		  }
		});
		Object.defineProperty(exports, "DEPRECATED_KEYS", {
		  enumerable: true,
		  get: function () {
		    return _utils.DEPRECATED_KEYS;
		  }
		});
		Object.defineProperty(exports, "FLIPPED_ALIAS_KEYS", {
		  enumerable: true,
		  get: function () {
		    return _utils.FLIPPED_ALIAS_KEYS;
		  }
		});
		Object.defineProperty(exports, "NODE_FIELDS", {
		  enumerable: true,
		  get: function () {
		    return _utils.NODE_FIELDS;
		  }
		});
		Object.defineProperty(exports, "NODE_PARENT_VALIDATIONS", {
		  enumerable: true,
		  get: function () {
		    return _utils.NODE_PARENT_VALIDATIONS;
		  }
		});
		Object.defineProperty(exports, "PLACEHOLDERS", {
		  enumerable: true,
		  get: function () {
		    return _placeholders.PLACEHOLDERS;
		  }
		});
		Object.defineProperty(exports, "PLACEHOLDERS_ALIAS", {
		  enumerable: true,
		  get: function () {
		    return _placeholders.PLACEHOLDERS_ALIAS;
		  }
		});
		Object.defineProperty(exports, "PLACEHOLDERS_FLIPPED_ALIAS", {
		  enumerable: true,
		  get: function () {
		    return _placeholders.PLACEHOLDERS_FLIPPED_ALIAS;
		  }
		});
		exports.TYPES = void 0;
		Object.defineProperty(exports, "VISITOR_KEYS", {
		  enumerable: true,
		  get: function () {
		    return _utils.VISITOR_KEYS;
		  }
		});
		requireCore();
		requireFlow();
		requireJsx();
		requireMisc();
		requireExperimental();
		requireTypescript();
		var _utils = requireUtils();
		var _placeholders = requirePlaceholders();
		var _deprecatedAliases = requireDeprecatedAliases();
		Object.keys(_deprecatedAliases.DEPRECATED_ALIASES).forEach(deprecatedAlias => {
		  _utils.FLIPPED_ALIAS_KEYS[deprecatedAlias] = _utils.FLIPPED_ALIAS_KEYS[_deprecatedAliases.DEPRECATED_ALIASES[deprecatedAlias]];
		});
		exports.TYPES = [].concat(Object.keys(_utils.VISITOR_KEYS), Object.keys(_utils.FLIPPED_ALIAS_KEYS), Object.keys(_utils.DEPRECATED_KEYS));

		
} (definitions));
	return definitions;
}

var hasRequiredValidate;

function requireValidate () {
	if (hasRequiredValidate) return validate$1;
	hasRequiredValidate = 1;

	Object.defineProperty(validate$1, "__esModule", {
	  value: true
	});
	validate$1.default = validate;
	validate$1.validateChild = validateChild;
	validate$1.validateField = validateField;
	validate$1.validateInternal = validateInternal;
	var _index = requireDefinitions();
	function validate(node, key, val) {
	  if (!node) return;
	  const fields = _index.NODE_FIELDS[node.type];
	  if (!fields) return;
	  const field = fields[key];
	  validateField(node, key, val, field);
	  validateChild(node, key, val);
	}
	function validateInternal(field, node, key, val, maybeNode) {
	  if (!(field != null && field.validate)) return;
	  if (field.optional && val == null) return;
	  field.validate(node, key, val);
	  if (maybeNode) {
	    var _NODE_PARENT_VALIDATI;
	    const type = val.type;
	    if (type == null) return;
	    (_NODE_PARENT_VALIDATI = _index.NODE_PARENT_VALIDATIONS[type]) == null || _NODE_PARENT_VALIDATI.call(_index.NODE_PARENT_VALIDATIONS, node, key, val);
	  }
	}
	function validateField(node, key, val, field) {
	  if (!(field != null && field.validate)) return;
	  if (field.optional && val == null) return;
	  field.validate(node, key, val);
	}
	function validateChild(node, key, val) {
	  var _NODE_PARENT_VALIDATI2;
	  const type = val == null ? void 0 : val.type;
	  if (type == null) return;
	  (_NODE_PARENT_VALIDATI2 = _index.NODE_PARENT_VALIDATIONS[type]) == null || _NODE_PARENT_VALIDATI2.call(_index.NODE_PARENT_VALIDATIONS, node, key, val);
	}

	
	return validate$1;
}

var hasRequiredGenerated$2;

function requireGenerated$2 () {
	if (hasRequiredGenerated$2) return generated$2;
	hasRequiredGenerated$2 = 1;

	Object.defineProperty(generated$2, "__esModule", {
	  value: true
	});
	generated$2.anyTypeAnnotation = anyTypeAnnotation;
	generated$2.argumentPlaceholder = argumentPlaceholder;
	generated$2.arrayExpression = arrayExpression;
	generated$2.arrayPattern = arrayPattern;
	generated$2.arrayTypeAnnotation = arrayTypeAnnotation;
	generated$2.arrowFunctionExpression = arrowFunctionExpression;
	generated$2.assignmentExpression = assignmentExpression;
	generated$2.assignmentPattern = assignmentPattern;
	generated$2.awaitExpression = awaitExpression;
	generated$2.bigIntLiteral = bigIntLiteral;
	generated$2.binaryExpression = binaryExpression;
	generated$2.bindExpression = bindExpression;
	generated$2.blockStatement = blockStatement;
	generated$2.booleanLiteral = booleanLiteral;
	generated$2.booleanLiteralTypeAnnotation = booleanLiteralTypeAnnotation;
	generated$2.booleanTypeAnnotation = booleanTypeAnnotation;
	generated$2.breakStatement = breakStatement;
	generated$2.callExpression = callExpression;
	generated$2.catchClause = catchClause;
	generated$2.classAccessorProperty = classAccessorProperty;
	generated$2.classBody = classBody;
	generated$2.classDeclaration = classDeclaration;
	generated$2.classExpression = classExpression;
	generated$2.classImplements = classImplements;
	generated$2.classMethod = classMethod;
	generated$2.classPrivateMethod = classPrivateMethod;
	generated$2.classPrivateProperty = classPrivateProperty;
	generated$2.classProperty = classProperty;
	generated$2.conditionalExpression = conditionalExpression;
	generated$2.continueStatement = continueStatement;
	generated$2.debuggerStatement = debuggerStatement;
	generated$2.decimalLiteral = decimalLiteral;
	generated$2.declareClass = declareClass;
	generated$2.declareExportAllDeclaration = declareExportAllDeclaration;
	generated$2.declareExportDeclaration = declareExportDeclaration;
	generated$2.declareFunction = declareFunction;
	generated$2.declareInterface = declareInterface;
	generated$2.declareModule = declareModule;
	generated$2.declareModuleExports = declareModuleExports;
	generated$2.declareOpaqueType = declareOpaqueType;
	generated$2.declareTypeAlias = declareTypeAlias;
	generated$2.declareVariable = declareVariable;
	generated$2.declaredPredicate = declaredPredicate;
	generated$2.decorator = decorator;
	generated$2.directive = directive;
	generated$2.directiveLiteral = directiveLiteral;
	generated$2.doExpression = doExpression;
	generated$2.doWhileStatement = doWhileStatement;
	generated$2.emptyStatement = emptyStatement;
	generated$2.emptyTypeAnnotation = emptyTypeAnnotation;
	generated$2.enumBooleanBody = enumBooleanBody;
	generated$2.enumBooleanMember = enumBooleanMember;
	generated$2.enumDeclaration = enumDeclaration;
	generated$2.enumDefaultedMember = enumDefaultedMember;
	generated$2.enumNumberBody = enumNumberBody;
	generated$2.enumNumberMember = enumNumberMember;
	generated$2.enumStringBody = enumStringBody;
	generated$2.enumStringMember = enumStringMember;
	generated$2.enumSymbolBody = enumSymbolBody;
	generated$2.existsTypeAnnotation = existsTypeAnnotation;
	generated$2.exportAllDeclaration = exportAllDeclaration;
	generated$2.exportDefaultDeclaration = exportDefaultDeclaration;
	generated$2.exportDefaultSpecifier = exportDefaultSpecifier;
	generated$2.exportNamedDeclaration = exportNamedDeclaration;
	generated$2.exportNamespaceSpecifier = exportNamespaceSpecifier;
	generated$2.exportSpecifier = exportSpecifier;
	generated$2.expressionStatement = expressionStatement;
	generated$2.file = file;
	generated$2.forInStatement = forInStatement;
	generated$2.forOfStatement = forOfStatement;
	generated$2.forStatement = forStatement;
	generated$2.functionDeclaration = functionDeclaration;
	generated$2.functionExpression = functionExpression;
	generated$2.functionTypeAnnotation = functionTypeAnnotation;
	generated$2.functionTypeParam = functionTypeParam;
	generated$2.genericTypeAnnotation = genericTypeAnnotation;
	generated$2.identifier = identifier;
	generated$2.ifStatement = ifStatement;
	generated$2.import = _import;
	generated$2.importAttribute = importAttribute;
	generated$2.importDeclaration = importDeclaration;
	generated$2.importDefaultSpecifier = importDefaultSpecifier;
	generated$2.importExpression = importExpression;
	generated$2.importNamespaceSpecifier = importNamespaceSpecifier;
	generated$2.importSpecifier = importSpecifier;
	generated$2.indexedAccessType = indexedAccessType;
	generated$2.inferredPredicate = inferredPredicate;
	generated$2.interfaceDeclaration = interfaceDeclaration;
	generated$2.interfaceExtends = interfaceExtends;
	generated$2.interfaceTypeAnnotation = interfaceTypeAnnotation;
	generated$2.interpreterDirective = interpreterDirective;
	generated$2.intersectionTypeAnnotation = intersectionTypeAnnotation;
	generated$2.jSXAttribute = generated$2.jsxAttribute = jsxAttribute;
	generated$2.jSXClosingElement = generated$2.jsxClosingElement = jsxClosingElement;
	generated$2.jSXClosingFragment = generated$2.jsxClosingFragment = jsxClosingFragment;
	generated$2.jSXElement = generated$2.jsxElement = jsxElement;
	generated$2.jSXEmptyExpression = generated$2.jsxEmptyExpression = jsxEmptyExpression;
	generated$2.jSXExpressionContainer = generated$2.jsxExpressionContainer = jsxExpressionContainer;
	generated$2.jSXFragment = generated$2.jsxFragment = jsxFragment;
	generated$2.jSXIdentifier = generated$2.jsxIdentifier = jsxIdentifier;
	generated$2.jSXMemberExpression = generated$2.jsxMemberExpression = jsxMemberExpression;
	generated$2.jSXNamespacedName = generated$2.jsxNamespacedName = jsxNamespacedName;
	generated$2.jSXOpeningElement = generated$2.jsxOpeningElement = jsxOpeningElement;
	generated$2.jSXOpeningFragment = generated$2.jsxOpeningFragment = jsxOpeningFragment;
	generated$2.jSXSpreadAttribute = generated$2.jsxSpreadAttribute = jsxSpreadAttribute;
	generated$2.jSXSpreadChild = generated$2.jsxSpreadChild = jsxSpreadChild;
	generated$2.jSXText = generated$2.jsxText = jsxText;
	generated$2.labeledStatement = labeledStatement;
	generated$2.logicalExpression = logicalExpression;
	generated$2.memberExpression = memberExpression;
	generated$2.metaProperty = metaProperty;
	generated$2.mixedTypeAnnotation = mixedTypeAnnotation;
	generated$2.moduleExpression = moduleExpression;
	generated$2.newExpression = newExpression;
	generated$2.noop = noop;
	generated$2.nullLiteral = nullLiteral;
	generated$2.nullLiteralTypeAnnotation = nullLiteralTypeAnnotation;
	generated$2.nullableTypeAnnotation = nullableTypeAnnotation;
	generated$2.numberLiteral = NumberLiteral;
	generated$2.numberLiteralTypeAnnotation = numberLiteralTypeAnnotation;
	generated$2.numberTypeAnnotation = numberTypeAnnotation;
	generated$2.numericLiteral = numericLiteral;
	generated$2.objectExpression = objectExpression;
	generated$2.objectMethod = objectMethod;
	generated$2.objectPattern = objectPattern;
	generated$2.objectProperty = objectProperty;
	generated$2.objectTypeAnnotation = objectTypeAnnotation;
	generated$2.objectTypeCallProperty = objectTypeCallProperty;
	generated$2.objectTypeIndexer = objectTypeIndexer;
	generated$2.objectTypeInternalSlot = objectTypeInternalSlot;
	generated$2.objectTypeProperty = objectTypeProperty;
	generated$2.objectTypeSpreadProperty = objectTypeSpreadProperty;
	generated$2.opaqueType = opaqueType;
	generated$2.optionalCallExpression = optionalCallExpression;
	generated$2.optionalIndexedAccessType = optionalIndexedAccessType;
	generated$2.optionalMemberExpression = optionalMemberExpression;
	generated$2.parenthesizedExpression = parenthesizedExpression;
	generated$2.pipelineBareFunction = pipelineBareFunction;
	generated$2.pipelinePrimaryTopicReference = pipelinePrimaryTopicReference;
	generated$2.pipelineTopicExpression = pipelineTopicExpression;
	generated$2.placeholder = placeholder;
	generated$2.privateName = privateName;
	generated$2.program = program;
	generated$2.qualifiedTypeIdentifier = qualifiedTypeIdentifier;
	generated$2.recordExpression = recordExpression;
	generated$2.regExpLiteral = regExpLiteral;
	generated$2.regexLiteral = RegexLiteral;
	generated$2.restElement = restElement;
	generated$2.restProperty = RestProperty;
	generated$2.returnStatement = returnStatement;
	generated$2.sequenceExpression = sequenceExpression;
	generated$2.spreadElement = spreadElement;
	generated$2.spreadProperty = SpreadProperty;
	generated$2.staticBlock = staticBlock;
	generated$2.stringLiteral = stringLiteral;
	generated$2.stringLiteralTypeAnnotation = stringLiteralTypeAnnotation;
	generated$2.stringTypeAnnotation = stringTypeAnnotation;
	generated$2.super = _super;
	generated$2.switchCase = switchCase;
	generated$2.switchStatement = switchStatement;
	generated$2.symbolTypeAnnotation = symbolTypeAnnotation;
	generated$2.taggedTemplateExpression = taggedTemplateExpression;
	generated$2.templateElement = templateElement;
	generated$2.templateLiteral = templateLiteral;
	generated$2.thisExpression = thisExpression;
	generated$2.thisTypeAnnotation = thisTypeAnnotation;
	generated$2.throwStatement = throwStatement;
	generated$2.topicReference = topicReference;
	generated$2.tryStatement = tryStatement;
	generated$2.tSAnyKeyword = generated$2.tsAnyKeyword = tsAnyKeyword;
	generated$2.tSArrayType = generated$2.tsArrayType = tsArrayType;
	generated$2.tSAsExpression = generated$2.tsAsExpression = tsAsExpression;
	generated$2.tSBigIntKeyword = generated$2.tsBigIntKeyword = tsBigIntKeyword;
	generated$2.tSBooleanKeyword = generated$2.tsBooleanKeyword = tsBooleanKeyword;
	generated$2.tSCallSignatureDeclaration = generated$2.tsCallSignatureDeclaration = tsCallSignatureDeclaration;
	generated$2.tSConditionalType = generated$2.tsConditionalType = tsConditionalType;
	generated$2.tSConstructSignatureDeclaration = generated$2.tsConstructSignatureDeclaration = tsConstructSignatureDeclaration;
	generated$2.tSConstructorType = generated$2.tsConstructorType = tsConstructorType;
	generated$2.tSDeclareFunction = generated$2.tsDeclareFunction = tsDeclareFunction;
	generated$2.tSDeclareMethod = generated$2.tsDeclareMethod = tsDeclareMethod;
	generated$2.tSEnumDeclaration = generated$2.tsEnumDeclaration = tsEnumDeclaration;
	generated$2.tSEnumMember = generated$2.tsEnumMember = tsEnumMember;
	generated$2.tSExportAssignment = generated$2.tsExportAssignment = tsExportAssignment;
	generated$2.tSExpressionWithTypeArguments = generated$2.tsExpressionWithTypeArguments = tsExpressionWithTypeArguments;
	generated$2.tSExternalModuleReference = generated$2.tsExternalModuleReference = tsExternalModuleReference;
	generated$2.tSFunctionType = generated$2.tsFunctionType = tsFunctionType;
	generated$2.tSImportEqualsDeclaration = generated$2.tsImportEqualsDeclaration = tsImportEqualsDeclaration;
	generated$2.tSImportType = generated$2.tsImportType = tsImportType;
	generated$2.tSIndexSignature = generated$2.tsIndexSignature = tsIndexSignature;
	generated$2.tSIndexedAccessType = generated$2.tsIndexedAccessType = tsIndexedAccessType;
	generated$2.tSInferType = generated$2.tsInferType = tsInferType;
	generated$2.tSInstantiationExpression = generated$2.tsInstantiationExpression = tsInstantiationExpression;
	generated$2.tSInterfaceBody = generated$2.tsInterfaceBody = tsInterfaceBody;
	generated$2.tSInterfaceDeclaration = generated$2.tsInterfaceDeclaration = tsInterfaceDeclaration;
	generated$2.tSIntersectionType = generated$2.tsIntersectionType = tsIntersectionType;
	generated$2.tSIntrinsicKeyword = generated$2.tsIntrinsicKeyword = tsIntrinsicKeyword;
	generated$2.tSLiteralType = generated$2.tsLiteralType = tsLiteralType;
	generated$2.tSMappedType = generated$2.tsMappedType = tsMappedType;
	generated$2.tSMethodSignature = generated$2.tsMethodSignature = tsMethodSignature;
	generated$2.tSModuleBlock = generated$2.tsModuleBlock = tsModuleBlock;
	generated$2.tSModuleDeclaration = generated$2.tsModuleDeclaration = tsModuleDeclaration;
	generated$2.tSNamedTupleMember = generated$2.tsNamedTupleMember = tsNamedTupleMember;
	generated$2.tSNamespaceExportDeclaration = generated$2.tsNamespaceExportDeclaration = tsNamespaceExportDeclaration;
	generated$2.tSNeverKeyword = generated$2.tsNeverKeyword = tsNeverKeyword;
	generated$2.tSNonNullExpression = generated$2.tsNonNullExpression = tsNonNullExpression;
	generated$2.tSNullKeyword = generated$2.tsNullKeyword = tsNullKeyword;
	generated$2.tSNumberKeyword = generated$2.tsNumberKeyword = tsNumberKeyword;
	generated$2.tSObjectKeyword = generated$2.tsObjectKeyword = tsObjectKeyword;
	generated$2.tSOptionalType = generated$2.tsOptionalType = tsOptionalType;
	generated$2.tSParameterProperty = generated$2.tsParameterProperty = tsParameterProperty;
	generated$2.tSParenthesizedType = generated$2.tsParenthesizedType = tsParenthesizedType;
	generated$2.tSPropertySignature = generated$2.tsPropertySignature = tsPropertySignature;
	generated$2.tSQualifiedName = generated$2.tsQualifiedName = tsQualifiedName;
	generated$2.tSRestType = generated$2.tsRestType = tsRestType;
	generated$2.tSSatisfiesExpression = generated$2.tsSatisfiesExpression = tsSatisfiesExpression;
	generated$2.tSStringKeyword = generated$2.tsStringKeyword = tsStringKeyword;
	generated$2.tSSymbolKeyword = generated$2.tsSymbolKeyword = tsSymbolKeyword;
	generated$2.tSThisType = generated$2.tsThisType = tsThisType;
	generated$2.tSTupleType = generated$2.tsTupleType = tsTupleType;
	generated$2.tSTypeAliasDeclaration = generated$2.tsTypeAliasDeclaration = tsTypeAliasDeclaration;
	generated$2.tSTypeAnnotation = generated$2.tsTypeAnnotation = tsTypeAnnotation;
	generated$2.tSTypeAssertion = generated$2.tsTypeAssertion = tsTypeAssertion;
	generated$2.tSTypeLiteral = generated$2.tsTypeLiteral = tsTypeLiteral;
	generated$2.tSTypeOperator = generated$2.tsTypeOperator = tsTypeOperator;
	generated$2.tSTypeParameter = generated$2.tsTypeParameter = tsTypeParameter;
	generated$2.tSTypeParameterDeclaration = generated$2.tsTypeParameterDeclaration = tsTypeParameterDeclaration;
	generated$2.tSTypeParameterInstantiation = generated$2.tsTypeParameterInstantiation = tsTypeParameterInstantiation;
	generated$2.tSTypePredicate = generated$2.tsTypePredicate = tsTypePredicate;
	generated$2.tSTypeQuery = generated$2.tsTypeQuery = tsTypeQuery;
	generated$2.tSTypeReference = generated$2.tsTypeReference = tsTypeReference;
	generated$2.tSUndefinedKeyword = generated$2.tsUndefinedKeyword = tsUndefinedKeyword;
	generated$2.tSUnionType = generated$2.tsUnionType = tsUnionType;
	generated$2.tSUnknownKeyword = generated$2.tsUnknownKeyword = tsUnknownKeyword;
	generated$2.tSVoidKeyword = generated$2.tsVoidKeyword = tsVoidKeyword;
	generated$2.tupleExpression = tupleExpression;
	generated$2.tupleTypeAnnotation = tupleTypeAnnotation;
	generated$2.typeAlias = typeAlias;
	generated$2.typeAnnotation = typeAnnotation;
	generated$2.typeCastExpression = typeCastExpression;
	generated$2.typeParameter = typeParameter;
	generated$2.typeParameterDeclaration = typeParameterDeclaration;
	generated$2.typeParameterInstantiation = typeParameterInstantiation;
	generated$2.typeofTypeAnnotation = typeofTypeAnnotation;
	generated$2.unaryExpression = unaryExpression;
	generated$2.unionTypeAnnotation = unionTypeAnnotation;
	generated$2.updateExpression = updateExpression;
	generated$2.v8IntrinsicIdentifier = v8IntrinsicIdentifier;
	generated$2.variableDeclaration = variableDeclaration;
	generated$2.variableDeclarator = variableDeclarator;
	generated$2.variance = variance;
	generated$2.voidTypeAnnotation = voidTypeAnnotation;
	generated$2.whileStatement = whileStatement;
	generated$2.withStatement = withStatement;
	generated$2.yieldExpression = yieldExpression;
	var _validate = requireValidate();
	var _deprecationWarning = requireDeprecationWarning();
	var utils = requireUtils();
	const {
	  validateInternal: validate
	} = _validate;
	const {
	  NODE_FIELDS
	} = utils;
	function arrayExpression(elements = []) {
	  const node = {
	    type: "ArrayExpression",
	    elements
	  };
	  const defs = NODE_FIELDS.ArrayExpression;
	  validate(defs.elements, node, "elements", elements, 1);
	  return node;
	}
	function assignmentExpression(operator, left, right) {
	  const node = {
	    type: "AssignmentExpression",
	    operator,
	    left,
	    right
	  };
	  const defs = NODE_FIELDS.AssignmentExpression;
	  validate(defs.operator, node, "operator", operator);
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function binaryExpression(operator, left, right) {
	  const node = {
	    type: "BinaryExpression",
	    operator,
	    left,
	    right
	  };
	  const defs = NODE_FIELDS.BinaryExpression;
	  validate(defs.operator, node, "operator", operator);
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function interpreterDirective(value) {
	  const node = {
	    type: "InterpreterDirective",
	    value
	  };
	  const defs = NODE_FIELDS.InterpreterDirective;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function directive(value) {
	  const node = {
	    type: "Directive",
	    value
	  };
	  const defs = NODE_FIELDS.Directive;
	  validate(defs.value, node, "value", value, 1);
	  return node;
	}
	function directiveLiteral(value) {
	  const node = {
	    type: "DirectiveLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.DirectiveLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function blockStatement(body, directives = []) {
	  const node = {
	    type: "BlockStatement",
	    body,
	    directives
	  };
	  const defs = NODE_FIELDS.BlockStatement;
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.directives, node, "directives", directives, 1);
	  return node;
	}
	function breakStatement(label = null) {
	  const node = {
	    type: "BreakStatement",
	    label
	  };
	  const defs = NODE_FIELDS.BreakStatement;
	  validate(defs.label, node, "label", label, 1);
	  return node;
	}
	function callExpression(callee, _arguments) {
	  const node = {
	    type: "CallExpression",
	    callee,
	    arguments: _arguments
	  };
	  const defs = NODE_FIELDS.CallExpression;
	  validate(defs.callee, node, "callee", callee, 1);
	  validate(defs.arguments, node, "arguments", _arguments, 1);
	  return node;
	}
	function catchClause(param = null, body) {
	  const node = {
	    type: "CatchClause",
	    param,
	    body
	  };
	  const defs = NODE_FIELDS.CatchClause;
	  validate(defs.param, node, "param", param, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function conditionalExpression(test, consequent, alternate) {
	  const node = {
	    type: "ConditionalExpression",
	    test,
	    consequent,
	    alternate
	  };
	  const defs = NODE_FIELDS.ConditionalExpression;
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.consequent, node, "consequent", consequent, 1);
	  validate(defs.alternate, node, "alternate", alternate, 1);
	  return node;
	}
	function continueStatement(label = null) {
	  const node = {
	    type: "ContinueStatement",
	    label
	  };
	  const defs = NODE_FIELDS.ContinueStatement;
	  validate(defs.label, node, "label", label, 1);
	  return node;
	}
	function debuggerStatement() {
	  return {
	    type: "DebuggerStatement"
	  };
	}
	function doWhileStatement(test, body) {
	  const node = {
	    type: "DoWhileStatement",
	    test,
	    body
	  };
	  const defs = NODE_FIELDS.DoWhileStatement;
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function emptyStatement() {
	  return {
	    type: "EmptyStatement"
	  };
	}
	function expressionStatement(expression) {
	  const node = {
	    type: "ExpressionStatement",
	    expression
	  };
	  const defs = NODE_FIELDS.ExpressionStatement;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function file(program, comments = null, tokens = null) {
	  const node = {
	    type: "File",
	    program,
	    comments,
	    tokens
	  };
	  const defs = NODE_FIELDS.File;
	  validate(defs.program, node, "program", program, 1);
	  validate(defs.comments, node, "comments", comments, 1);
	  validate(defs.tokens, node, "tokens", tokens);
	  return node;
	}
	function forInStatement(left, right, body) {
	  const node = {
	    type: "ForInStatement",
	    left,
	    right,
	    body
	  };
	  const defs = NODE_FIELDS.ForInStatement;
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function forStatement(init = null, test = null, update = null, body) {
	  const node = {
	    type: "ForStatement",
	    init,
	    test,
	    update,
	    body
	  };
	  const defs = NODE_FIELDS.ForStatement;
	  validate(defs.init, node, "init", init, 1);
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.update, node, "update", update, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function functionDeclaration(id = null, params, body, generator = false, async = false) {
	  const node = {
	    type: "FunctionDeclaration",
	    id,
	    params,
	    body,
	    generator,
	    async
	  };
	  const defs = NODE_FIELDS.FunctionDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.generator, node, "generator", generator);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function functionExpression(id = null, params, body, generator = false, async = false) {
	  const node = {
	    type: "FunctionExpression",
	    id,
	    params,
	    body,
	    generator,
	    async
	  };
	  const defs = NODE_FIELDS.FunctionExpression;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.generator, node, "generator", generator);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function identifier(name) {
	  const node = {
	    type: "Identifier",
	    name
	  };
	  const defs = NODE_FIELDS.Identifier;
	  validate(defs.name, node, "name", name);
	  return node;
	}
	function ifStatement(test, consequent, alternate = null) {
	  const node = {
	    type: "IfStatement",
	    test,
	    consequent,
	    alternate
	  };
	  const defs = NODE_FIELDS.IfStatement;
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.consequent, node, "consequent", consequent, 1);
	  validate(defs.alternate, node, "alternate", alternate, 1);
	  return node;
	}
	function labeledStatement(label, body) {
	  const node = {
	    type: "LabeledStatement",
	    label,
	    body
	  };
	  const defs = NODE_FIELDS.LabeledStatement;
	  validate(defs.label, node, "label", label, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function stringLiteral(value) {
	  const node = {
	    type: "StringLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.StringLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function numericLiteral(value) {
	  const node = {
	    type: "NumericLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.NumericLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function nullLiteral() {
	  return {
	    type: "NullLiteral"
	  };
	}
	function booleanLiteral(value) {
	  const node = {
	    type: "BooleanLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.BooleanLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function regExpLiteral(pattern, flags = "") {
	  const node = {
	    type: "RegExpLiteral",
	    pattern,
	    flags
	  };
	  const defs = NODE_FIELDS.RegExpLiteral;
	  validate(defs.pattern, node, "pattern", pattern);
	  validate(defs.flags, node, "flags", flags);
	  return node;
	}
	function logicalExpression(operator, left, right) {
	  const node = {
	    type: "LogicalExpression",
	    operator,
	    left,
	    right
	  };
	  const defs = NODE_FIELDS.LogicalExpression;
	  validate(defs.operator, node, "operator", operator);
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function memberExpression(object, property, computed = false, optional = null) {
	  const node = {
	    type: "MemberExpression",
	    object,
	    property,
	    computed,
	    optional
	  };
	  const defs = NODE_FIELDS.MemberExpression;
	  validate(defs.object, node, "object", object, 1);
	  validate(defs.property, node, "property", property, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.optional, node, "optional", optional);
	  return node;
	}
	function newExpression(callee, _arguments) {
	  const node = {
	    type: "NewExpression",
	    callee,
	    arguments: _arguments
	  };
	  const defs = NODE_FIELDS.NewExpression;
	  validate(defs.callee, node, "callee", callee, 1);
	  validate(defs.arguments, node, "arguments", _arguments, 1);
	  return node;
	}
	function program(body, directives = [], sourceType = "script", interpreter = null) {
	  const node = {
	    type: "Program",
	    body,
	    directives,
	    sourceType,
	    interpreter
	  };
	  const defs = NODE_FIELDS.Program;
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.directives, node, "directives", directives, 1);
	  validate(defs.sourceType, node, "sourceType", sourceType);
	  validate(defs.interpreter, node, "interpreter", interpreter, 1);
	  return node;
	}
	function objectExpression(properties) {
	  const node = {
	    type: "ObjectExpression",
	    properties
	  };
	  const defs = NODE_FIELDS.ObjectExpression;
	  validate(defs.properties, node, "properties", properties, 1);
	  return node;
	}
	function objectMethod(kind = "method", key, params, body, computed = false, generator = false, async = false) {
	  const node = {
	    type: "ObjectMethod",
	    kind,
	    key,
	    params,
	    body,
	    computed,
	    generator,
	    async
	  };
	  const defs = NODE_FIELDS.ObjectMethod;
	  validate(defs.kind, node, "kind", kind);
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.generator, node, "generator", generator);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function objectProperty(key, value, computed = false, shorthand = false, decorators = null) {
	  const node = {
	    type: "ObjectProperty",
	    key,
	    value,
	    computed,
	    shorthand,
	    decorators
	  };
	  const defs = NODE_FIELDS.ObjectProperty;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.shorthand, node, "shorthand", shorthand);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  return node;
	}
	function restElement(argument) {
	  const node = {
	    type: "RestElement",
	    argument
	  };
	  const defs = NODE_FIELDS.RestElement;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function returnStatement(argument = null) {
	  const node = {
	    type: "ReturnStatement",
	    argument
	  };
	  const defs = NODE_FIELDS.ReturnStatement;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function sequenceExpression(expressions) {
	  const node = {
	    type: "SequenceExpression",
	    expressions
	  };
	  const defs = NODE_FIELDS.SequenceExpression;
	  validate(defs.expressions, node, "expressions", expressions, 1);
	  return node;
	}
	function parenthesizedExpression(expression) {
	  const node = {
	    type: "ParenthesizedExpression",
	    expression
	  };
	  const defs = NODE_FIELDS.ParenthesizedExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function switchCase(test = null, consequent) {
	  const node = {
	    type: "SwitchCase",
	    test,
	    consequent
	  };
	  const defs = NODE_FIELDS.SwitchCase;
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.consequent, node, "consequent", consequent, 1);
	  return node;
	}
	function switchStatement(discriminant, cases) {
	  const node = {
	    type: "SwitchStatement",
	    discriminant,
	    cases
	  };
	  const defs = NODE_FIELDS.SwitchStatement;
	  validate(defs.discriminant, node, "discriminant", discriminant, 1);
	  validate(defs.cases, node, "cases", cases, 1);
	  return node;
	}
	function thisExpression() {
	  return {
	    type: "ThisExpression"
	  };
	}
	function throwStatement(argument) {
	  const node = {
	    type: "ThrowStatement",
	    argument
	  };
	  const defs = NODE_FIELDS.ThrowStatement;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function tryStatement(block, handler = null, finalizer = null) {
	  const node = {
	    type: "TryStatement",
	    block,
	    handler,
	    finalizer
	  };
	  const defs = NODE_FIELDS.TryStatement;
	  validate(defs.block, node, "block", block, 1);
	  validate(defs.handler, node, "handler", handler, 1);
	  validate(defs.finalizer, node, "finalizer", finalizer, 1);
	  return node;
	}
	function unaryExpression(operator, argument, prefix = true) {
	  const node = {
	    type: "UnaryExpression",
	    operator,
	    argument,
	    prefix
	  };
	  const defs = NODE_FIELDS.UnaryExpression;
	  validate(defs.operator, node, "operator", operator);
	  validate(defs.argument, node, "argument", argument, 1);
	  validate(defs.prefix, node, "prefix", prefix);
	  return node;
	}
	function updateExpression(operator, argument, prefix = false) {
	  const node = {
	    type: "UpdateExpression",
	    operator,
	    argument,
	    prefix
	  };
	  const defs = NODE_FIELDS.UpdateExpression;
	  validate(defs.operator, node, "operator", operator);
	  validate(defs.argument, node, "argument", argument, 1);
	  validate(defs.prefix, node, "prefix", prefix);
	  return node;
	}
	function variableDeclaration(kind, declarations) {
	  const node = {
	    type: "VariableDeclaration",
	    kind,
	    declarations
	  };
	  const defs = NODE_FIELDS.VariableDeclaration;
	  validate(defs.kind, node, "kind", kind);
	  validate(defs.declarations, node, "declarations", declarations, 1);
	  return node;
	}
	function variableDeclarator(id, init = null) {
	  const node = {
	    type: "VariableDeclarator",
	    id,
	    init
	  };
	  const defs = NODE_FIELDS.VariableDeclarator;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.init, node, "init", init, 1);
	  return node;
	}
	function whileStatement(test, body) {
	  const node = {
	    type: "WhileStatement",
	    test,
	    body
	  };
	  const defs = NODE_FIELDS.WhileStatement;
	  validate(defs.test, node, "test", test, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function withStatement(object, body) {
	  const node = {
	    type: "WithStatement",
	    object,
	    body
	  };
	  const defs = NODE_FIELDS.WithStatement;
	  validate(defs.object, node, "object", object, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function assignmentPattern(left, right) {
	  const node = {
	    type: "AssignmentPattern",
	    left,
	    right
	  };
	  const defs = NODE_FIELDS.AssignmentPattern;
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function arrayPattern(elements) {
	  const node = {
	    type: "ArrayPattern",
	    elements
	  };
	  const defs = NODE_FIELDS.ArrayPattern;
	  validate(defs.elements, node, "elements", elements, 1);
	  return node;
	}
	function arrowFunctionExpression(params, body, async = false) {
	  const node = {
	    type: "ArrowFunctionExpression",
	    params,
	    body,
	    async,
	    expression: null
	  };
	  const defs = NODE_FIELDS.ArrowFunctionExpression;
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function classBody(body) {
	  const node = {
	    type: "ClassBody",
	    body
	  };
	  const defs = NODE_FIELDS.ClassBody;
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function classExpression(id = null, superClass = null, body, decorators = null) {
	  const node = {
	    type: "ClassExpression",
	    id,
	    superClass,
	    body,
	    decorators
	  };
	  const defs = NODE_FIELDS.ClassExpression;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.superClass, node, "superClass", superClass, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  return node;
	}
	function classDeclaration(id = null, superClass = null, body, decorators = null) {
	  const node = {
	    type: "ClassDeclaration",
	    id,
	    superClass,
	    body,
	    decorators
	  };
	  const defs = NODE_FIELDS.ClassDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.superClass, node, "superClass", superClass, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  return node;
	}
	function exportAllDeclaration(source) {
	  const node = {
	    type: "ExportAllDeclaration",
	    source
	  };
	  const defs = NODE_FIELDS.ExportAllDeclaration;
	  validate(defs.source, node, "source", source, 1);
	  return node;
	}
	function exportDefaultDeclaration(declaration) {
	  const node = {
	    type: "ExportDefaultDeclaration",
	    declaration
	  };
	  const defs = NODE_FIELDS.ExportDefaultDeclaration;
	  validate(defs.declaration, node, "declaration", declaration, 1);
	  return node;
	}
	function exportNamedDeclaration(declaration = null, specifiers = [], source = null) {
	  const node = {
	    type: "ExportNamedDeclaration",
	    declaration,
	    specifiers,
	    source
	  };
	  const defs = NODE_FIELDS.ExportNamedDeclaration;
	  validate(defs.declaration, node, "declaration", declaration, 1);
	  validate(defs.specifiers, node, "specifiers", specifiers, 1);
	  validate(defs.source, node, "source", source, 1);
	  return node;
	}
	function exportSpecifier(local, exported) {
	  const node = {
	    type: "ExportSpecifier",
	    local,
	    exported
	  };
	  const defs = NODE_FIELDS.ExportSpecifier;
	  validate(defs.local, node, "local", local, 1);
	  validate(defs.exported, node, "exported", exported, 1);
	  return node;
	}
	function forOfStatement(left, right, body, _await = false) {
	  const node = {
	    type: "ForOfStatement",
	    left,
	    right,
	    body,
	    await: _await
	  };
	  const defs = NODE_FIELDS.ForOfStatement;
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.await, node, "await", _await);
	  return node;
	}
	function importDeclaration(specifiers, source) {
	  const node = {
	    type: "ImportDeclaration",
	    specifiers,
	    source
	  };
	  const defs = NODE_FIELDS.ImportDeclaration;
	  validate(defs.specifiers, node, "specifiers", specifiers, 1);
	  validate(defs.source, node, "source", source, 1);
	  return node;
	}
	function importDefaultSpecifier(local) {
	  const node = {
	    type: "ImportDefaultSpecifier",
	    local
	  };
	  const defs = NODE_FIELDS.ImportDefaultSpecifier;
	  validate(defs.local, node, "local", local, 1);
	  return node;
	}
	function importNamespaceSpecifier(local) {
	  const node = {
	    type: "ImportNamespaceSpecifier",
	    local
	  };
	  const defs = NODE_FIELDS.ImportNamespaceSpecifier;
	  validate(defs.local, node, "local", local, 1);
	  return node;
	}
	function importSpecifier(local, imported) {
	  const node = {
	    type: "ImportSpecifier",
	    local,
	    imported
	  };
	  const defs = NODE_FIELDS.ImportSpecifier;
	  validate(defs.local, node, "local", local, 1);
	  validate(defs.imported, node, "imported", imported, 1);
	  return node;
	}
	function importExpression(source, options = null) {
	  const node = {
	    type: "ImportExpression",
	    source,
	    options
	  };
	  const defs = NODE_FIELDS.ImportExpression;
	  validate(defs.source, node, "source", source, 1);
	  validate(defs.options, node, "options", options, 1);
	  return node;
	}
	function metaProperty(meta, property) {
	  const node = {
	    type: "MetaProperty",
	    meta,
	    property
	  };
	  const defs = NODE_FIELDS.MetaProperty;
	  validate(defs.meta, node, "meta", meta, 1);
	  validate(defs.property, node, "property", property, 1);
	  return node;
	}
	function classMethod(kind = "method", key, params, body, computed = false, _static = false, generator = false, async = false) {
	  const node = {
	    type: "ClassMethod",
	    kind,
	    key,
	    params,
	    body,
	    computed,
	    static: _static,
	    generator,
	    async
	  };
	  const defs = NODE_FIELDS.ClassMethod;
	  validate(defs.kind, node, "kind", kind);
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.static, node, "static", _static);
	  validate(defs.generator, node, "generator", generator);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function objectPattern(properties) {
	  const node = {
	    type: "ObjectPattern",
	    properties
	  };
	  const defs = NODE_FIELDS.ObjectPattern;
	  validate(defs.properties, node, "properties", properties, 1);
	  return node;
	}
	function spreadElement(argument) {
	  const node = {
	    type: "SpreadElement",
	    argument
	  };
	  const defs = NODE_FIELDS.SpreadElement;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function _super() {
	  return {
	    type: "Super"
	  };
	}
	function taggedTemplateExpression(tag, quasi) {
	  const node = {
	    type: "TaggedTemplateExpression",
	    tag,
	    quasi
	  };
	  const defs = NODE_FIELDS.TaggedTemplateExpression;
	  validate(defs.tag, node, "tag", tag, 1);
	  validate(defs.quasi, node, "quasi", quasi, 1);
	  return node;
	}
	function templateElement(value, tail = false) {
	  const node = {
	    type: "TemplateElement",
	    value,
	    tail
	  };
	  const defs = NODE_FIELDS.TemplateElement;
	  validate(defs.value, node, "value", value);
	  validate(defs.tail, node, "tail", tail);
	  return node;
	}
	function templateLiteral(quasis, expressions) {
	  const node = {
	    type: "TemplateLiteral",
	    quasis,
	    expressions
	  };
	  const defs = NODE_FIELDS.TemplateLiteral;
	  validate(defs.quasis, node, "quasis", quasis, 1);
	  validate(defs.expressions, node, "expressions", expressions, 1);
	  return node;
	}
	function yieldExpression(argument = null, delegate = false) {
	  const node = {
	    type: "YieldExpression",
	    argument,
	    delegate
	  };
	  const defs = NODE_FIELDS.YieldExpression;
	  validate(defs.argument, node, "argument", argument, 1);
	  validate(defs.delegate, node, "delegate", delegate);
	  return node;
	}
	function awaitExpression(argument) {
	  const node = {
	    type: "AwaitExpression",
	    argument
	  };
	  const defs = NODE_FIELDS.AwaitExpression;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function _import() {
	  return {
	    type: "Import"
	  };
	}
	function bigIntLiteral(value) {
	  const node = {
	    type: "BigIntLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.BigIntLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function exportNamespaceSpecifier(exported) {
	  const node = {
	    type: "ExportNamespaceSpecifier",
	    exported
	  };
	  const defs = NODE_FIELDS.ExportNamespaceSpecifier;
	  validate(defs.exported, node, "exported", exported, 1);
	  return node;
	}
	function optionalMemberExpression(object, property, computed = false, optional) {
	  const node = {
	    type: "OptionalMemberExpression",
	    object,
	    property,
	    computed,
	    optional
	  };
	  const defs = NODE_FIELDS.OptionalMemberExpression;
	  validate(defs.object, node, "object", object, 1);
	  validate(defs.property, node, "property", property, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.optional, node, "optional", optional);
	  return node;
	}
	function optionalCallExpression(callee, _arguments, optional) {
	  const node = {
	    type: "OptionalCallExpression",
	    callee,
	    arguments: _arguments,
	    optional
	  };
	  const defs = NODE_FIELDS.OptionalCallExpression;
	  validate(defs.callee, node, "callee", callee, 1);
	  validate(defs.arguments, node, "arguments", _arguments, 1);
	  validate(defs.optional, node, "optional", optional);
	  return node;
	}
	function classProperty(key, value = null, typeAnnotation = null, decorators = null, computed = false, _static = false) {
	  const node = {
	    type: "ClassProperty",
	    key,
	    value,
	    typeAnnotation,
	    decorators,
	    computed,
	    static: _static
	  };
	  const defs = NODE_FIELDS.ClassProperty;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.static, node, "static", _static);
	  return node;
	}
	function classAccessorProperty(key, value = null, typeAnnotation = null, decorators = null, computed = false, _static = false) {
	  const node = {
	    type: "ClassAccessorProperty",
	    key,
	    value,
	    typeAnnotation,
	    decorators,
	    computed,
	    static: _static
	  };
	  const defs = NODE_FIELDS.ClassAccessorProperty;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  validate(defs.computed, node, "computed", computed);
	  validate(defs.static, node, "static", _static);
	  return node;
	}
	function classPrivateProperty(key, value = null, decorators = null, _static = false) {
	  const node = {
	    type: "ClassPrivateProperty",
	    key,
	    value,
	    decorators,
	    static: _static
	  };
	  const defs = NODE_FIELDS.ClassPrivateProperty;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  validate(defs.static, node, "static", _static);
	  return node;
	}
	function classPrivateMethod(kind = "method", key, params, body, _static = false) {
	  const node = {
	    type: "ClassPrivateMethod",
	    kind,
	    key,
	    params,
	    body,
	    static: _static
	  };
	  const defs = NODE_FIELDS.ClassPrivateMethod;
	  validate(defs.kind, node, "kind", kind);
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.static, node, "static", _static);
	  return node;
	}
	function privateName(id) {
	  const node = {
	    type: "PrivateName",
	    id
	  };
	  const defs = NODE_FIELDS.PrivateName;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function staticBlock(body) {
	  const node = {
	    type: "StaticBlock",
	    body
	  };
	  const defs = NODE_FIELDS.StaticBlock;
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function anyTypeAnnotation() {
	  return {
	    type: "AnyTypeAnnotation"
	  };
	}
	function arrayTypeAnnotation(elementType) {
	  const node = {
	    type: "ArrayTypeAnnotation",
	    elementType
	  };
	  const defs = NODE_FIELDS.ArrayTypeAnnotation;
	  validate(defs.elementType, node, "elementType", elementType, 1);
	  return node;
	}
	function booleanTypeAnnotation() {
	  return {
	    type: "BooleanTypeAnnotation"
	  };
	}
	function booleanLiteralTypeAnnotation(value) {
	  const node = {
	    type: "BooleanLiteralTypeAnnotation",
	    value
	  };
	  const defs = NODE_FIELDS.BooleanLiteralTypeAnnotation;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function nullLiteralTypeAnnotation() {
	  return {
	    type: "NullLiteralTypeAnnotation"
	  };
	}
	function classImplements(id, typeParameters = null) {
	  const node = {
	    type: "ClassImplements",
	    id,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.ClassImplements;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function declareClass(id, typeParameters = null, _extends = null, body) {
	  const node = {
	    type: "DeclareClass",
	    id,
	    typeParameters,
	    extends: _extends,
	    body
	  };
	  const defs = NODE_FIELDS.DeclareClass;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.extends, node, "extends", _extends, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function declareFunction(id) {
	  const node = {
	    type: "DeclareFunction",
	    id
	  };
	  const defs = NODE_FIELDS.DeclareFunction;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function declareInterface(id, typeParameters = null, _extends = null, body) {
	  const node = {
	    type: "DeclareInterface",
	    id,
	    typeParameters,
	    extends: _extends,
	    body
	  };
	  const defs = NODE_FIELDS.DeclareInterface;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.extends, node, "extends", _extends, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function declareModule(id, body, kind = null) {
	  const node = {
	    type: "DeclareModule",
	    id,
	    body,
	    kind
	  };
	  const defs = NODE_FIELDS.DeclareModule;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.kind, node, "kind", kind);
	  return node;
	}
	function declareModuleExports(typeAnnotation) {
	  const node = {
	    type: "DeclareModuleExports",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.DeclareModuleExports;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function declareTypeAlias(id, typeParameters = null, right) {
	  const node = {
	    type: "DeclareTypeAlias",
	    id,
	    typeParameters,
	    right
	  };
	  const defs = NODE_FIELDS.DeclareTypeAlias;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function declareOpaqueType(id, typeParameters = null, supertype = null) {
	  const node = {
	    type: "DeclareOpaqueType",
	    id,
	    typeParameters,
	    supertype
	  };
	  const defs = NODE_FIELDS.DeclareOpaqueType;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.supertype, node, "supertype", supertype, 1);
	  return node;
	}
	function declareVariable(id) {
	  const node = {
	    type: "DeclareVariable",
	    id
	  };
	  const defs = NODE_FIELDS.DeclareVariable;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function declareExportDeclaration(declaration = null, specifiers = null, source = null, attributes = null) {
	  const node = {
	    type: "DeclareExportDeclaration",
	    declaration,
	    specifiers,
	    source,
	    attributes
	  };
	  const defs = NODE_FIELDS.DeclareExportDeclaration;
	  validate(defs.declaration, node, "declaration", declaration, 1);
	  validate(defs.specifiers, node, "specifiers", specifiers, 1);
	  validate(defs.source, node, "source", source, 1);
	  validate(defs.attributes, node, "attributes", attributes, 1);
	  return node;
	}
	function declareExportAllDeclaration(source, attributes = null) {
	  const node = {
	    type: "DeclareExportAllDeclaration",
	    source,
	    attributes
	  };
	  const defs = NODE_FIELDS.DeclareExportAllDeclaration;
	  validate(defs.source, node, "source", source, 1);
	  validate(defs.attributes, node, "attributes", attributes, 1);
	  return node;
	}
	function declaredPredicate(value) {
	  const node = {
	    type: "DeclaredPredicate",
	    value
	  };
	  const defs = NODE_FIELDS.DeclaredPredicate;
	  validate(defs.value, node, "value", value, 1);
	  return node;
	}
	function existsTypeAnnotation() {
	  return {
	    type: "ExistsTypeAnnotation"
	  };
	}
	function functionTypeAnnotation(typeParameters = null, params, rest = null, returnType) {
	  const node = {
	    type: "FunctionTypeAnnotation",
	    typeParameters,
	    params,
	    rest,
	    returnType
	  };
	  const defs = NODE_FIELDS.FunctionTypeAnnotation;
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.rest, node, "rest", rest, 1);
	  validate(defs.returnType, node, "returnType", returnType, 1);
	  return node;
	}
	function functionTypeParam(name = null, typeAnnotation) {
	  const node = {
	    type: "FunctionTypeParam",
	    name,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.FunctionTypeParam;
	  validate(defs.name, node, "name", name, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function genericTypeAnnotation(id, typeParameters = null) {
	  const node = {
	    type: "GenericTypeAnnotation",
	    id,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.GenericTypeAnnotation;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function inferredPredicate() {
	  return {
	    type: "InferredPredicate"
	  };
	}
	function interfaceExtends(id, typeParameters = null) {
	  const node = {
	    type: "InterfaceExtends",
	    id,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.InterfaceExtends;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function interfaceDeclaration(id, typeParameters = null, _extends = null, body) {
	  const node = {
	    type: "InterfaceDeclaration",
	    id,
	    typeParameters,
	    extends: _extends,
	    body
	  };
	  const defs = NODE_FIELDS.InterfaceDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.extends, node, "extends", _extends, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function interfaceTypeAnnotation(_extends = null, body) {
	  const node = {
	    type: "InterfaceTypeAnnotation",
	    extends: _extends,
	    body
	  };
	  const defs = NODE_FIELDS.InterfaceTypeAnnotation;
	  validate(defs.extends, node, "extends", _extends, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function intersectionTypeAnnotation(types) {
	  const node = {
	    type: "IntersectionTypeAnnotation",
	    types
	  };
	  const defs = NODE_FIELDS.IntersectionTypeAnnotation;
	  validate(defs.types, node, "types", types, 1);
	  return node;
	}
	function mixedTypeAnnotation() {
	  return {
	    type: "MixedTypeAnnotation"
	  };
	}
	function emptyTypeAnnotation() {
	  return {
	    type: "EmptyTypeAnnotation"
	  };
	}
	function nullableTypeAnnotation(typeAnnotation) {
	  const node = {
	    type: "NullableTypeAnnotation",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.NullableTypeAnnotation;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function numberLiteralTypeAnnotation(value) {
	  const node = {
	    type: "NumberLiteralTypeAnnotation",
	    value
	  };
	  const defs = NODE_FIELDS.NumberLiteralTypeAnnotation;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function numberTypeAnnotation() {
	  return {
	    type: "NumberTypeAnnotation"
	  };
	}
	function objectTypeAnnotation(properties, indexers = [], callProperties = [], internalSlots = [], exact = false) {
	  const node = {
	    type: "ObjectTypeAnnotation",
	    properties,
	    indexers,
	    callProperties,
	    internalSlots,
	    exact
	  };
	  const defs = NODE_FIELDS.ObjectTypeAnnotation;
	  validate(defs.properties, node, "properties", properties, 1);
	  validate(defs.indexers, node, "indexers", indexers, 1);
	  validate(defs.callProperties, node, "callProperties", callProperties, 1);
	  validate(defs.internalSlots, node, "internalSlots", internalSlots, 1);
	  validate(defs.exact, node, "exact", exact);
	  return node;
	}
	function objectTypeInternalSlot(id, value, optional, _static, method) {
	  const node = {
	    type: "ObjectTypeInternalSlot",
	    id,
	    value,
	    optional,
	    static: _static,
	    method
	  };
	  const defs = NODE_FIELDS.ObjectTypeInternalSlot;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.optional, node, "optional", optional);
	  validate(defs.static, node, "static", _static);
	  validate(defs.method, node, "method", method);
	  return node;
	}
	function objectTypeCallProperty(value) {
	  const node = {
	    type: "ObjectTypeCallProperty",
	    value,
	    static: null
	  };
	  const defs = NODE_FIELDS.ObjectTypeCallProperty;
	  validate(defs.value, node, "value", value, 1);
	  return node;
	}
	function objectTypeIndexer(id = null, key, value, variance = null) {
	  const node = {
	    type: "ObjectTypeIndexer",
	    id,
	    key,
	    value,
	    variance,
	    static: null
	  };
	  const defs = NODE_FIELDS.ObjectTypeIndexer;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.variance, node, "variance", variance, 1);
	  return node;
	}
	function objectTypeProperty(key, value, variance = null) {
	  const node = {
	    type: "ObjectTypeProperty",
	    key,
	    value,
	    variance,
	    kind: null,
	    method: null,
	    optional: null,
	    proto: null,
	    static: null
	  };
	  const defs = NODE_FIELDS.ObjectTypeProperty;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  validate(defs.variance, node, "variance", variance, 1);
	  return node;
	}
	function objectTypeSpreadProperty(argument) {
	  const node = {
	    type: "ObjectTypeSpreadProperty",
	    argument
	  };
	  const defs = NODE_FIELDS.ObjectTypeSpreadProperty;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function opaqueType(id, typeParameters = null, supertype = null, impltype) {
	  const node = {
	    type: "OpaqueType",
	    id,
	    typeParameters,
	    supertype,
	    impltype
	  };
	  const defs = NODE_FIELDS.OpaqueType;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.supertype, node, "supertype", supertype, 1);
	  validate(defs.impltype, node, "impltype", impltype, 1);
	  return node;
	}
	function qualifiedTypeIdentifier(id, qualification) {
	  const node = {
	    type: "QualifiedTypeIdentifier",
	    id,
	    qualification
	  };
	  const defs = NODE_FIELDS.QualifiedTypeIdentifier;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.qualification, node, "qualification", qualification, 1);
	  return node;
	}
	function stringLiteralTypeAnnotation(value) {
	  const node = {
	    type: "StringLiteralTypeAnnotation",
	    value
	  };
	  const defs = NODE_FIELDS.StringLiteralTypeAnnotation;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function stringTypeAnnotation() {
	  return {
	    type: "StringTypeAnnotation"
	  };
	}
	function symbolTypeAnnotation() {
	  return {
	    type: "SymbolTypeAnnotation"
	  };
	}
	function thisTypeAnnotation() {
	  return {
	    type: "ThisTypeAnnotation"
	  };
	}
	function tupleTypeAnnotation(types) {
	  const node = {
	    type: "TupleTypeAnnotation",
	    types
	  };
	  const defs = NODE_FIELDS.TupleTypeAnnotation;
	  validate(defs.types, node, "types", types, 1);
	  return node;
	}
	function typeofTypeAnnotation(argument) {
	  const node = {
	    type: "TypeofTypeAnnotation",
	    argument
	  };
	  const defs = NODE_FIELDS.TypeofTypeAnnotation;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function typeAlias(id, typeParameters = null, right) {
	  const node = {
	    type: "TypeAlias",
	    id,
	    typeParameters,
	    right
	  };
	  const defs = NODE_FIELDS.TypeAlias;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function typeAnnotation(typeAnnotation) {
	  const node = {
	    type: "TypeAnnotation",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TypeAnnotation;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function typeCastExpression(expression, typeAnnotation) {
	  const node = {
	    type: "TypeCastExpression",
	    expression,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TypeCastExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function typeParameter(bound = null, _default = null, variance = null) {
	  const node = {
	    type: "TypeParameter",
	    bound,
	    default: _default,
	    variance,
	    name: null
	  };
	  const defs = NODE_FIELDS.TypeParameter;
	  validate(defs.bound, node, "bound", bound, 1);
	  validate(defs.default, node, "default", _default, 1);
	  validate(defs.variance, node, "variance", variance, 1);
	  return node;
	}
	function typeParameterDeclaration(params) {
	  const node = {
	    type: "TypeParameterDeclaration",
	    params
	  };
	  const defs = NODE_FIELDS.TypeParameterDeclaration;
	  validate(defs.params, node, "params", params, 1);
	  return node;
	}
	function typeParameterInstantiation(params) {
	  const node = {
	    type: "TypeParameterInstantiation",
	    params
	  };
	  const defs = NODE_FIELDS.TypeParameterInstantiation;
	  validate(defs.params, node, "params", params, 1);
	  return node;
	}
	function unionTypeAnnotation(types) {
	  const node = {
	    type: "UnionTypeAnnotation",
	    types
	  };
	  const defs = NODE_FIELDS.UnionTypeAnnotation;
	  validate(defs.types, node, "types", types, 1);
	  return node;
	}
	function variance(kind) {
	  const node = {
	    type: "Variance",
	    kind
	  };
	  const defs = NODE_FIELDS.Variance;
	  validate(defs.kind, node, "kind", kind);
	  return node;
	}
	function voidTypeAnnotation() {
	  return {
	    type: "VoidTypeAnnotation"
	  };
	}
	function enumDeclaration(id, body) {
	  const node = {
	    type: "EnumDeclaration",
	    id,
	    body
	  };
	  const defs = NODE_FIELDS.EnumDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function enumBooleanBody(members) {
	  const node = {
	    type: "EnumBooleanBody",
	    members,
	    explicitType: null,
	    hasUnknownMembers: null
	  };
	  const defs = NODE_FIELDS.EnumBooleanBody;
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function enumNumberBody(members) {
	  const node = {
	    type: "EnumNumberBody",
	    members,
	    explicitType: null,
	    hasUnknownMembers: null
	  };
	  const defs = NODE_FIELDS.EnumNumberBody;
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function enumStringBody(members) {
	  const node = {
	    type: "EnumStringBody",
	    members,
	    explicitType: null,
	    hasUnknownMembers: null
	  };
	  const defs = NODE_FIELDS.EnumStringBody;
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function enumSymbolBody(members) {
	  const node = {
	    type: "EnumSymbolBody",
	    members,
	    hasUnknownMembers: null
	  };
	  const defs = NODE_FIELDS.EnumSymbolBody;
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function enumBooleanMember(id) {
	  const node = {
	    type: "EnumBooleanMember",
	    id,
	    init: null
	  };
	  const defs = NODE_FIELDS.EnumBooleanMember;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function enumNumberMember(id, init) {
	  const node = {
	    type: "EnumNumberMember",
	    id,
	    init
	  };
	  const defs = NODE_FIELDS.EnumNumberMember;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.init, node, "init", init, 1);
	  return node;
	}
	function enumStringMember(id, init) {
	  const node = {
	    type: "EnumStringMember",
	    id,
	    init
	  };
	  const defs = NODE_FIELDS.EnumStringMember;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.init, node, "init", init, 1);
	  return node;
	}
	function enumDefaultedMember(id) {
	  const node = {
	    type: "EnumDefaultedMember",
	    id
	  };
	  const defs = NODE_FIELDS.EnumDefaultedMember;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function indexedAccessType(objectType, indexType) {
	  const node = {
	    type: "IndexedAccessType",
	    objectType,
	    indexType
	  };
	  const defs = NODE_FIELDS.IndexedAccessType;
	  validate(defs.objectType, node, "objectType", objectType, 1);
	  validate(defs.indexType, node, "indexType", indexType, 1);
	  return node;
	}
	function optionalIndexedAccessType(objectType, indexType) {
	  const node = {
	    type: "OptionalIndexedAccessType",
	    objectType,
	    indexType,
	    optional: null
	  };
	  const defs = NODE_FIELDS.OptionalIndexedAccessType;
	  validate(defs.objectType, node, "objectType", objectType, 1);
	  validate(defs.indexType, node, "indexType", indexType, 1);
	  return node;
	}
	function jsxAttribute(name, value = null) {
	  const node = {
	    type: "JSXAttribute",
	    name,
	    value
	  };
	  const defs = NODE_FIELDS.JSXAttribute;
	  validate(defs.name, node, "name", name, 1);
	  validate(defs.value, node, "value", value, 1);
	  return node;
	}
	function jsxClosingElement(name) {
	  const node = {
	    type: "JSXClosingElement",
	    name
	  };
	  const defs = NODE_FIELDS.JSXClosingElement;
	  validate(defs.name, node, "name", name, 1);
	  return node;
	}
	function jsxElement(openingElement, closingElement = null, children, selfClosing = null) {
	  const node = {
	    type: "JSXElement",
	    openingElement,
	    closingElement,
	    children,
	    selfClosing
	  };
	  const defs = NODE_FIELDS.JSXElement;
	  validate(defs.openingElement, node, "openingElement", openingElement, 1);
	  validate(defs.closingElement, node, "closingElement", closingElement, 1);
	  validate(defs.children, node, "children", children, 1);
	  validate(defs.selfClosing, node, "selfClosing", selfClosing);
	  return node;
	}
	function jsxEmptyExpression() {
	  return {
	    type: "JSXEmptyExpression"
	  };
	}
	function jsxExpressionContainer(expression) {
	  const node = {
	    type: "JSXExpressionContainer",
	    expression
	  };
	  const defs = NODE_FIELDS.JSXExpressionContainer;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function jsxSpreadChild(expression) {
	  const node = {
	    type: "JSXSpreadChild",
	    expression
	  };
	  const defs = NODE_FIELDS.JSXSpreadChild;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function jsxIdentifier(name) {
	  const node = {
	    type: "JSXIdentifier",
	    name
	  };
	  const defs = NODE_FIELDS.JSXIdentifier;
	  validate(defs.name, node, "name", name);
	  return node;
	}
	function jsxMemberExpression(object, property) {
	  const node = {
	    type: "JSXMemberExpression",
	    object,
	    property
	  };
	  const defs = NODE_FIELDS.JSXMemberExpression;
	  validate(defs.object, node, "object", object, 1);
	  validate(defs.property, node, "property", property, 1);
	  return node;
	}
	function jsxNamespacedName(namespace, name) {
	  const node = {
	    type: "JSXNamespacedName",
	    namespace,
	    name
	  };
	  const defs = NODE_FIELDS.JSXNamespacedName;
	  validate(defs.namespace, node, "namespace", namespace, 1);
	  validate(defs.name, node, "name", name, 1);
	  return node;
	}
	function jsxOpeningElement(name, attributes, selfClosing = false) {
	  const node = {
	    type: "JSXOpeningElement",
	    name,
	    attributes,
	    selfClosing
	  };
	  const defs = NODE_FIELDS.JSXOpeningElement;
	  validate(defs.name, node, "name", name, 1);
	  validate(defs.attributes, node, "attributes", attributes, 1);
	  validate(defs.selfClosing, node, "selfClosing", selfClosing);
	  return node;
	}
	function jsxSpreadAttribute(argument) {
	  const node = {
	    type: "JSXSpreadAttribute",
	    argument
	  };
	  const defs = NODE_FIELDS.JSXSpreadAttribute;
	  validate(defs.argument, node, "argument", argument, 1);
	  return node;
	}
	function jsxText(value) {
	  const node = {
	    type: "JSXText",
	    value
	  };
	  const defs = NODE_FIELDS.JSXText;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function jsxFragment(openingFragment, closingFragment, children) {
	  const node = {
	    type: "JSXFragment",
	    openingFragment,
	    closingFragment,
	    children
	  };
	  const defs = NODE_FIELDS.JSXFragment;
	  validate(defs.openingFragment, node, "openingFragment", openingFragment, 1);
	  validate(defs.closingFragment, node, "closingFragment", closingFragment, 1);
	  validate(defs.children, node, "children", children, 1);
	  return node;
	}
	function jsxOpeningFragment() {
	  return {
	    type: "JSXOpeningFragment"
	  };
	}
	function jsxClosingFragment() {
	  return {
	    type: "JSXClosingFragment"
	  };
	}
	function noop() {
	  return {
	    type: "Noop"
	  };
	}
	function placeholder(expectedNode, name) {
	  const node = {
	    type: "Placeholder",
	    expectedNode,
	    name
	  };
	  const defs = NODE_FIELDS.Placeholder;
	  validate(defs.expectedNode, node, "expectedNode", expectedNode);
	  validate(defs.name, node, "name", name, 1);
	  return node;
	}
	function v8IntrinsicIdentifier(name) {
	  const node = {
	    type: "V8IntrinsicIdentifier",
	    name
	  };
	  const defs = NODE_FIELDS.V8IntrinsicIdentifier;
	  validate(defs.name, node, "name", name);
	  return node;
	}
	function argumentPlaceholder() {
	  return {
	    type: "ArgumentPlaceholder"
	  };
	}
	function bindExpression(object, callee) {
	  const node = {
	    type: "BindExpression",
	    object,
	    callee
	  };
	  const defs = NODE_FIELDS.BindExpression;
	  validate(defs.object, node, "object", object, 1);
	  validate(defs.callee, node, "callee", callee, 1);
	  return node;
	}
	function importAttribute(key, value) {
	  const node = {
	    type: "ImportAttribute",
	    key,
	    value
	  };
	  const defs = NODE_FIELDS.ImportAttribute;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.value, node, "value", value, 1);
	  return node;
	}
	function decorator(expression) {
	  const node = {
	    type: "Decorator",
	    expression
	  };
	  const defs = NODE_FIELDS.Decorator;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function doExpression(body, async = false) {
	  const node = {
	    type: "DoExpression",
	    body,
	    async
	  };
	  const defs = NODE_FIELDS.DoExpression;
	  validate(defs.body, node, "body", body, 1);
	  validate(defs.async, node, "async", async);
	  return node;
	}
	function exportDefaultSpecifier(exported) {
	  const node = {
	    type: "ExportDefaultSpecifier",
	    exported
	  };
	  const defs = NODE_FIELDS.ExportDefaultSpecifier;
	  validate(defs.exported, node, "exported", exported, 1);
	  return node;
	}
	function recordExpression(properties) {
	  const node = {
	    type: "RecordExpression",
	    properties
	  };
	  const defs = NODE_FIELDS.RecordExpression;
	  validate(defs.properties, node, "properties", properties, 1);
	  return node;
	}
	function tupleExpression(elements = []) {
	  const node = {
	    type: "TupleExpression",
	    elements
	  };
	  const defs = NODE_FIELDS.TupleExpression;
	  validate(defs.elements, node, "elements", elements, 1);
	  return node;
	}
	function decimalLiteral(value) {
	  const node = {
	    type: "DecimalLiteral",
	    value
	  };
	  const defs = NODE_FIELDS.DecimalLiteral;
	  validate(defs.value, node, "value", value);
	  return node;
	}
	function moduleExpression(body) {
	  const node = {
	    type: "ModuleExpression",
	    body
	  };
	  const defs = NODE_FIELDS.ModuleExpression;
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function topicReference() {
	  return {
	    type: "TopicReference"
	  };
	}
	function pipelineTopicExpression(expression) {
	  const node = {
	    type: "PipelineTopicExpression",
	    expression
	  };
	  const defs = NODE_FIELDS.PipelineTopicExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function pipelineBareFunction(callee) {
	  const node = {
	    type: "PipelineBareFunction",
	    callee
	  };
	  const defs = NODE_FIELDS.PipelineBareFunction;
	  validate(defs.callee, node, "callee", callee, 1);
	  return node;
	}
	function pipelinePrimaryTopicReference() {
	  return {
	    type: "PipelinePrimaryTopicReference"
	  };
	}
	function tsParameterProperty(parameter) {
	  const node = {
	    type: "TSParameterProperty",
	    parameter
	  };
	  const defs = NODE_FIELDS.TSParameterProperty;
	  validate(defs.parameter, node, "parameter", parameter, 1);
	  return node;
	}
	function tsDeclareFunction(id = null, typeParameters = null, params, returnType = null) {
	  const node = {
	    type: "TSDeclareFunction",
	    id,
	    typeParameters,
	    params,
	    returnType
	  };
	  const defs = NODE_FIELDS.TSDeclareFunction;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.returnType, node, "returnType", returnType, 1);
	  return node;
	}
	function tsDeclareMethod(decorators = null, key, typeParameters = null, params, returnType = null) {
	  const node = {
	    type: "TSDeclareMethod",
	    decorators,
	    key,
	    typeParameters,
	    params,
	    returnType
	  };
	  const defs = NODE_FIELDS.TSDeclareMethod;
	  validate(defs.decorators, node, "decorators", decorators, 1);
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.params, node, "params", params, 1);
	  validate(defs.returnType, node, "returnType", returnType, 1);
	  return node;
	}
	function tsQualifiedName(left, right) {
	  const node = {
	    type: "TSQualifiedName",
	    left,
	    right
	  };
	  const defs = NODE_FIELDS.TSQualifiedName;
	  validate(defs.left, node, "left", left, 1);
	  validate(defs.right, node, "right", right, 1);
	  return node;
	}
	function tsCallSignatureDeclaration(typeParameters = null, parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSCallSignatureDeclaration",
	    typeParameters,
	    parameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSCallSignatureDeclaration;
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsConstructSignatureDeclaration(typeParameters = null, parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSConstructSignatureDeclaration",
	    typeParameters,
	    parameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSConstructSignatureDeclaration;
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsPropertySignature(key, typeAnnotation = null) {
	  const node = {
	    type: "TSPropertySignature",
	    key,
	    typeAnnotation,
	    kind: null
	  };
	  const defs = NODE_FIELDS.TSPropertySignature;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsMethodSignature(key, typeParameters = null, parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSMethodSignature",
	    key,
	    typeParameters,
	    parameters,
	    typeAnnotation,
	    kind: null
	  };
	  const defs = NODE_FIELDS.TSMethodSignature;
	  validate(defs.key, node, "key", key, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsIndexSignature(parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSIndexSignature",
	    parameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSIndexSignature;
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsAnyKeyword() {
	  return {
	    type: "TSAnyKeyword"
	  };
	}
	function tsBooleanKeyword() {
	  return {
	    type: "TSBooleanKeyword"
	  };
	}
	function tsBigIntKeyword() {
	  return {
	    type: "TSBigIntKeyword"
	  };
	}
	function tsIntrinsicKeyword() {
	  return {
	    type: "TSIntrinsicKeyword"
	  };
	}
	function tsNeverKeyword() {
	  return {
	    type: "TSNeverKeyword"
	  };
	}
	function tsNullKeyword() {
	  return {
	    type: "TSNullKeyword"
	  };
	}
	function tsNumberKeyword() {
	  return {
	    type: "TSNumberKeyword"
	  };
	}
	function tsObjectKeyword() {
	  return {
	    type: "TSObjectKeyword"
	  };
	}
	function tsStringKeyword() {
	  return {
	    type: "TSStringKeyword"
	  };
	}
	function tsSymbolKeyword() {
	  return {
	    type: "TSSymbolKeyword"
	  };
	}
	function tsUndefinedKeyword() {
	  return {
	    type: "TSUndefinedKeyword"
	  };
	}
	function tsUnknownKeyword() {
	  return {
	    type: "TSUnknownKeyword"
	  };
	}
	function tsVoidKeyword() {
	  return {
	    type: "TSVoidKeyword"
	  };
	}
	function tsThisType() {
	  return {
	    type: "TSThisType"
	  };
	}
	function tsFunctionType(typeParameters = null, parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSFunctionType",
	    typeParameters,
	    parameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSFunctionType;
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsConstructorType(typeParameters = null, parameters, typeAnnotation = null) {
	  const node = {
	    type: "TSConstructorType",
	    typeParameters,
	    parameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSConstructorType;
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.parameters, node, "parameters", parameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsTypeReference(typeName, typeParameters = null) {
	  const node = {
	    type: "TSTypeReference",
	    typeName,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.TSTypeReference;
	  validate(defs.typeName, node, "typeName", typeName, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function tsTypePredicate(parameterName, typeAnnotation = null, asserts = null) {
	  const node = {
	    type: "TSTypePredicate",
	    parameterName,
	    typeAnnotation,
	    asserts
	  };
	  const defs = NODE_FIELDS.TSTypePredicate;
	  validate(defs.parameterName, node, "parameterName", parameterName, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  validate(defs.asserts, node, "asserts", asserts);
	  return node;
	}
	function tsTypeQuery(exprName, typeParameters = null) {
	  const node = {
	    type: "TSTypeQuery",
	    exprName,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.TSTypeQuery;
	  validate(defs.exprName, node, "exprName", exprName, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function tsTypeLiteral(members) {
	  const node = {
	    type: "TSTypeLiteral",
	    members
	  };
	  const defs = NODE_FIELDS.TSTypeLiteral;
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function tsArrayType(elementType) {
	  const node = {
	    type: "TSArrayType",
	    elementType
	  };
	  const defs = NODE_FIELDS.TSArrayType;
	  validate(defs.elementType, node, "elementType", elementType, 1);
	  return node;
	}
	function tsTupleType(elementTypes) {
	  const node = {
	    type: "TSTupleType",
	    elementTypes
	  };
	  const defs = NODE_FIELDS.TSTupleType;
	  validate(defs.elementTypes, node, "elementTypes", elementTypes, 1);
	  return node;
	}
	function tsOptionalType(typeAnnotation) {
	  const node = {
	    type: "TSOptionalType",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSOptionalType;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsRestType(typeAnnotation) {
	  const node = {
	    type: "TSRestType",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSRestType;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsNamedTupleMember(label, elementType, optional = false) {
	  const node = {
	    type: "TSNamedTupleMember",
	    label,
	    elementType,
	    optional
	  };
	  const defs = NODE_FIELDS.TSNamedTupleMember;
	  validate(defs.label, node, "label", label, 1);
	  validate(defs.elementType, node, "elementType", elementType, 1);
	  validate(defs.optional, node, "optional", optional);
	  return node;
	}
	function tsUnionType(types) {
	  const node = {
	    type: "TSUnionType",
	    types
	  };
	  const defs = NODE_FIELDS.TSUnionType;
	  validate(defs.types, node, "types", types, 1);
	  return node;
	}
	function tsIntersectionType(types) {
	  const node = {
	    type: "TSIntersectionType",
	    types
	  };
	  const defs = NODE_FIELDS.TSIntersectionType;
	  validate(defs.types, node, "types", types, 1);
	  return node;
	}
	function tsConditionalType(checkType, extendsType, trueType, falseType) {
	  const node = {
	    type: "TSConditionalType",
	    checkType,
	    extendsType,
	    trueType,
	    falseType
	  };
	  const defs = NODE_FIELDS.TSConditionalType;
	  validate(defs.checkType, node, "checkType", checkType, 1);
	  validate(defs.extendsType, node, "extendsType", extendsType, 1);
	  validate(defs.trueType, node, "trueType", trueType, 1);
	  validate(defs.falseType, node, "falseType", falseType, 1);
	  return node;
	}
	function tsInferType(typeParameter) {
	  const node = {
	    type: "TSInferType",
	    typeParameter
	  };
	  const defs = NODE_FIELDS.TSInferType;
	  validate(defs.typeParameter, node, "typeParameter", typeParameter, 1);
	  return node;
	}
	function tsParenthesizedType(typeAnnotation) {
	  const node = {
	    type: "TSParenthesizedType",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSParenthesizedType;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsTypeOperator(typeAnnotation) {
	  const node = {
	    type: "TSTypeOperator",
	    typeAnnotation,
	    operator: null
	  };
	  const defs = NODE_FIELDS.TSTypeOperator;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsIndexedAccessType(objectType, indexType) {
	  const node = {
	    type: "TSIndexedAccessType",
	    objectType,
	    indexType
	  };
	  const defs = NODE_FIELDS.TSIndexedAccessType;
	  validate(defs.objectType, node, "objectType", objectType, 1);
	  validate(defs.indexType, node, "indexType", indexType, 1);
	  return node;
	}
	function tsMappedType(typeParameter, typeAnnotation = null, nameType = null) {
	  const node = {
	    type: "TSMappedType",
	    typeParameter,
	    typeAnnotation,
	    nameType
	  };
	  const defs = NODE_FIELDS.TSMappedType;
	  validate(defs.typeParameter, node, "typeParameter", typeParameter, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  validate(defs.nameType, node, "nameType", nameType, 1);
	  return node;
	}
	function tsLiteralType(literal) {
	  const node = {
	    type: "TSLiteralType",
	    literal
	  };
	  const defs = NODE_FIELDS.TSLiteralType;
	  validate(defs.literal, node, "literal", literal, 1);
	  return node;
	}
	function tsExpressionWithTypeArguments(expression, typeParameters = null) {
	  const node = {
	    type: "TSExpressionWithTypeArguments",
	    expression,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.TSExpressionWithTypeArguments;
	  validate(defs.expression, node, "expression", expression, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function tsInterfaceDeclaration(id, typeParameters = null, _extends = null, body) {
	  const node = {
	    type: "TSInterfaceDeclaration",
	    id,
	    typeParameters,
	    extends: _extends,
	    body
	  };
	  const defs = NODE_FIELDS.TSInterfaceDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.extends, node, "extends", _extends, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function tsInterfaceBody(body) {
	  const node = {
	    type: "TSInterfaceBody",
	    body
	  };
	  const defs = NODE_FIELDS.TSInterfaceBody;
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function tsTypeAliasDeclaration(id, typeParameters = null, typeAnnotation) {
	  const node = {
	    type: "TSTypeAliasDeclaration",
	    id,
	    typeParameters,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSTypeAliasDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsInstantiationExpression(expression, typeParameters = null) {
	  const node = {
	    type: "TSInstantiationExpression",
	    expression,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.TSInstantiationExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function tsAsExpression(expression, typeAnnotation) {
	  const node = {
	    type: "TSAsExpression",
	    expression,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSAsExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsSatisfiesExpression(expression, typeAnnotation) {
	  const node = {
	    type: "TSSatisfiesExpression",
	    expression,
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSSatisfiesExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsTypeAssertion(typeAnnotation, expression) {
	  const node = {
	    type: "TSTypeAssertion",
	    typeAnnotation,
	    expression
	  };
	  const defs = NODE_FIELDS.TSTypeAssertion;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function tsEnumDeclaration(id, members) {
	  const node = {
	    type: "TSEnumDeclaration",
	    id,
	    members
	  };
	  const defs = NODE_FIELDS.TSEnumDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.members, node, "members", members, 1);
	  return node;
	}
	function tsEnumMember(id, initializer = null) {
	  const node = {
	    type: "TSEnumMember",
	    id,
	    initializer
	  };
	  const defs = NODE_FIELDS.TSEnumMember;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.initializer, node, "initializer", initializer, 1);
	  return node;
	}
	function tsModuleDeclaration(id, body) {
	  const node = {
	    type: "TSModuleDeclaration",
	    id,
	    body,
	    kind: null
	  };
	  const defs = NODE_FIELDS.TSModuleDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function tsModuleBlock(body) {
	  const node = {
	    type: "TSModuleBlock",
	    body
	  };
	  const defs = NODE_FIELDS.TSModuleBlock;
	  validate(defs.body, node, "body", body, 1);
	  return node;
	}
	function tsImportType(argument, qualifier = null, typeParameters = null) {
	  const node = {
	    type: "TSImportType",
	    argument,
	    qualifier,
	    typeParameters
	  };
	  const defs = NODE_FIELDS.TSImportType;
	  validate(defs.argument, node, "argument", argument, 1);
	  validate(defs.qualifier, node, "qualifier", qualifier, 1);
	  validate(defs.typeParameters, node, "typeParameters", typeParameters, 1);
	  return node;
	}
	function tsImportEqualsDeclaration(id, moduleReference) {
	  const node = {
	    type: "TSImportEqualsDeclaration",
	    id,
	    moduleReference,
	    isExport: null
	  };
	  const defs = NODE_FIELDS.TSImportEqualsDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  validate(defs.moduleReference, node, "moduleReference", moduleReference, 1);
	  return node;
	}
	function tsExternalModuleReference(expression) {
	  const node = {
	    type: "TSExternalModuleReference",
	    expression
	  };
	  const defs = NODE_FIELDS.TSExternalModuleReference;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function tsNonNullExpression(expression) {
	  const node = {
	    type: "TSNonNullExpression",
	    expression
	  };
	  const defs = NODE_FIELDS.TSNonNullExpression;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function tsExportAssignment(expression) {
	  const node = {
	    type: "TSExportAssignment",
	    expression
	  };
	  const defs = NODE_FIELDS.TSExportAssignment;
	  validate(defs.expression, node, "expression", expression, 1);
	  return node;
	}
	function tsNamespaceExportDeclaration(id) {
	  const node = {
	    type: "TSNamespaceExportDeclaration",
	    id
	  };
	  const defs = NODE_FIELDS.TSNamespaceExportDeclaration;
	  validate(defs.id, node, "id", id, 1);
	  return node;
	}
	function tsTypeAnnotation(typeAnnotation) {
	  const node = {
	    type: "TSTypeAnnotation",
	    typeAnnotation
	  };
	  const defs = NODE_FIELDS.TSTypeAnnotation;
	  validate(defs.typeAnnotation, node, "typeAnnotation", typeAnnotation, 1);
	  return node;
	}
	function tsTypeParameterInstantiation(params) {
	  const node = {
	    type: "TSTypeParameterInstantiation",
	    params
	  };
	  const defs = NODE_FIELDS.TSTypeParameterInstantiation;
	  validate(defs.params, node, "params", params, 1);
	  return node;
	}
	function tsTypeParameterDeclaration(params) {
	  const node = {
	    type: "TSTypeParameterDeclaration",
	    params
	  };
	  const defs = NODE_FIELDS.TSTypeParameterDeclaration;
	  validate(defs.params, node, "params", params, 1);
	  return node;
	}
	function tsTypeParameter(constraint = null, _default = null, name) {
	  const node = {
	    type: "TSTypeParameter",
	    constraint,
	    default: _default,
	    name
	  };
	  const defs = NODE_FIELDS.TSTypeParameter;
	  validate(defs.constraint, node, "constraint", constraint, 1);
	  validate(defs.default, node, "default", _default, 1);
	  validate(defs.name, node, "name", name);
	  return node;
	}
	function NumberLiteral(value) {
	  (0, _deprecationWarning.default)("NumberLiteral", "NumericLiteral", "The node type ");
	  return numericLiteral(value);
	}
	function RegexLiteral(pattern, flags = "") {
	  (0, _deprecationWarning.default)("RegexLiteral", "RegExpLiteral", "The node type ");
	  return regExpLiteral(pattern, flags);
	}
	function RestProperty(argument) {
	  (0, _deprecationWarning.default)("RestProperty", "RestElement", "The node type ");
	  return restElement(argument);
	}
	function SpreadProperty(argument) {
	  (0, _deprecationWarning.default)("SpreadProperty", "SpreadElement", "The node type ");
	  return spreadElement(argument);
	}

	
	return generated$2;
}

var hasRequiredCleanJSXElementLiteralChild;

function requireCleanJSXElementLiteralChild () {
	if (hasRequiredCleanJSXElementLiteralChild) return cleanJSXElementLiteralChild;
	hasRequiredCleanJSXElementLiteralChild = 1;

	Object.defineProperty(cleanJSXElementLiteralChild, "__esModule", {
	  value: true
	});
	cleanJSXElementLiteralChild.default = cleanJSXElementLiteralChild$1;
	var _index = requireGenerated$2();
	var _index2 = requireLib$1();
	function cleanJSXElementLiteralChild$1(child, args) {
	  const lines = child.value.split(/\r\n|\n|\r/);
	  let lastNonEmptyLine = 0;
	  for (let i = 0; i < lines.length; i++) {
	    if (/[^ \t]/.exec(lines[i])) {
	      lastNonEmptyLine = i;
	    }
	  }
	  let str = "";
	  for (let i = 0; i < lines.length; i++) {
	    const line = lines[i];
	    const isFirstLine = i === 0;
	    const isLastLine = i === lines.length - 1;
	    const isLastNonEmptyLine = i === lastNonEmptyLine;
	    let trimmedLine = line.replace(/\t/g, " ");
	    if (!isFirstLine) {
	      trimmedLine = trimmedLine.replace(/^ +/, "");
	    }
	    if (!isLastLine) {
	      trimmedLine = trimmedLine.replace(/ +$/, "");
	    }
	    if (trimmedLine) {
	      if (!isLastNonEmptyLine) {
	        trimmedLine += " ";
	      }
	      str += trimmedLine;
	    }
	  }
	  if (str) args.push((0, _index2.inherits)((0, _index.stringLiteral)(str), child));
	}

	
	return cleanJSXElementLiteralChild;
}

var hasRequiredBuildChildren;

function requireBuildChildren () {
	if (hasRequiredBuildChildren) return buildChildren;
	hasRequiredBuildChildren = 1;

	Object.defineProperty(buildChildren, "__esModule", {
	  value: true
	});
	buildChildren.default = buildChildren$1;
	var _index = requireGenerated$3();
	var _cleanJSXElementLiteralChild = requireCleanJSXElementLiteralChild();
	function buildChildren$1(node) {
	  const elements = [];
	  for (let i = 0; i < node.children.length; i++) {
	    let child = node.children[i];
	    if ((0, _index.isJSXText)(child)) {
	      (0, _cleanJSXElementLiteralChild.default)(child, elements);
	      continue;
	    }
	    if ((0, _index.isJSXExpressionContainer)(child)) child = child.expression;
	    if ((0, _index.isJSXEmptyExpression)(child)) continue;
	    elements.push(child);
	  }
	  return elements;
	}

	
	return buildChildren;
}

var assertNode = {};

var isNode = {};

var hasRequiredIsNode;

function requireIsNode () {
	if (hasRequiredIsNode) return isNode;
	hasRequiredIsNode = 1;

	Object.defineProperty(isNode, "__esModule", {
	  value: true
	});
	isNode.default = isNode$1;
	var _index = requireDefinitions();
	function isNode$1(node) {
	  return !!(node && _index.VISITOR_KEYS[node.type]);
	}

	
	return isNode;
}

var hasRequiredAssertNode;

function requireAssertNode () {
	if (hasRequiredAssertNode) return assertNode;
	hasRequiredAssertNode = 1;

	Object.defineProperty(assertNode, "__esModule", {
	  value: true
	});
	assertNode.default = assertNode$1;
	var _isNode = requireIsNode();
	function assertNode$1(node) {
	  if (!(0, _isNode.default)(node)) {
	    var _node$type;
	    const type = (_node$type = node == null ? void 0 : node.type) != null ? _node$type : JSON.stringify(node);
	    throw new TypeError(`Not a valid node of type "${type}"`);
	  }
	}

	
	return assertNode;
}

var generated$1 = {};

var hasRequiredGenerated$1;

function requireGenerated$1 () {
	if (hasRequiredGenerated$1) return generated$1;
	hasRequiredGenerated$1 = 1;

	Object.defineProperty(generated$1, "__esModule", {
	  value: true
	});
	generated$1.assertAccessor = assertAccessor;
	generated$1.assertAnyTypeAnnotation = assertAnyTypeAnnotation;
	generated$1.assertArgumentPlaceholder = assertArgumentPlaceholder;
	generated$1.assertArrayExpression = assertArrayExpression;
	generated$1.assertArrayPattern = assertArrayPattern;
	generated$1.assertArrayTypeAnnotation = assertArrayTypeAnnotation;
	generated$1.assertArrowFunctionExpression = assertArrowFunctionExpression;
	generated$1.assertAssignmentExpression = assertAssignmentExpression;
	generated$1.assertAssignmentPattern = assertAssignmentPattern;
	generated$1.assertAwaitExpression = assertAwaitExpression;
	generated$1.assertBigIntLiteral = assertBigIntLiteral;
	generated$1.assertBinary = assertBinary;
	generated$1.assertBinaryExpression = assertBinaryExpression;
	generated$1.assertBindExpression = assertBindExpression;
	generated$1.assertBlock = assertBlock;
	generated$1.assertBlockParent = assertBlockParent;
	generated$1.assertBlockStatement = assertBlockStatement;
	generated$1.assertBooleanLiteral = assertBooleanLiteral;
	generated$1.assertBooleanLiteralTypeAnnotation = assertBooleanLiteralTypeAnnotation;
	generated$1.assertBooleanTypeAnnotation = assertBooleanTypeAnnotation;
	generated$1.assertBreakStatement = assertBreakStatement;
	generated$1.assertCallExpression = assertCallExpression;
	generated$1.assertCatchClause = assertCatchClause;
	generated$1.assertClass = assertClass;
	generated$1.assertClassAccessorProperty = assertClassAccessorProperty;
	generated$1.assertClassBody = assertClassBody;
	generated$1.assertClassDeclaration = assertClassDeclaration;
	generated$1.assertClassExpression = assertClassExpression;
	generated$1.assertClassImplements = assertClassImplements;
	generated$1.assertClassMethod = assertClassMethod;
	generated$1.assertClassPrivateMethod = assertClassPrivateMethod;
	generated$1.assertClassPrivateProperty = assertClassPrivateProperty;
	generated$1.assertClassProperty = assertClassProperty;
	generated$1.assertCompletionStatement = assertCompletionStatement;
	generated$1.assertConditional = assertConditional;
	generated$1.assertConditionalExpression = assertConditionalExpression;
	generated$1.assertContinueStatement = assertContinueStatement;
	generated$1.assertDebuggerStatement = assertDebuggerStatement;
	generated$1.assertDecimalLiteral = assertDecimalLiteral;
	generated$1.assertDeclaration = assertDeclaration;
	generated$1.assertDeclareClass = assertDeclareClass;
	generated$1.assertDeclareExportAllDeclaration = assertDeclareExportAllDeclaration;
	generated$1.assertDeclareExportDeclaration = assertDeclareExportDeclaration;
	generated$1.assertDeclareFunction = assertDeclareFunction;
	generated$1.assertDeclareInterface = assertDeclareInterface;
	generated$1.assertDeclareModule = assertDeclareModule;
	generated$1.assertDeclareModuleExports = assertDeclareModuleExports;
	generated$1.assertDeclareOpaqueType = assertDeclareOpaqueType;
	generated$1.assertDeclareTypeAlias = assertDeclareTypeAlias;
	generated$1.assertDeclareVariable = assertDeclareVariable;
	generated$1.assertDeclaredPredicate = assertDeclaredPredicate;
	generated$1.assertDecorator = assertDecorator;
	generated$1.assertDirective = assertDirective;
	generated$1.assertDirectiveLiteral = assertDirectiveLiteral;
	generated$1.assertDoExpression = assertDoExpression;
	generated$1.assertDoWhileStatement = assertDoWhileStatement;
	generated$1.assertEmptyStatement = assertEmptyStatement;
	generated$1.assertEmptyTypeAnnotation = assertEmptyTypeAnnotation;
	generated$1.assertEnumBody = assertEnumBody;
	generated$1.assertEnumBooleanBody = assertEnumBooleanBody;
	generated$1.assertEnumBooleanMember = assertEnumBooleanMember;
	generated$1.assertEnumDeclaration = assertEnumDeclaration;
	generated$1.assertEnumDefaultedMember = assertEnumDefaultedMember;
	generated$1.assertEnumMember = assertEnumMember;
	generated$1.assertEnumNumberBody = assertEnumNumberBody;
	generated$1.assertEnumNumberMember = assertEnumNumberMember;
	generated$1.assertEnumStringBody = assertEnumStringBody;
	generated$1.assertEnumStringMember = assertEnumStringMember;
	generated$1.assertEnumSymbolBody = assertEnumSymbolBody;
	generated$1.assertExistsTypeAnnotation = assertExistsTypeAnnotation;
	generated$1.assertExportAllDeclaration = assertExportAllDeclaration;
	generated$1.assertExportDeclaration = assertExportDeclaration;
	generated$1.assertExportDefaultDeclaration = assertExportDefaultDeclaration;
	generated$1.assertExportDefaultSpecifier = assertExportDefaultSpecifier;
	generated$1.assertExportNamedDeclaration = assertExportNamedDeclaration;
	generated$1.assertExportNamespaceSpecifier = assertExportNamespaceSpecifier;
	generated$1.assertExportSpecifier = assertExportSpecifier;
	generated$1.assertExpression = assertExpression;
	generated$1.assertExpressionStatement = assertExpressionStatement;
	generated$1.assertExpressionWrapper = assertExpressionWrapper;
	generated$1.assertFile = assertFile;
	generated$1.assertFlow = assertFlow;
	generated$1.assertFlowBaseAnnotation = assertFlowBaseAnnotation;
	generated$1.assertFlowDeclaration = assertFlowDeclaration;
	generated$1.assertFlowPredicate = assertFlowPredicate;
	generated$1.assertFlowType = assertFlowType;
	generated$1.assertFor = assertFor;
	generated$1.assertForInStatement = assertForInStatement;
	generated$1.assertForOfStatement = assertForOfStatement;
	generated$1.assertForStatement = assertForStatement;
	generated$1.assertForXStatement = assertForXStatement;
	generated$1.assertFunction = assertFunction;
	generated$1.assertFunctionDeclaration = assertFunctionDeclaration;
	generated$1.assertFunctionExpression = assertFunctionExpression;
	generated$1.assertFunctionParent = assertFunctionParent;
	generated$1.assertFunctionTypeAnnotation = assertFunctionTypeAnnotation;
	generated$1.assertFunctionTypeParam = assertFunctionTypeParam;
	generated$1.assertGenericTypeAnnotation = assertGenericTypeAnnotation;
	generated$1.assertIdentifier = assertIdentifier;
	generated$1.assertIfStatement = assertIfStatement;
	generated$1.assertImmutable = assertImmutable;
	generated$1.assertImport = assertImport;
	generated$1.assertImportAttribute = assertImportAttribute;
	generated$1.assertImportDeclaration = assertImportDeclaration;
	generated$1.assertImportDefaultSpecifier = assertImportDefaultSpecifier;
	generated$1.assertImportExpression = assertImportExpression;
	generated$1.assertImportNamespaceSpecifier = assertImportNamespaceSpecifier;
	generated$1.assertImportOrExportDeclaration = assertImportOrExportDeclaration;
	generated$1.assertImportSpecifier = assertImportSpecifier;
	generated$1.assertIndexedAccessType = assertIndexedAccessType;
	generated$1.assertInferredPredicate = assertInferredPredicate;
	generated$1.assertInterfaceDeclaration = assertInterfaceDeclaration;
	generated$1.assertInterfaceExtends = assertInterfaceExtends;
	generated$1.assertInterfaceTypeAnnotation = assertInterfaceTypeAnnotation;
	generated$1.assertInterpreterDirective = assertInterpreterDirective;
	generated$1.assertIntersectionTypeAnnotation = assertIntersectionTypeAnnotation;
	generated$1.assertJSX = assertJSX;
	generated$1.assertJSXAttribute = assertJSXAttribute;
	generated$1.assertJSXClosingElement = assertJSXClosingElement;
	generated$1.assertJSXClosingFragment = assertJSXClosingFragment;
	generated$1.assertJSXElement = assertJSXElement;
	generated$1.assertJSXEmptyExpression = assertJSXEmptyExpression;
	generated$1.assertJSXExpressionContainer = assertJSXExpressionContainer;
	generated$1.assertJSXFragment = assertJSXFragment;
	generated$1.assertJSXIdentifier = assertJSXIdentifier;
	generated$1.assertJSXMemberExpression = assertJSXMemberExpression;
	generated$1.assertJSXNamespacedName = assertJSXNamespacedName;
	generated$1.assertJSXOpeningElement = assertJSXOpeningElement;
	generated$1.assertJSXOpeningFragment = assertJSXOpeningFragment;
	generated$1.assertJSXSpreadAttribute = assertJSXSpreadAttribute;
	generated$1.assertJSXSpreadChild = assertJSXSpreadChild;
	generated$1.assertJSXText = assertJSXText;
	generated$1.assertLVal = assertLVal;
	generated$1.assertLabeledStatement = assertLabeledStatement;
	generated$1.assertLiteral = assertLiteral;
	generated$1.assertLogicalExpression = assertLogicalExpression;
	generated$1.assertLoop = assertLoop;
	generated$1.assertMemberExpression = assertMemberExpression;
	generated$1.assertMetaProperty = assertMetaProperty;
	generated$1.assertMethod = assertMethod;
	generated$1.assertMiscellaneous = assertMiscellaneous;
	generated$1.assertMixedTypeAnnotation = assertMixedTypeAnnotation;
	generated$1.assertModuleDeclaration = assertModuleDeclaration;
	generated$1.assertModuleExpression = assertModuleExpression;
	generated$1.assertModuleSpecifier = assertModuleSpecifier;
	generated$1.assertNewExpression = assertNewExpression;
	generated$1.assertNoop = assertNoop;
	generated$1.assertNullLiteral = assertNullLiteral;
	generated$1.assertNullLiteralTypeAnnotation = assertNullLiteralTypeAnnotation;
	generated$1.assertNullableTypeAnnotation = assertNullableTypeAnnotation;
	generated$1.assertNumberLiteral = assertNumberLiteral;
	generated$1.assertNumberLiteralTypeAnnotation = assertNumberLiteralTypeAnnotation;
	generated$1.assertNumberTypeAnnotation = assertNumberTypeAnnotation;
	generated$1.assertNumericLiteral = assertNumericLiteral;
	generated$1.assertObjectExpression = assertObjectExpression;
	generated$1.assertObjectMember = assertObjectMember;
	generated$1.assertObjectMethod = assertObjectMethod;
	generated$1.assertObjectPattern = assertObjectPattern;
	generated$1.assertObjectProperty = assertObjectProperty;
	generated$1.assertObjectTypeAnnotation = assertObjectTypeAnnotation;
	generated$1.assertObjectTypeCallProperty = assertObjectTypeCallProperty;
	generated$1.assertObjectTypeIndexer = assertObjectTypeIndexer;
	generated$1.assertObjectTypeInternalSlot = assertObjectTypeInternalSlot;
	generated$1.assertObjectTypeProperty = assertObjectTypeProperty;
	generated$1.assertObjectTypeSpreadProperty = assertObjectTypeSpreadProperty;
	generated$1.assertOpaqueType = assertOpaqueType;
	generated$1.assertOptionalCallExpression = assertOptionalCallExpression;
	generated$1.assertOptionalIndexedAccessType = assertOptionalIndexedAccessType;
	generated$1.assertOptionalMemberExpression = assertOptionalMemberExpression;
	generated$1.assertParenthesizedExpression = assertParenthesizedExpression;
	generated$1.assertPattern = assertPattern;
	generated$1.assertPatternLike = assertPatternLike;
	generated$1.assertPipelineBareFunction = assertPipelineBareFunction;
	generated$1.assertPipelinePrimaryTopicReference = assertPipelinePrimaryTopicReference;
	generated$1.assertPipelineTopicExpression = assertPipelineTopicExpression;
	generated$1.assertPlaceholder = assertPlaceholder;
	generated$1.assertPrivate = assertPrivate;
	generated$1.assertPrivateName = assertPrivateName;
	generated$1.assertProgram = assertProgram;
	generated$1.assertProperty = assertProperty;
	generated$1.assertPureish = assertPureish;
	generated$1.assertQualifiedTypeIdentifier = assertQualifiedTypeIdentifier;
	generated$1.assertRecordExpression = assertRecordExpression;
	generated$1.assertRegExpLiteral = assertRegExpLiteral;
	generated$1.assertRegexLiteral = assertRegexLiteral;
	generated$1.assertRestElement = assertRestElement;
	generated$1.assertRestProperty = assertRestProperty;
	generated$1.assertReturnStatement = assertReturnStatement;
	generated$1.assertScopable = assertScopable;
	generated$1.assertSequenceExpression = assertSequenceExpression;
	generated$1.assertSpreadElement = assertSpreadElement;
	generated$1.assertSpreadProperty = assertSpreadProperty;
	generated$1.assertStandardized = assertStandardized;
	generated$1.assertStatement = assertStatement;
	generated$1.assertStaticBlock = assertStaticBlock;
	generated$1.assertStringLiteral = assertStringLiteral;
	generated$1.assertStringLiteralTypeAnnotation = assertStringLiteralTypeAnnotation;
	generated$1.assertStringTypeAnnotation = assertStringTypeAnnotation;
	generated$1.assertSuper = assertSuper;
	generated$1.assertSwitchCase = assertSwitchCase;
	generated$1.assertSwitchStatement = assertSwitchStatement;
	generated$1.assertSymbolTypeAnnotation = assertSymbolTypeAnnotation;
	generated$1.assertTSAnyKeyword = assertTSAnyKeyword;
	generated$1.assertTSArrayType = assertTSArrayType;
	generated$1.assertTSAsExpression = assertTSAsExpression;
	generated$1.assertTSBaseType = assertTSBaseType;
	generated$1.assertTSBigIntKeyword = assertTSBigIntKeyword;
	generated$1.assertTSBooleanKeyword = assertTSBooleanKeyword;
	generated$1.assertTSCallSignatureDeclaration = assertTSCallSignatureDeclaration;
	generated$1.assertTSConditionalType = assertTSConditionalType;
	generated$1.assertTSConstructSignatureDeclaration = assertTSConstructSignatureDeclaration;
	generated$1.assertTSConstructorType = assertTSConstructorType;
	generated$1.assertTSDeclareFunction = assertTSDeclareFunction;
	generated$1.assertTSDeclareMethod = assertTSDeclareMethod;
	generated$1.assertTSEntityName = assertTSEntityName;
	generated$1.assertTSEnumDeclaration = assertTSEnumDeclaration;
	generated$1.assertTSEnumMember = assertTSEnumMember;
	generated$1.assertTSExportAssignment = assertTSExportAssignment;
	generated$1.assertTSExpressionWithTypeArguments = assertTSExpressionWithTypeArguments;
	generated$1.assertTSExternalModuleReference = assertTSExternalModuleReference;
	generated$1.assertTSFunctionType = assertTSFunctionType;
	generated$1.assertTSImportEqualsDeclaration = assertTSImportEqualsDeclaration;
	generated$1.assertTSImportType = assertTSImportType;
	generated$1.assertTSIndexSignature = assertTSIndexSignature;
	generated$1.assertTSIndexedAccessType = assertTSIndexedAccessType;
	generated$1.assertTSInferType = assertTSInferType;
	generated$1.assertTSInstantiationExpression = assertTSInstantiationExpression;
	generated$1.assertTSInterfaceBody = assertTSInterfaceBody;
	generated$1.assertTSInterfaceDeclaration = assertTSInterfaceDeclaration;
	generated$1.assertTSIntersectionType = assertTSIntersectionType;
	generated$1.assertTSIntrinsicKeyword = assertTSIntrinsicKeyword;
	generated$1.assertTSLiteralType = assertTSLiteralType;
	generated$1.assertTSMappedType = assertTSMappedType;
	generated$1.assertTSMethodSignature = assertTSMethodSignature;
	generated$1.assertTSModuleBlock = assertTSModuleBlock;
	generated$1.assertTSModuleDeclaration = assertTSModuleDeclaration;
	generated$1.assertTSNamedTupleMember = assertTSNamedTupleMember;
	generated$1.assertTSNamespaceExportDeclaration = assertTSNamespaceExportDeclaration;
	generated$1.assertTSNeverKeyword = assertTSNeverKeyword;
	generated$1.assertTSNonNullExpression = assertTSNonNullExpression;
	generated$1.assertTSNullKeyword = assertTSNullKeyword;
	generated$1.assertTSNumberKeyword = assertTSNumberKeyword;
	generated$1.assertTSObjectKeyword = assertTSObjectKeyword;
	generated$1.assertTSOptionalType = assertTSOptionalType;
	generated$1.assertTSParameterProperty = assertTSParameterProperty;
	generated$1.assertTSParenthesizedType = assertTSParenthesizedType;
	generated$1.assertTSPropertySignature = assertTSPropertySignature;
	generated$1.assertTSQualifiedName = assertTSQualifiedName;
	generated$1.assertTSRestType = assertTSRestType;
	generated$1.assertTSSatisfiesExpression = assertTSSatisfiesExpression;
	generated$1.assertTSStringKeyword = assertTSStringKeyword;
	generated$1.assertTSSymbolKeyword = assertTSSymbolKeyword;
	generated$1.assertTSThisType = assertTSThisType;
	generated$1.assertTSTupleType = assertTSTupleType;
	generated$1.assertTSType = assertTSType;
	generated$1.assertTSTypeAliasDeclaration = assertTSTypeAliasDeclaration;
	generated$1.assertTSTypeAnnotation = assertTSTypeAnnotation;
	generated$1.assertTSTypeAssertion = assertTSTypeAssertion;
	generated$1.assertTSTypeElement = assertTSTypeElement;
	generated$1.assertTSTypeLiteral = assertTSTypeLiteral;
	generated$1.assertTSTypeOperator = assertTSTypeOperator;
	generated$1.assertTSTypeParameter = assertTSTypeParameter;
	generated$1.assertTSTypeParameterDeclaration = assertTSTypeParameterDeclaration;
	generated$1.assertTSTypeParameterInstantiation = assertTSTypeParameterInstantiation;
	generated$1.assertTSTypePredicate = assertTSTypePredicate;
	generated$1.assertTSTypeQuery = assertTSTypeQuery;
	generated$1.assertTSTypeReference = assertTSTypeReference;
	generated$1.assertTSUndefinedKeyword = assertTSUndefinedKeyword;
	generated$1.assertTSUnionType = assertTSUnionType;
	generated$1.assertTSUnknownKeyword = assertTSUnknownKeyword;
	generated$1.assertTSVoidKeyword = assertTSVoidKeyword;
	generated$1.assertTaggedTemplateExpression = assertTaggedTemplateExpression;
	generated$1.assertTemplateElement = assertTemplateElement;
	generated$1.assertTemplateLiteral = assertTemplateLiteral;
	generated$1.assertTerminatorless = assertTerminatorless;
	generated$1.assertThisExpression = assertThisExpression;
	generated$1.assertThisTypeAnnotation = assertThisTypeAnnotation;
	generated$1.assertThrowStatement = assertThrowStatement;
	generated$1.assertTopicReference = assertTopicReference;
	generated$1.assertTryStatement = assertTryStatement;
	generated$1.assertTupleExpression = assertTupleExpression;
	generated$1.assertTupleTypeAnnotation = assertTupleTypeAnnotation;
	generated$1.assertTypeAlias = assertTypeAlias;
	generated$1.assertTypeAnnotation = assertTypeAnnotation;
	generated$1.assertTypeCastExpression = assertTypeCastExpression;
	generated$1.assertTypeParameter = assertTypeParameter;
	generated$1.assertTypeParameterDeclaration = assertTypeParameterDeclaration;
	generated$1.assertTypeParameterInstantiation = assertTypeParameterInstantiation;
	generated$1.assertTypeScript = assertTypeScript;
	generated$1.assertTypeofTypeAnnotation = assertTypeofTypeAnnotation;
	generated$1.assertUnaryExpression = assertUnaryExpression;
	generated$1.assertUnaryLike = assertUnaryLike;
	generated$1.assertUnionTypeAnnotation = assertUnionTypeAnnotation;
	generated$1.assertUpdateExpression = assertUpdateExpression;
	generated$1.assertUserWhitespacable = assertUserWhitespacable;
	generated$1.assertV8IntrinsicIdentifier = assertV8IntrinsicIdentifier;
	generated$1.assertVariableDeclaration = assertVariableDeclaration;
	generated$1.assertVariableDeclarator = assertVariableDeclarator;
	generated$1.assertVariance = assertVariance;
	generated$1.assertVoidTypeAnnotation = assertVoidTypeAnnotation;
	generated$1.assertWhile = assertWhile;
	generated$1.assertWhileStatement = assertWhileStatement;
	generated$1.assertWithStatement = assertWithStatement;
	generated$1.assertYieldExpression = assertYieldExpression;
	var _is = requireIs();
	var _deprecationWarning = requireDeprecationWarning();
	function assert(type, node, opts) {
	  if (!(0, _is.default)(type, node, opts)) {
	    throw new Error(`Expected type "${type}" with option ${JSON.stringify(opts)}, ` + `but instead got "${node.type}".`);
	  }
	}
	function assertArrayExpression(node, opts) {
	  assert("ArrayExpression", node, opts);
	}
	function assertAssignmentExpression(node, opts) {
	  assert("AssignmentExpression", node, opts);
	}
	function assertBinaryExpression(node, opts) {
	  assert("BinaryExpression", node, opts);
	}
	function assertInterpreterDirective(node, opts) {
	  assert("InterpreterDirective", node, opts);
	}
	function assertDirective(node, opts) {
	  assert("Directive", node, opts);
	}
	function assertDirectiveLiteral(node, opts) {
	  assert("DirectiveLiteral", node, opts);
	}
	function assertBlockStatement(node, opts) {
	  assert("BlockStatement", node, opts);
	}
	function assertBreakStatement(node, opts) {
	  assert("BreakStatement", node, opts);
	}
	function assertCallExpression(node, opts) {
	  assert("CallExpression", node, opts);
	}
	function assertCatchClause(node, opts) {
	  assert("CatchClause", node, opts);
	}
	function assertConditionalExpression(node, opts) {
	  assert("ConditionalExpression", node, opts);
	}
	function assertContinueStatement(node, opts) {
	  assert("ContinueStatement", node, opts);
	}
	function assertDebuggerStatement(node, opts) {
	  assert("DebuggerStatement", node, opts);
	}
	function assertDoWhileStatement(node, opts) {
	  assert("DoWhileStatement", node, opts);
	}
	function assertEmptyStatement(node, opts) {
	  assert("EmptyStatement", node, opts);
	}
	function assertExpressionStatement(node, opts) {
	  assert("ExpressionStatement", node, opts);
	}
	function assertFile(node, opts) {
	  assert("File", node, opts);
	}
	function assertForInStatement(node, opts) {
	  assert("ForInStatement", node, opts);
	}
	function assertForStatement(node, opts) {
	  assert("ForStatement", node, opts);
	}
	function assertFunctionDeclaration(node, opts) {
	  assert("FunctionDeclaration", node, opts);
	}
	function assertFunctionExpression(node, opts) {
	  assert("FunctionExpression", node, opts);
	}
	function assertIdentifier(node, opts) {
	  assert("Identifier", node, opts);
	}
	function assertIfStatement(node, opts) {
	  assert("IfStatement", node, opts);
	}
	function assertLabeledStatement(node, opts) {
	  assert("LabeledStatement", node, opts);
	}
	function assertStringLiteral(node, opts) {
	  assert("StringLiteral", node, opts);
	}
	function assertNumericLiteral(node, opts) {
	  assert("NumericLiteral", node, opts);
	}
	function assertNullLiteral(node, opts) {
	  assert("NullLiteral", node, opts);
	}
	function assertBooleanLiteral(node, opts) {
	  assert("BooleanLiteral", node, opts);
	}
	function assertRegExpLiteral(node, opts) {
	  assert("RegExpLiteral", node, opts);
	}
	function assertLogicalExpression(node, opts) {
	  assert("LogicalExpression", node, opts);
	}
	function assertMemberExpression(node, opts) {
	  assert("MemberExpression", node, opts);
	}
	function assertNewExpression(node, opts) {
	  assert("NewExpression", node, opts);
	}
	function assertProgram(node, opts) {
	  assert("Program", node, opts);
	}
	function assertObjectExpression(node, opts) {
	  assert("ObjectExpression", node, opts);
	}
	function assertObjectMethod(node, opts) {
	  assert("ObjectMethod", node, opts);
	}
	function assertObjectProperty(node, opts) {
	  assert("ObjectProperty", node, opts);
	}
	function assertRestElement(node, opts) {
	  assert("RestElement", node, opts);
	}
	function assertReturnStatement(node, opts) {
	  assert("ReturnStatement", node, opts);
	}
	function assertSequenceExpression(node, opts) {
	  assert("SequenceExpression", node, opts);
	}
	function assertParenthesizedExpression(node, opts) {
	  assert("ParenthesizedExpression", node, opts);
	}
	function assertSwitchCase(node, opts) {
	  assert("SwitchCase", node, opts);
	}
	function assertSwitchStatement(node, opts) {
	  assert("SwitchStatement", node, opts);
	}
	function assertThisExpression(node, opts) {
	  assert("ThisExpression", node, opts);
	}
	function assertThrowStatement(node, opts) {
	  assert("ThrowStatement", node, opts);
	}
	function assertTryStatement(node, opts) {
	  assert("TryStatement", node, opts);
	}
	function assertUnaryExpression(node, opts) {
	  assert("UnaryExpression", node, opts);
	}
	function assertUpdateExpression(node, opts) {
	  assert("UpdateExpression", node, opts);
	}
	function assertVariableDeclaration(node, opts) {
	  assert("VariableDeclaration", node, opts);
	}
	function assertVariableDeclarator(node, opts) {
	  assert("VariableDeclarator", node, opts);
	}
	function assertWhileStatement(node, opts) {
	  assert("WhileStatement", node, opts);
	}
	function assertWithStatement(node, opts) {
	  assert("WithStatement", node, opts);
	}
	function assertAssignmentPattern(node, opts) {
	  assert("AssignmentPattern", node, opts);
	}
	function assertArrayPattern(node, opts) {
	  assert("ArrayPattern", node, opts);
	}
	function assertArrowFunctionExpression(node, opts) {
	  assert("ArrowFunctionExpression", node, opts);
	}
	function assertClassBody(node, opts) {
	  assert("ClassBody", node, opts);
	}
	function assertClassExpression(node, opts) {
	  assert("ClassExpression", node, opts);
	}
	function assertClassDeclaration(node, opts) {
	  assert("ClassDeclaration", node, opts);
	}
	function assertExportAllDeclaration(node, opts) {
	  assert("ExportAllDeclaration", node, opts);
	}
	function assertExportDefaultDeclaration(node, opts) {
	  assert("ExportDefaultDeclaration", node, opts);
	}
	function assertExportNamedDeclaration(node, opts) {
	  assert("ExportNamedDeclaration", node, opts);
	}
	function assertExportSpecifier(node, opts) {
	  assert("ExportSpecifier", node, opts);
	}
	function assertForOfStatement(node, opts) {
	  assert("ForOfStatement", node, opts);
	}
	function assertImportDeclaration(node, opts) {
	  assert("ImportDeclaration", node, opts);
	}
	function assertImportDefaultSpecifier(node, opts) {
	  assert("ImportDefaultSpecifier", node, opts);
	}
	function assertImportNamespaceSpecifier(node, opts) {
	  assert("ImportNamespaceSpecifier", node, opts);
	}
	function assertImportSpecifier(node, opts) {
	  assert("ImportSpecifier", node, opts);
	}
	function assertImportExpression(node, opts) {
	  assert("ImportExpression", node, opts);
	}
	function assertMetaProperty(node, opts) {
	  assert("MetaProperty", node, opts);
	}
	function assertClassMethod(node, opts) {
	  assert("ClassMethod", node, opts);
	}
	function assertObjectPattern(node, opts) {
	  assert("ObjectPattern", node, opts);
	}
	function assertSpreadElement(node, opts) {
	  assert("SpreadElement", node, opts);
	}
	function assertSuper(node, opts) {
	  assert("Super", node, opts);
	}
	function assertTaggedTemplateExpression(node, opts) {
	  assert("TaggedTemplateExpression", node, opts);
	}
	function assertTemplateElement(node, opts) {
	  assert("TemplateElement", node, opts);
	}
	function assertTemplateLiteral(node, opts) {
	  assert("TemplateLiteral", node, opts);
	}
	function assertYieldExpression(node, opts) {
	  assert("YieldExpression", node, opts);
	}
	function assertAwaitExpression(node, opts) {
	  assert("AwaitExpression", node, opts);
	}
	function assertImport(node, opts) {
	  assert("Import", node, opts);
	}
	function assertBigIntLiteral(node, opts) {
	  assert("BigIntLiteral", node, opts);
	}
	function assertExportNamespaceSpecifier(node, opts) {
	  assert("ExportNamespaceSpecifier", node, opts);
	}
	function assertOptionalMemberExpression(node, opts) {
	  assert("OptionalMemberExpression", node, opts);
	}
	function assertOptionalCallExpression(node, opts) {
	  assert("OptionalCallExpression", node, opts);
	}
	function assertClassProperty(node, opts) {
	  assert("ClassProperty", node, opts);
	}
	function assertClassAccessorProperty(node, opts) {
	  assert("ClassAccessorProperty", node, opts);
	}
	function assertClassPrivateProperty(node, opts) {
	  assert("ClassPrivateProperty", node, opts);
	}
	function assertClassPrivateMethod(node, opts) {
	  assert("ClassPrivateMethod", node, opts);
	}
	function assertPrivateName(node, opts) {
	  assert("PrivateName", node, opts);
	}
	function assertStaticBlock(node, opts) {
	  assert("StaticBlock", node, opts);
	}
	function assertAnyTypeAnnotation(node, opts) {
	  assert("AnyTypeAnnotation", node, opts);
	}
	function assertArrayTypeAnnotation(node, opts) {
	  assert("ArrayTypeAnnotation", node, opts);
	}
	function assertBooleanTypeAnnotation(node, opts) {
	  assert("BooleanTypeAnnotation", node, opts);
	}
	function assertBooleanLiteralTypeAnnotation(node, opts) {
	  assert("BooleanLiteralTypeAnnotation", node, opts);
	}
	function assertNullLiteralTypeAnnotation(node, opts) {
	  assert("NullLiteralTypeAnnotation", node, opts);
	}
	function assertClassImplements(node, opts) {
	  assert("ClassImplements", node, opts);
	}
	function assertDeclareClass(node, opts) {
	  assert("DeclareClass", node, opts);
	}
	function assertDeclareFunction(node, opts) {
	  assert("DeclareFunction", node, opts);
	}
	function assertDeclareInterface(node, opts) {
	  assert("DeclareInterface", node, opts);
	}
	function assertDeclareModule(node, opts) {
	  assert("DeclareModule", node, opts);
	}
	function assertDeclareModuleExports(node, opts) {
	  assert("DeclareModuleExports", node, opts);
	}
	function assertDeclareTypeAlias(node, opts) {
	  assert("DeclareTypeAlias", node, opts);
	}
	function assertDeclareOpaqueType(node, opts) {
	  assert("DeclareOpaqueType", node, opts);
	}
	function assertDeclareVariable(node, opts) {
	  assert("DeclareVariable", node, opts);
	}
	function assertDeclareExportDeclaration(node, opts) {
	  assert("DeclareExportDeclaration", node, opts);
	}
	function assertDeclareExportAllDeclaration(node, opts) {
	  assert("DeclareExportAllDeclaration", node, opts);
	}
	function assertDeclaredPredicate(node, opts) {
	  assert("DeclaredPredicate", node, opts);
	}
	function assertExistsTypeAnnotation(node, opts) {
	  assert("ExistsTypeAnnotation", node, opts);
	}
	function assertFunctionTypeAnnotation(node, opts) {
	  assert("FunctionTypeAnnotation", node, opts);
	}
	function assertFunctionTypeParam(node, opts) {
	  assert("FunctionTypeParam", node, opts);
	}
	function assertGenericTypeAnnotation(node, opts) {
	  assert("GenericTypeAnnotation", node, opts);
	}
	function assertInferredPredicate(node, opts) {
	  assert("InferredPredicate", node, opts);
	}
	function assertInterfaceExtends(node, opts) {
	  assert("InterfaceExtends", node, opts);
	}
	function assertInterfaceDeclaration(node, opts) {
	  assert("InterfaceDeclaration", node, opts);
	}
	function assertInterfaceTypeAnnotation(node, opts) {
	  assert("InterfaceTypeAnnotation", node, opts);
	}
	function assertIntersectionTypeAnnotation(node, opts) {
	  assert("IntersectionTypeAnnotation", node, opts);
	}
	function assertMixedTypeAnnotation(node, opts) {
	  assert("MixedTypeAnnotation", node, opts);
	}
	function assertEmptyTypeAnnotation(node, opts) {
	  assert("EmptyTypeAnnotation", node, opts);
	}
	function assertNullableTypeAnnotation(node, opts) {
	  assert("NullableTypeAnnotation", node, opts);
	}
	function assertNumberLiteralTypeAnnotation(node, opts) {
	  assert("NumberLiteralTypeAnnotation", node, opts);
	}
	function assertNumberTypeAnnotation(node, opts) {
	  assert("NumberTypeAnnotation", node, opts);
	}
	function assertObjectTypeAnnotation(node, opts) {
	  assert("ObjectTypeAnnotation", node, opts);
	}
	function assertObjectTypeInternalSlot(node, opts) {
	  assert("ObjectTypeInternalSlot", node, opts);
	}
	function assertObjectTypeCallProperty(node, opts) {
	  assert("ObjectTypeCallProperty", node, opts);
	}
	function assertObjectTypeIndexer(node, opts) {
	  assert("ObjectTypeIndexer", node, opts);
	}
	function assertObjectTypeProperty(node, opts) {
	  assert("ObjectTypeProperty", node, opts);
	}
	function assertObjectTypeSpreadProperty(node, opts) {
	  assert("ObjectTypeSpreadProperty", node, opts);
	}
	function assertOpaqueType(node, opts) {
	  assert("OpaqueType", node, opts);
	}
	function assertQualifiedTypeIdentifier(node, opts) {
	  assert("QualifiedTypeIdentifier", node, opts);
	}
	function assertStringLiteralTypeAnnotation(node, opts) {
	  assert("StringLiteralTypeAnnotation", node, opts);
	}
	function assertStringTypeAnnotation(node, opts) {
	  assert("StringTypeAnnotation", node, opts);
	}
	function assertSymbolTypeAnnotation(node, opts) {
	  assert("SymbolTypeAnnotation", node, opts);
	}
	function assertThisTypeAnnotation(node, opts) {
	  assert("ThisTypeAnnotation", node, opts);
	}
	function assertTupleTypeAnnotation(node, opts) {
	  assert("TupleTypeAnnotation", node, opts);
	}
	function assertTypeofTypeAnnotation(node, opts) {
	  assert("TypeofTypeAnnotation", node, opts);
	}
	function assertTypeAlias(node, opts) {
	  assert("TypeAlias", node, opts);
	}
	function assertTypeAnnotation(node, opts) {
	  assert("TypeAnnotation", node, opts);
	}
	function assertTypeCastExpression(node, opts) {
	  assert("TypeCastExpression", node, opts);
	}
	function assertTypeParameter(node, opts) {
	  assert("TypeParameter", node, opts);
	}
	function assertTypeParameterDeclaration(node, opts) {
	  assert("TypeParameterDeclaration", node, opts);
	}
	function assertTypeParameterInstantiation(node, opts) {
	  assert("TypeParameterInstantiation", node, opts);
	}
	function assertUnionTypeAnnotation(node, opts) {
	  assert("UnionTypeAnnotation", node, opts);
	}
	function assertVariance(node, opts) {
	  assert("Variance", node, opts);
	}
	function assertVoidTypeAnnotation(node, opts) {
	  assert("VoidTypeAnnotation", node, opts);
	}
	function assertEnumDeclaration(node, opts) {
	  assert("EnumDeclaration", node, opts);
	}
	function assertEnumBooleanBody(node, opts) {
	  assert("EnumBooleanBody", node, opts);
	}
	function assertEnumNumberBody(node, opts) {
	  assert("EnumNumberBody", node, opts);
	}
	function assertEnumStringBody(node, opts) {
	  assert("EnumStringBody", node, opts);
	}
	function assertEnumSymbolBody(node, opts) {
	  assert("EnumSymbolBody", node, opts);
	}
	function assertEnumBooleanMember(node, opts) {
	  assert("EnumBooleanMember", node, opts);
	}
	function assertEnumNumberMember(node, opts) {
	  assert("EnumNumberMember", node, opts);
	}
	function assertEnumStringMember(node, opts) {
	  assert("EnumStringMember", node, opts);
	}
	function assertEnumDefaultedMember(node, opts) {
	  assert("EnumDefaultedMember", node, opts);
	}
	function assertIndexedAccessType(node, opts) {
	  assert("IndexedAccessType", node, opts);
	}
	function assertOptionalIndexedAccessType(node, opts) {
	  assert("OptionalIndexedAccessType", node, opts);
	}
	function assertJSXAttribute(node, opts) {
	  assert("JSXAttribute", node, opts);
	}
	function assertJSXClosingElement(node, opts) {
	  assert("JSXClosingElement", node, opts);
	}
	function assertJSXElement(node, opts) {
	  assert("JSXElement", node, opts);
	}
	function assertJSXEmptyExpression(node, opts) {
	  assert("JSXEmptyExpression", node, opts);
	}
	function assertJSXExpressionContainer(node, opts) {
	  assert("JSXExpressionContainer", node, opts);
	}
	function assertJSXSpreadChild(node, opts) {
	  assert("JSXSpreadChild", node, opts);
	}
	function assertJSXIdentifier(node, opts) {
	  assert("JSXIdentifier", node, opts);
	}
	function assertJSXMemberExpression(node, opts) {
	  assert("JSXMemberExpression", node, opts);
	}
	function assertJSXNamespacedName(node, opts) {
	  assert("JSXNamespacedName", node, opts);
	}
	function assertJSXOpeningElement(node, opts) {
	  assert("JSXOpeningElement", node, opts);
	}
	function assertJSXSpreadAttribute(node, opts) {
	  assert("JSXSpreadAttribute", node, opts);
	}
	function assertJSXText(node, opts) {
	  assert("JSXText", node, opts);
	}
	function assertJSXFragment(node, opts) {
	  assert("JSXFragment", node, opts);
	}
	function assertJSXOpeningFragment(node, opts) {
	  assert("JSXOpeningFragment", node, opts);
	}
	function assertJSXClosingFragment(node, opts) {
	  assert("JSXClosingFragment", node, opts);
	}
	function assertNoop(node, opts) {
	  assert("Noop", node, opts);
	}
	function assertPlaceholder(node, opts) {
	  assert("Placeholder", node, opts);
	}
	function assertV8IntrinsicIdentifier(node, opts) {
	  assert("V8IntrinsicIdentifier", node, opts);
	}
	function assertArgumentPlaceholder(node, opts) {
	  assert("ArgumentPlaceholder", node, opts);
	}
	function assertBindExpression(node, opts) {
	  assert("BindExpression", node, opts);
	}
	function assertImportAttribute(node, opts) {
	  assert("ImportAttribute", node, opts);
	}
	function assertDecorator(node, opts) {
	  assert("Decorator", node, opts);
	}
	function assertDoExpression(node, opts) {
	  assert("DoExpression", node, opts);
	}
	function assertExportDefaultSpecifier(node, opts) {
	  assert("ExportDefaultSpecifier", node, opts);
	}
	function assertRecordExpression(node, opts) {
	  assert("RecordExpression", node, opts);
	}
	function assertTupleExpression(node, opts) {
	  assert("TupleExpression", node, opts);
	}
	function assertDecimalLiteral(node, opts) {
	  assert("DecimalLiteral", node, opts);
	}
	function assertModuleExpression(node, opts) {
	  assert("ModuleExpression", node, opts);
	}
	function assertTopicReference(node, opts) {
	  assert("TopicReference", node, opts);
	}
	function assertPipelineTopicExpression(node, opts) {
	  assert("PipelineTopicExpression", node, opts);
	}
	function assertPipelineBareFunction(node, opts) {
	  assert("PipelineBareFunction", node, opts);
	}
	function assertPipelinePrimaryTopicReference(node, opts) {
	  assert("PipelinePrimaryTopicReference", node, opts);
	}
	function assertTSParameterProperty(node, opts) {
	  assert("TSParameterProperty", node, opts);
	}
	function assertTSDeclareFunction(node, opts) {
	  assert("TSDeclareFunction", node, opts);
	}
	function assertTSDeclareMethod(node, opts) {
	  assert("TSDeclareMethod", node, opts);
	}
	function assertTSQualifiedName(node, opts) {
	  assert("TSQualifiedName", node, opts);
	}
	function assertTSCallSignatureDeclaration(node, opts) {
	  assert("TSCallSignatureDeclaration", node, opts);
	}
	function assertTSConstructSignatureDeclaration(node, opts) {
	  assert("TSConstructSignatureDeclaration", node, opts);
	}
	function assertTSPropertySignature(node, opts) {
	  assert("TSPropertySignature", node, opts);
	}
	function assertTSMethodSignature(node, opts) {
	  assert("TSMethodSignature", node, opts);
	}
	function assertTSIndexSignature(node, opts) {
	  assert("TSIndexSignature", node, opts);
	}
	function assertTSAnyKeyword(node, opts) {
	  assert("TSAnyKeyword", node, opts);
	}
	function assertTSBooleanKeyword(node, opts) {
	  assert("TSBooleanKeyword", node, opts);
	}
	function assertTSBigIntKeyword(node, opts) {
	  assert("TSBigIntKeyword", node, opts);
	}
	function assertTSIntrinsicKeyword(node, opts) {
	  assert("TSIntrinsicKeyword", node, opts);
	}
	function assertTSNeverKeyword(node, opts) {
	  assert("TSNeverKeyword", node, opts);
	}
	function assertTSNullKeyword(node, opts) {
	  assert("TSNullKeyword", node, opts);
	}
	function assertTSNumberKeyword(node, opts) {
	  assert("TSNumberKeyword", node, opts);
	}
	function assertTSObjectKeyword(node, opts) {
	  assert("TSObjectKeyword", node, opts);
	}
	function assertTSStringKeyword(node, opts) {
	  assert("TSStringKeyword", node, opts);
	}
	function assertTSSymbolKeyword(node, opts) {
	  assert("TSSymbolKeyword", node, opts);
	}
	function assertTSUndefinedKeyword(node, opts) {
	  assert("TSUndefinedKeyword", node, opts);
	}
	function assertTSUnknownKeyword(node, opts) {
	  assert("TSUnknownKeyword", node, opts);
	}
	function assertTSVoidKeyword(node, opts) {
	  assert("TSVoidKeyword", node, opts);
	}
	function assertTSThisType(node, opts) {
	  assert("TSThisType", node, opts);
	}
	function assertTSFunctionType(node, opts) {
	  assert("TSFunctionType", node, opts);
	}
	function assertTSConstructorType(node, opts) {
	  assert("TSConstructorType", node, opts);
	}
	function assertTSTypeReference(node, opts) {
	  assert("TSTypeReference", node, opts);
	}
	function assertTSTypePredicate(node, opts) {
	  assert("TSTypePredicate", node, opts);
	}
	function assertTSTypeQuery(node, opts) {
	  assert("TSTypeQuery", node, opts);
	}
	function assertTSTypeLiteral(node, opts) {
	  assert("TSTypeLiteral", node, opts);
	}
	function assertTSArrayType(node, opts) {
	  assert("TSArrayType", node, opts);
	}
	function assertTSTupleType(node, opts) {
	  assert("TSTupleType", node, opts);
	}
	function assertTSOptionalType(node, opts) {
	  assert("TSOptionalType", node, opts);
	}
	function assertTSRestType(node, opts) {
	  assert("TSRestType", node, opts);
	}
	function assertTSNamedTupleMember(node, opts) {
	  assert("TSNamedTupleMember", node, opts);
	}
	function assertTSUnionType(node, opts) {
	  assert("TSUnionType", node, opts);
	}
	function assertTSIntersectionType(node, opts) {
	  assert("TSIntersectionType", node, opts);
	}
	function assertTSConditionalType(node, opts) {
	  assert("TSConditionalType", node, opts);
	}
	function assertTSInferType(node, opts) {
	  assert("TSInferType", node, opts);
	}
	function assertTSParenthesizedType(node, opts) {
	  assert("TSParenthesizedType", node, opts);
	}
	function assertTSTypeOperator(node, opts) {
	  assert("TSTypeOperator", node, opts);
	}
	function assertTSIndexedAccessType(node, opts) {
	  assert("TSIndexedAccessType", node, opts);
	}
	function assertTSMappedType(node, opts) {
	  assert("TSMappedType", node, opts);
	}
	function assertTSLiteralType(node, opts) {
	  assert("TSLiteralType", node, opts);
	}
	function assertTSExpressionWithTypeArguments(node, opts) {
	  assert("TSExpressionWithTypeArguments", node, opts);
	}
	function assertTSInterfaceDeclaration(node, opts) {
	  assert("TSInterfaceDeclaration", node, opts);
	}
	function assertTSInterfaceBody(node, opts) {
	  assert("TSInterfaceBody", node, opts);
	}
	function assertTSTypeAliasDeclaration(node, opts) {
	  assert("TSTypeAliasDeclaration", node, opts);
	}
	function assertTSInstantiationExpression(node, opts) {
	  assert("TSInstantiationExpression", node, opts);
	}
	function assertTSAsExpression(node, opts) {
	  assert("TSAsExpression", node, opts);
	}
	function assertTSSatisfiesExpression(node, opts) {
	  assert("TSSatisfiesExpression", node, opts);
	}
	function assertTSTypeAssertion(node, opts) {
	  assert("TSTypeAssertion", node, opts);
	}
	function assertTSEnumDeclaration(node, opts) {
	  assert("TSEnumDeclaration", node, opts);
	}
	function assertTSEnumMember(node, opts) {
	  assert("TSEnumMember", node, opts);
	}
	function assertTSModuleDeclaration(node, opts) {
	  assert("TSModuleDeclaration", node, opts);
	}
	function assertTSModuleBlock(node, opts) {
	  assert("TSModuleBlock", node, opts);
	}
	function assertTSImportType(node, opts) {
	  assert("TSImportType", node, opts);
	}
	function assertTSImportEqualsDeclaration(node, opts) {
	  assert("TSImportEqualsDeclaration", node, opts);
	}
	function assertTSExternalModuleReference(node, opts) {
	  assert("TSExternalModuleReference", node, opts);
	}
	function assertTSNonNullExpression(node, opts) {
	  assert("TSNonNullExpression", node, opts);
	}
	function assertTSExportAssignment(node, opts) {
	  assert("TSExportAssignment", node, opts);
	}
	function assertTSNamespaceExportDeclaration(node, opts) {
	  assert("TSNamespaceExportDeclaration", node, opts);
	}
	function assertTSTypeAnnotation(node, opts) {
	  assert("TSTypeAnnotation", node, opts);
	}
	function assertTSTypeParameterInstantiation(node, opts) {
	  assert("TSTypeParameterInstantiation", node, opts);
	}
	function assertTSTypeParameterDeclaration(node, opts) {
	  assert("TSTypeParameterDeclaration", node, opts);
	}
	function assertTSTypeParameter(node, opts) {
	  assert("TSTypeParameter", node, opts);
	}
	function assertStandardized(node, opts) {
	  assert("Standardized", node, opts);
	}
	function assertExpression(node, opts) {
	  assert("Expression", node, opts);
	}
	function assertBinary(node, opts) {
	  assert("Binary", node, opts);
	}
	function assertScopable(node, opts) {
	  assert("Scopable", node, opts);
	}
	function assertBlockParent(node, opts) {
	  assert("BlockParent", node, opts);
	}
	function assertBlock(node, opts) {
	  assert("Block", node, opts);
	}
	function assertStatement(node, opts) {
	  assert("Statement", node, opts);
	}
	function assertTerminatorless(node, opts) {
	  assert("Terminatorless", node, opts);
	}
	function assertCompletionStatement(node, opts) {
	  assert("CompletionStatement", node, opts);
	}
	function assertConditional(node, opts) {
	  assert("Conditional", node, opts);
	}
	function assertLoop(node, opts) {
	  assert("Loop", node, opts);
	}
	function assertWhile(node, opts) {
	  assert("While", node, opts);
	}
	function assertExpressionWrapper(node, opts) {
	  assert("ExpressionWrapper", node, opts);
	}
	function assertFor(node, opts) {
	  assert("For", node, opts);
	}
	function assertForXStatement(node, opts) {
	  assert("ForXStatement", node, opts);
	}
	function assertFunction(node, opts) {
	  assert("Function", node, opts);
	}
	function assertFunctionParent(node, opts) {
	  assert("FunctionParent", node, opts);
	}
	function assertPureish(node, opts) {
	  assert("Pureish", node, opts);
	}
	function assertDeclaration(node, opts) {
	  assert("Declaration", node, opts);
	}
	function assertPatternLike(node, opts) {
	  assert("PatternLike", node, opts);
	}
	function assertLVal(node, opts) {
	  assert("LVal", node, opts);
	}
	function assertTSEntityName(node, opts) {
	  assert("TSEntityName", node, opts);
	}
	function assertLiteral(node, opts) {
	  assert("Literal", node, opts);
	}
	function assertImmutable(node, opts) {
	  assert("Immutable", node, opts);
	}
	function assertUserWhitespacable(node, opts) {
	  assert("UserWhitespacable", node, opts);
	}
	function assertMethod(node, opts) {
	  assert("Method", node, opts);
	}
	function assertObjectMember(node, opts) {
	  assert("ObjectMember", node, opts);
	}
	function assertProperty(node, opts) {
	  assert("Property", node, opts);
	}
	function assertUnaryLike(node, opts) {
	  assert("UnaryLike", node, opts);
	}
	function assertPattern(node, opts) {
	  assert("Pattern", node, opts);
	}
	function assertClass(node, opts) {
	  assert("Class", node, opts);
	}
	function assertImportOrExportDeclaration(node, opts) {
	  assert("ImportOrExportDeclaration", node, opts);
	}
	function assertExportDeclaration(node, opts) {
	  assert("ExportDeclaration", node, opts);
	}
	function assertModuleSpecifier(node, opts) {
	  assert("ModuleSpecifier", node, opts);
	}
	function assertAccessor(node, opts) {
	  assert("Accessor", node, opts);
	}
	function assertPrivate(node, opts) {
	  assert("Private", node, opts);
	}
	function assertFlow(node, opts) {
	  assert("Flow", node, opts);
	}
	function assertFlowType(node, opts) {
	  assert("FlowType", node, opts);
	}
	function assertFlowBaseAnnotation(node, opts) {
	  assert("FlowBaseAnnotation", node, opts);
	}
	function assertFlowDeclaration(node, opts) {
	  assert("FlowDeclaration", node, opts);
	}
	function assertFlowPredicate(node, opts) {
	  assert("FlowPredicate", node, opts);
	}
	function assertEnumBody(node, opts) {
	  assert("EnumBody", node, opts);
	}
	function assertEnumMember(node, opts) {
	  assert("EnumMember", node, opts);
	}
	function assertJSX(node, opts) {
	  assert("JSX", node, opts);
	}
	function assertMiscellaneous(node, opts) {
	  assert("Miscellaneous", node, opts);
	}
	function assertTypeScript(node, opts) {
	  assert("TypeScript", node, opts);
	}
	function assertTSTypeElement(node, opts) {
	  assert("TSTypeElement", node, opts);
	}
	function assertTSType(node, opts) {
	  assert("TSType", node, opts);
	}
	function assertTSBaseType(node, opts) {
	  assert("TSBaseType", node, opts);
	}
	function assertNumberLiteral(node, opts) {
	  (0, _deprecationWarning.default)("assertNumberLiteral", "assertNumericLiteral");
	  assert("NumberLiteral", node, opts);
	}
	function assertRegexLiteral(node, opts) {
	  (0, _deprecationWarning.default)("assertRegexLiteral", "assertRegExpLiteral");
	  assert("RegexLiteral", node, opts);
	}
	function assertRestProperty(node, opts) {
	  (0, _deprecationWarning.default)("assertRestProperty", "assertRestElement");
	  assert("RestProperty", node, opts);
	}
	function assertSpreadProperty(node, opts) {
	  (0, _deprecationWarning.default)("assertSpreadProperty", "assertSpreadElement");
	  assert("SpreadProperty", node, opts);
	}
	function assertModuleDeclaration(node, opts) {
	  (0, _deprecationWarning.default)("assertModuleDeclaration", "assertImportOrExportDeclaration");
	  assert("ModuleDeclaration", node, opts);
	}

	
	return generated$1;
}

var createTypeAnnotationBasedOnTypeof = {};

var hasRequiredCreateTypeAnnotationBasedOnTypeof;

function requireCreateTypeAnnotationBasedOnTypeof () {
	if (hasRequiredCreateTypeAnnotationBasedOnTypeof) return createTypeAnnotationBasedOnTypeof;
	hasRequiredCreateTypeAnnotationBasedOnTypeof = 1;

	Object.defineProperty(createTypeAnnotationBasedOnTypeof, "__esModule", {
	  value: true
	});
	createTypeAnnotationBasedOnTypeof.default = void 0;
	var _index = requireGenerated$2();
	createTypeAnnotationBasedOnTypeof.default = createTypeAnnotationBasedOnTypeof$1;
	function createTypeAnnotationBasedOnTypeof$1(type) {
	  switch (type) {
	    case "string":
	      return (0, _index.stringTypeAnnotation)();
	    case "number":
	      return (0, _index.numberTypeAnnotation)();
	    case "undefined":
	      return (0, _index.voidTypeAnnotation)();
	    case "boolean":
	      return (0, _index.booleanTypeAnnotation)();
	    case "function":
	      return (0, _index.genericTypeAnnotation)((0, _index.identifier)("Function"));
	    case "object":
	      return (0, _index.genericTypeAnnotation)((0, _index.identifier)("Object"));
	    case "symbol":
	      return (0, _index.genericTypeAnnotation)((0, _index.identifier)("Symbol"));
	    case "bigint":
	      return (0, _index.anyTypeAnnotation)();
	  }
	  throw new Error("Invalid typeof value: " + type);
	}

	
	return createTypeAnnotationBasedOnTypeof;
}

var createFlowUnionType = {};

var removeTypeDuplicates$1 = {};

var hasRequiredRemoveTypeDuplicates$1;

function requireRemoveTypeDuplicates$1 () {
	if (hasRequiredRemoveTypeDuplicates$1) return removeTypeDuplicates$1;
	hasRequiredRemoveTypeDuplicates$1 = 1;

	Object.defineProperty(removeTypeDuplicates$1, "__esModule", {
	  value: true
	});
	removeTypeDuplicates$1.default = removeTypeDuplicates;
	var _index = requireGenerated$3();
	function getQualifiedName(node) {
	  return (0, _index.isIdentifier)(node) ? node.name : `${node.id.name}.${getQualifiedName(node.qualification)}`;
	}
	function removeTypeDuplicates(nodesIn) {
	  const nodes = Array.from(nodesIn);
	  const generics = new Map();
	  const bases = new Map();
	  const typeGroups = new Set();
	  const types = [];
	  for (let i = 0; i < nodes.length; i++) {
	    const node = nodes[i];
	    if (!node) continue;
	    if (types.includes(node)) {
	      continue;
	    }
	    if ((0, _index.isAnyTypeAnnotation)(node)) {
	      return [node];
	    }
	    if ((0, _index.isFlowBaseAnnotation)(node)) {
	      bases.set(node.type, node);
	      continue;
	    }
	    if ((0, _index.isUnionTypeAnnotation)(node)) {
	      if (!typeGroups.has(node.types)) {
	        nodes.push(...node.types);
	        typeGroups.add(node.types);
	      }
	      continue;
	    }
	    if ((0, _index.isGenericTypeAnnotation)(node)) {
	      const name = getQualifiedName(node.id);
	      if (generics.has(name)) {
	        let existing = generics.get(name);
	        if (existing.typeParameters) {
	          if (node.typeParameters) {
	            existing.typeParameters.params.push(...node.typeParameters.params);
	            existing.typeParameters.params = removeTypeDuplicates(existing.typeParameters.params);
	          }
	        } else {
	          existing = node.typeParameters;
	        }
	      } else {
	        generics.set(name, node);
	      }
	      continue;
	    }
	    types.push(node);
	  }
	  for (const [, baseType] of bases) {
	    types.push(baseType);
	  }
	  for (const [, genericName] of generics) {
	    types.push(genericName);
	  }
	  return types;
	}

	
	return removeTypeDuplicates$1;
}

var hasRequiredCreateFlowUnionType;

function requireCreateFlowUnionType () {
	if (hasRequiredCreateFlowUnionType) return createFlowUnionType;
	hasRequiredCreateFlowUnionType = 1;

	Object.defineProperty(createFlowUnionType, "__esModule", {
	  value: true
	});
	createFlowUnionType.default = createFlowUnionType$1;
	var _index = requireGenerated$2();
	var _removeTypeDuplicates = requireRemoveTypeDuplicates$1();
	function createFlowUnionType$1(types) {
	  const flattened = (0, _removeTypeDuplicates.default)(types);
	  if (flattened.length === 1) {
	    return flattened[0];
	  } else {
	    return (0, _index.unionTypeAnnotation)(flattened);
	  }
	}

	
	return createFlowUnionType;
}

var createTSUnionType = {};

var removeTypeDuplicates = {};

var hasRequiredRemoveTypeDuplicates;

function requireRemoveTypeDuplicates () {
	if (hasRequiredRemoveTypeDuplicates) return removeTypeDuplicates;
	hasRequiredRemoveTypeDuplicates = 1;

	Object.defineProperty(removeTypeDuplicates, "__esModule", {
	  value: true
	});
	removeTypeDuplicates.default = removeTypeDuplicates$1;
	var _index = requireGenerated$3();
	function getQualifiedName(node) {
	  return (0, _index.isIdentifier)(node) ? node.name : `${node.right.name}.${getQualifiedName(node.left)}`;
	}
	function removeTypeDuplicates$1(nodesIn) {
	  const nodes = Array.from(nodesIn);
	  const generics = new Map();
	  const bases = new Map();
	  const typeGroups = new Set();
	  const types = [];
	  for (let i = 0; i < nodes.length; i++) {
	    const node = nodes[i];
	    if (!node) continue;
	    if (types.includes(node)) {
	      continue;
	    }
	    if ((0, _index.isTSAnyKeyword)(node)) {
	      return [node];
	    }
	    if ((0, _index.isTSBaseType)(node)) {
	      bases.set(node.type, node);
	      continue;
	    }
	    if ((0, _index.isTSUnionType)(node)) {
	      if (!typeGroups.has(node.types)) {
	        nodes.push(...node.types);
	        typeGroups.add(node.types);
	      }
	      continue;
	    }
	    if ((0, _index.isTSTypeReference)(node) && node.typeParameters) {
	      const name = getQualifiedName(node.typeName);
	      if (generics.has(name)) {
	        let existing = generics.get(name);
	        if (existing.typeParameters) {
	          if (node.typeParameters) {
	            existing.typeParameters.params.push(...node.typeParameters.params);
	            existing.typeParameters.params = removeTypeDuplicates$1(existing.typeParameters.params);
	          }
	        } else {
	          existing = node.typeParameters;
	        }
	      } else {
	        generics.set(name, node);
	      }
	      continue;
	    }
	    types.push(node);
	  }
	  for (const [, baseType] of bases) {
	    types.push(baseType);
	  }
	  for (const [, genericName] of generics) {
	    types.push(genericName);
	  }
	  return types;
	}

	
	return removeTypeDuplicates;
}

var hasRequiredCreateTSUnionType;

function requireCreateTSUnionType () {
	if (hasRequiredCreateTSUnionType) return createTSUnionType;
	hasRequiredCreateTSUnionType = 1;

	Object.defineProperty(createTSUnionType, "__esModule", {
	  value: true
	});
	createTSUnionType.default = createTSUnionType$1;
	var _index = requireGenerated$2();
	var _removeTypeDuplicates = requireRemoveTypeDuplicates();
	var _index2 = requireGenerated$3();
	function createTSUnionType$1(typeAnnotations) {
	  const types = typeAnnotations.map(type => {
	    return (0, _index2.isTSTypeAnnotation)(type) ? type.typeAnnotation : type;
	  });
	  const flattened = (0, _removeTypeDuplicates.default)(types);
	  if (flattened.length === 1) {
	    return flattened[0];
	  } else {
	    return (0, _index.tsUnionType)(flattened);
	  }
	}

	
	return createTSUnionType;
}

var uppercase = {};

var hasRequiredUppercase;

function requireUppercase () {
	if (hasRequiredUppercase) return uppercase;
	hasRequiredUppercase = 1;
	(function (exports) {

		Object.defineProperty(exports, "__esModule", {
		  value: true
		});
		Object.defineProperty(exports, "AnyTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.anyTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ArgumentPlaceholder", {
		  enumerable: true,
		  get: function () {
		    return _index.argumentPlaceholder;
		  }
		});
		Object.defineProperty(exports, "ArrayExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.arrayExpression;
		  }
		});
		Object.defineProperty(exports, "ArrayPattern", {
		  enumerable: true,
		  get: function () {
		    return _index.arrayPattern;
		  }
		});
		Object.defineProperty(exports, "ArrayTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.arrayTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ArrowFunctionExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.arrowFunctionExpression;
		  }
		});
		Object.defineProperty(exports, "AssignmentExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.assignmentExpression;
		  }
		});
		Object.defineProperty(exports, "AssignmentPattern", {
		  enumerable: true,
		  get: function () {
		    return _index.assignmentPattern;
		  }
		});
		Object.defineProperty(exports, "AwaitExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.awaitExpression;
		  }
		});
		Object.defineProperty(exports, "BigIntLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.bigIntLiteral;
		  }
		});
		Object.defineProperty(exports, "BinaryExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.binaryExpression;
		  }
		});
		Object.defineProperty(exports, "BindExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.bindExpression;
		  }
		});
		Object.defineProperty(exports, "BlockStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.blockStatement;
		  }
		});
		Object.defineProperty(exports, "BooleanLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.booleanLiteral;
		  }
		});
		Object.defineProperty(exports, "BooleanLiteralTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.booleanLiteralTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "BooleanTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.booleanTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "BreakStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.breakStatement;
		  }
		});
		Object.defineProperty(exports, "CallExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.callExpression;
		  }
		});
		Object.defineProperty(exports, "CatchClause", {
		  enumerable: true,
		  get: function () {
		    return _index.catchClause;
		  }
		});
		Object.defineProperty(exports, "ClassAccessorProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.classAccessorProperty;
		  }
		});
		Object.defineProperty(exports, "ClassBody", {
		  enumerable: true,
		  get: function () {
		    return _index.classBody;
		  }
		});
		Object.defineProperty(exports, "ClassDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.classDeclaration;
		  }
		});
		Object.defineProperty(exports, "ClassExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.classExpression;
		  }
		});
		Object.defineProperty(exports, "ClassImplements", {
		  enumerable: true,
		  get: function () {
		    return _index.classImplements;
		  }
		});
		Object.defineProperty(exports, "ClassMethod", {
		  enumerable: true,
		  get: function () {
		    return _index.classMethod;
		  }
		});
		Object.defineProperty(exports, "ClassPrivateMethod", {
		  enumerable: true,
		  get: function () {
		    return _index.classPrivateMethod;
		  }
		});
		Object.defineProperty(exports, "ClassPrivateProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.classPrivateProperty;
		  }
		});
		Object.defineProperty(exports, "ClassProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.classProperty;
		  }
		});
		Object.defineProperty(exports, "ConditionalExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.conditionalExpression;
		  }
		});
		Object.defineProperty(exports, "ContinueStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.continueStatement;
		  }
		});
		Object.defineProperty(exports, "DebuggerStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.debuggerStatement;
		  }
		});
		Object.defineProperty(exports, "DecimalLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.decimalLiteral;
		  }
		});
		Object.defineProperty(exports, "DeclareClass", {
		  enumerable: true,
		  get: function () {
		    return _index.declareClass;
		  }
		});
		Object.defineProperty(exports, "DeclareExportAllDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.declareExportAllDeclaration;
		  }
		});
		Object.defineProperty(exports, "DeclareExportDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.declareExportDeclaration;
		  }
		});
		Object.defineProperty(exports, "DeclareFunction", {
		  enumerable: true,
		  get: function () {
		    return _index.declareFunction;
		  }
		});
		Object.defineProperty(exports, "DeclareInterface", {
		  enumerable: true,
		  get: function () {
		    return _index.declareInterface;
		  }
		});
		Object.defineProperty(exports, "DeclareModule", {
		  enumerable: true,
		  get: function () {
		    return _index.declareModule;
		  }
		});
		Object.defineProperty(exports, "DeclareModuleExports", {
		  enumerable: true,
		  get: function () {
		    return _index.declareModuleExports;
		  }
		});
		Object.defineProperty(exports, "DeclareOpaqueType", {
		  enumerable: true,
		  get: function () {
		    return _index.declareOpaqueType;
		  }
		});
		Object.defineProperty(exports, "DeclareTypeAlias", {
		  enumerable: true,
		  get: function () {
		    return _index.declareTypeAlias;
		  }
		});
		Object.defineProperty(exports, "DeclareVariable", {
		  enumerable: true,
		  get: function () {
		    return _index.declareVariable;
		  }
		});
		Object.defineProperty(exports, "DeclaredPredicate", {
		  enumerable: true,
		  get: function () {
		    return _index.declaredPredicate;
		  }
		});
		Object.defineProperty(exports, "Decorator", {
		  enumerable: true,
		  get: function () {
		    return _index.decorator;
		  }
		});
		Object.defineProperty(exports, "Directive", {
		  enumerable: true,
		  get: function () {
		    return _index.directive;
		  }
		});
		Object.defineProperty(exports, "DirectiveLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.directiveLiteral;
		  }
		});
		Object.defineProperty(exports, "DoExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.doExpression;
		  }
		});
		Object.defineProperty(exports, "DoWhileStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.doWhileStatement;
		  }
		});
		Object.defineProperty(exports, "EmptyStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.emptyStatement;
		  }
		});
		Object.defineProperty(exports, "EmptyTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.emptyTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "EnumBooleanBody", {
		  enumerable: true,
		  get: function () {
		    return _index.enumBooleanBody;
		  }
		});
		Object.defineProperty(exports, "EnumBooleanMember", {
		  enumerable: true,
		  get: function () {
		    return _index.enumBooleanMember;
		  }
		});
		Object.defineProperty(exports, "EnumDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.enumDeclaration;
		  }
		});
		Object.defineProperty(exports, "EnumDefaultedMember", {
		  enumerable: true,
		  get: function () {
		    return _index.enumDefaultedMember;
		  }
		});
		Object.defineProperty(exports, "EnumNumberBody", {
		  enumerable: true,
		  get: function () {
		    return _index.enumNumberBody;
		  }
		});
		Object.defineProperty(exports, "EnumNumberMember", {
		  enumerable: true,
		  get: function () {
		    return _index.enumNumberMember;
		  }
		});
		Object.defineProperty(exports, "EnumStringBody", {
		  enumerable: true,
		  get: function () {
		    return _index.enumStringBody;
		  }
		});
		Object.defineProperty(exports, "EnumStringMember", {
		  enumerable: true,
		  get: function () {
		    return _index.enumStringMember;
		  }
		});
		Object.defineProperty(exports, "EnumSymbolBody", {
		  enumerable: true,
		  get: function () {
		    return _index.enumSymbolBody;
		  }
		});
		Object.defineProperty(exports, "ExistsTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.existsTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ExportAllDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.exportAllDeclaration;
		  }
		});
		Object.defineProperty(exports, "ExportDefaultDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.exportDefaultDeclaration;
		  }
		});
		Object.defineProperty(exports, "ExportDefaultSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.exportDefaultSpecifier;
		  }
		});
		Object.defineProperty(exports, "ExportNamedDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.exportNamedDeclaration;
		  }
		});
		Object.defineProperty(exports, "ExportNamespaceSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.exportNamespaceSpecifier;
		  }
		});
		Object.defineProperty(exports, "ExportSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.exportSpecifier;
		  }
		});
		Object.defineProperty(exports, "ExpressionStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.expressionStatement;
		  }
		});
		Object.defineProperty(exports, "File", {
		  enumerable: true,
		  get: function () {
		    return _index.file;
		  }
		});
		Object.defineProperty(exports, "ForInStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.forInStatement;
		  }
		});
		Object.defineProperty(exports, "ForOfStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.forOfStatement;
		  }
		});
		Object.defineProperty(exports, "ForStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.forStatement;
		  }
		});
		Object.defineProperty(exports, "FunctionDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.functionDeclaration;
		  }
		});
		Object.defineProperty(exports, "FunctionExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.functionExpression;
		  }
		});
		Object.defineProperty(exports, "FunctionTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.functionTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "FunctionTypeParam", {
		  enumerable: true,
		  get: function () {
		    return _index.functionTypeParam;
		  }
		});
		Object.defineProperty(exports, "GenericTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.genericTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "Identifier", {
		  enumerable: true,
		  get: function () {
		    return _index.identifier;
		  }
		});
		Object.defineProperty(exports, "IfStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.ifStatement;
		  }
		});
		Object.defineProperty(exports, "Import", {
		  enumerable: true,
		  get: function () {
		    return _index.import;
		  }
		});
		Object.defineProperty(exports, "ImportAttribute", {
		  enumerable: true,
		  get: function () {
		    return _index.importAttribute;
		  }
		});
		Object.defineProperty(exports, "ImportDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.importDeclaration;
		  }
		});
		Object.defineProperty(exports, "ImportDefaultSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.importDefaultSpecifier;
		  }
		});
		Object.defineProperty(exports, "ImportExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.importExpression;
		  }
		});
		Object.defineProperty(exports, "ImportNamespaceSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.importNamespaceSpecifier;
		  }
		});
		Object.defineProperty(exports, "ImportSpecifier", {
		  enumerable: true,
		  get: function () {
		    return _index.importSpecifier;
		  }
		});
		Object.defineProperty(exports, "IndexedAccessType", {
		  enumerable: true,
		  get: function () {
		    return _index.indexedAccessType;
		  }
		});
		Object.defineProperty(exports, "InferredPredicate", {
		  enumerable: true,
		  get: function () {
		    return _index.inferredPredicate;
		  }
		});
		Object.defineProperty(exports, "InterfaceDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.interfaceDeclaration;
		  }
		});
		Object.defineProperty(exports, "InterfaceExtends", {
		  enumerable: true,
		  get: function () {
		    return _index.interfaceExtends;
		  }
		});
		Object.defineProperty(exports, "InterfaceTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.interfaceTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "InterpreterDirective", {
		  enumerable: true,
		  get: function () {
		    return _index.interpreterDirective;
		  }
		});
		Object.defineProperty(exports, "IntersectionTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.intersectionTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "JSXAttribute", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxAttribute;
		  }
		});
		Object.defineProperty(exports, "JSXClosingElement", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxClosingElement;
		  }
		});
		Object.defineProperty(exports, "JSXClosingFragment", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxClosingFragment;
		  }
		});
		Object.defineProperty(exports, "JSXElement", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxElement;
		  }
		});
		Object.defineProperty(exports, "JSXEmptyExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxEmptyExpression;
		  }
		});
		Object.defineProperty(exports, "JSXExpressionContainer", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxExpressionContainer;
		  }
		});
		Object.defineProperty(exports, "JSXFragment", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxFragment;
		  }
		});
		Object.defineProperty(exports, "JSXIdentifier", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxIdentifier;
		  }
		});
		Object.defineProperty(exports, "JSXMemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxMemberExpression;
		  }
		});
		Object.defineProperty(exports, "JSXNamespacedName", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxNamespacedName;
		  }
		});
		Object.defineProperty(exports, "JSXOpeningElement", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxOpeningElement;
		  }
		});
		Object.defineProperty(exports, "JSXOpeningFragment", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxOpeningFragment;
		  }
		});
		Object.defineProperty(exports, "JSXSpreadAttribute", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxSpreadAttribute;
		  }
		});
		Object.defineProperty(exports, "JSXSpreadChild", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxSpreadChild;
		  }
		});
		Object.defineProperty(exports, "JSXText", {
		  enumerable: true,
		  get: function () {
		    return _index.jsxText;
		  }
		});
		Object.defineProperty(exports, "LabeledStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.labeledStatement;
		  }
		});
		Object.defineProperty(exports, "LogicalExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.logicalExpression;
		  }
		});
		Object.defineProperty(exports, "MemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.memberExpression;
		  }
		});
		Object.defineProperty(exports, "MetaProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.metaProperty;
		  }
		});
		Object.defineProperty(exports, "MixedTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.mixedTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ModuleExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.moduleExpression;
		  }
		});
		Object.defineProperty(exports, "NewExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.newExpression;
		  }
		});
		Object.defineProperty(exports, "Noop", {
		  enumerable: true,
		  get: function () {
		    return _index.noop;
		  }
		});
		Object.defineProperty(exports, "NullLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.nullLiteral;
		  }
		});
		Object.defineProperty(exports, "NullLiteralTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.nullLiteralTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "NullableTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.nullableTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "NumberLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.numberLiteral;
		  }
		});
		Object.defineProperty(exports, "NumberLiteralTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.numberLiteralTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "NumberTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.numberTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "NumericLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.numericLiteral;
		  }
		});
		Object.defineProperty(exports, "ObjectExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.objectExpression;
		  }
		});
		Object.defineProperty(exports, "ObjectMethod", {
		  enumerable: true,
		  get: function () {
		    return _index.objectMethod;
		  }
		});
		Object.defineProperty(exports, "ObjectPattern", {
		  enumerable: true,
		  get: function () {
		    return _index.objectPattern;
		  }
		});
		Object.defineProperty(exports, "ObjectProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.objectProperty;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeCallProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeCallProperty;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeIndexer", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeIndexer;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeInternalSlot", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeInternalSlot;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeProperty;
		  }
		});
		Object.defineProperty(exports, "ObjectTypeSpreadProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.objectTypeSpreadProperty;
		  }
		});
		Object.defineProperty(exports, "OpaqueType", {
		  enumerable: true,
		  get: function () {
		    return _index.opaqueType;
		  }
		});
		Object.defineProperty(exports, "OptionalCallExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.optionalCallExpression;
		  }
		});
		Object.defineProperty(exports, "OptionalIndexedAccessType", {
		  enumerable: true,
		  get: function () {
		    return _index.optionalIndexedAccessType;
		  }
		});
		Object.defineProperty(exports, "OptionalMemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.optionalMemberExpression;
		  }
		});
		Object.defineProperty(exports, "ParenthesizedExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.parenthesizedExpression;
		  }
		});
		Object.defineProperty(exports, "PipelineBareFunction", {
		  enumerable: true,
		  get: function () {
		    return _index.pipelineBareFunction;
		  }
		});
		Object.defineProperty(exports, "PipelinePrimaryTopicReference", {
		  enumerable: true,
		  get: function () {
		    return _index.pipelinePrimaryTopicReference;
		  }
		});
		Object.defineProperty(exports, "PipelineTopicExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.pipelineTopicExpression;
		  }
		});
		Object.defineProperty(exports, "Placeholder", {
		  enumerable: true,
		  get: function () {
		    return _index.placeholder;
		  }
		});
		Object.defineProperty(exports, "PrivateName", {
		  enumerable: true,
		  get: function () {
		    return _index.privateName;
		  }
		});
		Object.defineProperty(exports, "Program", {
		  enumerable: true,
		  get: function () {
		    return _index.program;
		  }
		});
		Object.defineProperty(exports, "QualifiedTypeIdentifier", {
		  enumerable: true,
		  get: function () {
		    return _index.qualifiedTypeIdentifier;
		  }
		});
		Object.defineProperty(exports, "RecordExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.recordExpression;
		  }
		});
		Object.defineProperty(exports, "RegExpLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.regExpLiteral;
		  }
		});
		Object.defineProperty(exports, "RegexLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.regexLiteral;
		  }
		});
		Object.defineProperty(exports, "RestElement", {
		  enumerable: true,
		  get: function () {
		    return _index.restElement;
		  }
		});
		Object.defineProperty(exports, "RestProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.restProperty;
		  }
		});
		Object.defineProperty(exports, "ReturnStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.returnStatement;
		  }
		});
		Object.defineProperty(exports, "SequenceExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.sequenceExpression;
		  }
		});
		Object.defineProperty(exports, "SpreadElement", {
		  enumerable: true,
		  get: function () {
		    return _index.spreadElement;
		  }
		});
		Object.defineProperty(exports, "SpreadProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.spreadProperty;
		  }
		});
		Object.defineProperty(exports, "StaticBlock", {
		  enumerable: true,
		  get: function () {
		    return _index.staticBlock;
		  }
		});
		Object.defineProperty(exports, "StringLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.stringLiteral;
		  }
		});
		Object.defineProperty(exports, "StringLiteralTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.stringLiteralTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "StringTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.stringTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "Super", {
		  enumerable: true,
		  get: function () {
		    return _index.super;
		  }
		});
		Object.defineProperty(exports, "SwitchCase", {
		  enumerable: true,
		  get: function () {
		    return _index.switchCase;
		  }
		});
		Object.defineProperty(exports, "SwitchStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.switchStatement;
		  }
		});
		Object.defineProperty(exports, "SymbolTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.symbolTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "TSAnyKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsAnyKeyword;
		  }
		});
		Object.defineProperty(exports, "TSArrayType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsArrayType;
		  }
		});
		Object.defineProperty(exports, "TSAsExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.tsAsExpression;
		  }
		});
		Object.defineProperty(exports, "TSBigIntKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsBigIntKeyword;
		  }
		});
		Object.defineProperty(exports, "TSBooleanKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsBooleanKeyword;
		  }
		});
		Object.defineProperty(exports, "TSCallSignatureDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsCallSignatureDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSConditionalType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsConditionalType;
		  }
		});
		Object.defineProperty(exports, "TSConstructSignatureDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsConstructSignatureDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSConstructorType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsConstructorType;
		  }
		});
		Object.defineProperty(exports, "TSDeclareFunction", {
		  enumerable: true,
		  get: function () {
		    return _index.tsDeclareFunction;
		  }
		});
		Object.defineProperty(exports, "TSDeclareMethod", {
		  enumerable: true,
		  get: function () {
		    return _index.tsDeclareMethod;
		  }
		});
		Object.defineProperty(exports, "TSEnumDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsEnumDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSEnumMember", {
		  enumerable: true,
		  get: function () {
		    return _index.tsEnumMember;
		  }
		});
		Object.defineProperty(exports, "TSExportAssignment", {
		  enumerable: true,
		  get: function () {
		    return _index.tsExportAssignment;
		  }
		});
		Object.defineProperty(exports, "TSExpressionWithTypeArguments", {
		  enumerable: true,
		  get: function () {
		    return _index.tsExpressionWithTypeArguments;
		  }
		});
		Object.defineProperty(exports, "TSExternalModuleReference", {
		  enumerable: true,
		  get: function () {
		    return _index.tsExternalModuleReference;
		  }
		});
		Object.defineProperty(exports, "TSFunctionType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsFunctionType;
		  }
		});
		Object.defineProperty(exports, "TSImportEqualsDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsImportEqualsDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSImportType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsImportType;
		  }
		});
		Object.defineProperty(exports, "TSIndexSignature", {
		  enumerable: true,
		  get: function () {
		    return _index.tsIndexSignature;
		  }
		});
		Object.defineProperty(exports, "TSIndexedAccessType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsIndexedAccessType;
		  }
		});
		Object.defineProperty(exports, "TSInferType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsInferType;
		  }
		});
		Object.defineProperty(exports, "TSInstantiationExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.tsInstantiationExpression;
		  }
		});
		Object.defineProperty(exports, "TSInterfaceBody", {
		  enumerable: true,
		  get: function () {
		    return _index.tsInterfaceBody;
		  }
		});
		Object.defineProperty(exports, "TSInterfaceDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsInterfaceDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSIntersectionType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsIntersectionType;
		  }
		});
		Object.defineProperty(exports, "TSIntrinsicKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsIntrinsicKeyword;
		  }
		});
		Object.defineProperty(exports, "TSLiteralType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsLiteralType;
		  }
		});
		Object.defineProperty(exports, "TSMappedType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsMappedType;
		  }
		});
		Object.defineProperty(exports, "TSMethodSignature", {
		  enumerable: true,
		  get: function () {
		    return _index.tsMethodSignature;
		  }
		});
		Object.defineProperty(exports, "TSModuleBlock", {
		  enumerable: true,
		  get: function () {
		    return _index.tsModuleBlock;
		  }
		});
		Object.defineProperty(exports, "TSModuleDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsModuleDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSNamedTupleMember", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNamedTupleMember;
		  }
		});
		Object.defineProperty(exports, "TSNamespaceExportDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNamespaceExportDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSNeverKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNeverKeyword;
		  }
		});
		Object.defineProperty(exports, "TSNonNullExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNonNullExpression;
		  }
		});
		Object.defineProperty(exports, "TSNullKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNullKeyword;
		  }
		});
		Object.defineProperty(exports, "TSNumberKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsNumberKeyword;
		  }
		});
		Object.defineProperty(exports, "TSObjectKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsObjectKeyword;
		  }
		});
		Object.defineProperty(exports, "TSOptionalType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsOptionalType;
		  }
		});
		Object.defineProperty(exports, "TSParameterProperty", {
		  enumerable: true,
		  get: function () {
		    return _index.tsParameterProperty;
		  }
		});
		Object.defineProperty(exports, "TSParenthesizedType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsParenthesizedType;
		  }
		});
		Object.defineProperty(exports, "TSPropertySignature", {
		  enumerable: true,
		  get: function () {
		    return _index.tsPropertySignature;
		  }
		});
		Object.defineProperty(exports, "TSQualifiedName", {
		  enumerable: true,
		  get: function () {
		    return _index.tsQualifiedName;
		  }
		});
		Object.defineProperty(exports, "TSRestType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsRestType;
		  }
		});
		Object.defineProperty(exports, "TSSatisfiesExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.tsSatisfiesExpression;
		  }
		});
		Object.defineProperty(exports, "TSStringKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsStringKeyword;
		  }
		});
		Object.defineProperty(exports, "TSSymbolKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsSymbolKeyword;
		  }
		});
		Object.defineProperty(exports, "TSThisType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsThisType;
		  }
		});
		Object.defineProperty(exports, "TSTupleType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTupleType;
		  }
		});
		Object.defineProperty(exports, "TSTypeAliasDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeAliasDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "TSTypeAssertion", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeAssertion;
		  }
		});
		Object.defineProperty(exports, "TSTypeLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeLiteral;
		  }
		});
		Object.defineProperty(exports, "TSTypeOperator", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeOperator;
		  }
		});
		Object.defineProperty(exports, "TSTypeParameter", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeParameter;
		  }
		});
		Object.defineProperty(exports, "TSTypeParameterDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeParameterDeclaration;
		  }
		});
		Object.defineProperty(exports, "TSTypeParameterInstantiation", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeParameterInstantiation;
		  }
		});
		Object.defineProperty(exports, "TSTypePredicate", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypePredicate;
		  }
		});
		Object.defineProperty(exports, "TSTypeQuery", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeQuery;
		  }
		});
		Object.defineProperty(exports, "TSTypeReference", {
		  enumerable: true,
		  get: function () {
		    return _index.tsTypeReference;
		  }
		});
		Object.defineProperty(exports, "TSUndefinedKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsUndefinedKeyword;
		  }
		});
		Object.defineProperty(exports, "TSUnionType", {
		  enumerable: true,
		  get: function () {
		    return _index.tsUnionType;
		  }
		});
		Object.defineProperty(exports, "TSUnknownKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsUnknownKeyword;
		  }
		});
		Object.defineProperty(exports, "TSVoidKeyword", {
		  enumerable: true,
		  get: function () {
		    return _index.tsVoidKeyword;
		  }
		});
		Object.defineProperty(exports, "TaggedTemplateExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.taggedTemplateExpression;
		  }
		});
		Object.defineProperty(exports, "TemplateElement", {
		  enumerable: true,
		  get: function () {
		    return _index.templateElement;
		  }
		});
		Object.defineProperty(exports, "TemplateLiteral", {
		  enumerable: true,
		  get: function () {
		    return _index.templateLiteral;
		  }
		});
		Object.defineProperty(exports, "ThisExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.thisExpression;
		  }
		});
		Object.defineProperty(exports, "ThisTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.thisTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "ThrowStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.throwStatement;
		  }
		});
		Object.defineProperty(exports, "TopicReference", {
		  enumerable: true,
		  get: function () {
		    return _index.topicReference;
		  }
		});
		Object.defineProperty(exports, "TryStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.tryStatement;
		  }
		});
		Object.defineProperty(exports, "TupleExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.tupleExpression;
		  }
		});
		Object.defineProperty(exports, "TupleTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.tupleTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "TypeAlias", {
		  enumerable: true,
		  get: function () {
		    return _index.typeAlias;
		  }
		});
		Object.defineProperty(exports, "TypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.typeAnnotation;
		  }
		});
		Object.defineProperty(exports, "TypeCastExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.typeCastExpression;
		  }
		});
		Object.defineProperty(exports, "TypeParameter", {
		  enumerable: true,
		  get: function () {
		    return _index.typeParameter;
		  }
		});
		Object.defineProperty(exports, "TypeParameterDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.typeParameterDeclaration;
		  }
		});
		Object.defineProperty(exports, "TypeParameterInstantiation", {
		  enumerable: true,
		  get: function () {
		    return _index.typeParameterInstantiation;
		  }
		});
		Object.defineProperty(exports, "TypeofTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.typeofTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "UnaryExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.unaryExpression;
		  }
		});
		Object.defineProperty(exports, "UnionTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.unionTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "UpdateExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.updateExpression;
		  }
		});
		Object.defineProperty(exports, "V8IntrinsicIdentifier", {
		  enumerable: true,
		  get: function () {
		    return _index.v8IntrinsicIdentifier;
		  }
		});
		Object.defineProperty(exports, "VariableDeclaration", {
		  enumerable: true,
		  get: function () {
		    return _index.variableDeclaration;
		  }
		});
		Object.defineProperty(exports, "VariableDeclarator", {
		  enumerable: true,
		  get: function () {
		    return _index.variableDeclarator;
		  }
		});
		Object.defineProperty(exports, "Variance", {
		  enumerable: true,
		  get: function () {
		    return _index.variance;
		  }
		});
		Object.defineProperty(exports, "VoidTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _index.voidTypeAnnotation;
		  }
		});
		Object.defineProperty(exports, "WhileStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.whileStatement;
		  }
		});
		Object.defineProperty(exports, "WithStatement", {
		  enumerable: true,
		  get: function () {
		    return _index.withStatement;
		  }
		});
		Object.defineProperty(exports, "YieldExpression", {
		  enumerable: true,
		  get: function () {
		    return _index.yieldExpression;
		  }
		});
		var _index = requireGenerated$2();

		
} (uppercase));
	return uppercase;
}

var productions = {};

var hasRequiredProductions;

function requireProductions () {
	if (hasRequiredProductions) return productions;
	hasRequiredProductions = 1;

	Object.defineProperty(productions, "__esModule", {
	  value: true
	});
	productions.buildUndefinedNode = buildUndefinedNode;
	var _index = requireGenerated$2();
	function buildUndefinedNode() {
	  return (0, _index.unaryExpression)("void", (0, _index.numericLiteral)(0), true);
	}

	
	return productions;
}

var cloneNode = {};

var hasRequiredCloneNode;

function requireCloneNode () {
	if (hasRequiredCloneNode) return cloneNode;
	hasRequiredCloneNode = 1;

	Object.defineProperty(cloneNode, "__esModule", {
	  value: true
	});
	cloneNode.default = cloneNode$1;
	var _index = requireDefinitions();
	var _index2 = requireGenerated$3();
	const {
	  hasOwn
	} = {
	  hasOwn: Function.call.bind(Object.prototype.hasOwnProperty)
	};
	function cloneIfNode(obj, deep, withoutLoc, commentsCache) {
	  if (obj && typeof obj.type === "string") {
	    return cloneNodeInternal(obj, deep, withoutLoc, commentsCache);
	  }
	  return obj;
	}
	function cloneIfNodeOrArray(obj, deep, withoutLoc, commentsCache) {
	  if (Array.isArray(obj)) {
	    return obj.map(node => cloneIfNode(node, deep, withoutLoc, commentsCache));
	  }
	  return cloneIfNode(obj, deep, withoutLoc, commentsCache);
	}
	function cloneNode$1(node, deep = true, withoutLoc = false) {
	  return cloneNodeInternal(node, deep, withoutLoc, new Map());
	}
	function cloneNodeInternal(node, deep = true, withoutLoc = false, commentsCache) {
	  if (!node) return node;
	  const {
	    type
	  } = node;
	  const newNode = {
	    type: node.type
	  };
	  if ((0, _index2.isIdentifier)(node)) {
	    newNode.name = node.name;
	    if (hasOwn(node, "optional") && typeof node.optional === "boolean") {
	      newNode.optional = node.optional;
	    }
	    if (hasOwn(node, "typeAnnotation")) {
	      newNode.typeAnnotation = deep ? cloneIfNodeOrArray(node.typeAnnotation, true, withoutLoc, commentsCache) : node.typeAnnotation;
	    }
	    if (hasOwn(node, "decorators")) {
	      newNode.decorators = deep ? cloneIfNodeOrArray(node.decorators, true, withoutLoc, commentsCache) : node.decorators;
	    }
	  } else if (!hasOwn(_index.NODE_FIELDS, type)) {
	    throw new Error(`Unknown node type: "${type}"`);
	  } else {
	    for (const field of Object.keys(_index.NODE_FIELDS[type])) {
	      if (hasOwn(node, field)) {
	        if (deep) {
	          newNode[field] = (0, _index2.isFile)(node) && field === "comments" ? maybeCloneComments(node.comments, deep, withoutLoc, commentsCache) : cloneIfNodeOrArray(node[field], true, withoutLoc, commentsCache);
	        } else {
	          newNode[field] = node[field];
	        }
	      }
	    }
	  }
	  if (hasOwn(node, "loc")) {
	    if (withoutLoc) {
	      newNode.loc = null;
	    } else {
	      newNode.loc = node.loc;
	    }
	  }
	  if (hasOwn(node, "leadingComments")) {
	    newNode.leadingComments = maybeCloneComments(node.leadingComments, deep, withoutLoc, commentsCache);
	  }
	  if (hasOwn(node, "innerComments")) {
	    newNode.innerComments = maybeCloneComments(node.innerComments, deep, withoutLoc, commentsCache);
	  }
	  if (hasOwn(node, "trailingComments")) {
	    newNode.trailingComments = maybeCloneComments(node.trailingComments, deep, withoutLoc, commentsCache);
	  }
	  if (hasOwn(node, "extra")) {
	    newNode.extra = Object.assign({}, node.extra);
	  }
	  return newNode;
	}
	function maybeCloneComments(comments, deep, withoutLoc, commentsCache) {
	  if (!comments || !deep) {
	    return comments;
	  }
	  return comments.map(comment => {
	    const cache = commentsCache.get(comment);
	    if (cache) return cache;
	    const {
	      type,
	      value,
	      loc
	    } = comment;
	    const ret = {
	      type,
	      value,
	      loc
	    };
	    if (withoutLoc) {
	      ret.loc = null;
	    }
	    commentsCache.set(comment, ret);
	    return ret;
	  });
	}

	
	return cloneNode;
}

var clone = {};

var hasRequiredClone;

function requireClone () {
	if (hasRequiredClone) return clone;
	hasRequiredClone = 1;

	Object.defineProperty(clone, "__esModule", {
	  value: true
	});
	clone.default = clone$1;
	var _cloneNode = requireCloneNode();
	function clone$1(node) {
	  return (0, _cloneNode.default)(node, false);
	}

	
	return clone;
}

var cloneDeep = {};

var hasRequiredCloneDeep;

function requireCloneDeep () {
	if (hasRequiredCloneDeep) return cloneDeep;
	hasRequiredCloneDeep = 1;

	Object.defineProperty(cloneDeep, "__esModule", {
	  value: true
	});
	cloneDeep.default = cloneDeep$1;
	var _cloneNode = requireCloneNode();
	function cloneDeep$1(node) {
	  return (0, _cloneNode.default)(node);
	}

	
	return cloneDeep;
}

var cloneDeepWithoutLoc = {};

var hasRequiredCloneDeepWithoutLoc;

function requireCloneDeepWithoutLoc () {
	if (hasRequiredCloneDeepWithoutLoc) return cloneDeepWithoutLoc;
	hasRequiredCloneDeepWithoutLoc = 1;

	Object.defineProperty(cloneDeepWithoutLoc, "__esModule", {
	  value: true
	});
	cloneDeepWithoutLoc.default = cloneDeepWithoutLoc$1;
	var _cloneNode = requireCloneNode();
	function cloneDeepWithoutLoc$1(node) {
	  return (0, _cloneNode.default)(node, true, true);
	}

	
	return cloneDeepWithoutLoc;
}

var cloneWithoutLoc = {};

var hasRequiredCloneWithoutLoc;

function requireCloneWithoutLoc () {
	if (hasRequiredCloneWithoutLoc) return cloneWithoutLoc;
	hasRequiredCloneWithoutLoc = 1;

	Object.defineProperty(cloneWithoutLoc, "__esModule", {
	  value: true
	});
	cloneWithoutLoc.default = cloneWithoutLoc$1;
	var _cloneNode = requireCloneNode();
	function cloneWithoutLoc$1(node) {
	  return (0, _cloneNode.default)(node, false, true);
	}

	
	return cloneWithoutLoc;
}

var addComment = {};

var addComments = {};

var hasRequiredAddComments;

function requireAddComments () {
	if (hasRequiredAddComments) return addComments;
	hasRequiredAddComments = 1;

	Object.defineProperty(addComments, "__esModule", {
	  value: true
	});
	addComments.default = addComments$1;
	function addComments$1(node, type, comments) {
	  if (!comments || !node) return node;
	  const key = `${type}Comments`;
	  if (node[key]) {
	    if (type === "leading") {
	      node[key] = comments.concat(node[key]);
	    } else {
	      node[key].push(...comments);
	    }
	  } else {
	    node[key] = comments;
	  }
	  return node;
	}

	
	return addComments;
}

var hasRequiredAddComment;

function requireAddComment () {
	if (hasRequiredAddComment) return addComment;
	hasRequiredAddComment = 1;

	Object.defineProperty(addComment, "__esModule", {
	  value: true
	});
	addComment.default = addComment$1;
	var _addComments = requireAddComments();
	function addComment$1(node, type, content, line) {
	  return (0, _addComments.default)(node, type, [{
	    type: line ? "CommentLine" : "CommentBlock",
	    value: content
	  }]);
	}

	
	return addComment;
}

var inheritInnerComments = {};

var inherit = {};

var hasRequiredInherit;

function requireInherit () {
	if (hasRequiredInherit) return inherit;
	hasRequiredInherit = 1;

	Object.defineProperty(inherit, "__esModule", {
	  value: true
	});
	inherit.default = inherit$1;
	function inherit$1(key, child, parent) {
	  if (child && parent) {
	    child[key] = Array.from(new Set([].concat(child[key], parent[key]).filter(Boolean)));
	  }
	}

	
	return inherit;
}

var hasRequiredInheritInnerComments;

function requireInheritInnerComments () {
	if (hasRequiredInheritInnerComments) return inheritInnerComments;
	hasRequiredInheritInnerComments = 1;

	Object.defineProperty(inheritInnerComments, "__esModule", {
	  value: true
	});
	inheritInnerComments.default = inheritInnerComments$1;
	var _inherit = requireInherit();
	function inheritInnerComments$1(child, parent) {
	  (0, _inherit.default)("innerComments", child, parent);
	}

	
	return inheritInnerComments;
}

var inheritLeadingComments = {};

var hasRequiredInheritLeadingComments;

function requireInheritLeadingComments () {
	if (hasRequiredInheritLeadingComments) return inheritLeadingComments;
	hasRequiredInheritLeadingComments = 1;

	Object.defineProperty(inheritLeadingComments, "__esModule", {
	  value: true
	});
	inheritLeadingComments.default = inheritLeadingComments$1;
	var _inherit = requireInherit();
	function inheritLeadingComments$1(child, parent) {
	  (0, _inherit.default)("leadingComments", child, parent);
	}

	
	return inheritLeadingComments;
}

var inheritsComments = {};

var inheritTrailingComments = {};

var hasRequiredInheritTrailingComments;

function requireInheritTrailingComments () {
	if (hasRequiredInheritTrailingComments) return inheritTrailingComments;
	hasRequiredInheritTrailingComments = 1;

	Object.defineProperty(inheritTrailingComments, "__esModule", {
	  value: true
	});
	inheritTrailingComments.default = inheritTrailingComments$1;
	var _inherit = requireInherit();
	function inheritTrailingComments$1(child, parent) {
	  (0, _inherit.default)("trailingComments", child, parent);
	}

	
	return inheritTrailingComments;
}

var hasRequiredInheritsComments;

function requireInheritsComments () {
	if (hasRequiredInheritsComments) return inheritsComments;
	hasRequiredInheritsComments = 1;

	Object.defineProperty(inheritsComments, "__esModule", {
	  value: true
	});
	inheritsComments.default = inheritsComments$1;
	var _inheritTrailingComments = requireInheritTrailingComments();
	var _inheritLeadingComments = requireInheritLeadingComments();
	var _inheritInnerComments = requireInheritInnerComments();
	function inheritsComments$1(child, parent) {
	  (0, _inheritTrailingComments.default)(child, parent);
	  (0, _inheritLeadingComments.default)(child, parent);
	  (0, _inheritInnerComments.default)(child, parent);
	  return child;
	}

	
	return inheritsComments;
}

var removeComments = {};

var hasRequiredRemoveComments;

function requireRemoveComments () {
	if (hasRequiredRemoveComments) return removeComments;
	hasRequiredRemoveComments = 1;

	Object.defineProperty(removeComments, "__esModule", {
	  value: true
	});
	removeComments.default = removeComments$1;
	var _index = requireConstants();
	function removeComments$1(node) {
	  _index.COMMENT_KEYS.forEach(key => {
	    node[key] = null;
	  });
	  return node;
	}

	
	return removeComments;
}

var generated = {};

var hasRequiredGenerated;

function requireGenerated () {
	if (hasRequiredGenerated) return generated;
	hasRequiredGenerated = 1;

	Object.defineProperty(generated, "__esModule", {
	  value: true
	});
	generated.WHILE_TYPES = generated.USERWHITESPACABLE_TYPES = generated.UNARYLIKE_TYPES = generated.TYPESCRIPT_TYPES = generated.TSTYPE_TYPES = generated.TSTYPEELEMENT_TYPES = generated.TSENTITYNAME_TYPES = generated.TSBASETYPE_TYPES = generated.TERMINATORLESS_TYPES = generated.STATEMENT_TYPES = generated.STANDARDIZED_TYPES = generated.SCOPABLE_TYPES = generated.PUREISH_TYPES = generated.PROPERTY_TYPES = generated.PRIVATE_TYPES = generated.PATTERN_TYPES = generated.PATTERNLIKE_TYPES = generated.OBJECTMEMBER_TYPES = generated.MODULESPECIFIER_TYPES = generated.MODULEDECLARATION_TYPES = generated.MISCELLANEOUS_TYPES = generated.METHOD_TYPES = generated.LVAL_TYPES = generated.LOOP_TYPES = generated.LITERAL_TYPES = generated.JSX_TYPES = generated.IMPORTOREXPORTDECLARATION_TYPES = generated.IMMUTABLE_TYPES = generated.FUNCTION_TYPES = generated.FUNCTIONPARENT_TYPES = generated.FOR_TYPES = generated.FORXSTATEMENT_TYPES = generated.FLOW_TYPES = generated.FLOWTYPE_TYPES = generated.FLOWPREDICATE_TYPES = generated.FLOWDECLARATION_TYPES = generated.FLOWBASEANNOTATION_TYPES = generated.EXPRESSION_TYPES = generated.EXPRESSIONWRAPPER_TYPES = generated.EXPORTDECLARATION_TYPES = generated.ENUMMEMBER_TYPES = generated.ENUMBODY_TYPES = generated.DECLARATION_TYPES = generated.CONDITIONAL_TYPES = generated.COMPLETIONSTATEMENT_TYPES = generated.CLASS_TYPES = generated.BLOCK_TYPES = generated.BLOCKPARENT_TYPES = generated.BINARY_TYPES = generated.ACCESSOR_TYPES = void 0;
	var _index = requireDefinitions();
	generated.STANDARDIZED_TYPES = _index.FLIPPED_ALIAS_KEYS["Standardized"];
	generated.EXPRESSION_TYPES = _index.FLIPPED_ALIAS_KEYS["Expression"];
	generated.BINARY_TYPES = _index.FLIPPED_ALIAS_KEYS["Binary"];
	generated.SCOPABLE_TYPES = _index.FLIPPED_ALIAS_KEYS["Scopable"];
	generated.BLOCKPARENT_TYPES = _index.FLIPPED_ALIAS_KEYS["BlockParent"];
	generated.BLOCK_TYPES = _index.FLIPPED_ALIAS_KEYS["Block"];
	generated.STATEMENT_TYPES = _index.FLIPPED_ALIAS_KEYS["Statement"];
	generated.TERMINATORLESS_TYPES = _index.FLIPPED_ALIAS_KEYS["Terminatorless"];
	generated.COMPLETIONSTATEMENT_TYPES = _index.FLIPPED_ALIAS_KEYS["CompletionStatement"];
	generated.CONDITIONAL_TYPES = _index.FLIPPED_ALIAS_KEYS["Conditional"];
	generated.LOOP_TYPES = _index.FLIPPED_ALIAS_KEYS["Loop"];
	generated.WHILE_TYPES = _index.FLIPPED_ALIAS_KEYS["While"];
	generated.EXPRESSIONWRAPPER_TYPES = _index.FLIPPED_ALIAS_KEYS["ExpressionWrapper"];
	generated.FOR_TYPES = _index.FLIPPED_ALIAS_KEYS["For"];
	generated.FORXSTATEMENT_TYPES = _index.FLIPPED_ALIAS_KEYS["ForXStatement"];
	generated.FUNCTION_TYPES = _index.FLIPPED_ALIAS_KEYS["Function"];
	generated.FUNCTIONPARENT_TYPES = _index.FLIPPED_ALIAS_KEYS["FunctionParent"];
	generated.PUREISH_TYPES = _index.FLIPPED_ALIAS_KEYS["Pureish"];
	generated.DECLARATION_TYPES = _index.FLIPPED_ALIAS_KEYS["Declaration"];
	generated.PATTERNLIKE_TYPES = _index.FLIPPED_ALIAS_KEYS["PatternLike"];
	generated.LVAL_TYPES = _index.FLIPPED_ALIAS_KEYS["LVal"];
	generated.TSENTITYNAME_TYPES = _index.FLIPPED_ALIAS_KEYS["TSEntityName"];
	generated.LITERAL_TYPES = _index.FLIPPED_ALIAS_KEYS["Literal"];
	generated.IMMUTABLE_TYPES = _index.FLIPPED_ALIAS_KEYS["Immutable"];
	generated.USERWHITESPACABLE_TYPES = _index.FLIPPED_ALIAS_KEYS["UserWhitespacable"];
	generated.METHOD_TYPES = _index.FLIPPED_ALIAS_KEYS["Method"];
	generated.OBJECTMEMBER_TYPES = _index.FLIPPED_ALIAS_KEYS["ObjectMember"];
	generated.PROPERTY_TYPES = _index.FLIPPED_ALIAS_KEYS["Property"];
	generated.UNARYLIKE_TYPES = _index.FLIPPED_ALIAS_KEYS["UnaryLike"];
	generated.PATTERN_TYPES = _index.FLIPPED_ALIAS_KEYS["Pattern"];
	generated.CLASS_TYPES = _index.FLIPPED_ALIAS_KEYS["Class"];
	const IMPORTOREXPORTDECLARATION_TYPES = generated.IMPORTOREXPORTDECLARATION_TYPES = _index.FLIPPED_ALIAS_KEYS["ImportOrExportDeclaration"];
	generated.EXPORTDECLARATION_TYPES = _index.FLIPPED_ALIAS_KEYS["ExportDeclaration"];
	generated.MODULESPECIFIER_TYPES = _index.FLIPPED_ALIAS_KEYS["ModuleSpecifier"];
	generated.ACCESSOR_TYPES = _index.FLIPPED_ALIAS_KEYS["Accessor"];
	generated.PRIVATE_TYPES = _index.FLIPPED_ALIAS_KEYS["Private"];
	generated.FLOW_TYPES = _index.FLIPPED_ALIAS_KEYS["Flow"];
	generated.FLOWTYPE_TYPES = _index.FLIPPED_ALIAS_KEYS["FlowType"];
	generated.FLOWBASEANNOTATION_TYPES = _index.FLIPPED_ALIAS_KEYS["FlowBaseAnnotation"];
	generated.FLOWDECLARATION_TYPES = _index.FLIPPED_ALIAS_KEYS["FlowDeclaration"];
	generated.FLOWPREDICATE_TYPES = _index.FLIPPED_ALIAS_KEYS["FlowPredicate"];
	generated.ENUMBODY_TYPES = _index.FLIPPED_ALIAS_KEYS["EnumBody"];
	generated.ENUMMEMBER_TYPES = _index.FLIPPED_ALIAS_KEYS["EnumMember"];
	generated.JSX_TYPES = _index.FLIPPED_ALIAS_KEYS["JSX"];
	generated.MISCELLANEOUS_TYPES = _index.FLIPPED_ALIAS_KEYS["Miscellaneous"];
	generated.TYPESCRIPT_TYPES = _index.FLIPPED_ALIAS_KEYS["TypeScript"];
	generated.TSTYPEELEMENT_TYPES = _index.FLIPPED_ALIAS_KEYS["TSTypeElement"];
	generated.TSTYPE_TYPES = _index.FLIPPED_ALIAS_KEYS["TSType"];
	generated.TSBASETYPE_TYPES = _index.FLIPPED_ALIAS_KEYS["TSBaseType"];
	generated.MODULEDECLARATION_TYPES = IMPORTOREXPORTDECLARATION_TYPES;

	
	return generated;
}

var ensureBlock = {};

var toBlock = {};

var hasRequiredToBlock;

function requireToBlock () {
	if (hasRequiredToBlock) return toBlock;
	hasRequiredToBlock = 1;

	Object.defineProperty(toBlock, "__esModule", {
	  value: true
	});
	toBlock.default = toBlock$1;
	var _index = requireGenerated$3();
	var _index2 = requireGenerated$2();
	function toBlock$1(node, parent) {
	  if ((0, _index.isBlockStatement)(node)) {
	    return node;
	  }
	  let blockNodes = [];
	  if ((0, _index.isEmptyStatement)(node)) {
	    blockNodes = [];
	  } else {
	    if (!(0, _index.isStatement)(node)) {
	      if ((0, _index.isFunction)(parent)) {
	        node = (0, _index2.returnStatement)(node);
	      } else {
	        node = (0, _index2.expressionStatement)(node);
	      }
	    }
	    blockNodes = [node];
	  }
	  return (0, _index2.blockStatement)(blockNodes);
	}

	
	return toBlock;
}

var hasRequiredEnsureBlock;

function requireEnsureBlock () {
	if (hasRequiredEnsureBlock) return ensureBlock;
	hasRequiredEnsureBlock = 1;

	Object.defineProperty(ensureBlock, "__esModule", {
	  value: true
	});
	ensureBlock.default = ensureBlock$1;
	var _toBlock = requireToBlock();
	function ensureBlock$1(node, key = "body") {
	  const result = (0, _toBlock.default)(node[key], node);
	  node[key] = result;
	  return result;
	}

	
	return ensureBlock;
}

var toBindingIdentifierName = {};

var toIdentifier = {};

var hasRequiredToIdentifier;

function requireToIdentifier () {
	if (hasRequiredToIdentifier) return toIdentifier;
	hasRequiredToIdentifier = 1;

	Object.defineProperty(toIdentifier, "__esModule", {
	  value: true
	});
	toIdentifier.default = toIdentifier$1;
	var _isValidIdentifier = requireIsValidIdentifier();
	var _helperValidatorIdentifier = requireLib$3();
	function toIdentifier$1(input) {
	  input = input + "";
	  let name = "";
	  for (const c of input) {
	    name += (0, _helperValidatorIdentifier.isIdentifierChar)(c.codePointAt(0)) ? c : "-";
	  }
	  name = name.replace(/^[-0-9]+/, "");
	  name = name.replace(/[-\s]+(.)?/g, function (match, c) {
	    return c ? c.toUpperCase() : "";
	  });
	  if (!(0, _isValidIdentifier.default)(name)) {
	    name = `_${name}`;
	  }
	  return name || "_";
	}

	
	return toIdentifier;
}

var hasRequiredToBindingIdentifierName;

function requireToBindingIdentifierName () {
	if (hasRequiredToBindingIdentifierName) return toBindingIdentifierName;
	hasRequiredToBindingIdentifierName = 1;

	Object.defineProperty(toBindingIdentifierName, "__esModule", {
	  value: true
	});
	toBindingIdentifierName.default = toBindingIdentifierName$1;
	var _toIdentifier = requireToIdentifier();
	function toBindingIdentifierName$1(name) {
	  name = (0, _toIdentifier.default)(name);
	  if (name === "eval" || name === "arguments") name = "_" + name;
	  return name;
	}

	
	return toBindingIdentifierName;
}

var toComputedKey = {};

var hasRequiredToComputedKey;

function requireToComputedKey () {
	if (hasRequiredToComputedKey) return toComputedKey;
	hasRequiredToComputedKey = 1;

	Object.defineProperty(toComputedKey, "__esModule", {
	  value: true
	});
	toComputedKey.default = toComputedKey$1;
	var _index = requireGenerated$3();
	var _index2 = requireGenerated$2();
	function toComputedKey$1(node, key = node.key || node.property) {
	  if (!node.computed && (0, _index.isIdentifier)(key)) key = (0, _index2.stringLiteral)(key.name);
	  return key;
	}

	
	return toComputedKey;
}

var toExpression = {};

var hasRequiredToExpression;

function requireToExpression () {
	if (hasRequiredToExpression) return toExpression;
	hasRequiredToExpression = 1;

	Object.defineProperty(toExpression, "__esModule", {
	  value: true
	});
	toExpression.default = void 0;
	var _index = requireGenerated$3();
	toExpression.default = toExpression$1;
	function toExpression$1(node) {
	  if ((0, _index.isExpressionStatement)(node)) {
	    node = node.expression;
	  }
	  if ((0, _index.isExpression)(node)) {
	    return node;
	  }
	  if ((0, _index.isClass)(node)) {
	    node.type = "ClassExpression";
	  } else if ((0, _index.isFunction)(node)) {
	    node.type = "FunctionExpression";
	  }
	  if (!(0, _index.isExpression)(node)) {
	    throw new Error(`cannot turn ${node.type} to an expression`);
	  }
	  return node;
	}

	
	return toExpression;
}

var toKeyAlias = {};

var removePropertiesDeep = {};

var traverseFast = {};

var hasRequiredTraverseFast;

function requireTraverseFast () {
	if (hasRequiredTraverseFast) return traverseFast;
	hasRequiredTraverseFast = 1;

	Object.defineProperty(traverseFast, "__esModule", {
	  value: true
	});
	traverseFast.default = traverseFast$1;
	var _index = requireDefinitions();
	function traverseFast$1(node, enter, opts) {
	  if (!node) return;
	  const keys = _index.VISITOR_KEYS[node.type];
	  if (!keys) return;
	  opts = opts || {};
	  enter(node, opts);
	  for (const key of keys) {
	    const subNode = node[key];
	    if (Array.isArray(subNode)) {
	      for (const node of subNode) {
	        traverseFast$1(node, enter, opts);
	      }
	    } else {
	      traverseFast$1(subNode, enter, opts);
	    }
	  }
	}

	
	return traverseFast;
}

var removeProperties = {};

var hasRequiredRemoveProperties;

function requireRemoveProperties () {
	if (hasRequiredRemoveProperties) return removeProperties;
	hasRequiredRemoveProperties = 1;

	Object.defineProperty(removeProperties, "__esModule", {
	  value: true
	});
	removeProperties.default = removeProperties$1;
	var _index = requireConstants();
	const CLEAR_KEYS = ["tokens", "start", "end", "loc", "raw", "rawValue"];
	const CLEAR_KEYS_PLUS_COMMENTS = [..._index.COMMENT_KEYS, "comments", ...CLEAR_KEYS];
	function removeProperties$1(node, opts = {}) {
	  const map = opts.preserveComments ? CLEAR_KEYS : CLEAR_KEYS_PLUS_COMMENTS;
	  for (const key of map) {
	    if (node[key] != null) node[key] = undefined;
	  }
	  for (const key of Object.keys(node)) {
	    if (key[0] === "_" && node[key] != null) node[key] = undefined;
	  }
	  const symbols = Object.getOwnPropertySymbols(node);
	  for (const sym of symbols) {
	    node[sym] = null;
	  }
	}

	
	return removeProperties;
}

var hasRequiredRemovePropertiesDeep;

function requireRemovePropertiesDeep () {
	if (hasRequiredRemovePropertiesDeep) return removePropertiesDeep;
	hasRequiredRemovePropertiesDeep = 1;

	Object.defineProperty(removePropertiesDeep, "__esModule", {
	  value: true
	});
	removePropertiesDeep.default = removePropertiesDeep$1;
	var _traverseFast = requireTraverseFast();
	var _removeProperties = requireRemoveProperties();
	function removePropertiesDeep$1(tree, opts) {
	  (0, _traverseFast.default)(tree, _removeProperties.default, opts);
	  return tree;
	}

	
	return removePropertiesDeep;
}

var hasRequiredToKeyAlias;

function requireToKeyAlias () {
	if (hasRequiredToKeyAlias) return toKeyAlias;
	hasRequiredToKeyAlias = 1;

	Object.defineProperty(toKeyAlias, "__esModule", {
	  value: true
	});
	toKeyAlias.default = toKeyAlias$1;
	var _index = requireGenerated$3();
	var _cloneNode = requireCloneNode();
	var _removePropertiesDeep = requireRemovePropertiesDeep();
	function toKeyAlias$1(node, key = node.key) {
	  let alias;
	  if (node.kind === "method") {
	    return toKeyAlias$1.increment() + "";
	  } else if ((0, _index.isIdentifier)(key)) {
	    alias = key.name;
	  } else if ((0, _index.isStringLiteral)(key)) {
	    alias = JSON.stringify(key.value);
	  } else {
	    alias = JSON.stringify((0, _removePropertiesDeep.default)((0, _cloneNode.default)(key)));
	  }
	  if (node.computed) {
	    alias = `[${alias}]`;
	  }
	  if (node.static) {
	    alias = `static:${alias}`;
	  }
	  return alias;
	}
	toKeyAlias$1.uid = 0;
	toKeyAlias$1.increment = function () {
	  if (toKeyAlias$1.uid >= Number.MAX_SAFE_INTEGER) {
	    return toKeyAlias$1.uid = 0;
	  } else {
	    return toKeyAlias$1.uid++;
	  }
	};

	
	return toKeyAlias;
}

var toStatement = {};

var hasRequiredToStatement;

function requireToStatement () {
	if (hasRequiredToStatement) return toStatement;
	hasRequiredToStatement = 1;

	Object.defineProperty(toStatement, "__esModule", {
	  value: true
	});
	toStatement.default = void 0;
	var _index = requireGenerated$3();
	var _index2 = requireGenerated$2();
	toStatement.default = toStatement$1;
	function toStatement$1(node, ignore) {
	  if ((0, _index.isStatement)(node)) {
	    return node;
	  }
	  let mustHaveId = false;
	  let newType;
	  if ((0, _index.isClass)(node)) {
	    mustHaveId = true;
	    newType = "ClassDeclaration";
	  } else if ((0, _index.isFunction)(node)) {
	    mustHaveId = true;
	    newType = "FunctionDeclaration";
	  } else if ((0, _index.isAssignmentExpression)(node)) {
	    return (0, _index2.expressionStatement)(node);
	  }
	  if (mustHaveId && !node.id) {
	    newType = false;
	  }
	  if (!newType) {
	    if (ignore) {
	      return false;
	    } else {
	      throw new Error(`cannot turn ${node.type} to a statement`);
	    }
	  }
	  node.type = newType;
	  return node;
	}

	
	return toStatement;
}

var valueToNode = {};

var hasRequiredValueToNode;

function requireValueToNode () {
	if (hasRequiredValueToNode) return valueToNode;
	hasRequiredValueToNode = 1;

	Object.defineProperty(valueToNode, "__esModule", {
	  value: true
	});
	valueToNode.default = void 0;
	var _isValidIdentifier = requireIsValidIdentifier();
	var _index = requireGenerated$2();
	valueToNode.default = valueToNode$1;
	const objectToString = Function.call.bind(Object.prototype.toString);
	function isRegExp(value) {
	  return objectToString(value) === "[object RegExp]";
	}
	function isPlainObject(value) {
	  if (typeof value !== "object" || value === null || Object.prototype.toString.call(value) !== "[object Object]") {
	    return false;
	  }
	  const proto = Object.getPrototypeOf(value);
	  return proto === null || Object.getPrototypeOf(proto) === null;
	}
	function valueToNode$1(value) {
	  if (value === undefined) {
	    return (0, _index.identifier)("undefined");
	  }
	  if (value === true || value === false) {
	    return (0, _index.booleanLiteral)(value);
	  }
	  if (value === null) {
	    return (0, _index.nullLiteral)();
	  }
	  if (typeof value === "string") {
	    return (0, _index.stringLiteral)(value);
	  }
	  if (typeof value === "number") {
	    let result;
	    if (Number.isFinite(value)) {
	      result = (0, _index.numericLiteral)(Math.abs(value));
	    } else {
	      let numerator;
	      if (Number.isNaN(value)) {
	        numerator = (0, _index.numericLiteral)(0);
	      } else {
	        numerator = (0, _index.numericLiteral)(1);
	      }
	      result = (0, _index.binaryExpression)("/", numerator, (0, _index.numericLiteral)(0));
	    }
	    if (value < 0 || Object.is(value, -0)) {
	      result = (0, _index.unaryExpression)("-", result);
	    }
	    return result;
	  }
	  if (isRegExp(value)) {
	    const pattern = value.source;
	    const flags = /\/([a-z]*)$/.exec(value.toString())[1];
	    return (0, _index.regExpLiteral)(pattern, flags);
	  }
	  if (Array.isArray(value)) {
	    return (0, _index.arrayExpression)(value.map(valueToNode$1));
	  }
	  if (isPlainObject(value)) {
	    const props = [];
	    for (const key of Object.keys(value)) {
	      let nodeKey;
	      if ((0, _isValidIdentifier.default)(key)) {
	        nodeKey = (0, _index.identifier)(key);
	      } else {
	        nodeKey = (0, _index.stringLiteral)(key);
	      }
	      props.push((0, _index.objectProperty)(nodeKey, valueToNode$1(value[key])));
	    }
	    return (0, _index.objectExpression)(props);
	  }
	  throw new Error("don't know how to turn this value into a node");
	}

	
	return valueToNode;
}

var appendToMemberExpression = {};

var hasRequiredAppendToMemberExpression;

function requireAppendToMemberExpression () {
	if (hasRequiredAppendToMemberExpression) return appendToMemberExpression;
	hasRequiredAppendToMemberExpression = 1;

	Object.defineProperty(appendToMemberExpression, "__esModule", {
	  value: true
	});
	appendToMemberExpression.default = appendToMemberExpression$1;
	var _index = requireGenerated$2();
	function appendToMemberExpression$1(member, append, computed = false) {
	  member.object = (0, _index.memberExpression)(member.object, member.property, member.computed);
	  member.property = append;
	  member.computed = !!computed;
	  return member;
	}

	
	return appendToMemberExpression;
}

var inherits = {};

var hasRequiredInherits;

function requireInherits () {
	if (hasRequiredInherits) return inherits;
	hasRequiredInherits = 1;

	Object.defineProperty(inherits, "__esModule", {
	  value: true
	});
	inherits.default = inherits$1;
	var _index = requireConstants();
	var _inheritsComments = requireInheritsComments();
	function inherits$1(child, parent) {
	  if (!child || !parent) return child;
	  for (const key of _index.INHERIT_KEYS.optional) {
	    if (child[key] == null) {
	      child[key] = parent[key];
	    }
	  }
	  for (const key of Object.keys(parent)) {
	    if (key[0] === "_" && key !== "__clone") {
	      child[key] = parent[key];
	    }
	  }
	  for (const key of _index.INHERIT_KEYS.force) {
	    child[key] = parent[key];
	  }
	  (0, _inheritsComments.default)(child, parent);
	  return child;
	}

	
	return inherits;
}

var prependToMemberExpression = {};

var hasRequiredPrependToMemberExpression;

function requirePrependToMemberExpression () {
	if (hasRequiredPrependToMemberExpression) return prependToMemberExpression;
	hasRequiredPrependToMemberExpression = 1;

	Object.defineProperty(prependToMemberExpression, "__esModule", {
	  value: true
	});
	prependToMemberExpression.default = prependToMemberExpression$1;
	var _index = requireGenerated$2();
	var _index2 = requireLib$1();
	function prependToMemberExpression$1(member, prepend) {
	  if ((0, _index2.isSuper)(member.object)) {
	    throw new Error("Cannot prepend node to super property access (`super.foo`).");
	  }
	  member.object = (0, _index.memberExpression)(prepend, member.object);
	  return member;
	}

	
	return prependToMemberExpression;
}

var getAssignmentIdentifiers = {};

var hasRequiredGetAssignmentIdentifiers;

function requireGetAssignmentIdentifiers () {
	if (hasRequiredGetAssignmentIdentifiers) return getAssignmentIdentifiers;
	hasRequiredGetAssignmentIdentifiers = 1;

	Object.defineProperty(getAssignmentIdentifiers, "__esModule", {
	  value: true
	});
	getAssignmentIdentifiers.default = getAssignmentIdentifiers$1;
	function getAssignmentIdentifiers$1(node) {
	  const search = [].concat(node);
	  const ids = Object.create(null);
	  while (search.length) {
	    const id = search.pop();
	    if (!id) continue;
	    switch (id.type) {
	      case "ArrayPattern":
	        search.push(...id.elements);
	        break;
	      case "AssignmentExpression":
	      case "AssignmentPattern":
	      case "ForInStatement":
	      case "ForOfStatement":
	        search.push(id.left);
	        break;
	      case "ObjectPattern":
	        search.push(...id.properties);
	        break;
	      case "ObjectProperty":
	        search.push(id.value);
	        break;
	      case "RestElement":
	      case "UpdateExpression":
	        search.push(id.argument);
	        break;
	      case "UnaryExpression":
	        if (id.operator === "delete") {
	          search.push(id.argument);
	        }
	        break;
	      case "Identifier":
	        ids[id.name] = id;
	        break;
	    }
	  }
	  return ids;
	}

	
	return getAssignmentIdentifiers;
}

var getBindingIdentifiers = {};

var hasRequiredGetBindingIdentifiers;

function requireGetBindingIdentifiers () {
	if (hasRequiredGetBindingIdentifiers) return getBindingIdentifiers;
	hasRequiredGetBindingIdentifiers = 1;

	Object.defineProperty(getBindingIdentifiers, "__esModule", {
	  value: true
	});
	getBindingIdentifiers.default = getBindingIdentifiers$1;
	var _index = requireGenerated$3();
	function getBindingIdentifiers$1(node, duplicates, outerOnly, newBindingsOnly) {
	  const search = [].concat(node);
	  const ids = Object.create(null);
	  while (search.length) {
	    const id = search.shift();
	    if (!id) continue;
	    if (newBindingsOnly && ((0, _index.isAssignmentExpression)(id) || (0, _index.isUnaryExpression)(id) || (0, _index.isUpdateExpression)(id))) {
	      continue;
	    }
	    if ((0, _index.isIdentifier)(id)) {
	      if (duplicates) {
	        const _ids = ids[id.name] = ids[id.name] || [];
	        _ids.push(id);
	      } else {
	        ids[id.name] = id;
	      }
	      continue;
	    }
	    if ((0, _index.isExportDeclaration)(id) && !(0, _index.isExportAllDeclaration)(id)) {
	      if ((0, _index.isDeclaration)(id.declaration)) {
	        search.push(id.declaration);
	      }
	      continue;
	    }
	    if (outerOnly) {
	      if ((0, _index.isFunctionDeclaration)(id)) {
	        search.push(id.id);
	        continue;
	      }
	      if ((0, _index.isFunctionExpression)(id)) {
	        continue;
	      }
	    }
	    const keys = getBindingIdentifiers$1.keys[id.type];
	    if (keys) {
	      for (let i = 0; i < keys.length; i++) {
	        const key = keys[i];
	        const nodes = id[key];
	        if (nodes) {
	          if (Array.isArray(nodes)) {
	            search.push(...nodes);
	          } else {
	            search.push(nodes);
	          }
	        }
	      }
	    }
	  }
	  return ids;
	}
	const keys = {
	  DeclareClass: ["id"],
	  DeclareFunction: ["id"],
	  DeclareModule: ["id"],
	  DeclareVariable: ["id"],
	  DeclareInterface: ["id"],
	  DeclareTypeAlias: ["id"],
	  DeclareOpaqueType: ["id"],
	  InterfaceDeclaration: ["id"],
	  TypeAlias: ["id"],
	  OpaqueType: ["id"],
	  CatchClause: ["param"],
	  LabeledStatement: ["label"],
	  UnaryExpression: ["argument"],
	  AssignmentExpression: ["left"],
	  ImportSpecifier: ["local"],
	  ImportNamespaceSpecifier: ["local"],
	  ImportDefaultSpecifier: ["local"],
	  ImportDeclaration: ["specifiers"],
	  ExportSpecifier: ["exported"],
	  ExportNamespaceSpecifier: ["exported"],
	  ExportDefaultSpecifier: ["exported"],
	  FunctionDeclaration: ["id", "params"],
	  FunctionExpression: ["id", "params"],
	  ArrowFunctionExpression: ["params"],
	  ObjectMethod: ["params"],
	  ClassMethod: ["params"],
	  ClassPrivateMethod: ["params"],
	  ForInStatement: ["left"],
	  ForOfStatement: ["left"],
	  ClassDeclaration: ["id"],
	  ClassExpression: ["id"],
	  RestElement: ["argument"],
	  UpdateExpression: ["argument"],
	  ObjectProperty: ["value"],
	  AssignmentPattern: ["left"],
	  ArrayPattern: ["elements"],
	  ObjectPattern: ["properties"],
	  VariableDeclaration: ["declarations"],
	  VariableDeclarator: ["id"]
	};
	getBindingIdentifiers$1.keys = keys;

	
	return getBindingIdentifiers;
}

var getOuterBindingIdentifiers = {};

var hasRequiredGetOuterBindingIdentifiers;

function requireGetOuterBindingIdentifiers () {
	if (hasRequiredGetOuterBindingIdentifiers) return getOuterBindingIdentifiers;
	hasRequiredGetOuterBindingIdentifiers = 1;

	Object.defineProperty(getOuterBindingIdentifiers, "__esModule", {
	  value: true
	});
	getOuterBindingIdentifiers.default = void 0;
	var _getBindingIdentifiers = requireGetBindingIdentifiers();
	getOuterBindingIdentifiers.default = getOuterBindingIdentifiers$1;
	function getOuterBindingIdentifiers$1(node, duplicates) {
	  return (0, _getBindingIdentifiers.default)(node, duplicates, true);
	}

	
	return getOuterBindingIdentifiers;
}

var getFunctionName$3 = {};

var hasRequiredGetFunctionName;

function requireGetFunctionName () {
	if (hasRequiredGetFunctionName) return getFunctionName$3;
	hasRequiredGetFunctionName = 1;

	Object.defineProperty(getFunctionName$3, "__esModule", {
	  value: true
	});
	getFunctionName$3.default = getFunctionName;
	var _index = requireGenerated$3();
	function getNameFromLiteralId(id) {
	  if ((0, _index.isNullLiteral)(id)) {
	    return "null";
	  }
	  if ((0, _index.isRegExpLiteral)(id)) {
	    return `/${id.pattern}/${id.flags}`;
	  }
	  if ((0, _index.isTemplateLiteral)(id)) {
	    return id.quasis.map(quasi => quasi.value.raw).join("");
	  }
	  if (id.value !== undefined) {
	    return String(id.value);
	  }
	  return null;
	}
	function getObjectMemberKey(node) {
	  if (!node.computed || (0, _index.isLiteral)(node.key)) {
	    return node.key;
	  }
	}
	function getFunctionName(node, parent) {
	  if ("id" in node && node.id) {
	    return {
	      name: node.id.name,
	      originalNode: node.id
	    };
	  }
	  let prefix = "";
	  let id;
	  if ((0, _index.isObjectProperty)(parent, {
	    value: node
	  })) {
	    id = getObjectMemberKey(parent);
	  } else if ((0, _index.isObjectMethod)(node) || (0, _index.isClassMethod)(node)) {
	    id = getObjectMemberKey(node);
	    if (node.kind === "get") prefix = "get ";else if (node.kind === "set") prefix = "set ";
	  } else if ((0, _index.isVariableDeclarator)(parent, {
	    init: node
	  })) {
	    id = parent.id;
	  } else if ((0, _index.isAssignmentExpression)(parent, {
	    operator: "=",
	    right: node
	  })) {
	    id = parent.left;
	  }
	  if (!id) return null;
	  const name = (0, _index.isLiteral)(id) ? getNameFromLiteralId(id) : (0, _index.isIdentifier)(id) ? id.name : (0, _index.isPrivateName)(id) ? id.id.name : null;
	  if (name == null) return null;
	  return {
	    name: prefix + name,
	    originalNode: id
	  };
	}

	
	return getFunctionName$3;
}

var traverse = {};

var hasRequiredTraverse;

function requireTraverse () {
	if (hasRequiredTraverse) return traverse;
	hasRequiredTraverse = 1;

	Object.defineProperty(traverse, "__esModule", {
	  value: true
	});
	traverse.default = traverse$1;
	var _index = requireDefinitions();
	function traverse$1(node, handlers, state) {
	  if (typeof handlers === "function") {
	    handlers = {
	      enter: handlers
	    };
	  }
	  const {
	    enter,
	    exit
	  } = handlers;
	  traverseSimpleImpl(node, enter, exit, state, []);
	}
	function traverseSimpleImpl(node, enter, exit, state, ancestors) {
	  const keys = _index.VISITOR_KEYS[node.type];
	  if (!keys) return;
	  if (enter) enter(node, ancestors, state);
	  for (const key of keys) {
	    const subNode = node[key];
	    if (Array.isArray(subNode)) {
	      for (let i = 0; i < subNode.length; i++) {
	        const child = subNode[i];
	        if (!child) continue;
	        ancestors.push({
	          node,
	          key,
	          index: i
	        });
	        traverseSimpleImpl(child, enter, exit, state, ancestors);
	        ancestors.pop();
	      }
	    } else if (subNode) {
	      ancestors.push({
	        node,
	        key
	      });
	      traverseSimpleImpl(subNode, enter, exit, state, ancestors);
	      ancestors.pop();
	    }
	  }
	  if (exit) exit(node, ancestors, state);
	}

	
	return traverse;
}

var isBinding = {};

var hasRequiredIsBinding;

function requireIsBinding () {
	if (hasRequiredIsBinding) return isBinding;
	hasRequiredIsBinding = 1;

	Object.defineProperty(isBinding, "__esModule", {
	  value: true
	});
	isBinding.default = isBinding$1;
	var _getBindingIdentifiers = requireGetBindingIdentifiers();
	function isBinding$1(node, parent, grandparent) {
	  if (grandparent && node.type === "Identifier" && parent.type === "ObjectProperty" && grandparent.type === "ObjectExpression") {
	    return false;
	  }
	  const keys = _getBindingIdentifiers.default.keys[parent.type];
	  if (keys) {
	    for (let i = 0; i < keys.length; i++) {
	      const key = keys[i];
	      const val = parent[key];
	      if (Array.isArray(val)) {
	        if (val.includes(node)) return true;
	      } else {
	        if (val === node) return true;
	      }
	    }
	  }
	  return false;
	}

	
	return isBinding;
}

var isBlockScoped = {};

var isLet = {};

var hasRequiredIsLet;

function requireIsLet () {
	if (hasRequiredIsLet) return isLet;
	hasRequiredIsLet = 1;

	Object.defineProperty(isLet, "__esModule", {
	  value: true
	});
	isLet.default = isLet$1;
	var _index = requireGenerated$3();
	var _index2 = requireConstants();
	function isLet$1(node) {
	  return (0, _index.isVariableDeclaration)(node) && (node.kind !== "var" || node[_index2.BLOCK_SCOPED_SYMBOL]);
	}

	
	return isLet;
}

var hasRequiredIsBlockScoped;

function requireIsBlockScoped () {
	if (hasRequiredIsBlockScoped) return isBlockScoped;
	hasRequiredIsBlockScoped = 1;

	Object.defineProperty(isBlockScoped, "__esModule", {
	  value: true
	});
	isBlockScoped.default = isBlockScoped$1;
	var _index = requireGenerated$3();
	var _isLet = requireIsLet();
	function isBlockScoped$1(node) {
	  return (0, _index.isFunctionDeclaration)(node) || (0, _index.isClassDeclaration)(node) || (0, _isLet.default)(node);
	}

	
	return isBlockScoped;
}

var isImmutable = {};

var hasRequiredIsImmutable;

function requireIsImmutable () {
	if (hasRequiredIsImmutable) return isImmutable;
	hasRequiredIsImmutable = 1;

	Object.defineProperty(isImmutable, "__esModule", {
	  value: true
	});
	isImmutable.default = isImmutable$1;
	var _isType = requireIsType();
	var _index = requireGenerated$3();
	function isImmutable$1(node) {
	  if ((0, _isType.default)(node.type, "Immutable")) return true;
	  if ((0, _index.isIdentifier)(node)) {
	    if (node.name === "undefined") {
	      return true;
	    } else {
	      return false;
	    }
	  }
	  return false;
	}

	
	return isImmutable;
}

var isNodesEquivalent = {};

var hasRequiredIsNodesEquivalent;

function requireIsNodesEquivalent () {
	if (hasRequiredIsNodesEquivalent) return isNodesEquivalent;
	hasRequiredIsNodesEquivalent = 1;

	Object.defineProperty(isNodesEquivalent, "__esModule", {
	  value: true
	});
	isNodesEquivalent.default = isNodesEquivalent$1;
	var _index = requireDefinitions();
	function isNodesEquivalent$1(a, b) {
	  if (typeof a !== "object" || typeof b !== "object" || a == null || b == null) {
	    return a === b;
	  }
	  if (a.type !== b.type) {
	    return false;
	  }
	  const fields = Object.keys(_index.NODE_FIELDS[a.type] || a.type);
	  const visitorKeys = _index.VISITOR_KEYS[a.type];
	  for (const field of fields) {
	    const val_a = a[field];
	    const val_b = b[field];
	    if (typeof val_a !== typeof val_b) {
	      return false;
	    }
	    if (val_a == null && val_b == null) {
	      continue;
	    } else if (val_a == null || val_b == null) {
	      return false;
	    }
	    if (Array.isArray(val_a)) {
	      if (!Array.isArray(val_b)) {
	        return false;
	      }
	      if (val_a.length !== val_b.length) {
	        return false;
	      }
	      for (let i = 0; i < val_a.length; i++) {
	        if (!isNodesEquivalent$1(val_a[i], val_b[i])) {
	          return false;
	        }
	      }
	      continue;
	    }
	    if (typeof val_a === "object" && !(visitorKeys != null && visitorKeys.includes(field))) {
	      for (const key of Object.keys(val_a)) {
	        if (val_a[key] !== val_b[key]) {
	          return false;
	        }
	      }
	      continue;
	    }
	    if (!isNodesEquivalent$1(val_a, val_b)) {
	      return false;
	    }
	  }
	  return true;
	}

	
	return isNodesEquivalent;
}

var isReferenced = {};

var hasRequiredIsReferenced;

function requireIsReferenced () {
	if (hasRequiredIsReferenced) return isReferenced;
	hasRequiredIsReferenced = 1;

	Object.defineProperty(isReferenced, "__esModule", {
	  value: true
	});
	isReferenced.default = isReferenced$1;
	function isReferenced$1(node, parent, grandparent) {
	  switch (parent.type) {
	    case "MemberExpression":
	    case "OptionalMemberExpression":
	      if (parent.property === node) {
	        return !!parent.computed;
	      }
	      return parent.object === node;
	    case "JSXMemberExpression":
	      return parent.object === node;
	    case "VariableDeclarator":
	      return parent.init === node;
	    case "ArrowFunctionExpression":
	      return parent.body === node;
	    case "PrivateName":
	      return false;
	    case "ClassMethod":
	    case "ClassPrivateMethod":
	    case "ObjectMethod":
	      if (parent.key === node) {
	        return !!parent.computed;
	      }
	      return false;
	    case "ObjectProperty":
	      if (parent.key === node) {
	        return !!parent.computed;
	      }
	      return !grandparent || grandparent.type !== "ObjectPattern";
	    case "ClassProperty":
	    case "ClassAccessorProperty":
	      if (parent.key === node) {
	        return !!parent.computed;
	      }
	      return true;
	    case "ClassPrivateProperty":
	      return parent.key !== node;
	    case "ClassDeclaration":
	    case "ClassExpression":
	      return parent.superClass === node;
	    case "AssignmentExpression":
	      return parent.right === node;
	    case "AssignmentPattern":
	      return parent.right === node;
	    case "LabeledStatement":
	      return false;
	    case "CatchClause":
	      return false;
	    case "RestElement":
	      return false;
	    case "BreakStatement":
	    case "ContinueStatement":
	      return false;
	    case "FunctionDeclaration":
	    case "FunctionExpression":
	      return false;
	    case "ExportNamespaceSpecifier":
	    case "ExportDefaultSpecifier":
	      return false;
	    case "ExportSpecifier":
	      if (grandparent != null && grandparent.source) {
	        return false;
	      }
	      return parent.local === node;
	    case "ImportDefaultSpecifier":
	    case "ImportNamespaceSpecifier":
	    case "ImportSpecifier":
	      return false;
	    case "ImportAttribute":
	      return false;
	    case "JSXAttribute":
	      return false;
	    case "ObjectPattern":
	    case "ArrayPattern":
	      return false;
	    case "MetaProperty":
	      return false;
	    case "ObjectTypeProperty":
	      return parent.key !== node;
	    case "TSEnumMember":
	      return parent.id !== node;
	    case "TSPropertySignature":
	      if (parent.key === node) {
	        return !!parent.computed;
	      }
	      return true;
	  }
	  return true;
	}

	
	return isReferenced;
}

var isScope = {};

var hasRequiredIsScope;

function requireIsScope () {
	if (hasRequiredIsScope) return isScope;
	hasRequiredIsScope = 1;

	Object.defineProperty(isScope, "__esModule", {
	  value: true
	});
	isScope.default = isScope$1;
	var _index = requireGenerated$3();
	function isScope$1(node, parent) {
	  if ((0, _index.isBlockStatement)(node) && ((0, _index.isFunction)(parent) || (0, _index.isCatchClause)(parent))) {
	    return false;
	  }
	  if ((0, _index.isPattern)(node) && ((0, _index.isFunction)(parent) || (0, _index.isCatchClause)(parent))) {
	    return true;
	  }
	  return (0, _index.isScopable)(node);
	}

	
	return isScope;
}

var isSpecifierDefault = {};

var hasRequiredIsSpecifierDefault;

function requireIsSpecifierDefault () {
	if (hasRequiredIsSpecifierDefault) return isSpecifierDefault;
	hasRequiredIsSpecifierDefault = 1;

	Object.defineProperty(isSpecifierDefault, "__esModule", {
	  value: true
	});
	isSpecifierDefault.default = isSpecifierDefault$1;
	var _index = requireGenerated$3();
	function isSpecifierDefault$1(specifier) {
	  return (0, _index.isImportDefaultSpecifier)(specifier) || (0, _index.isIdentifier)(specifier.imported || specifier.exported, {
	    name: "default"
	  });
	}

	
	return isSpecifierDefault;
}

var isValidES3Identifier = {};

var hasRequiredIsValidES3Identifier;

function requireIsValidES3Identifier () {
	if (hasRequiredIsValidES3Identifier) return isValidES3Identifier;
	hasRequiredIsValidES3Identifier = 1;

	Object.defineProperty(isValidES3Identifier, "__esModule", {
	  value: true
	});
	isValidES3Identifier.default = isValidES3Identifier$1;
	var _isValidIdentifier = requireIsValidIdentifier();
	const RESERVED_WORDS_ES3_ONLY = new Set(["abstract", "boolean", "byte", "char", "double", "enum", "final", "float", "goto", "implements", "int", "interface", "long", "native", "package", "private", "protected", "public", "short", "static", "synchronized", "throws", "transient", "volatile"]);
	function isValidES3Identifier$1(name) {
	  return (0, _isValidIdentifier.default)(name) && !RESERVED_WORDS_ES3_ONLY.has(name);
	}

	
	return isValidES3Identifier;
}

var isVar = {};

var hasRequiredIsVar;

function requireIsVar () {
	if (hasRequiredIsVar) return isVar;
	hasRequiredIsVar = 1;

	Object.defineProperty(isVar, "__esModule", {
	  value: true
	});
	isVar.default = isVar$1;
	var _index = requireGenerated$3();
	var _index2 = requireConstants();
	function isVar$1(node) {
	  return (0, _index.isVariableDeclaration)(node, {
	    kind: "var"
	  }) && !node[_index2.BLOCK_SCOPED_SYMBOL];
	}

	
	return isVar;
}

var toSequenceExpression = {};

var gatherSequenceExpressions = {};

var hasRequiredGatherSequenceExpressions;

function requireGatherSequenceExpressions () {
	if (hasRequiredGatherSequenceExpressions) return gatherSequenceExpressions;
	hasRequiredGatherSequenceExpressions = 1;

	Object.defineProperty(gatherSequenceExpressions, "__esModule", {
	  value: true
	});
	gatherSequenceExpressions.default = gatherSequenceExpressions$1;
	var _getBindingIdentifiers = requireGetBindingIdentifiers();
	var _index = requireGenerated$3();
	var _index2 = requireGenerated$2();
	var _productions = requireProductions();
	var _cloneNode = requireCloneNode();
	function gatherSequenceExpressions$1(nodes, declars) {
	  const exprs = [];
	  let ensureLastUndefined = true;
	  for (const node of nodes) {
	    if (!(0, _index.isEmptyStatement)(node)) {
	      ensureLastUndefined = false;
	    }
	    if ((0, _index.isExpression)(node)) {
	      exprs.push(node);
	    } else if ((0, _index.isExpressionStatement)(node)) {
	      exprs.push(node.expression);
	    } else if ((0, _index.isVariableDeclaration)(node)) {
	      if (node.kind !== "var") return;
	      for (const declar of node.declarations) {
	        const bindings = (0, _getBindingIdentifiers.default)(declar);
	        for (const key of Object.keys(bindings)) {
	          declars.push({
	            kind: node.kind,
	            id: (0, _cloneNode.default)(bindings[key])
	          });
	        }
	        if (declar.init) {
	          exprs.push((0, _index2.assignmentExpression)("=", declar.id, declar.init));
	        }
	      }
	      ensureLastUndefined = true;
	    } else if ((0, _index.isIfStatement)(node)) {
	      const consequent = node.consequent ? gatherSequenceExpressions$1([node.consequent], declars) : (0, _productions.buildUndefinedNode)();
	      const alternate = node.alternate ? gatherSequenceExpressions$1([node.alternate], declars) : (0, _productions.buildUndefinedNode)();
	      if (!consequent || !alternate) return;
	      exprs.push((0, _index2.conditionalExpression)(node.test, consequent, alternate));
	    } else if ((0, _index.isBlockStatement)(node)) {
	      const body = gatherSequenceExpressions$1(node.body, declars);
	      if (!body) return;
	      exprs.push(body);
	    } else if ((0, _index.isEmptyStatement)(node)) {
	      if (nodes.indexOf(node) === 0) {
	        ensureLastUndefined = true;
	      }
	    } else {
	      return;
	    }
	  }
	  if (ensureLastUndefined) {
	    exprs.push((0, _productions.buildUndefinedNode)());
	  }
	  if (exprs.length === 1) {
	    return exprs[0];
	  } else {
	    return (0, _index2.sequenceExpression)(exprs);
	  }
	}

	
	return gatherSequenceExpressions;
}

var hasRequiredToSequenceExpression;

function requireToSequenceExpression () {
	if (hasRequiredToSequenceExpression) return toSequenceExpression;
	hasRequiredToSequenceExpression = 1;

	Object.defineProperty(toSequenceExpression, "__esModule", {
	  value: true
	});
	toSequenceExpression.default = toSequenceExpression$1;
	var _gatherSequenceExpressions = requireGatherSequenceExpressions();
	function toSequenceExpression$1(nodes, scope) {
	  if (!(nodes != null && nodes.length)) return;
	  const declars = [];
	  const result = (0, _gatherSequenceExpressions.default)(nodes, declars);
	  if (!result) return;
	  for (const declar of declars) {
	    scope.push(declar);
	  }
	  return result;
	}

	
	return toSequenceExpression;
}

var hasRequiredLib$1;

function requireLib$1 () {
	if (hasRequiredLib$1) return lib$3;
	hasRequiredLib$1 = 1;
	(function (exports) {

		Object.defineProperty(exports, "__esModule", {
		  value: true
		});
		var _exportNames = {
		  react: true,
		  assertNode: true,
		  createTypeAnnotationBasedOnTypeof: true,
		  createUnionTypeAnnotation: true,
		  createFlowUnionType: true,
		  createTSUnionType: true,
		  cloneNode: true,
		  clone: true,
		  cloneDeep: true,
		  cloneDeepWithoutLoc: true,
		  cloneWithoutLoc: true,
		  addComment: true,
		  addComments: true,
		  inheritInnerComments: true,
		  inheritLeadingComments: true,
		  inheritsComments: true,
		  inheritTrailingComments: true,
		  removeComments: true,
		  ensureBlock: true,
		  toBindingIdentifierName: true,
		  toBlock: true,
		  toComputedKey: true,
		  toExpression: true,
		  toIdentifier: true,
		  toKeyAlias: true,
		  toStatement: true,
		  valueToNode: true,
		  appendToMemberExpression: true,
		  inherits: true,
		  prependToMemberExpression: true,
		  removeProperties: true,
		  removePropertiesDeep: true,
		  removeTypeDuplicates: true,
		  getAssignmentIdentifiers: true,
		  getBindingIdentifiers: true,
		  getOuterBindingIdentifiers: true,
		  getFunctionName: true,
		  traverse: true,
		  traverseFast: true,
		  shallowEqual: true,
		  is: true,
		  isBinding: true,
		  isBlockScoped: true,
		  isImmutable: true,
		  isLet: true,
		  isNode: true,
		  isNodesEquivalent: true,
		  isPlaceholderType: true,
		  isReferenced: true,
		  isScope: true,
		  isSpecifierDefault: true,
		  isType: true,
		  isValidES3Identifier: true,
		  isValidIdentifier: true,
		  isVar: true,
		  matchesPattern: true,
		  validate: true,
		  buildMatchMemberExpression: true,
		  __internal__deprecationWarning: true
		};
		Object.defineProperty(exports, "__internal__deprecationWarning", {
		  enumerable: true,
		  get: function () {
		    return _deprecationWarning.default;
		  }
		});
		Object.defineProperty(exports, "addComment", {
		  enumerable: true,
		  get: function () {
		    return _addComment.default;
		  }
		});
		Object.defineProperty(exports, "addComments", {
		  enumerable: true,
		  get: function () {
		    return _addComments.default;
		  }
		});
		Object.defineProperty(exports, "appendToMemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _appendToMemberExpression.default;
		  }
		});
		Object.defineProperty(exports, "assertNode", {
		  enumerable: true,
		  get: function () {
		    return _assertNode.default;
		  }
		});
		Object.defineProperty(exports, "buildMatchMemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _buildMatchMemberExpression.default;
		  }
		});
		Object.defineProperty(exports, "clone", {
		  enumerable: true,
		  get: function () {
		    return _clone.default;
		  }
		});
		Object.defineProperty(exports, "cloneDeep", {
		  enumerable: true,
		  get: function () {
		    return _cloneDeep.default;
		  }
		});
		Object.defineProperty(exports, "cloneDeepWithoutLoc", {
		  enumerable: true,
		  get: function () {
		    return _cloneDeepWithoutLoc.default;
		  }
		});
		Object.defineProperty(exports, "cloneNode", {
		  enumerable: true,
		  get: function () {
		    return _cloneNode.default;
		  }
		});
		Object.defineProperty(exports, "cloneWithoutLoc", {
		  enumerable: true,
		  get: function () {
		    return _cloneWithoutLoc.default;
		  }
		});
		Object.defineProperty(exports, "createFlowUnionType", {
		  enumerable: true,
		  get: function () {
		    return _createFlowUnionType.default;
		  }
		});
		Object.defineProperty(exports, "createTSUnionType", {
		  enumerable: true,
		  get: function () {
		    return _createTSUnionType.default;
		  }
		});
		Object.defineProperty(exports, "createTypeAnnotationBasedOnTypeof", {
		  enumerable: true,
		  get: function () {
		    return _createTypeAnnotationBasedOnTypeof.default;
		  }
		});
		Object.defineProperty(exports, "createUnionTypeAnnotation", {
		  enumerable: true,
		  get: function () {
		    return _createFlowUnionType.default;
		  }
		});
		Object.defineProperty(exports, "ensureBlock", {
		  enumerable: true,
		  get: function () {
		    return _ensureBlock.default;
		  }
		});
		Object.defineProperty(exports, "getAssignmentIdentifiers", {
		  enumerable: true,
		  get: function () {
		    return _getAssignmentIdentifiers.default;
		  }
		});
		Object.defineProperty(exports, "getBindingIdentifiers", {
		  enumerable: true,
		  get: function () {
		    return _getBindingIdentifiers.default;
		  }
		});
		Object.defineProperty(exports, "getFunctionName", {
		  enumerable: true,
		  get: function () {
		    return _getFunctionName.default;
		  }
		});
		Object.defineProperty(exports, "getOuterBindingIdentifiers", {
		  enumerable: true,
		  get: function () {
		    return _getOuterBindingIdentifiers.default;
		  }
		});
		Object.defineProperty(exports, "inheritInnerComments", {
		  enumerable: true,
		  get: function () {
		    return _inheritInnerComments.default;
		  }
		});
		Object.defineProperty(exports, "inheritLeadingComments", {
		  enumerable: true,
		  get: function () {
		    return _inheritLeadingComments.default;
		  }
		});
		Object.defineProperty(exports, "inheritTrailingComments", {
		  enumerable: true,
		  get: function () {
		    return _inheritTrailingComments.default;
		  }
		});
		Object.defineProperty(exports, "inherits", {
		  enumerable: true,
		  get: function () {
		    return _inherits.default;
		  }
		});
		Object.defineProperty(exports, "inheritsComments", {
		  enumerable: true,
		  get: function () {
		    return _inheritsComments.default;
		  }
		});
		Object.defineProperty(exports, "is", {
		  enumerable: true,
		  get: function () {
		    return _is.default;
		  }
		});
		Object.defineProperty(exports, "isBinding", {
		  enumerable: true,
		  get: function () {
		    return _isBinding.default;
		  }
		});
		Object.defineProperty(exports, "isBlockScoped", {
		  enumerable: true,
		  get: function () {
		    return _isBlockScoped.default;
		  }
		});
		Object.defineProperty(exports, "isImmutable", {
		  enumerable: true,
		  get: function () {
		    return _isImmutable.default;
		  }
		});
		Object.defineProperty(exports, "isLet", {
		  enumerable: true,
		  get: function () {
		    return _isLet.default;
		  }
		});
		Object.defineProperty(exports, "isNode", {
		  enumerable: true,
		  get: function () {
		    return _isNode.default;
		  }
		});
		Object.defineProperty(exports, "isNodesEquivalent", {
		  enumerable: true,
		  get: function () {
		    return _isNodesEquivalent.default;
		  }
		});
		Object.defineProperty(exports, "isPlaceholderType", {
		  enumerable: true,
		  get: function () {
		    return _isPlaceholderType.default;
		  }
		});
		Object.defineProperty(exports, "isReferenced", {
		  enumerable: true,
		  get: function () {
		    return _isReferenced.default;
		  }
		});
		Object.defineProperty(exports, "isScope", {
		  enumerable: true,
		  get: function () {
		    return _isScope.default;
		  }
		});
		Object.defineProperty(exports, "isSpecifierDefault", {
		  enumerable: true,
		  get: function () {
		    return _isSpecifierDefault.default;
		  }
		});
		Object.defineProperty(exports, "isType", {
		  enumerable: true,
		  get: function () {
		    return _isType.default;
		  }
		});
		Object.defineProperty(exports, "isValidES3Identifier", {
		  enumerable: true,
		  get: function () {
		    return _isValidES3Identifier.default;
		  }
		});
		Object.defineProperty(exports, "isValidIdentifier", {
		  enumerable: true,
		  get: function () {
		    return _isValidIdentifier.default;
		  }
		});
		Object.defineProperty(exports, "isVar", {
		  enumerable: true,
		  get: function () {
		    return _isVar.default;
		  }
		});
		Object.defineProperty(exports, "matchesPattern", {
		  enumerable: true,
		  get: function () {
		    return _matchesPattern.default;
		  }
		});
		Object.defineProperty(exports, "prependToMemberExpression", {
		  enumerable: true,
		  get: function () {
		    return _prependToMemberExpression.default;
		  }
		});
		exports.react = void 0;
		Object.defineProperty(exports, "removeComments", {
		  enumerable: true,
		  get: function () {
		    return _removeComments.default;
		  }
		});
		Object.defineProperty(exports, "removeProperties", {
		  enumerable: true,
		  get: function () {
		    return _removeProperties.default;
		  }
		});
		Object.defineProperty(exports, "removePropertiesDeep", {
		  enumerable: true,
		  get: function () {
		    return _removePropertiesDeep.default;
		  }
		});
		Object.defineProperty(exports, "removeTypeDuplicates", {
		  enumerable: true,
		  get: function () {
		    return _removeTypeDuplicates.default;
		  }
		});
		Object.defineProperty(exports, "shallowEqual", {
		  enumerable: true,
		  get: function () {
		    return _shallowEqual.default;
		  }
		});
		Object.defineProperty(exports, "toBindingIdentifierName", {
		  enumerable: true,
		  get: function () {
		    return _toBindingIdentifierName.default;
		  }
		});
		Object.defineProperty(exports, "toBlock", {
		  enumerable: true,
		  get: function () {
		    return _toBlock.default;
		  }
		});
		Object.defineProperty(exports, "toComputedKey", {
		  enumerable: true,
		  get: function () {
		    return _toComputedKey.default;
		  }
		});
		Object.defineProperty(exports, "toExpression", {
		  enumerable: true,
		  get: function () {
		    return _toExpression.default;
		  }
		});
		Object.defineProperty(exports, "toIdentifier", {
		  enumerable: true,
		  get: function () {
		    return _toIdentifier.default;
		  }
		});
		Object.defineProperty(exports, "toKeyAlias", {
		  enumerable: true,
		  get: function () {
		    return _toKeyAlias.default;
		  }
		});
		Object.defineProperty(exports, "toStatement", {
		  enumerable: true,
		  get: function () {
		    return _toStatement.default;
		  }
		});
		Object.defineProperty(exports, "traverse", {
		  enumerable: true,
		  get: function () {
		    return _traverse.default;
		  }
		});
		Object.defineProperty(exports, "traverseFast", {
		  enumerable: true,
		  get: function () {
		    return _traverseFast.default;
		  }
		});
		Object.defineProperty(exports, "validate", {
		  enumerable: true,
		  get: function () {
		    return _validate.default;
		  }
		});
		Object.defineProperty(exports, "valueToNode", {
		  enumerable: true,
		  get: function () {
		    return _valueToNode.default;
		  }
		});
		var _isReactComponent = requireIsReactComponent();
		var _isCompatTag = requireIsCompatTag();
		var _buildChildren = requireBuildChildren();
		var _assertNode = requireAssertNode();
		var _index = requireGenerated$1();
		Object.keys(_index).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index[key];
		    }
		  });
		});
		var _createTypeAnnotationBasedOnTypeof = requireCreateTypeAnnotationBasedOnTypeof();
		var _createFlowUnionType = requireCreateFlowUnionType();
		var _createTSUnionType = requireCreateTSUnionType();
		var _index2 = requireGenerated$2();
		Object.keys(_index2).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index2[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index2[key];
		    }
		  });
		});
		var _uppercase = requireUppercase();
		Object.keys(_uppercase).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _uppercase[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _uppercase[key];
		    }
		  });
		});
		var _productions = requireProductions();
		Object.keys(_productions).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _productions[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _productions[key];
		    }
		  });
		});
		var _cloneNode = requireCloneNode();
		var _clone = requireClone();
		var _cloneDeep = requireCloneDeep();
		var _cloneDeepWithoutLoc = requireCloneDeepWithoutLoc();
		var _cloneWithoutLoc = requireCloneWithoutLoc();
		var _addComment = requireAddComment();
		var _addComments = requireAddComments();
		var _inheritInnerComments = requireInheritInnerComments();
		var _inheritLeadingComments = requireInheritLeadingComments();
		var _inheritsComments = requireInheritsComments();
		var _inheritTrailingComments = requireInheritTrailingComments();
		var _removeComments = requireRemoveComments();
		var _index3 = requireGenerated();
		Object.keys(_index3).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index3[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index3[key];
		    }
		  });
		});
		var _index4 = requireConstants();
		Object.keys(_index4).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index4[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index4[key];
		    }
		  });
		});
		var _ensureBlock = requireEnsureBlock();
		var _toBindingIdentifierName = requireToBindingIdentifierName();
		var _toBlock = requireToBlock();
		var _toComputedKey = requireToComputedKey();
		var _toExpression = requireToExpression();
		var _toIdentifier = requireToIdentifier();
		var _toKeyAlias = requireToKeyAlias();
		var _toStatement = requireToStatement();
		var _valueToNode = requireValueToNode();
		var _index5 = requireDefinitions();
		Object.keys(_index5).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index5[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index5[key];
		    }
		  });
		});
		var _appendToMemberExpression = requireAppendToMemberExpression();
		var _inherits = requireInherits();
		var _prependToMemberExpression = requirePrependToMemberExpression();
		var _removeProperties = requireRemoveProperties();
		var _removePropertiesDeep = requireRemovePropertiesDeep();
		var _removeTypeDuplicates = requireRemoveTypeDuplicates$1();
		var _getAssignmentIdentifiers = requireGetAssignmentIdentifiers();
		var _getBindingIdentifiers = requireGetBindingIdentifiers();
		var _getOuterBindingIdentifiers = requireGetOuterBindingIdentifiers();
		var _getFunctionName = requireGetFunctionName();
		var _traverse = requireTraverse();
		Object.keys(_traverse).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _traverse[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _traverse[key];
		    }
		  });
		});
		var _traverseFast = requireTraverseFast();
		var _shallowEqual = requireShallowEqual();
		var _is = requireIs();
		var _isBinding = requireIsBinding();
		var _isBlockScoped = requireIsBlockScoped();
		var _isImmutable = requireIsImmutable();
		var _isLet = requireIsLet();
		var _isNode = requireIsNode();
		var _isNodesEquivalent = requireIsNodesEquivalent();
		var _isPlaceholderType = requireIsPlaceholderType();
		var _isReferenced = requireIsReferenced();
		var _isScope = requireIsScope();
		var _isSpecifierDefault = requireIsSpecifierDefault();
		var _isType = requireIsType();
		var _isValidES3Identifier = requireIsValidES3Identifier();
		var _isValidIdentifier = requireIsValidIdentifier();
		var _isVar = requireIsVar();
		var _matchesPattern = requireMatchesPattern();
		var _validate = requireValidate();
		var _buildMatchMemberExpression = requireBuildMatchMemberExpression();
		var _index6 = requireGenerated$3();
		Object.keys(_index6).forEach(function (key) {
		  if (key === "default" || key === "__esModule") return;
		  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
		  if (key in exports && exports[key] === _index6[key]) return;
		  Object.defineProperty(exports, key, {
		    enumerable: true,
		    get: function () {
		      return _index6[key];
		    }
		  });
		});
		var _deprecationWarning = requireDeprecationWarning();
		exports.react = {
		  isReactComponent: _isReactComponent.default,
		  isCompatTag: _isCompatTag.default,
		  buildChildren: _buildChildren.default
		};
		{
		  exports.toSequenceExpression = requireToSequenceExpression().default;
		}
		if (process.env.BABEL_TYPES_8_BREAKING) {
		  console.warn("BABEL_TYPES_8_BREAKING is not supported anymore. Use the latest Babel 8.0.0 pre-release instead!");
		}

		
} (lib$3));
	return lib$3;
}

var libExports$1 = requireLib$1();

var lib = {};

var picocolorsExports = {};
var picocolors = {
  get exports(){ return picocolorsExports; },
  set exports(v){ picocolorsExports = v; },
};

var hasRequiredPicocolors;

function requirePicocolors () {
	if (hasRequiredPicocolors) return picocolorsExports;
	hasRequiredPicocolors = 1;
	let p = process || {}, argv = p.argv || [], env = p.env || {};
	let isColorSupported =
		!(!!env.NO_COLOR || argv.includes("--no-color")) &&
		(!!env.FORCE_COLOR || argv.includes("--color") || p.platform === "win32" || ((p.stdout || {}).isTTY && env.TERM !== "dumb") || !!env.CI);

	let formatter = (open, close, replace = open) =>
		input => {
			let string = "" + input, index = string.indexOf(close, open.length);
			return ~index ? open + replaceClose(string, close, replace, index) + close : open + string + close
		};

	let replaceClose = (string, close, replace, index) => {
		let result = "", cursor = 0;
		do {
			result += string.substring(cursor, index) + replace;
			cursor = index + close.length;
			index = string.indexOf(close, cursor);
		} while (~index)
		return result + string.substring(cursor)
	};

	let createColors = (enabled = isColorSupported) => {
		let f = enabled ? formatter : () => String;
		return {
			isColorSupported: enabled,
			reset: f("\x1b[0m", "\x1b[0m"),
			bold: f("\x1b[1m", "\x1b[22m", "\x1b[22m\x1b[1m"),
			dim: f("\x1b[2m", "\x1b[22m", "\x1b[22m\x1b[2m"),
			italic: f("\x1b[3m", "\x1b[23m"),
			underline: f("\x1b[4m", "\x1b[24m"),
			inverse: f("\x1b[7m", "\x1b[27m"),
			hidden: f("\x1b[8m", "\x1b[28m"),
			strikethrough: f("\x1b[9m", "\x1b[29m"),

			black: f("\x1b[30m", "\x1b[39m"),
			red: f("\x1b[31m", "\x1b[39m"),
			green: f("\x1b[32m", "\x1b[39m"),
			yellow: f("\x1b[33m", "\x1b[39m"),
			blue: f("\x1b[34m", "\x1b[39m"),
			magenta: f("\x1b[35m", "\x1b[39m"),
			cyan: f("\x1b[36m", "\x1b[39m"),
			white: f("\x1b[37m", "\x1b[39m"),
			gray: f("\x1b[90m", "\x1b[39m"),

			bgBlack: f("\x1b[40m", "\x1b[49m"),
			bgRed: f("\x1b[41m", "\x1b[49m"),
			bgGreen: f("\x1b[42m", "\x1b[49m"),
			bgYellow: f("\x1b[43m", "\x1b[49m"),
			bgBlue: f("\x1b[44m", "\x1b[49m"),
			bgMagenta: f("\x1b[45m", "\x1b[49m"),
			bgCyan: f("\x1b[46m", "\x1b[49m"),
			bgWhite: f("\x1b[47m", "\x1b[49m"),

			blackBright: f("\x1b[90m", "\x1b[39m"),
			redBright: f("\x1b[91m", "\x1b[39m"),
			greenBright: f("\x1b[92m", "\x1b[39m"),
			yellowBright: f("\x1b[93m", "\x1b[39m"),
			blueBright: f("\x1b[94m", "\x1b[39m"),
			magentaBright: f("\x1b[95m", "\x1b[39m"),
			cyanBright: f("\x1b[96m", "\x1b[39m"),
			whiteBright: f("\x1b[97m", "\x1b[39m"),

			bgBlackBright: f("\x1b[100m", "\x1b[49m"),
			bgRedBright: f("\x1b[101m", "\x1b[49m"),
			bgGreenBright: f("\x1b[102m", "\x1b[49m"),
			bgYellowBright: f("\x1b[103m", "\x1b[49m"),
			bgBlueBright: f("\x1b[104m", "\x1b[49m"),
			bgMagentaBright: f("\x1b[105m", "\x1b[49m"),
			bgCyanBright: f("\x1b[106m", "\x1b[49m"),
			bgWhiteBright: f("\x1b[107m", "\x1b[49m"),
		}
	};

	picocolors.exports = createColors();
	picocolorsExports.createColors = createColors;
	return picocolorsExports;
}

var jsTokens = {};

var hasRequiredJsTokens;

function requireJsTokens () {
	if (hasRequiredJsTokens) return jsTokens;
	hasRequiredJsTokens = 1;
	// Copyright 2014, 2015, 2016, 2017, 2018 Simon Lydell
	// License: MIT. (See LICENSE.)

	Object.defineProperty(jsTokens, "__esModule", {
	  value: true
	});

	// This regex comes from regex.coffee, and is inserted here by generate-index.js
	// (run `npm run build`).
	jsTokens.default = /((['"])(?:(?!\2|\\).|\\(?:\r\n|[\s\S]))*(\2)?|`(?:[^`\\$]|\\[\s\S]|\$(?!\{)|\$\{(?:[^{}]|\{[^}]*\}?)*\}?)*(`)?)|(\/\/.*)|(\/\*(?:[^*]|\*(?!\/))*(\*\/)?)|(\/(?!\*)(?:\[(?:(?![\]\\]).|\\.)*\]|(?![\/\]\\]).|\\.)+\/(?:(?!\s*(?:\b|[\u0080-\uFFFF$\\'"~({]|[+\-!](?!=)|\.?\d))|[gmiyus]{1,6}\b(?![\u0080-\uFFFF$\\]|\s*(?:[+\-*%&|^<>!=?({]|\/(?![\/*])))))|(0[xX][\da-fA-F]+|0[oO][0-7]+|0[bB][01]+|(?:\d*\.\d+|\d+\.?)(?:[eE][+-]?\d+)?)|((?!\d)(?:(?!\s)[$\w\u0080-\uFFFF]|\\u[\da-fA-F]{4}|\\u\{[\da-fA-F]+\})+)|(--|\+\+|&&|\|\||=>|\.{3}|(?:[+\-\/%&|^]|\*{1,2}|<{1,2}|>{1,3}|!=?|={1,2})=?|[?~.,:;[\](){}])|(\s+)|(^$|[\s\S])/g;

	jsTokens.matchToToken = function(match) {
	  var token = {type: "invalid", value: match[0], closed: undefined};
	       if (match[ 1]) token.type = "string" , token.closed = !!(match[3] || match[4]);
	  else if (match[ 5]) token.type = "comment";
	  else if (match[ 6]) token.type = "comment", token.closed = !!match[7];
	  else if (match[ 8]) token.type = "regex";
	  else if (match[ 9]) token.type = "number";
	  else if (match[10]) token.type = "name";
	  else if (match[11]) token.type = "punctuator";
	  else if (match[12]) token.type = "whitespace";
	  return token
	};
	return jsTokens;
}

var hasRequiredLib;

function requireLib () {
	if (hasRequiredLib) return lib;
	hasRequiredLib = 1;

	Object.defineProperty(lib, '__esModule', { value: true });

	var picocolors = requirePicocolors();
	var jsTokens = requireJsTokens();
	var helperValidatorIdentifier = requireLib$3();

	function isColorSupported() {
	  return (typeof process === "object" && (process.env.FORCE_COLOR === "0" || process.env.FORCE_COLOR === "false") ? false : picocolors.isColorSupported
	  );
	}
	const compose = (f, g) => v => f(g(v));
	function buildDefs(colors) {
	  return {
	    keyword: colors.cyan,
	    capitalized: colors.yellow,
	    jsxIdentifier: colors.yellow,
	    punctuator: colors.yellow,
	    number: colors.magenta,
	    string: colors.green,
	    regex: colors.magenta,
	    comment: colors.gray,
	    invalid: compose(compose(colors.white, colors.bgRed), colors.bold),
	    gutter: colors.gray,
	    marker: compose(colors.red, colors.bold),
	    message: compose(colors.red, colors.bold),
	    reset: colors.reset
	  };
	}
	const defsOn = buildDefs(picocolors.createColors(true));
	const defsOff = buildDefs(picocolors.createColors(false));
	function getDefs(enabled) {
	  return enabled ? defsOn : defsOff;
	}

	const sometimesKeywords = new Set(["as", "async", "from", "get", "of", "set"]);
	const NEWLINE$1 = /\r\n|[\n\r\u2028\u2029]/;
	const BRACKET = /^[()[\]{}]$/;
	let tokenize;
	{
	  const JSX_TAG = /^[a-z][\w-]*$/i;
	  const getTokenType = function (token, offset, text) {
	    if (token.type === "name") {
	      if (helperValidatorIdentifier.isKeyword(token.value) || helperValidatorIdentifier.isStrictReservedWord(token.value, true) || sometimesKeywords.has(token.value)) {
	        return "keyword";
	      }
	      if (JSX_TAG.test(token.value) && (text[offset - 1] === "<" || text.slice(offset - 2, offset) === "</")) {
	        return "jsxIdentifier";
	      }
	      if (token.value[0] !== token.value[0].toLowerCase()) {
	        return "capitalized";
	      }
	    }
	    if (token.type === "punctuator" && BRACKET.test(token.value)) {
	      return "bracket";
	    }
	    if (token.type === "invalid" && (token.value === "@" || token.value === "#")) {
	      return "punctuator";
	    }
	    return token.type;
	  };
	  tokenize = function* (text) {
	    let match;
	    while (match = jsTokens.default.exec(text)) {
	      const token = jsTokens.matchToToken(match);
	      yield {
	        type: getTokenType(token, match.index, text),
	        value: token.value
	      };
	    }
	  };
	}
	function highlight(text) {
	  if (text === "") return "";
	  const defs = getDefs(true);
	  let highlighted = "";
	  for (const {
	    type,
	    value
	  } of tokenize(text)) {
	    if (type in defs) {
	      highlighted += value.split(NEWLINE$1).map(str => defs[type](str)).join("\n");
	    } else {
	      highlighted += value;
	    }
	  }
	  return highlighted;
	}

	let deprecationWarningShown = false;
	const NEWLINE = /\r\n|[\n\r\u2028\u2029]/;
	function getMarkerLines(loc, source, opts) {
	  const startLoc = Object.assign({
	    column: 0,
	    line: -1
	  }, loc.start);
	  const endLoc = Object.assign({}, startLoc, loc.end);
	  const {
	    linesAbove = 2,
	    linesBelow = 3
	  } = opts || {};
	  const startLine = startLoc.line;
	  const startColumn = startLoc.column;
	  const endLine = endLoc.line;
	  const endColumn = endLoc.column;
	  let start = Math.max(startLine - (linesAbove + 1), 0);
	  let end = Math.min(source.length, endLine + linesBelow);
	  if (startLine === -1) {
	    start = 0;
	  }
	  if (endLine === -1) {
	    end = source.length;
	  }
	  const lineDiff = endLine - startLine;
	  const markerLines = {};
	  if (lineDiff) {
	    for (let i = 0; i <= lineDiff; i++) {
	      const lineNumber = i + startLine;
	      if (!startColumn) {
	        markerLines[lineNumber] = true;
	      } else if (i === 0) {
	        const sourceLength = source[lineNumber - 1].length;
	        markerLines[lineNumber] = [startColumn, sourceLength - startColumn + 1];
	      } else if (i === lineDiff) {
	        markerLines[lineNumber] = [0, endColumn];
	      } else {
	        const sourceLength = source[lineNumber - i].length;
	        markerLines[lineNumber] = [0, sourceLength];
	      }
	    }
	  } else {
	    if (startColumn === endColumn) {
	      if (startColumn) {
	        markerLines[startLine] = [startColumn, 0];
	      } else {
	        markerLines[startLine] = true;
	      }
	    } else {
	      markerLines[startLine] = [startColumn, endColumn - startColumn];
	    }
	  }
	  return {
	    start,
	    end,
	    markerLines
	  };
	}
	function codeFrameColumns(rawLines, loc, opts = {}) {
	  const shouldHighlight = opts.forceColor || isColorSupported() && opts.highlightCode;
	  const defs = getDefs(shouldHighlight);
	  const lines = rawLines.split(NEWLINE);
	  const {
	    start,
	    end,
	    markerLines
	  } = getMarkerLines(loc, lines, opts);
	  const hasColumns = loc.start && typeof loc.start.column === "number";
	  const numberMaxWidth = String(end).length;
	  const highlightedLines = shouldHighlight ? highlight(rawLines) : rawLines;
	  let frame = highlightedLines.split(NEWLINE, end).slice(start, end).map((line, index) => {
	    const number = start + 1 + index;
	    const paddedNumber = ` ${number}`.slice(-numberMaxWidth);
	    const gutter = ` ${paddedNumber} |`;
	    const hasMarker = markerLines[number];
	    const lastMarkerLine = !markerLines[number + 1];
	    if (hasMarker) {
	      let markerLine = "";
	      if (Array.isArray(hasMarker)) {
	        const markerSpacing = line.slice(0, Math.max(hasMarker[0] - 1, 0)).replace(/[^\t]/g, " ");
	        const numberOfMarkers = hasMarker[1] || 1;
	        markerLine = ["\n ", defs.gutter(gutter.replace(/\d/g, " ")), " ", markerSpacing, defs.marker("^").repeat(numberOfMarkers)].join("");
	        if (lastMarkerLine && opts.message) {
	          markerLine += " " + defs.message(opts.message);
	        }
	      }
	      return [defs.marker(">"), defs.gutter(gutter), line.length > 0 ? ` ${line}` : "", markerLine].join("");
	    } else {
	      return ` ${defs.gutter(gutter)}${line.length > 0 ? ` ${line}` : ""}`;
	    }
	  }).join("\n");
	  if (opts.message && !hasColumns) {
	    frame = `${" ".repeat(numberMaxWidth + 1)}${opts.message}\n${frame}`;
	  }
	  if (shouldHighlight) {
	    return defs.reset(frame);
	  } else {
	    return frame;
	  }
	}
	function index (rawLines, lineNumber, colNumber, opts = {}) {
	  if (!deprecationWarningShown) {
	    deprecationWarningShown = true;
	    const message = "Passing lineNumber and colNumber is deprecated to @babel/code-frame. Please use `codeFrameColumns`.";
	    if (process.emitWarning) {
	      process.emitWarning(message, "DeprecationWarning");
	    } else {
	      const deprecationError = new Error(message);
	      deprecationError.name = "DeprecationWarning";
	      console.warn(new Error(message));
	    }
	  }
	  colNumber = Math.max(colNumber, 0);
	  const location = {
	    start: {
	      column: colNumber,
	      line: lineNumber
	    }
	  };
	  return codeFrameColumns(rawLines, location, opts);
	}

	lib.codeFrameColumns = codeFrameColumns;
	lib.default = index;
	lib.highlight = highlight;
	
	return lib;
}

var libExports = requireLib();

/*! *****************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */
/* global Reflect, Promise */


function __classPrivateFieldGet(receiver, state, kind, f) {
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
    return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
}

function __classPrivateFieldSet(receiver, state, value, kind, f) {
    if (kind === "m") throw new TypeError("Private method is not writable");
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a setter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
    return (kind === "a" ? f.call(receiver, value) : f ? f.value = value : state.set(receiver, value)), value;
}

var _OkImpl_val, _ErrImpl_val;
function Ok(val) {
    return new OkImpl(val);
}
class OkImpl {
    constructor(val) {
        _OkImpl_val.set(this, void 0);
        __classPrivateFieldSet(this, _OkImpl_val, val, "f");
    }
    map(fn) {
        return new OkImpl(fn(__classPrivateFieldGet(this, _OkImpl_val, "f")));
    }
    mapErr(_fn) {
        return this;
    }
    mapOr(_fallback, fn) {
        return fn(__classPrivateFieldGet(this, _OkImpl_val, "f"));
    }
    mapOrElse(_fallback, fn) {
        return fn(__classPrivateFieldGet(this, _OkImpl_val, "f"));
    }
    andThen(fn) {
        return fn(__classPrivateFieldGet(this, _OkImpl_val, "f"));
    }
    and(res) {
        return res;
    }
    or(_res) {
        return this;
    }
    orElse(_fn) {
        return this;
    }
    isOk() {
        return true;
    }
    isErr() {
        return false;
    }
    expect(_msg) {
        return __classPrivateFieldGet(this, _OkImpl_val, "f");
    }
    expectErr(msg) {
        throw new Error(`${msg}: ${__classPrivateFieldGet(this, _OkImpl_val, "f")}`);
    }
    unwrap() {
        return __classPrivateFieldGet(this, _OkImpl_val, "f");
    }
    unwrapOr(_fallback) {
        return __classPrivateFieldGet(this, _OkImpl_val, "f");
    }
    unwrapOrElse(_fallback) {
        return __classPrivateFieldGet(this, _OkImpl_val, "f");
    }
    unwrapErr() {
        if (__classPrivateFieldGet(this, _OkImpl_val, "f") instanceof Error) {
            throw __classPrivateFieldGet(this, _OkImpl_val, "f");
        }
        throw new Error(`Can't unwrap \`Ok\` to \`Err\`: ${__classPrivateFieldGet(this, _OkImpl_val, "f")}`);
    }
}
_OkImpl_val = new WeakMap();
function Err(val) {
    return new ErrImpl(val);
}
class ErrImpl {
    constructor(val) {
        _ErrImpl_val.set(this, void 0);
        __classPrivateFieldSet(this, _ErrImpl_val, val, "f");
    }
    map(_fn) {
        return this;
    }
    mapErr(fn) {
        return new ErrImpl(fn(__classPrivateFieldGet(this, _ErrImpl_val, "f")));
    }
    mapOr(fallback, _fn) {
        return fallback;
    }
    mapOrElse(fallback, _fn) {
        return fallback();
    }
    andThen(_fn) {
        return this;
    }
    and(_res) {
        return this;
    }
    or(res) {
        return res;
    }
    orElse(fn) {
        return fn(__classPrivateFieldGet(this, _ErrImpl_val, "f"));
    }
    isOk() {
        return false;
    }
    isErr() {
        return true;
    }
    expect(msg) {
        throw new Error(`${msg}: ${__classPrivateFieldGet(this, _ErrImpl_val, "f")}`);
    }
    expectErr(_msg) {
        return __classPrivateFieldGet(this, _ErrImpl_val, "f");
    }
    unwrap() {
        if (__classPrivateFieldGet(this, _ErrImpl_val, "f") instanceof Error) {
            throw __classPrivateFieldGet(this, _ErrImpl_val, "f");
        }
        throw new Error(`Can't unwrap \`Err\` to \`Ok\`: ${__classPrivateFieldGet(this, _ErrImpl_val, "f")}`);
    }
    unwrapOr(fallback) {
        return fallback;
    }
    unwrapOrElse(fallback) {
        return fallback(__classPrivateFieldGet(this, _ErrImpl_val, "f"));
    }
    unwrapErr() {
        return __classPrivateFieldGet(this, _ErrImpl_val, "f");
    }
}
_ErrImpl_val = new WeakMap();

function assertExhaustive$1(_, errorMsg) {
    throw new Error(errorMsg);
}
function retainWhere(array, predicate) {
    let writeIndex = 0;
    for (let readIndex = 0; readIndex < array.length; readIndex++) {
        const item = array[readIndex];
        if (predicate(item, readIndex) === true) {
            array[writeIndex++] = item;
        }
    }
    array.length = writeIndex;
}
function retainWhere_Set(items, predicate) {
    for (const item of items) {
        if (!predicate(item)) {
            items.delete(item);
        }
    }
}
function getOrInsertWith(m, key, makeDefault) {
    if (m.has(key)) {
        return m.get(key);
    }
    else {
        const defaultValue = makeDefault();
        m.set(key, defaultValue);
        return defaultValue;
    }
}
function getOrInsertDefault(m, key, defaultValue) {
    if (m.has(key)) {
        return m.get(key);
    }
    else {
        m.set(key, defaultValue);
        return defaultValue;
    }
}
function Set_equal(a, b) {
    if (a.size !== b.size) {
        return false;
    }
    for (const item of a) {
        if (!b.has(item)) {
            return false;
        }
    }
    return true;
}
function Set_union(a, b) {
    const union = new Set(a);
    for (const item of b) {
        union.add(item);
    }
    return union;
}
function Set_intersect(sets) {
    if (sets.length === 0 || sets.some(s => s.size === 0)) {
        return new Set();
    }
    else if (sets.length === 1) {
        return new Set(sets[0]);
    }
    const result = new Set();
    const first = sets[0];
    outer: for (const e of first) {
        for (let i = 1; i < sets.length; i++) {
            if (!sets[i].has(e)) {
                continue outer;
            }
        }
        result.add(e);
    }
    return result;
}
function Set_isSuperset(a, b) {
    for (const v of b) {
        if (!a.has(v)) {
            return false;
        }
    }
    return true;
}
function Iterable_some(iter, pred) {
    for (const item of iter) {
        if (pred(item)) {
            return true;
        }
    }
    return false;
}
function Set_filter(source, fn) {
    const result = new Set();
    for (const entry of source) {
        if (fn(entry)) {
            result.add(entry);
        }
    }
    return result;
}
function hasNode(input) {
    return input.node != null;
}
function hasOwnProperty$1(obj, key) {
    return Object.prototype.hasOwnProperty.call(obj, key);
}

const CODEFRAME_LINES_ABOVE = 2;
const CODEFRAME_LINES_BELOW = 3;
const CODEFRAME_MAX_LINES = 10;
const CODEFRAME_ABBREVIATED_SOURCE_LINES = 5;
var ErrorSeverity;
(function (ErrorSeverity) {
    ErrorSeverity["Error"] = "Error";
    ErrorSeverity["Warning"] = "Warning";
    ErrorSeverity["Hint"] = "Hint";
    ErrorSeverity["Off"] = "Off";
})(ErrorSeverity || (ErrorSeverity = {}));
var CompilerSuggestionOperation;
(function (CompilerSuggestionOperation) {
    CompilerSuggestionOperation[CompilerSuggestionOperation["InsertBefore"] = 0] = "InsertBefore";
    CompilerSuggestionOperation[CompilerSuggestionOperation["InsertAfter"] = 1] = "InsertAfter";
    CompilerSuggestionOperation[CompilerSuggestionOperation["Remove"] = 2] = "Remove";
    CompilerSuggestionOperation[CompilerSuggestionOperation["Replace"] = 3] = "Replace";
})(CompilerSuggestionOperation || (CompilerSuggestionOperation = {}));
class CompilerDiagnostic {
    constructor(options) {
        this.options = options;
    }
    static create(options) {
        return new CompilerDiagnostic(Object.assign(Object.assign({}, options), { details: [] }));
    }
    get reason() {
        return this.options.reason;
    }
    get description() {
        return this.options.description;
    }
    get severity() {
        return getRuleForCategory(this.category).severity;
    }
    get suggestions() {
        return this.options.suggestions;
    }
    get category() {
        return this.options.category;
    }
    withDetails(...details) {
        this.options.details.push(...details);
        return this;
    }
    primaryLocation() {
        const firstErrorDetail = this.options.details.filter(d => d.kind === 'error')[0];
        return firstErrorDetail != null && firstErrorDetail.kind === 'error'
            ? firstErrorDetail.loc
            : null;
    }
    printErrorMessage(source, options) {
        var _a, _b;
        const buffer = [printErrorSummary(this.category, this.reason)];
        if (this.description != null) {
            buffer.push('\n\n', `${this.description}.`);
        }
        for (const detail of this.options.details) {
            switch (detail.kind) {
                case 'error': {
                    const loc = detail.loc;
                    if (loc == null || typeof loc === 'symbol') {
                        continue;
                    }
                    let codeFrame;
                    try {
                        codeFrame = printCodeFrame(source, loc, (_a = detail.message) !== null && _a !== void 0 ? _a : '');
                    }
                    catch (e) {
                        codeFrame = (_b = detail.message) !== null && _b !== void 0 ? _b : '';
                    }
                    buffer.push('\n\n');
                    if (loc.filename != null) {
                        const line = loc.start.line;
                        const column = options.eslint
                            ? loc.start.column + 1
                            : loc.start.column;
                        buffer.push(`${loc.filename}:${line}:${column}\n`);
                    }
                    buffer.push(codeFrame);
                    break;
                }
                case 'hint': {
                    buffer.push('\n\n');
                    buffer.push(detail.message);
                    break;
                }
                default: {
                    assertExhaustive$1(detail, `Unexpected detail kind ${detail.kind}`);
                }
            }
        }
        return buffer.join('');
    }
    toString() {
        const buffer = [printErrorSummary(this.category, this.reason)];
        if (this.description != null) {
            buffer.push(`. ${this.description}.`);
        }
        const loc = this.primaryLocation();
        if (loc != null && typeof loc !== 'symbol') {
            buffer.push(` (${loc.start.line}:${loc.start.column})`);
        }
        return buffer.join('');
    }
}
class CompilerErrorDetail {
    constructor(options) {
        this.options = options;
    }
    get reason() {
        return this.options.reason;
    }
    get description() {
        return this.options.description;
    }
    get severity() {
        return getRuleForCategory(this.category).severity;
    }
    get loc() {
        return this.options.loc;
    }
    get suggestions() {
        return this.options.suggestions;
    }
    get category() {
        return this.options.category;
    }
    primaryLocation() {
        return this.loc;
    }
    printErrorMessage(source, options) {
        const buffer = [printErrorSummary(this.category, this.reason)];
        if (this.description != null) {
            buffer.push(`\n\n${this.description}.`);
        }
        const loc = this.loc;
        if (loc != null && typeof loc !== 'symbol') {
            let codeFrame;
            try {
                codeFrame = printCodeFrame(source, loc, this.reason);
            }
            catch (e) {
                codeFrame = '';
            }
            buffer.push(`\n\n`);
            if (loc.filename != null) {
                const line = loc.start.line;
                const column = options.eslint ? loc.start.column + 1 : loc.start.column;
                buffer.push(`${loc.filename}:${line}:${column}\n`);
            }
            buffer.push(codeFrame);
            buffer.push('\n\n');
        }
        return buffer.join('');
    }
    toString() {
        const buffer = [printErrorSummary(this.category, this.reason)];
        if (this.description != null) {
            buffer.push(`. ${this.description}.`);
        }
        const loc = this.loc;
        if (loc != null && typeof loc !== 'symbol') {
            buffer.push(` (${loc.start.line}:${loc.start.column})`);
        }
        return buffer.join('');
    }
}
class CompilerError extends Error {
    static invariant(condition, options) {
        if (!condition) {
            const errors = new CompilerError();
            errors.pushDiagnostic(CompilerDiagnostic.create({
                reason: options.reason,
                description: options.description,
                category: ErrorCategory.Invariant,
            }).withDetails(...options.details));
            throw errors;
        }
    }
    static throwDiagnostic(options) {
        const errors = new CompilerError();
        errors.pushDiagnostic(new CompilerDiagnostic(options));
        throw errors;
    }
    static throwTodo(options) {
        const errors = new CompilerError();
        errors.pushErrorDetail(new CompilerErrorDetail(Object.assign(Object.assign({}, options), { category: ErrorCategory.Todo })));
        throw errors;
    }
    static throwInvalidJS(options) {
        const errors = new CompilerError();
        errors.pushErrorDetail(new CompilerErrorDetail(Object.assign(Object.assign({}, options), { category: ErrorCategory.Syntax })));
        throw errors;
    }
    static throwInvalidReact(options) {
        const errors = new CompilerError();
        errors.pushErrorDetail(new CompilerErrorDetail(options));
        throw errors;
    }
    static throwInvalidConfig(options) {
        const errors = new CompilerError();
        errors.pushErrorDetail(new CompilerErrorDetail(Object.assign(Object.assign({}, options), { category: ErrorCategory.Config })));
        throw errors;
    }
    static throw(options) {
        const errors = new CompilerError();
        errors.pushErrorDetail(new CompilerErrorDetail(options));
        throw errors;
    }
    constructor(...args) {
        super(...args);
        this.details = [];
        this.disabledDetails = [];
        this.printedMessage = null;
        this.name = 'ReactCompilerError';
        this.details = [];
        this.disabledDetails = [];
    }
    get message() {
        var _a;
        return (_a = this.printedMessage) !== null && _a !== void 0 ? _a : this.toString();
    }
    set message(_message) { }
    toString() {
        if (this.printedMessage) {
            return this.printedMessage;
        }
        if (Array.isArray(this.details)) {
            return this.details.map(detail => detail.toString()).join('\n\n');
        }
        return this.name;
    }
    withPrintedMessage(source, options) {
        this.printedMessage = this.printErrorMessage(source, options);
        return this;
    }
    printErrorMessage(source, options) {
        if (options.eslint && this.details.length === 1) {
            return this.details[0].printErrorMessage(source, options);
        }
        return (`Found ${this.details.length} error${this.details.length === 1 ? '' : 's'}:\n\n` +
            this.details
                .map(detail => detail.printErrorMessage(source, options).trim())
                .join('\n\n'));
    }
    merge(other) {
        this.details.push(...other.details);
        this.disabledDetails.push(...other.disabledDetails);
    }
    pushDiagnostic(diagnostic) {
        if (diagnostic.severity === ErrorSeverity.Off) {
            this.disabledDetails.push(diagnostic);
        }
        else {
            this.details.push(diagnostic);
        }
    }
    push(options) {
        var _a;
        const detail = new CompilerErrorDetail({
            category: options.category,
            reason: options.reason,
            description: (_a = options.description) !== null && _a !== void 0 ? _a : null,
            suggestions: options.suggestions,
            loc: typeof options.loc === 'symbol' ? null : options.loc,
        });
        return this.pushErrorDetail(detail);
    }
    pushErrorDetail(detail) {
        if (detail.severity === ErrorSeverity.Off) {
            this.disabledDetails.push(detail);
        }
        else {
            this.details.push(detail);
        }
        return detail;
    }
    hasAnyErrors() {
        return this.details.length > 0;
    }
    asResult() {
        return this.hasAnyErrors() ? Err(this) : Ok(undefined);
    }
    hasErrors() {
        for (const detail of this.details) {
            if (detail.severity === ErrorSeverity.Error) {
                return true;
            }
        }
        return false;
    }
    hasWarning() {
        let res = false;
        for (const detail of this.details) {
            if (detail.severity === ErrorSeverity.Error) {
                return false;
            }
            if (detail.severity === ErrorSeverity.Warning) {
                res = true;
            }
        }
        return res;
    }
    hasHints() {
        let res = false;
        for (const detail of this.details) {
            if (detail.severity === ErrorSeverity.Error) {
                return false;
            }
            if (detail.severity === ErrorSeverity.Warning) {
                return false;
            }
            if (detail.severity === ErrorSeverity.Hint) {
                res = true;
            }
        }
        return res;
    }
}
function printCodeFrame(source, loc, message) {
    const printed = libExports.codeFrameColumns(source, {
        start: {
            line: loc.start.line,
            column: loc.start.column + 1,
        },
        end: {
            line: loc.end.line,
            column: loc.end.column + 1,
        },
    }, {
        message,
        linesAbove: CODEFRAME_LINES_ABOVE,
        linesBelow: CODEFRAME_LINES_BELOW,
    });
    const lines = printed.split(/\r?\n/);
    if (loc.end.line - loc.start.line < CODEFRAME_MAX_LINES) {
        return printed;
    }
    const pipeIndex = lines[0].indexOf('|');
    return [
        ...lines.slice(0, CODEFRAME_LINES_ABOVE + CODEFRAME_ABBREVIATED_SOURCE_LINES),
        ' '.repeat(pipeIndex) + '…',
        ...lines.slice(-(CODEFRAME_LINES_BELOW + CODEFRAME_ABBREVIATED_SOURCE_LINES)),
    ].join('\n');
}
function printErrorSummary(category, message) {
    let heading;
    switch (category) {
        case ErrorCategory.AutomaticEffectDependencies:
        case ErrorCategory.CapitalizedCalls:
        case ErrorCategory.Config:
        case ErrorCategory.EffectDerivationsOfState:
        case ErrorCategory.EffectSetState:
        case ErrorCategory.ErrorBoundaries:
        case ErrorCategory.Factories:
        case ErrorCategory.FBT:
        case ErrorCategory.Fire:
        case ErrorCategory.Gating:
        case ErrorCategory.Globals:
        case ErrorCategory.Hooks:
        case ErrorCategory.Immutability:
        case ErrorCategory.Purity:
        case ErrorCategory.Refs:
        case ErrorCategory.RenderSetState:
        case ErrorCategory.StaticComponents:
        case ErrorCategory.Suppression:
        case ErrorCategory.Syntax:
        case ErrorCategory.UseMemo:
        case ErrorCategory.VoidUseMemo: {
            heading = 'Error';
            break;
        }
        case ErrorCategory.EffectDependencies:
        case ErrorCategory.IncompatibleLibrary:
        case ErrorCategory.PreserveManualMemo:
        case ErrorCategory.UnsupportedSyntax: {
            heading = 'Compilation Skipped';
            break;
        }
        case ErrorCategory.Invariant: {
            heading = 'Invariant';
            break;
        }
        case ErrorCategory.Todo: {
            heading = 'Todo';
            break;
        }
        default: {
            assertExhaustive$1(category, `Unhandled category '${category}'`);
        }
    }
    return `${heading}: ${message}`;
}
var ErrorCategory;
(function (ErrorCategory) {
    ErrorCategory["Hooks"] = "Hooks";
    ErrorCategory["CapitalizedCalls"] = "CapitalizedCalls";
    ErrorCategory["StaticComponents"] = "StaticComponents";
    ErrorCategory["UseMemo"] = "UseMemo";
    ErrorCategory["VoidUseMemo"] = "VoidUseMemo";
    ErrorCategory["Factories"] = "Factories";
    ErrorCategory["PreserveManualMemo"] = "PreserveManualMemo";
    ErrorCategory["IncompatibleLibrary"] = "IncompatibleLibrary";
    ErrorCategory["Immutability"] = "Immutability";
    ErrorCategory["Globals"] = "Globals";
    ErrorCategory["Refs"] = "Refs";
    ErrorCategory["EffectDependencies"] = "EffectDependencies";
    ErrorCategory["EffectSetState"] = "EffectSetState";
    ErrorCategory["EffectDerivationsOfState"] = "EffectDerivationsOfState";
    ErrorCategory["ErrorBoundaries"] = "ErrorBoundaries";
    ErrorCategory["Purity"] = "Purity";
    ErrorCategory["RenderSetState"] = "RenderSetState";
    ErrorCategory["Invariant"] = "Invariant";
    ErrorCategory["Todo"] = "Todo";
    ErrorCategory["Syntax"] = "Syntax";
    ErrorCategory["UnsupportedSyntax"] = "UnsupportedSyntax";
    ErrorCategory["Config"] = "Config";
    ErrorCategory["Gating"] = "Gating";
    ErrorCategory["Suppression"] = "Suppression";
    ErrorCategory["AutomaticEffectDependencies"] = "AutomaticEffectDependencies";
    ErrorCategory["Fire"] = "Fire";
    ErrorCategory["FBT"] = "FBT";
})(ErrorCategory || (ErrorCategory = {}));
var LintRulePreset;
(function (LintRulePreset) {
    LintRulePreset["Recommended"] = "recommended";
    LintRulePreset["RecommendedLatest"] = "recommended-latest";
    LintRulePreset["Off"] = "off";
})(LintRulePreset || (LintRulePreset = {}));
const RULE_NAME_PATTERN = /^[a-z]+(-[a-z]+)*$/;
function getRuleForCategory(category) {
    const rule = getRuleForCategoryImpl(category);
    invariant(RULE_NAME_PATTERN.test(rule.name), `Invalid rule name, got '${rule.name}' but rules must match ${RULE_NAME_PATTERN.toString()}`);
    return rule;
}
function getRuleForCategoryImpl(category) {
    switch (category) {
        case ErrorCategory.AutomaticEffectDependencies: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'automatic-effect-dependencies',
                description: 'Verifies that automatic effect dependencies are compiled if opted-in',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.CapitalizedCalls: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'capitalized-calls',
                description: 'Validates against calling capitalized functions/methods instead of using JSX',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Config: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'config',
                description: 'Validates the compiler configuration options',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.EffectDependencies: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'memoized-effect-dependencies',
                description: 'Validates that effect dependencies are memoized',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.EffectDerivationsOfState: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'no-deriving-state-in-effects',
                description: 'Validates against deriving values from state in an effect',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.EffectSetState: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'set-state-in-effect',
                description: 'Validates against calling setState synchronously in an effect, which can lead to re-renders that degrade performance',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.ErrorBoundaries: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'error-boundaries',
                description: 'Validates usage of error boundaries instead of try/catch for errors in child components',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Factories: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'component-hook-factories',
                description: 'Validates against higher order functions defining nested components or hooks. ' +
                    'Components and hooks should be defined at the module level',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.FBT: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'fbt',
                description: 'Validates usage of fbt',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Fire: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'fire',
                description: 'Validates usage of `fire`',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Gating: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'gating',
                description: 'Validates configuration of [gating mode](https://react.dev/reference/react-compiler/gating)',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Globals: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'globals',
                description: 'Validates against assignment/mutation of globals during render, part of ensuring that ' +
                    '[side effects must render outside of render](https://react.dev/reference/rules/components-and-hooks-must-be-pure#side-effects-must-run-outside-of-render)',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Hooks: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'hooks',
                description: 'Validates the rules of hooks',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Immutability: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'immutability',
                description: 'Validates against mutating props, state, and other values that [are immutable](https://react.dev/reference/rules/components-and-hooks-must-be-pure#props-and-state-are-immutable)',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Invariant: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'invariant',
                description: 'Internal invariants',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.PreserveManualMemo: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'preserve-manual-memoization',
                description: 'Validates that existing manual memoized is preserved by the compiler. ' +
                    'React Compiler will only compile components and hooks if its inference ' +
                    '[matches or exceeds the existing manual memoization](https://react.dev/learn/react-compiler/introduction#what-should-i-do-about-usememo-usecallback-and-reactmemo)',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Purity: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'purity',
                description: 'Validates that [components/hooks are pure](https://react.dev/reference/rules/components-and-hooks-must-be-pure) by checking that they do not call known-impure functions',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Refs: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'refs',
                description: 'Validates correct usage of refs, not reading/writing during render. See the "pitfalls" section in [`useRef()` usage](https://react.dev/reference/react/useRef#usage)',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.RenderSetState: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'set-state-in-render',
                description: 'Validates against setting state during render, which can trigger additional renders and potential infinite render loops',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.StaticComponents: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'static-components',
                description: 'Validates that components are static, not recreated every render. Components that are recreated dynamically can reset state and trigger excessive re-rendering',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.Suppression: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'rule-suppression',
                description: 'Validates against suppression of other rules',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Syntax: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'syntax',
                description: 'Validates against invalid syntax',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.Todo: {
            return {
                category,
                severity: ErrorSeverity.Hint,
                name: 'todo',
                description: 'Unimplemented features',
                preset: LintRulePreset.Off,
            };
        }
        case ErrorCategory.UnsupportedSyntax: {
            return {
                category,
                severity: ErrorSeverity.Warning,
                name: 'unsupported-syntax',
                description: 'Validates against syntax that we do not plan to support in React Compiler',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.UseMemo: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'use-memo',
                description: 'Validates usage of the useMemo() hook against common mistakes. See [`useMemo()` docs](https://react.dev/reference/react/useMemo) for more information.',
                preset: LintRulePreset.Recommended,
            };
        }
        case ErrorCategory.VoidUseMemo: {
            return {
                category,
                severity: ErrorSeverity.Error,
                name: 'void-use-memo',
                description: 'Validates that useMemos always return a value and that the result of the useMemo is used by the component/hook. See [`useMemo()` docs](https://react.dev/reference/react/useMemo) for more information.',
                preset: LintRulePreset.RecommendedLatest,
            };
        }
        case ErrorCategory.IncompatibleLibrary: {
            return {
                category,
                severity: ErrorSeverity.Warning,
                name: 'incompatible-library',
                description: 'Validates against usage of libraries which are incompatible with memoization (manual or automatic)',
                preset: LintRulePreset.Recommended,
            };
        }
        default: {
            assertExhaustive$1(category, `Unsupported category ${category}`);
        }
    }
}
const LintRules = Object.keys(ErrorCategory).map(category => getRuleForCategory(category));

function insertAdditionalFunctionDeclaration(fnPath, compiled, programContext, gatingFunctionIdentifierName) {
    var _a, _b;
    const originalFnName = fnPath.node.id;
    const originalFnParams = fnPath.node.params;
    const compiledParams = fnPath.node.params;
    CompilerError.invariant(originalFnName != null && compiled.id != null, {
        reason: 'Expected function declarations that are referenced elsewhere to have a named identifier',
        description: null,
        details: [
            {
                kind: 'error',
                loc: (_a = fnPath.node.loc) !== null && _a !== void 0 ? _a : null,
                message: null,
            },
        ],
    });
    CompilerError.invariant(originalFnParams.length === compiledParams.length, {
        reason: 'Expected React Compiler optimized function declarations to have the same number of parameters as source',
        description: null,
        details: [
            {
                kind: 'error',
                loc: (_b = fnPath.node.loc) !== null && _b !== void 0 ? _b : null,
                message: null,
            },
        ],
    });
    const gatingCondition = libExports$1.identifier(programContext.newUid(`${gatingFunctionIdentifierName}_result`));
    const unoptimizedFnName = libExports$1.identifier(programContext.newUid(`${originalFnName.name}_unoptimized`));
    const optimizedFnName = libExports$1.identifier(programContext.newUid(`${originalFnName.name}_optimized`));
    compiled.id.name = optimizedFnName.name;
    fnPath.get('id').replaceInline(unoptimizedFnName);
    const newParams = [];
    const genNewArgs = [];
    for (let i = 0; i < originalFnParams.length; i++) {
        const argName = `arg${i}`;
        if (originalFnParams[i].type === 'RestElement') {
            newParams.push(libExports$1.restElement(libExports$1.identifier(argName)));
            genNewArgs.push(() => libExports$1.spreadElement(libExports$1.identifier(argName)));
        }
        else {
            newParams.push(libExports$1.identifier(argName));
            genNewArgs.push(() => libExports$1.identifier(argName));
        }
    }
    fnPath.insertAfter(libExports$1.functionDeclaration(originalFnName, newParams, libExports$1.blockStatement([
        libExports$1.ifStatement(gatingCondition, libExports$1.returnStatement(libExports$1.callExpression(compiled.id, genNewArgs.map(fn => fn()))), libExports$1.returnStatement(libExports$1.callExpression(unoptimizedFnName, genNewArgs.map(fn => fn())))),
    ])));
    fnPath.insertBefore(libExports$1.variableDeclaration('const', [
        libExports$1.variableDeclarator(gatingCondition, libExports$1.callExpression(libExports$1.identifier(gatingFunctionIdentifierName), [])),
    ]));
    fnPath.insertBefore(compiled);
}
function insertGatedFunctionDeclaration(fnPath, compiled, programContext, gating, referencedBeforeDeclaration) {
    var _a;
    const gatingImportedName = programContext.addImportSpecifier(gating).name;
    if (referencedBeforeDeclaration && fnPath.isFunctionDeclaration()) {
        CompilerError.invariant(compiled.type === 'FunctionDeclaration', {
            reason: 'Expected compiled node type to match input type',
            description: `Got ${compiled.type} but expected FunctionDeclaration`,
            details: [
                {
                    kind: 'error',
                    loc: (_a = fnPath.node.loc) !== null && _a !== void 0 ? _a : null,
                    message: null,
                },
            ],
        });
        insertAdditionalFunctionDeclaration(fnPath, compiled, programContext, gatingImportedName);
    }
    else {
        const gatingExpression = libExports$1.conditionalExpression(libExports$1.callExpression(libExports$1.identifier(gatingImportedName), []), buildFunctionExpression(compiled), buildFunctionExpression(fnPath.node));
        if (fnPath.parentPath.node.type !== 'ExportDefaultDeclaration' &&
            fnPath.node.type === 'FunctionDeclaration' &&
            fnPath.node.id != null) {
            fnPath.replaceWith(libExports$1.variableDeclaration('const', [
                libExports$1.variableDeclarator(fnPath.node.id, gatingExpression),
            ]));
        }
        else if (fnPath.parentPath.node.type === 'ExportDefaultDeclaration' &&
            fnPath.node.type !== 'ArrowFunctionExpression' &&
            fnPath.node.id != null) {
            fnPath.insertAfter(libExports$1.exportDefaultDeclaration(libExports$1.identifier(fnPath.node.id.name)));
            fnPath.parentPath.replaceWith(libExports$1.variableDeclaration('const', [
                libExports$1.variableDeclarator(libExports$1.identifier(fnPath.node.id.name), gatingExpression),
            ]));
        }
        else {
            fnPath.replaceWith(gatingExpression);
        }
    }
}
function buildFunctionExpression(node) {
    var _a, _b;
    if (node.type === 'ArrowFunctionExpression' ||
        node.type === 'FunctionExpression') {
        return node;
    }
    else {
        const fn = {
            type: 'FunctionExpression',
            async: node.async,
            generator: node.generator,
            loc: (_a = node.loc) !== null && _a !== void 0 ? _a : null,
            id: (_b = node.id) !== null && _b !== void 0 ? _b : null,
            params: node.params,
            body: node.body,
        };
        return fn;
    }
}

function makeTypeId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected instruction id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
let typeCounter = 0;
function makeType() {
    return {
        kind: 'Type',
        id: makeTypeId(typeCounter++),
    };
}
function typeEquals(tA, tB) {
    if (tA.kind !== tB.kind)
        return false;
    return (typeVarEquals(tA, tB) ||
        funcTypeEquals(tA, tB) ||
        objectTypeEquals(tA, tB) ||
        primitiveTypeEquals(tA, tB) ||
        polyTypeEquals(tA, tB) ||
        phiTypeEquals(tA, tB) ||
        propTypeEquals(tA, tB) ||
        objectMethodTypeEquals(tA, tB));
}
function typeVarEquals(tA, tB) {
    if (tA.kind === 'Type' && tB.kind === 'Type') {
        return tA.id === tB.id;
    }
    return false;
}
function typeKindCheck(tA, tb, type) {
    return tA.kind === type && tb.kind === type;
}
function objectMethodTypeEquals(tA, tB) {
    return typeKindCheck(tA, tB, 'ObjectMethod');
}
function propTypeEquals(tA, tB) {
    if (tA.kind === 'Property' && tB.kind === 'Property') {
        if (!typeEquals(tA.objectType, tB.objectType)) {
            return false;
        }
        return (tA.propertyName === tB.propertyName && tA.objectName === tB.objectName);
    }
    return false;
}
function primitiveTypeEquals(tA, tB) {
    return typeKindCheck(tA, tB, 'Primitive');
}
function polyTypeEquals(tA, tB) {
    return typeKindCheck(tA, tB, 'Poly');
}
function objectTypeEquals(tA, tB) {
    if (tA.kind === 'Object' && tB.kind == 'Object') {
        return tA.shapeId === tB.shapeId;
    }
    return false;
}
function funcTypeEquals(tA, tB) {
    if (tA.kind !== 'Function' || tB.kind !== 'Function') {
        return false;
    }
    return typeEquals(tA.return, tB.return);
}
function phiTypeEquals(tA, tB) {
    if (tA.kind === 'Phi' && tB.kind === 'Phi') {
        if (tA.operands.length !== tB.operands.length) {
            return false;
        }
        let operands = new Set(tA.operands);
        for (let i = 0; i < tB.operands.length; i++) {
            if (!operands.has(tB.operands[i])) {
                return false;
            }
        }
    }
    return false;
}

const RESERVED_WORDS = new Set([
    'break',
    'case',
    'catch',
    'class',
    'const',
    'continue',
    'debugger',
    'default',
    'delete',
    'do',
    'else',
    'enum',
    'export',
    'extends',
    'false',
    'finally',
    'for',
    'function',
    'if',
    'import',
    'in',
    'instanceof',
    'new',
    'null',
    'return',
    'super',
    'switch',
    'this',
    'throw',
    'true',
    'try',
    'typeof',
    'var',
    'void',
    'while',
    'with',
]);
const STRICT_MODE_RESERVED_WORDS = new Set([
    'let',
    'static',
    'implements',
    'interface',
    'package',
    'private',
    'protected',
    'public',
]);
const STRICT_MODE_RESTRICTED_WORDS = new Set(['eval', 'arguments']);
function isReservedWord(identifierName) {
    return (RESERVED_WORDS.has(identifierName) ||
        STRICT_MODE_RESERVED_WORDS.has(identifierName) ||
        STRICT_MODE_RESTRICTED_WORDS.has(identifierName));
}

const GeneratedSource = Symbol();
function isStatementBlockKind(kind) {
    return kind === 'block' || kind === 'catch';
}
function isExpressionBlockKind(kind) {
    return !isStatementBlockKind(kind);
}
var GotoVariant;
(function (GotoVariant) {
    GotoVariant["Break"] = "Break";
    GotoVariant["Continue"] = "Continue";
    GotoVariant["Try"] = "Try";
})(GotoVariant || (GotoVariant = {}));
var InstructionKind;
(function (InstructionKind) {
    InstructionKind["Const"] = "Const";
    InstructionKind["Let"] = "Let";
    InstructionKind["Reassign"] = "Reassign";
    InstructionKind["Catch"] = "Catch";
    InstructionKind["HoistedConst"] = "HoistedConst";
    InstructionKind["HoistedLet"] = "HoistedLet";
    InstructionKind["HoistedFunction"] = "HoistedFunction";
    InstructionKind["Function"] = "Function";
})(InstructionKind || (InstructionKind = {}));
function convertHoistedLValueKind(kind) {
    switch (kind) {
        case InstructionKind.HoistedLet:
            return InstructionKind.Let;
        case InstructionKind.HoistedConst:
            return InstructionKind.Const;
        case InstructionKind.HoistedFunction:
            return InstructionKind.Function;
        case InstructionKind.Let:
        case InstructionKind.Const:
        case InstructionKind.Function:
        case InstructionKind.Reassign:
        case InstructionKind.Catch:
            return null;
        default:
            assertExhaustive$1(kind, 'Unexpected lvalue kind');
    }
}
function makeTemporaryIdentifier(id, loc) {
    return {
        id,
        name: null,
        declarationId: makeDeclarationId(id),
        mutableRange: { start: makeInstructionId(0), end: makeInstructionId(0) },
        scope: null,
        type: makeType(),
        loc,
    };
}
function forkTemporaryIdentifier(id, source) {
    return Object.assign(Object.assign({}, source), { mutableRange: { start: makeInstructionId(0), end: makeInstructionId(0) }, id });
}
function validateIdentifierName(name) {
    if (isReservedWord(name)) {
        const error = new CompilerError();
        error.pushDiagnostic(CompilerDiagnostic.create({
            category: ErrorCategory.Syntax,
            reason: 'Expected a non-reserved identifier name',
            description: `\`${name}\` is a reserved word in JavaScript and cannot be used as an identifier name`,
            suggestions: null,
        }).withDetails({
            kind: 'error',
            loc: GeneratedSource,
            message: 'reserved word',
        }));
        return Err(error);
    }
    else if (!libExports$1.isValidIdentifier(name)) {
        const error = new CompilerError();
        error.pushDiagnostic(CompilerDiagnostic.create({
            category: ErrorCategory.Syntax,
            reason: `Expected a valid identifier name`,
            description: `\`${name}\` is not a valid JavaScript identifier`,
            suggestions: null,
        }).withDetails({
            kind: 'error',
            loc: GeneratedSource,
            message: 'reserved word',
        }));
    }
    return Ok({
        kind: 'named',
        value: name,
    });
}
function makeIdentifierName(name) {
    return validateIdentifierName(name).unwrap();
}
function promoteTemporary(identifier) {
    CompilerError.invariant(identifier.name === null, {
        reason: `Expected a temporary (unnamed) identifier`,
        description: `Identifier already has a name, \`${identifier.name}\``,
        details: [
            {
                kind: 'error',
                loc: GeneratedSource,
                message: null,
            },
        ],
        suggestions: null,
    });
    identifier.name = {
        kind: 'promoted',
        value: `#t${identifier.declarationId}`,
    };
}
function isPromotedTemporary(name) {
    return name.startsWith('#t');
}
function promoteTemporaryJsxTag(identifier) {
    CompilerError.invariant(identifier.name === null, {
        reason: `Expected a temporary (unnamed) identifier`,
        description: `Identifier already has a name, \`${identifier.name}\``,
        details: [
            {
                kind: 'error',
                loc: GeneratedSource,
                message: null,
            },
        ],
        suggestions: null,
    });
    identifier.name = {
        kind: 'promoted',
        value: `#T${identifier.declarationId}`,
    };
}
function isPromotedJsxTemporary(name) {
    return name.startsWith('#T');
}
var ValueReason;
(function (ValueReason) {
    ValueReason["Global"] = "global";
    ValueReason["JsxCaptured"] = "jsx-captured";
    ValueReason["HookCaptured"] = "hook-captured";
    ValueReason["HookReturn"] = "hook-return";
    ValueReason["Effect"] = "effect";
    ValueReason["KnownReturnSignature"] = "known-return-signature";
    ValueReason["Context"] = "context";
    ValueReason["State"] = "state";
    ValueReason["ReducerState"] = "reducer-state";
    ValueReason["ReactiveFunctionArgument"] = "reactive-function-argument";
    ValueReason["Other"] = "other";
})(ValueReason || (ValueReason = {}));
var ValueKind;
(function (ValueKind) {
    ValueKind["MaybeFrozen"] = "maybefrozen";
    ValueKind["Frozen"] = "frozen";
    ValueKind["Primitive"] = "primitive";
    ValueKind["Global"] = "global";
    ValueKind["Mutable"] = "mutable";
    ValueKind["Context"] = "context";
})(ValueKind || (ValueKind = {}));
const ValueKindSchema = v4.z.enum([
    ValueKind.MaybeFrozen,
    ValueKind.Frozen,
    ValueKind.Primitive,
    ValueKind.Global,
    ValueKind.Mutable,
    ValueKind.Context,
]);
const ValueReasonSchema = v4.z.enum([
    ValueReason.Context,
    ValueReason.Effect,
    ValueReason.Global,
    ValueReason.HookCaptured,
    ValueReason.HookReturn,
    ValueReason.JsxCaptured,
    ValueReason.KnownReturnSignature,
    ValueReason.Other,
    ValueReason.ReactiveFunctionArgument,
    ValueReason.ReducerState,
    ValueReason.State,
]);
var Effect;
(function (Effect) {
    Effect["Unknown"] = "<unknown>";
    Effect["Freeze"] = "freeze";
    Effect["Read"] = "read";
    Effect["Capture"] = "capture";
    Effect["ConditionallyMutateIterator"] = "mutate-iterator?";
    Effect["ConditionallyMutate"] = "mutate?";
    Effect["Mutate"] = "mutate";
    Effect["Store"] = "store";
})(Effect || (Effect = {}));
const EffectSchema = v4.z.enum([
    Effect.Read,
    Effect.Mutate,
    Effect.ConditionallyMutate,
    Effect.ConditionallyMutateIterator,
    Effect.Capture,
    Effect.Store,
    Effect.Freeze,
]);
function isMutableEffect(effect, location) {
    switch (effect) {
        case Effect.Capture:
        case Effect.Store:
        case Effect.ConditionallyMutate:
        case Effect.ConditionallyMutateIterator:
        case Effect.Mutate: {
            return true;
        }
        case Effect.Unknown: {
            CompilerError.invariant(false, {
                reason: 'Unexpected unknown effect',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: location,
                        message: null,
                    },
                ],
                suggestions: null,
            });
        }
        case Effect.Read:
        case Effect.Freeze: {
            return false;
        }
        default: {
            assertExhaustive$1(effect, `Unexpected effect \`${effect}\``);
        }
    }
}
function makePropertyLiteral(value) {
    return value;
}
function areEqualPaths(a, b) {
    return (a.length === b.length &&
        a.every((item, ix) => item.property === b[ix].property && item.optional === b[ix].optional));
}
function getPlaceScope(id, place) {
    const scope = place.identifier.scope;
    if (scope !== null && isScopeActive(scope, id)) {
        return scope;
    }
    return null;
}
function isScopeActive(scope, id) {
    return id >= scope.range.start && id < scope.range.end;
}
function makeBlockId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected block id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
function makeScopeId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected block id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
function makeIdentifierId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected identifier id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
function makeDeclarationId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected declaration id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
function makeInstructionId(id) {
    CompilerError.invariant(id >= 0 && Number.isInteger(id), {
        reason: 'Expected instruction id to be a non-negative integer',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    return id;
}
function isObjectMethodType(id) {
    return id.type.kind == 'ObjectMethod';
}
function isPrimitiveType(id) {
    return id.type.kind === 'Primitive';
}
function isArrayType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInArray';
}
function isMapType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInMap';
}
function isSetType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInSet';
}
function isPropsType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInProps';
}
function isRefValueType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInRefValue';
}
function isUseRefType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInUseRefId';
}
function isUseStateType(id) {
    return id.type.kind === 'Object' && id.type.shapeId === 'BuiltInUseState';
}
function isJsxType(type) {
    return type.kind === 'Object' && type.shapeId === 'BuiltInJsx';
}
function isRefOrRefValue(id) {
    return isUseRefType(id) || isRefValueType(id);
}
function isRefOrRefLikeMutableType(type) {
    return (type.kind === 'Object' &&
        (type.shapeId === 'BuiltInUseRefId' ||
            type.shapeId == 'ReanimatedSharedValueId'));
}
function isSetStateType(id) {
    return id.type.kind === 'Function' && id.type.shapeId === 'BuiltInSetState';
}
function isStartTransitionType(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInStartTransition');
}
function isSetActionStateType(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInSetActionState');
}
function isUseReducerType(id) {
    return id.type.kind === 'Function' && id.type.shapeId === 'BuiltInUseReducer';
}
function isDispatcherType(id) {
    return id.type.kind === 'Function' && id.type.shapeId === 'BuiltInDispatch';
}
function isFireFunctionType(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInFireFunction');
}
function isEffectEventFunctionType(id) {
    return (id.type.kind === 'Function' &&
        id.type.shapeId === 'BuiltInEffectEventFunction');
}
function isStableType(id) {
    return (isSetStateType(id) ||
        isSetActionStateType(id) ||
        isDispatcherType(id) ||
        isUseRefType(id) ||
        isStartTransitionType(id));
}
function isStableTypeContainer(id) {
    const type_ = id.type;
    if (type_.kind !== 'Object') {
        return false;
    }
    return (isUseStateType(id) ||
        type_.shapeId === 'BuiltInUseActionState' ||
        isUseReducerType(id) ||
        type_.shapeId === 'BuiltInUseTransition');
}
function evaluatesToStableTypeOrContainer(env, { value }) {
    if (value.kind === 'CallExpression' || value.kind === 'MethodCall') {
        const callee = value.kind === 'CallExpression' ? value.callee : value.property;
        const calleeHookKind = getHookKind(env, callee.identifier);
        switch (calleeHookKind) {
            case 'useState':
            case 'useReducer':
            case 'useActionState':
            case 'useRef':
            case 'useTransition':
                return true;
        }
    }
    return false;
}
function isUseEffectHookType(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInUseEffectHook');
}
function isUseLayoutEffectHookType(id) {
    return (id.type.kind === 'Function' &&
        id.type.shapeId === 'BuiltInUseLayoutEffectHook');
}
function isUseInsertionEffectHookType(id) {
    return (id.type.kind === 'Function' &&
        id.type.shapeId === 'BuiltInUseInsertionEffectHook');
}
function isUseContextHookType(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInUseContextHook');
}
function getHookKind(env, id) {
    return getHookKindForType(env, id.type);
}
function isUseOperator(id) {
    return (id.type.kind === 'Function' && id.type.shapeId === 'BuiltInUseOperator');
}
function getHookKindForType(env, type) {
    var _a;
    if (type.kind === 'Function') {
        const signature = env.getFunctionSignature(type);
        return (_a = signature === null || signature === void 0 ? void 0 : signature.hookKind) !== null && _a !== void 0 ? _a : null;
    }
    return null;
}

function printReactiveScopeSummary(scope) {
    const items = [];
    items.push('scope');
    items.push(`@${scope.id}`);
    items.push(`[${scope.range.start}:${scope.range.end}]`);
    items.push(`dependencies=[${Array.from(scope.dependencies)
        .map(dep => printDependency(dep))
        .join(', ')}]`);
    items.push(`declarations=[${Array.from(scope.declarations)
        .map(([, decl]) => printIdentifier(Object.assign(Object.assign({}, decl.identifier), { scope: decl.scope })))
        .join(', ')}]`);
    items.push(`reassignments=[${Array.from(scope.reassignments).map(reassign => printIdentifier(reassign))}]`);
    if (scope.earlyReturnValue !== null) {
        items.push(`earlyReturn={id: ${printIdentifier(scope.earlyReturnValue.value)}, label: ${scope.earlyReturnValue.label}}}`);
    }
    return items.join(' ');
}
function printDependency(dependency) {
    const identifier = printIdentifier(dependency.identifier) +
        printType(dependency.identifier.type);
    return `${identifier}${dependency.path.map(token => `${token.optional ? '?.' : '.'}${token.property}`).join('')}`;
}

function printFunction(fn) {
    const output = [];
    let definition = '';
    if (fn.id !== null) {
        definition += fn.id;
    }
    else {
        definition += '<<anonymous>>';
    }
    if (fn.nameHint != null) {
        definition += ` ${fn.nameHint}`;
    }
    if (fn.params.length !== 0) {
        definition +=
            '(' +
                fn.params
                    .map(param => {
                    if (param.kind === 'Identifier') {
                        return printPlace(param);
                    }
                    else {
                        return `...${printPlace(param.place)}`;
                    }
                })
                    .join(', ') +
                ')';
    }
    else {
        definition += '()';
    }
    definition += `: ${printPlace(fn.returns)}`;
    output.push(definition);
    output.push(...fn.directives);
    output.push(printHIR(fn.body));
    return output.join('\n');
}
function printHIR(ir, options = null) {
    var _a;
    let output = [];
    let indent = ' '.repeat((_a = options === null || options === void 0 ? void 0 : options.indent) !== null && _a !== void 0 ? _a : 0);
    const push = (text, indent = '  ') => {
        output.push(`${indent}${text}`);
    };
    for (const [blockId, block] of ir.blocks) {
        output.push(`bb${blockId} (${block.kind}):`);
        if (block.preds.size > 0) {
            const preds = ['predecessor blocks:'];
            for (const pred of block.preds) {
                preds.push(`bb${pred}`);
            }
            push(preds.join(' '));
        }
        for (const phi of block.phis) {
            push(printPhi(phi));
        }
        for (const instr of block.instructions) {
            push(printInstruction(instr));
        }
        const terminal = printTerminal(block.terminal);
        if (Array.isArray(terminal)) {
            terminal.forEach(line => push(line));
        }
        else {
            push(terminal);
        }
    }
    return output.map(line => indent + line).join('\n');
}
function printInstruction(instr) {
    const id = `[${instr.id}]`;
    let value = printInstructionValue(instr.value);
    if (instr.effects != null) {
        value += `\n    ${instr.effects.map(printAliasingEffect).join('\n    ')}`;
    }
    if (instr.lvalue !== null) {
        return `${id} ${printPlace(instr.lvalue)} = ${value}`;
    }
    else {
        return `${id} ${value}`;
    }
}
function printPhi(phi) {
    const items = [];
    items.push(printPlace(phi.place));
    items.push(printMutableRange(phi.place.identifier));
    items.push(printType(phi.place.identifier.type));
    items.push(': phi(');
    const phis = [];
    for (const [blockId, place] of phi.operands) {
        phis.push(`bb${blockId}: ${printPlace(place)}`);
    }
    items.push(phis.join(', '));
    items.push(')');
    return items.join('');
}
function printTerminal(terminal) {
    let value;
    switch (terminal.kind) {
        case 'if': {
            value = `[${terminal.id}] If (${printPlace(terminal.test)}) then:bb${terminal.consequent} else:bb${terminal.alternate}${terminal.fallthrough ? ` fallthrough=bb${terminal.fallthrough}` : ''}`;
            break;
        }
        case 'branch': {
            value = `[${terminal.id}] Branch (${printPlace(terminal.test)}) then:bb${terminal.consequent} else:bb${terminal.alternate} fallthrough:bb${terminal.fallthrough}`;
            break;
        }
        case 'logical': {
            value = `[${terminal.id}] Logical ${terminal.operator} test:bb${terminal.test} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'ternary': {
            value = `[${terminal.id}] Ternary test:bb${terminal.test} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'optional': {
            value = `[${terminal.id}] Optional (optional=${terminal.optional}) test:bb${terminal.test} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'throw': {
            value = `[${terminal.id}] Throw ${printPlace(terminal.value)}`;
            break;
        }
        case 'return': {
            value = `[${terminal.id}] Return ${terminal.returnVariant}${terminal.value != null ? ' ' + printPlace(terminal.value) : ''}`;
            if (terminal.effects != null) {
                value += `\n    ${terminal.effects.map(printAliasingEffect).join('\n    ')}`;
            }
            break;
        }
        case 'goto': {
            value = `[${terminal.id}] Goto${terminal.variant === GotoVariant.Continue ? '(Continue)' : ''} bb${terminal.block}`;
            break;
        }
        case 'switch': {
            const output = [];
            output.push(`[${terminal.id}] Switch (${printPlace(terminal.test)})`);
            terminal.cases.forEach(case_ => {
                if (case_.test !== null) {
                    output.push(`  Case ${printPlace(case_.test)}: bb${case_.block}`);
                }
                else {
                    output.push(`  Default: bb${case_.block}`);
                }
            });
            if (terminal.fallthrough) {
                output.push(`  Fallthrough: bb${terminal.fallthrough}`);
            }
            value = output;
            break;
        }
        case 'do-while': {
            value = `[${terminal.id}] DoWhile loop=${`bb${terminal.loop}`} test=bb${terminal.test} fallthrough=${`bb${terminal.fallthrough}`}`;
            break;
        }
        case 'while': {
            value = `[${terminal.id}] While test=bb${terminal.test} loop=${terminal.loop !== null ? `bb${terminal.loop}` : ''} fallthrough=${terminal.fallthrough ? `bb${terminal.fallthrough}` : ''}`;
            break;
        }
        case 'for': {
            value = `[${terminal.id}] For init=bb${terminal.init} test=bb${terminal.test} loop=bb${terminal.loop} update=bb${terminal.update} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'for-of': {
            value = `[${terminal.id}] ForOf init=bb${terminal.init} test=bb${terminal.test} loop=bb${terminal.loop} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'for-in': {
            value = `[${terminal.id}] ForIn init=bb${terminal.init} loop=bb${terminal.loop} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'label': {
            value = `[${terminal.id}] Label block=bb${terminal.block} fallthrough=${terminal.fallthrough ? `bb${terminal.fallthrough}` : ''}`;
            break;
        }
        case 'sequence': {
            value = `[${terminal.id}] Sequence block=bb${terminal.block} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'unreachable': {
            value = `[${terminal.id}] Unreachable`;
            break;
        }
        case 'unsupported': {
            value = `[${terminal.id}] Unsupported`;
            break;
        }
        case 'maybe-throw': {
            value = `[${terminal.id}] MaybeThrow continuation=bb${terminal.continuation} handler=bb${terminal.handler}`;
            if (terminal.effects != null) {
                value += `\n    ${terminal.effects.map(printAliasingEffect).join('\n    ')}`;
            }
            break;
        }
        case 'scope': {
            value = `[${terminal.id}] Scope ${printReactiveScopeSummary(terminal.scope)} block=bb${terminal.block} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'pruned-scope': {
            value = `[${terminal.id}] <pruned> Scope ${printReactiveScopeSummary(terminal.scope)} block=bb${terminal.block} fallthrough=bb${terminal.fallthrough}`;
            break;
        }
        case 'try': {
            value = `[${terminal.id}] Try block=bb${terminal.block} handler=bb${terminal.handler}${terminal.handlerBinding !== null
                ? ` handlerBinding=(${printPlace(terminal.handlerBinding)})`
                : ''} fallthrough=${terminal.fallthrough != null ? `bb${terminal.fallthrough}` : ''}`;
            break;
        }
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal}\``);
        }
    }
    return value;
}
function printHole() {
    return '<hole>';
}
function printObjectPropertyKey(key) {
    switch (key.kind) {
        case 'identifier':
            return key.name;
        case 'string':
            return `"${key.name}"`;
        case 'computed': {
            return `[${printPlace(key.name)}]`;
        }
        case 'number': {
            return String(key.name);
        }
    }
}
function printInstructionValue(instrValue) {
    var _a, _b, _c, _d, _e;
    let value = '';
    switch (instrValue.kind) {
        case 'ArrayExpression': {
            value = `Array [${instrValue.elements
                .map(element => {
                if (element.kind === 'Identifier') {
                    return printPlace(element);
                }
                else if (element.kind === 'Hole') {
                    return printHole();
                }
                else {
                    return `...${printPlace(element.place)}`;
                }
            })
                .join(', ')}]`;
            break;
        }
        case 'ObjectExpression': {
            const properties = [];
            if (instrValue.properties !== null) {
                for (const property of instrValue.properties) {
                    if (property.kind === 'ObjectProperty') {
                        properties.push(`${printObjectPropertyKey(property.key)}: ${printPlace(property.place)}`);
                    }
                    else {
                        properties.push(`...${printPlace(property.place)}`);
                    }
                }
            }
            value = `Object { ${properties.join(', ')} }`;
            break;
        }
        case 'UnaryExpression': {
            value = `Unary ${printPlace(instrValue.value)}`;
            break;
        }
        case 'BinaryExpression': {
            value = `Binary ${printPlace(instrValue.left)} ${instrValue.operator} ${printPlace(instrValue.right)}`;
            break;
        }
        case 'NewExpression': {
            value = `New ${printPlace(instrValue.callee)}(${instrValue.args
                .map(arg => printPattern(arg))
                .join(', ')})`;
            break;
        }
        case 'CallExpression': {
            value = `Call ${printPlace(instrValue.callee)}(${instrValue.args
                .map(arg => printPattern(arg))
                .join(', ')})`;
            break;
        }
        case 'MethodCall': {
            value = `MethodCall ${printPlace(instrValue.receiver)}.${printPlace(instrValue.property)}(${instrValue.args.map(arg => printPattern(arg)).join(', ')})`;
            break;
        }
        case 'JSXText': {
            value = `JSXText ${JSON.stringify(instrValue.value)}`;
            break;
        }
        case 'Primitive': {
            if (instrValue.value === undefined) {
                value = '<undefined>';
            }
            else {
                value = JSON.stringify(instrValue.value);
            }
            break;
        }
        case 'TypeCastExpression': {
            value = `TypeCast ${printPlace(instrValue.value)}: ${printType(instrValue.type)}`;
            break;
        }
        case 'JsxExpression': {
            const propItems = [];
            for (const attribute of instrValue.props) {
                if (attribute.kind === 'JsxAttribute') {
                    propItems.push(`${attribute.name}={${attribute.place !== null ? printPlace(attribute.place) : '<empty>'}}`);
                }
                else {
                    propItems.push(`...${printPlace(attribute.argument)}`);
                }
            }
            const tag = instrValue.tag.kind === 'Identifier'
                ? printPlace(instrValue.tag)
                : instrValue.tag.name;
            const props = propItems.length !== 0 ? ' ' + propItems.join(' ') : '';
            if (instrValue.children !== null) {
                const children = instrValue.children.map(child => {
                    return `{${printPlace(child)}}`;
                });
                value = `JSX <${tag}${props}${props.length > 0 ? ' ' : ''}>${children.join('')}</${tag}>`;
            }
            else {
                value = `JSX <${tag}${props}${props.length > 0 ? ' ' : ''}/>`;
            }
            break;
        }
        case 'JsxFragment': {
            value = `JsxFragment [${instrValue.children
                .map(child => printPlace(child))
                .join(', ')}]`;
            break;
        }
        case 'UnsupportedNode': {
            value = `UnsupportedNode ${instrValue.node.type}`;
            break;
        }
        case 'LoadLocal': {
            value = `LoadLocal ${printPlace(instrValue.place)}`;
            break;
        }
        case 'DeclareLocal': {
            value = `DeclareLocal ${instrValue.lvalue.kind} ${printPlace(instrValue.lvalue.place)}`;
            break;
        }
        case 'DeclareContext': {
            value = `DeclareContext ${instrValue.lvalue.kind} ${printPlace(instrValue.lvalue.place)}`;
            break;
        }
        case 'StoreLocal': {
            value = `StoreLocal ${instrValue.lvalue.kind} ${printPlace(instrValue.lvalue.place)} = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'LoadContext': {
            value = `LoadContext ${printPlace(instrValue.place)}`;
            break;
        }
        case 'StoreContext': {
            value = `StoreContext ${instrValue.lvalue.kind} ${printPlace(instrValue.lvalue.place)} = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'Destructure': {
            value = `Destructure ${instrValue.lvalue.kind} ${printPattern(instrValue.lvalue.pattern)} = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'PropertyLoad': {
            value = `PropertyLoad ${printPlace(instrValue.object)}.${instrValue.property}`;
            break;
        }
        case 'PropertyStore': {
            value = `PropertyStore ${printPlace(instrValue.object)}.${instrValue.property} = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'PropertyDelete': {
            value = `PropertyDelete ${printPlace(instrValue.object)}.${instrValue.property}`;
            break;
        }
        case 'ComputedLoad': {
            value = `ComputedLoad ${printPlace(instrValue.object)}[${printPlace(instrValue.property)}]`;
            break;
        }
        case 'ComputedStore': {
            value = `ComputedStore ${printPlace(instrValue.object)}[${printPlace(instrValue.property)}] = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'ComputedDelete': {
            value = `ComputedDelete ${printPlace(instrValue.object)}[${printPlace(instrValue.property)}]`;
            break;
        }
        case 'ObjectMethod':
        case 'FunctionExpression': {
            const kind = instrValue.kind === 'FunctionExpression' ? 'Function' : 'ObjectMethod';
            const name = getFunctionName$2(instrValue, '');
            const fn = printFunction(instrValue.loweredFunc.func)
                .split('\n')
                .map(line => `      ${line}`)
                .join('\n');
            const context = instrValue.loweredFunc.func.context
                .map(dep => printPlace(dep))
                .join(',');
            const aliasingEffects = (_c = (_b = (_a = instrValue.loweredFunc.func.aliasingEffects) === null || _a === void 0 ? void 0 : _a.map(printAliasingEffect)) === null || _b === void 0 ? void 0 : _b.join(', ')) !== null && _c !== void 0 ? _c : '';
            value = `${kind} ${name} @context[${context}] @aliasingEffects=[${aliasingEffects}]\n${fn}`;
            break;
        }
        case 'TaggedTemplateExpression': {
            value = `${printPlace(instrValue.tag)}\`${instrValue.value.raw}\``;
            break;
        }
        case 'LogicalExpression': {
            value = `Logical ${printInstructionValue(instrValue.left)} ${instrValue.operator} ${printInstructionValue(instrValue.right)}`;
            break;
        }
        case 'SequenceExpression': {
            value = [
                `Sequence`,
                ...instrValue.instructions.map(instr => `    ${printInstruction(instr)}`),
                `    ${printInstructionValue(instrValue.value)}`,
            ].join('\n');
            break;
        }
        case 'ConditionalExpression': {
            value = `Ternary ${printInstructionValue(instrValue.test)} ? ${printInstructionValue(instrValue.consequent)} : ${printInstructionValue(instrValue.alternate)}`;
            break;
        }
        case 'TemplateLiteral': {
            value = '`';
            CompilerError.invariant(instrValue.subexprs.length === instrValue.quasis.length - 1, {
                reason: 'Bad assumption about quasi length.',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: instrValue.loc,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            for (let i = 0; i < instrValue.subexprs.length; i++) {
                value += instrValue.quasis[i].raw;
                value += `\${${printPlace(instrValue.subexprs[i])}}`;
            }
            value += instrValue.quasis.at(-1).raw + '`';
            break;
        }
        case 'LoadGlobal': {
            switch (instrValue.binding.kind) {
                case 'Global': {
                    value = `LoadGlobal(global) ${instrValue.binding.name}`;
                    break;
                }
                case 'ModuleLocal': {
                    value = `LoadGlobal(module) ${instrValue.binding.name}`;
                    break;
                }
                case 'ImportDefault': {
                    value = `LoadGlobal import ${instrValue.binding.name} from '${instrValue.binding.module}'`;
                    break;
                }
                case 'ImportNamespace': {
                    value = `LoadGlobal import * as ${instrValue.binding.name} from '${instrValue.binding.module}'`;
                    break;
                }
                case 'ImportSpecifier': {
                    if (instrValue.binding.imported !== instrValue.binding.name) {
                        value = `LoadGlobal import { ${instrValue.binding.imported} as ${instrValue.binding.name} } from '${instrValue.binding.module}'`;
                    }
                    else {
                        value = `LoadGlobal import { ${instrValue.binding.name} } from '${instrValue.binding.module}'`;
                    }
                    break;
                }
                default: {
                    assertExhaustive$1(instrValue.binding, `Unexpected binding kind \`${instrValue.binding.kind}\``);
                }
            }
            break;
        }
        case 'StoreGlobal': {
            value = `StoreGlobal ${instrValue.name} = ${printPlace(instrValue.value)}`;
            break;
        }
        case 'OptionalExpression': {
            value = `OptionalExpression ${printInstructionValue(instrValue.value)}`;
            break;
        }
        case 'RegExpLiteral': {
            value = `RegExp /${instrValue.pattern}/${instrValue.flags}`;
            break;
        }
        case 'MetaProperty': {
            value = `MetaProperty ${instrValue.meta}.${instrValue.property}`;
            break;
        }
        case 'Await': {
            value = `Await ${printPlace(instrValue.value)}`;
            break;
        }
        case 'GetIterator': {
            value = `GetIterator collection=${printPlace(instrValue.collection)}`;
            break;
        }
        case 'IteratorNext': {
            value = `IteratorNext iterator=${printPlace(instrValue.iterator)} collection=${printPlace(instrValue.collection)}`;
            break;
        }
        case 'NextPropertyOf': {
            value = `NextPropertyOf ${printPlace(instrValue.value)}`;
            break;
        }
        case 'Debugger': {
            value = `Debugger`;
            break;
        }
        case 'PostfixUpdate': {
            value = `PostfixUpdate ${printPlace(instrValue.lvalue)} = ${printPlace(instrValue.value)} ${instrValue.operation}`;
            break;
        }
        case 'PrefixUpdate': {
            value = `PrefixUpdate ${printPlace(instrValue.lvalue)} = ${instrValue.operation} ${printPlace(instrValue.value)}`;
            break;
        }
        case 'StartMemoize': {
            value = `StartMemoize deps=${(_e = (_d = instrValue.deps) === null || _d === void 0 ? void 0 : _d.map(dep => printManualMemoDependency(dep, false))) !== null && _e !== void 0 ? _e : '(none)'}`;
            break;
        }
        case 'FinishMemoize': {
            value = `FinishMemoize decl=${printPlace(instrValue.decl)}${instrValue.pruned ? ' pruned' : ''}`;
            break;
        }
        default: {
            assertExhaustive$1(instrValue, `Unexpected instruction kind '${instrValue.kind}'`);
        }
    }
    return value;
}
function isMutable$1(range) {
    return range.end > range.start + 1;
}
function printMutableRange(identifier) {
    var _b, _c;
    const range = (_c = (_b = identifier.scope) === null || _b === void 0 ? void 0 : _b.range) !== null && _c !== void 0 ? _c : identifier.mutableRange;
    return isMutable$1(range) ? `[${range.start}:${range.end}]` : '';
}
function printPattern(pattern) {
    switch (pattern.kind) {
        case 'ArrayPattern': {
            return ('[ ' +
                pattern.items
                    .map(item => {
                    if (item.kind === 'Hole') {
                        return '<hole>';
                    }
                    return printPattern(item);
                })
                    .join(', ') +
                ' ]');
        }
        case 'ObjectPattern': {
            return ('{ ' +
                pattern.properties
                    .map(item => {
                    switch (item.kind) {
                        case 'ObjectProperty': {
                            return `${printObjectPropertyKey(item.key)}: ${printPattern(item.place)}`;
                        }
                        case 'Spread': {
                            return printPattern(item);
                        }
                        default: {
                            assertExhaustive$1(item, 'Unexpected object property kind');
                        }
                    }
                })
                    .join(', ') +
                ' }');
        }
        case 'Spread': {
            return `...${printPlace(pattern.place)}`;
        }
        case 'Identifier': {
            return printPlace(pattern);
        }
        default: {
            assertExhaustive$1(pattern, `Unexpected pattern kind \`${pattern.kind}\``);
        }
    }
}
function printPlace(place) {
    const items = [
        place.effect,
        ' ',
        printIdentifier(place.identifier),
        printMutableRange(place.identifier),
        printType(place.identifier.type),
        place.reactive ? '{reactive}' : null,
    ];
    return items.filter(x => x != null).join('');
}
function printIdentifier(id) {
    return `${printName(id.name)}\$${id.id}${printScope(id.scope)}`;
}
function printName(name) {
    if (name === null) {
        return '';
    }
    return name.value;
}
function printScope(scope) {
    return `${scope !== null ? `_@${scope.id}` : ''}`;
}
function printManualMemoDependency(val, nameOnly) {
    var _a;
    let rootStr;
    if (val.root.kind === 'Global') {
        rootStr = val.root.identifierName;
    }
    else {
        CompilerError.invariant(((_a = val.root.value.identifier.name) === null || _a === void 0 ? void 0 : _a.kind) === 'named', {
            reason: 'DepsValidation: expected named local variable in depslist',
            description: null,
            suggestions: null,
            details: [
                {
                    kind: 'error',
                    loc: val.root.value.loc,
                    message: null,
                },
            ],
        });
        rootStr = nameOnly
            ? val.root.value.identifier.name.value
            : printIdentifier(val.root.value.identifier);
    }
    return `${rootStr}${val.path.map(v => `${v.optional ? '?.' : '.'}${v.property}`).join('')}`;
}
function printType(type) {
    if (type.kind === 'Type')
        return '';
    if (type.kind === 'Object' && type.shapeId != null) {
        return `:T${type.kind}<${type.shapeId}>`;
    }
    else if (type.kind === 'Function' && type.shapeId != null) {
        const returnType = printType(type.return);
        return `:T${type.kind}<${type.shapeId}>()${returnType !== '' ? `:  ${returnType}` : ''}`;
    }
    else {
        return `:T${type.kind}`;
    }
}
function printSourceLocation(loc) {
    if (typeof loc === 'symbol') {
        return 'generated';
    }
    else {
        return `${loc.start.line}:${loc.start.column}:${loc.end.line}:${loc.end.column}`;
    }
}
function printSourceLocationLine(loc) {
    if (typeof loc === 'symbol') {
        return 'generated';
    }
    else {
        return `${loc.start.line}:${loc.end.line}`;
    }
}
function getFunctionName$2(instrValue, defaultValue) {
    var _a;
    switch (instrValue.kind) {
        case 'FunctionExpression':
            return (_a = instrValue.name) !== null && _a !== void 0 ? _a : defaultValue;
        case 'ObjectMethod':
            return defaultValue;
    }
}
function printAliasingEffect(effect) {
    var _a;
    switch (effect.kind) {
        case 'Assign': {
            return `Assign ${printPlaceForAliasEffect(effect.into)} = ${printPlaceForAliasEffect(effect.from)}`;
        }
        case 'Alias': {
            return `Alias ${printPlaceForAliasEffect(effect.into)} <- ${printPlaceForAliasEffect(effect.from)}`;
        }
        case 'MaybeAlias': {
            return `MaybeAlias ${printPlaceForAliasEffect(effect.into)} <- ${printPlaceForAliasEffect(effect.from)}`;
        }
        case 'Capture': {
            return `Capture ${printPlaceForAliasEffect(effect.into)} <- ${printPlaceForAliasEffect(effect.from)}`;
        }
        case 'ImmutableCapture': {
            return `ImmutableCapture ${printPlaceForAliasEffect(effect.into)} <- ${printPlaceForAliasEffect(effect.from)}`;
        }
        case 'Create': {
            return `Create ${printPlaceForAliasEffect(effect.into)} = ${effect.value}`;
        }
        case 'CreateFrom': {
            return `Create ${printPlaceForAliasEffect(effect.into)} = kindOf(${printPlaceForAliasEffect(effect.from)})`;
        }
        case 'CreateFunction': {
            return `Function ${printPlaceForAliasEffect(effect.into)} = Function captures=[${effect.captures.map(printPlaceForAliasEffect).join(', ')}]`;
        }
        case 'Apply': {
            const receiverCallee = effect.receiver.identifier.id === effect.function.identifier.id
                ? printPlaceForAliasEffect(effect.receiver)
                : `${printPlaceForAliasEffect(effect.receiver)}.${printPlaceForAliasEffect(effect.function)}`;
            const args = effect.args
                .map(arg => {
                if (arg.kind === 'Identifier') {
                    return printPlaceForAliasEffect(arg);
                }
                else if (arg.kind === 'Hole') {
                    return ' ';
                }
                return `...${printPlaceForAliasEffect(arg.place)}`;
            })
                .join(', ');
            let signature = '';
            if (effect.signature != null) {
                if (effect.signature.aliasing != null) {
                    signature = printAliasingSignature(effect.signature.aliasing);
                }
                else {
                    signature = JSON.stringify(effect.signature, null, 2);
                }
            }
            return `Apply ${printPlaceForAliasEffect(effect.into)} = ${receiverCallee}(${args})${signature != '' ? '\n     ' : ''}${signature}`;
        }
        case 'Freeze': {
            return `Freeze ${printPlaceForAliasEffect(effect.value)} ${effect.reason}`;
        }
        case 'Mutate':
        case 'MutateConditionally':
        case 'MutateTransitive':
        case 'MutateTransitiveConditionally': {
            return `${effect.kind} ${printPlaceForAliasEffect(effect.value)}${effect.kind === 'Mutate' && ((_a = effect.reason) === null || _a === void 0 ? void 0 : _a.kind) === 'AssignCurrentProperty' ? ' (assign `.current`)' : ''}`;
        }
        case 'MutateFrozen': {
            return `MutateFrozen ${printPlaceForAliasEffect(effect.place)} reason=${JSON.stringify(effect.error.reason)}`;
        }
        case 'MutateGlobal': {
            return `MutateGlobal ${printPlaceForAliasEffect(effect.place)} reason=${JSON.stringify(effect.error.reason)}`;
        }
        case 'Impure': {
            return `Impure ${printPlaceForAliasEffect(effect.place)} reason=${JSON.stringify(effect.error.reason)}`;
        }
        case 'Render': {
            return `Render ${printPlaceForAliasEffect(effect.place)}`;
        }
        default: {
            assertExhaustive$1(effect, `Unexpected kind '${effect.kind}'`);
        }
    }
}
function printPlaceForAliasEffect(place) {
    return printIdentifier(place.identifier);
}
function printAliasingSignature(signature) {
    const tokens = ['function '];
    if (signature.temporaries.length !== 0) {
        tokens.push('<');
        tokens.push(signature.temporaries.map(temp => `$${temp.identifier.id}`).join(', '));
        tokens.push('>');
    }
    tokens.push('(');
    tokens.push('this=$' + String(signature.receiver));
    for (const param of signature.params) {
        tokens.push(', $' + String(param));
    }
    if (signature.rest != null) {
        tokens.push(`, ...$${String(signature.rest)}`);
    }
    tokens.push('): ');
    tokens.push('$' + String(signature.returns) + ':');
    for (const effect of signature.effects) {
        tokens.push('\n  ' + printAliasingEffect(effect));
    }
    return tokens.join('');
}

var _ScopeBlockTraversal_activeScopes;
function* eachInstructionLValue(instr) {
    if (instr.lvalue !== null) {
        yield instr.lvalue;
    }
    yield* eachInstructionValueLValue(instr.value);
}
function* eachInstructionValueLValue(value) {
    switch (value.kind) {
        case 'DeclareContext':
        case 'StoreContext':
        case 'DeclareLocal':
        case 'StoreLocal': {
            yield value.lvalue.place;
            break;
        }
        case 'Destructure': {
            yield* eachPatternOperand(value.lvalue.pattern);
            break;
        }
        case 'PostfixUpdate':
        case 'PrefixUpdate': {
            yield value.lvalue;
            break;
        }
    }
}
function* eachInstructionOperand(instr) {
    yield* eachInstructionValueOperand(instr.value);
}
function* eachInstructionValueOperand(instrValue) {
    switch (instrValue.kind) {
        case 'NewExpression':
        case 'CallExpression': {
            yield instrValue.callee;
            yield* eachCallArgument(instrValue.args);
            break;
        }
        case 'BinaryExpression': {
            yield instrValue.left;
            yield instrValue.right;
            break;
        }
        case 'MethodCall': {
            yield instrValue.receiver;
            yield instrValue.property;
            yield* eachCallArgument(instrValue.args);
            break;
        }
        case 'DeclareContext':
        case 'DeclareLocal': {
            break;
        }
        case 'LoadLocal':
        case 'LoadContext': {
            yield instrValue.place;
            break;
        }
        case 'StoreLocal': {
            yield instrValue.value;
            break;
        }
        case 'StoreContext': {
            yield instrValue.lvalue.place;
            yield instrValue.value;
            break;
        }
        case 'StoreGlobal': {
            yield instrValue.value;
            break;
        }
        case 'Destructure': {
            yield instrValue.value;
            break;
        }
        case 'PropertyLoad': {
            yield instrValue.object;
            break;
        }
        case 'PropertyDelete': {
            yield instrValue.object;
            break;
        }
        case 'PropertyStore': {
            yield instrValue.object;
            yield instrValue.value;
            break;
        }
        case 'ComputedLoad': {
            yield instrValue.object;
            yield instrValue.property;
            break;
        }
        case 'ComputedDelete': {
            yield instrValue.object;
            yield instrValue.property;
            break;
        }
        case 'ComputedStore': {
            yield instrValue.object;
            yield instrValue.property;
            yield instrValue.value;
            break;
        }
        case 'UnaryExpression': {
            yield instrValue.value;
            break;
        }
        case 'JsxExpression': {
            if (instrValue.tag.kind === 'Identifier') {
                yield instrValue.tag;
            }
            for (const attribute of instrValue.props) {
                switch (attribute.kind) {
                    case 'JsxAttribute': {
                        yield attribute.place;
                        break;
                    }
                    case 'JsxSpreadAttribute': {
                        yield attribute.argument;
                        break;
                    }
                    default: {
                        assertExhaustive$1(attribute, `Unexpected attribute kind \`${attribute.kind}\``);
                    }
                }
            }
            if (instrValue.children) {
                yield* instrValue.children;
            }
            break;
        }
        case 'JsxFragment': {
            yield* instrValue.children;
            break;
        }
        case 'ObjectExpression': {
            for (const property of instrValue.properties) {
                if (property.kind === 'ObjectProperty' &&
                    property.key.kind === 'computed') {
                    yield property.key.name;
                }
                yield property.place;
            }
            break;
        }
        case 'ArrayExpression': {
            for (const element of instrValue.elements) {
                if (element.kind === 'Identifier') {
                    yield element;
                }
                else if (element.kind === 'Spread') {
                    yield element.place;
                }
            }
            break;
        }
        case 'ObjectMethod':
        case 'FunctionExpression': {
            yield* instrValue.loweredFunc.func.context;
            break;
        }
        case 'TaggedTemplateExpression': {
            yield instrValue.tag;
            break;
        }
        case 'TypeCastExpression': {
            yield instrValue.value;
            break;
        }
        case 'TemplateLiteral': {
            yield* instrValue.subexprs;
            break;
        }
        case 'Await': {
            yield instrValue.value;
            break;
        }
        case 'GetIterator': {
            yield instrValue.collection;
            break;
        }
        case 'IteratorNext': {
            yield instrValue.iterator;
            yield instrValue.collection;
            break;
        }
        case 'NextPropertyOf': {
            yield instrValue.value;
            break;
        }
        case 'PostfixUpdate':
        case 'PrefixUpdate': {
            yield instrValue.value;
            break;
        }
        case 'StartMemoize': {
            if (instrValue.deps != null) {
                for (const dep of instrValue.deps) {
                    if (dep.root.kind === 'NamedLocal') {
                        yield dep.root.value;
                    }
                }
            }
            break;
        }
        case 'FinishMemoize': {
            yield instrValue.decl;
            break;
        }
        case 'Debugger':
        case 'RegExpLiteral':
        case 'MetaProperty':
        case 'LoadGlobal':
        case 'UnsupportedNode':
        case 'Primitive':
        case 'JSXText': {
            break;
        }
        default: {
            assertExhaustive$1(instrValue, `Unexpected instruction kind \`${instrValue.kind}\``);
        }
    }
}
function* eachCallArgument(args) {
    for (const arg of args) {
        if (arg.kind === 'Identifier') {
            yield arg;
        }
        else {
            yield arg.place;
        }
    }
}
function doesPatternContainSpreadElement(pattern) {
    switch (pattern.kind) {
        case 'ArrayPattern': {
            for (const item of pattern.items) {
                if (item.kind === 'Spread') {
                    return true;
                }
            }
            break;
        }
        case 'ObjectPattern': {
            for (const property of pattern.properties) {
                if (property.kind === 'Spread') {
                    return true;
                }
            }
            break;
        }
        default: {
            assertExhaustive$1(pattern, `Unexpected pattern kind \`${pattern.kind}\``);
        }
    }
    return false;
}
function* eachPatternOperand(pattern) {
    switch (pattern.kind) {
        case 'ArrayPattern': {
            for (const item of pattern.items) {
                if (item.kind === 'Identifier') {
                    yield item;
                }
                else if (item.kind === 'Spread') {
                    yield item.place;
                }
                else if (item.kind === 'Hole') {
                    continue;
                }
                else {
                    assertExhaustive$1(item, `Unexpected item kind \`${item.kind}\``);
                }
            }
            break;
        }
        case 'ObjectPattern': {
            for (const property of pattern.properties) {
                if (property.kind === 'ObjectProperty') {
                    yield property.place;
                }
                else if (property.kind === 'Spread') {
                    yield property.place;
                }
                else {
                    assertExhaustive$1(property, `Unexpected item kind \`${property.kind}\``);
                }
            }
            break;
        }
        default: {
            assertExhaustive$1(pattern, `Unexpected pattern kind \`${pattern.kind}\``);
        }
    }
}
function* eachPatternItem(pattern) {
    switch (pattern.kind) {
        case 'ArrayPattern': {
            for (const item of pattern.items) {
                if (item.kind === 'Identifier') {
                    yield item;
                }
                else if (item.kind === 'Spread') {
                    yield item;
                }
                else if (item.kind === 'Hole') {
                    continue;
                }
                else {
                    assertExhaustive$1(item, `Unexpected item kind \`${item.kind}\``);
                }
            }
            break;
        }
        case 'ObjectPattern': {
            for (const property of pattern.properties) {
                if (property.kind === 'ObjectProperty') {
                    yield property.place;
                }
                else if (property.kind === 'Spread') {
                    yield property;
                }
                else {
                    assertExhaustive$1(property, `Unexpected item kind \`${property.kind}\``);
                }
            }
            break;
        }
        default: {
            assertExhaustive$1(pattern, `Unexpected pattern kind \`${pattern.kind}\``);
        }
    }
}
function mapInstructionLValues(instr, fn) {
    switch (instr.value.kind) {
        case 'DeclareLocal':
        case 'StoreLocal': {
            const lvalue = instr.value.lvalue;
            lvalue.place = fn(lvalue.place);
            break;
        }
        case 'Destructure': {
            mapPatternOperands(instr.value.lvalue.pattern, fn);
            break;
        }
        case 'PostfixUpdate':
        case 'PrefixUpdate': {
            instr.value.lvalue = fn(instr.value.lvalue);
            break;
        }
    }
    if (instr.lvalue !== null) {
        instr.lvalue = fn(instr.lvalue);
    }
}
function mapInstructionOperands(instr, fn) {
    mapInstructionValueOperands(instr.value, fn);
}
function mapInstructionValueOperands(instrValue, fn) {
    switch (instrValue.kind) {
        case 'BinaryExpression': {
            instrValue.left = fn(instrValue.left);
            instrValue.right = fn(instrValue.right);
            break;
        }
        case 'PropertyLoad': {
            instrValue.object = fn(instrValue.object);
            break;
        }
        case 'PropertyDelete': {
            instrValue.object = fn(instrValue.object);
            break;
        }
        case 'PropertyStore': {
            instrValue.object = fn(instrValue.object);
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'ComputedLoad': {
            instrValue.object = fn(instrValue.object);
            instrValue.property = fn(instrValue.property);
            break;
        }
        case 'ComputedDelete': {
            instrValue.object = fn(instrValue.object);
            instrValue.property = fn(instrValue.property);
            break;
        }
        case 'ComputedStore': {
            instrValue.object = fn(instrValue.object);
            instrValue.property = fn(instrValue.property);
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'DeclareContext':
        case 'DeclareLocal': {
            break;
        }
        case 'LoadLocal':
        case 'LoadContext': {
            instrValue.place = fn(instrValue.place);
            break;
        }
        case 'StoreLocal': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'StoreContext': {
            instrValue.lvalue.place = fn(instrValue.lvalue.place);
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'StoreGlobal': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'Destructure': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'NewExpression':
        case 'CallExpression': {
            instrValue.callee = fn(instrValue.callee);
            instrValue.args = mapCallArguments(instrValue.args, fn);
            break;
        }
        case 'MethodCall': {
            instrValue.receiver = fn(instrValue.receiver);
            instrValue.property = fn(instrValue.property);
            instrValue.args = mapCallArguments(instrValue.args, fn);
            break;
        }
        case 'UnaryExpression': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'JsxExpression': {
            if (instrValue.tag.kind === 'Identifier') {
                instrValue.tag = fn(instrValue.tag);
            }
            for (const attribute of instrValue.props) {
                switch (attribute.kind) {
                    case 'JsxAttribute': {
                        attribute.place = fn(attribute.place);
                        break;
                    }
                    case 'JsxSpreadAttribute': {
                        attribute.argument = fn(attribute.argument);
                        break;
                    }
                    default: {
                        assertExhaustive$1(attribute, `Unexpected attribute kind \`${attribute.kind}\``);
                    }
                }
            }
            if (instrValue.children) {
                instrValue.children = instrValue.children.map(p => fn(p));
            }
            break;
        }
        case 'ObjectExpression': {
            for (const property of instrValue.properties) {
                if (property.kind === 'ObjectProperty' &&
                    property.key.kind === 'computed') {
                    property.key.name = fn(property.key.name);
                }
                property.place = fn(property.place);
            }
            break;
        }
        case 'ArrayExpression': {
            instrValue.elements = instrValue.elements.map(element => {
                if (element.kind === 'Identifier') {
                    return fn(element);
                }
                else if (element.kind === 'Spread') {
                    element.place = fn(element.place);
                    return element;
                }
                else {
                    return element;
                }
            });
            break;
        }
        case 'JsxFragment': {
            instrValue.children = instrValue.children.map(e => fn(e));
            break;
        }
        case 'ObjectMethod':
        case 'FunctionExpression': {
            instrValue.loweredFunc.func.context =
                instrValue.loweredFunc.func.context.map(d => fn(d));
            break;
        }
        case 'TaggedTemplateExpression': {
            instrValue.tag = fn(instrValue.tag);
            break;
        }
        case 'TypeCastExpression': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'TemplateLiteral': {
            instrValue.subexprs = instrValue.subexprs.map(fn);
            break;
        }
        case 'Await': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'GetIterator': {
            instrValue.collection = fn(instrValue.collection);
            break;
        }
        case 'IteratorNext': {
            instrValue.iterator = fn(instrValue.iterator);
            instrValue.collection = fn(instrValue.collection);
            break;
        }
        case 'NextPropertyOf': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'PostfixUpdate':
        case 'PrefixUpdate': {
            instrValue.value = fn(instrValue.value);
            break;
        }
        case 'StartMemoize': {
            if (instrValue.deps != null) {
                for (const dep of instrValue.deps) {
                    if (dep.root.kind === 'NamedLocal') {
                        dep.root.value = fn(dep.root.value);
                    }
                }
            }
            break;
        }
        case 'FinishMemoize': {
            instrValue.decl = fn(instrValue.decl);
            break;
        }
        case 'Debugger':
        case 'RegExpLiteral':
        case 'MetaProperty':
        case 'LoadGlobal':
        case 'UnsupportedNode':
        case 'Primitive':
        case 'JSXText': {
            break;
        }
        default: {
            assertExhaustive$1(instrValue, 'Unexpected instruction kind');
        }
    }
}
function mapCallArguments(args, fn) {
    return args.map(arg => {
        if (arg.kind === 'Identifier') {
            return fn(arg);
        }
        else {
            arg.place = fn(arg.place);
            return arg;
        }
    });
}
function mapPatternOperands(pattern, fn) {
    switch (pattern.kind) {
        case 'ArrayPattern': {
            pattern.items = pattern.items.map(item => {
                if (item.kind === 'Identifier') {
                    return fn(item);
                }
                else if (item.kind === 'Spread') {
                    item.place = fn(item.place);
                    return item;
                }
                else {
                    return item;
                }
            });
            break;
        }
        case 'ObjectPattern': {
            for (const property of pattern.properties) {
                property.place = fn(property.place);
            }
            break;
        }
        default: {
            assertExhaustive$1(pattern, `Unexpected pattern kind \`${pattern.kind}\``);
        }
    }
}
function mapTerminalSuccessors(terminal, fn) {
    switch (terminal.kind) {
        case 'goto': {
            const target = fn(terminal.block);
            return {
                kind: 'goto',
                block: target,
                variant: terminal.variant,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'if': {
            const consequent = fn(terminal.consequent);
            const alternate = fn(terminal.alternate);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'if',
                test: terminal.test,
                consequent,
                alternate,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'branch': {
            const consequent = fn(terminal.consequent);
            const alternate = fn(terminal.alternate);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'branch',
                test: terminal.test,
                consequent,
                alternate,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'switch': {
            const cases = terminal.cases.map(case_ => {
                const target = fn(case_.block);
                return {
                    test: case_.test,
                    block: target,
                };
            });
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'switch',
                test: terminal.test,
                cases,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'logical': {
            const test = fn(terminal.test);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'logical',
                test,
                fallthrough,
                operator: terminal.operator,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'ternary': {
            const test = fn(terminal.test);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'ternary',
                test,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'optional': {
            const test = fn(terminal.test);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'optional',
                optional: terminal.optional,
                test,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'return': {
            return {
                kind: 'return',
                returnVariant: terminal.returnVariant,
                loc: terminal.loc,
                value: terminal.value,
                id: makeInstructionId(0),
                effects: terminal.effects,
            };
        }
        case 'throw': {
            return terminal;
        }
        case 'do-while': {
            const loop = fn(terminal.loop);
            const test = fn(terminal.test);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'do-while',
                loc: terminal.loc,
                test,
                loop,
                fallthrough,
                id: makeInstructionId(0),
            };
        }
        case 'while': {
            const test = fn(terminal.test);
            const loop = fn(terminal.loop);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'while',
                loc: terminal.loc,
                test,
                loop,
                fallthrough,
                id: makeInstructionId(0),
            };
        }
        case 'for': {
            const init = fn(terminal.init);
            const test = fn(terminal.test);
            const update = terminal.update !== null ? fn(terminal.update) : null;
            const loop = fn(terminal.loop);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'for',
                loc: terminal.loc,
                init,
                test,
                update,
                loop,
                fallthrough,
                id: makeInstructionId(0),
            };
        }
        case 'for-of': {
            const init = fn(terminal.init);
            const loop = fn(terminal.loop);
            const test = fn(terminal.test);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'for-of',
                loc: terminal.loc,
                init,
                test,
                loop,
                fallthrough,
                id: makeInstructionId(0),
            };
        }
        case 'for-in': {
            const init = fn(terminal.init);
            const loop = fn(terminal.loop);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'for-in',
                loc: terminal.loc,
                init,
                loop,
                fallthrough,
                id: makeInstructionId(0),
            };
        }
        case 'label': {
            const block = fn(terminal.block);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'label',
                block,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'sequence': {
            const block = fn(terminal.block);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'sequence',
                block,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'maybe-throw': {
            const continuation = fn(terminal.continuation);
            const handler = fn(terminal.handler);
            return {
                kind: 'maybe-throw',
                continuation,
                handler,
                id: makeInstructionId(0),
                loc: terminal.loc,
                effects: terminal.effects,
            };
        }
        case 'try': {
            const block = fn(terminal.block);
            const handler = fn(terminal.handler);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: 'try',
                block,
                handlerBinding: terminal.handlerBinding,
                handler,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'scope':
        case 'pruned-scope': {
            const block = fn(terminal.block);
            const fallthrough = fn(terminal.fallthrough);
            return {
                kind: terminal.kind,
                scope: terminal.scope,
                block,
                fallthrough,
                id: makeInstructionId(0),
                loc: terminal.loc,
            };
        }
        case 'unreachable':
        case 'unsupported': {
            return terminal;
        }
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal.kind}\``);
        }
    }
}
function terminalHasFallthrough(terminal) {
    switch (terminal.kind) {
        case 'maybe-throw':
        case 'goto':
        case 'return':
        case 'throw':
        case 'unreachable':
        case 'unsupported': {
            return false;
        }
        case 'branch':
        case 'try':
        case 'do-while':
        case 'for-of':
        case 'for-in':
        case 'for':
        case 'if':
        case 'label':
        case 'logical':
        case 'optional':
        case 'sequence':
        case 'switch':
        case 'ternary':
        case 'while':
        case 'scope':
        case 'pruned-scope': {
            return true;
        }
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal.kind}\``);
        }
    }
}
function terminalFallthrough(terminal) {
    if (terminalHasFallthrough(terminal)) {
        return terminal.fallthrough;
    }
    else {
        return null;
    }
}
function* eachTerminalSuccessor(terminal) {
    switch (terminal.kind) {
        case 'goto': {
            yield terminal.block;
            break;
        }
        case 'if': {
            yield terminal.consequent;
            yield terminal.alternate;
            break;
        }
        case 'branch': {
            yield terminal.consequent;
            yield terminal.alternate;
            break;
        }
        case 'switch': {
            for (const case_ of terminal.cases) {
                yield case_.block;
            }
            break;
        }
        case 'optional':
        case 'ternary':
        case 'logical': {
            yield terminal.test;
            break;
        }
        case 'return': {
            break;
        }
        case 'throw': {
            break;
        }
        case 'do-while': {
            yield terminal.loop;
            break;
        }
        case 'while': {
            yield terminal.test;
            break;
        }
        case 'for': {
            yield terminal.init;
            break;
        }
        case 'for-of': {
            yield terminal.init;
            break;
        }
        case 'for-in': {
            yield terminal.init;
            break;
        }
        case 'label': {
            yield terminal.block;
            break;
        }
        case 'sequence': {
            yield terminal.block;
            break;
        }
        case 'maybe-throw': {
            yield terminal.continuation;
            yield terminal.handler;
            break;
        }
        case 'try': {
            yield terminal.block;
            break;
        }
        case 'scope':
        case 'pruned-scope': {
            yield terminal.block;
            break;
        }
        case 'unreachable':
        case 'unsupported':
            break;
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal.kind}\``);
        }
    }
}
function mapTerminalOperands(terminal, fn) {
    switch (terminal.kind) {
        case 'if': {
            terminal.test = fn(terminal.test);
            break;
        }
        case 'branch': {
            terminal.test = fn(terminal.test);
            break;
        }
        case 'switch': {
            terminal.test = fn(terminal.test);
            for (const case_ of terminal.cases) {
                if (case_.test === null) {
                    continue;
                }
                case_.test = fn(case_.test);
            }
            break;
        }
        case 'return':
        case 'throw': {
            terminal.value = fn(terminal.value);
            break;
        }
        case 'try': {
            if (terminal.handlerBinding !== null) {
                terminal.handlerBinding = fn(terminal.handlerBinding);
            }
            else {
                terminal.handlerBinding = null;
            }
            break;
        }
        case 'maybe-throw':
        case 'sequence':
        case 'label':
        case 'optional':
        case 'ternary':
        case 'logical':
        case 'do-while':
        case 'while':
        case 'for':
        case 'for-of':
        case 'for-in':
        case 'goto':
        case 'unreachable':
        case 'unsupported':
        case 'scope':
        case 'pruned-scope': {
            break;
        }
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal.kind}\``);
        }
    }
}
function* eachTerminalOperand(terminal) {
    switch (terminal.kind) {
        case 'if': {
            yield terminal.test;
            break;
        }
        case 'branch': {
            yield terminal.test;
            break;
        }
        case 'switch': {
            yield terminal.test;
            for (const case_ of terminal.cases) {
                if (case_.test === null) {
                    continue;
                }
                yield case_.test;
            }
            break;
        }
        case 'return':
        case 'throw': {
            yield terminal.value;
            break;
        }
        case 'try': {
            if (terminal.handlerBinding !== null) {
                yield terminal.handlerBinding;
            }
            break;
        }
        case 'maybe-throw':
        case 'sequence':
        case 'label':
        case 'optional':
        case 'ternary':
        case 'logical':
        case 'do-while':
        case 'while':
        case 'for':
        case 'for-of':
        case 'for-in':
        case 'goto':
        case 'unreachable':
        case 'unsupported':
        case 'scope':
        case 'pruned-scope': {
            break;
        }
        default: {
            assertExhaustive$1(terminal, `Unexpected terminal kind \`${terminal.kind}\``);
        }
    }
}
class ScopeBlockTraversal {
    constructor() {
        _ScopeBlockTraversal_activeScopes.set(this, []);
        this.blockInfos = new Map();
    }
    recordScopes(block) {
        var _a, _b;
        const blockInfo = this.blockInfos.get(block.id);
        if ((blockInfo === null || blockInfo === void 0 ? void 0 : blockInfo.kind) === 'begin') {
            __classPrivateFieldGet(this, _ScopeBlockTraversal_activeScopes, "f").push(blockInfo.scope.id);
        }
        else if ((blockInfo === null || blockInfo === void 0 ? void 0 : blockInfo.kind) === 'end') {
            const top = __classPrivateFieldGet(this, _ScopeBlockTraversal_activeScopes, "f").at(-1);
            CompilerError.invariant(blockInfo.scope.id === top, {
                reason: 'Expected traversed block fallthrough to match top-most active scope',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: (_b = (_a = block.instructions[0]) === null || _a === void 0 ? void 0 : _a.loc) !== null && _b !== void 0 ? _b : block.terminal.id,
                        message: null,
                    },
                ],
            });
            __classPrivateFieldGet(this, _ScopeBlockTraversal_activeScopes, "f").pop();
        }
        if (block.terminal.kind === 'scope' ||
            block.terminal.kind === 'pruned-scope') {
            CompilerError.invariant(!this.blockInfos.has(block.terminal.block) &&
                !this.blockInfos.has(block.terminal.fallthrough), {
                reason: 'Expected unique scope blocks and fallthroughs',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: block.terminal.loc,
                        message: null,
                    },
                ],
            });
            this.blockInfos.set(block.terminal.block, {
                kind: 'begin',
                scope: block.terminal.scope,
                pruned: block.terminal.kind === 'pruned-scope',
                fallthrough: block.terminal.fallthrough,
            });
            this.blockInfos.set(block.terminal.fallthrough, {
                kind: 'end',
                scope: block.terminal.scope,
                pruned: block.terminal.kind === 'pruned-scope',
            });
        }
    }
    isScopeActive(scopeId) {
        return __classPrivateFieldGet(this, _ScopeBlockTraversal_activeScopes, "f").indexOf(scopeId) !== -1;
    }
    get currentScope() {
        var _a;
        return (_a = __classPrivateFieldGet(this, _ScopeBlockTraversal_activeScopes, "f").at(-1)) !== null && _a !== void 0 ? _a : null;
    }
}
_ScopeBlockTraversal_activeScopes = new WeakMap();

function assertConsistentIdentifiers(fn) {
    const identifiers = new Map();
    const assignments = new Set();
    for (const [, block] of fn.body.blocks) {
        for (const phi of block.phis) {
            validate(identifiers, phi.place.identifier);
            for (const [, operand] of phi.operands) {
                validate(identifiers, operand.identifier);
            }
        }
        for (const instr of block.instructions) {
            CompilerError.invariant(instr.lvalue.identifier.name === null, {
                reason: `Expected all lvalues to be temporaries`,
                description: `Found named lvalue \`${instr.lvalue.identifier.name}\``,
                details: [
                    {
                        kind: 'error',
                        loc: instr.lvalue.loc,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            CompilerError.invariant(!assignments.has(instr.lvalue.identifier.id), {
                reason: `Expected lvalues to be assigned exactly once`,
                description: `Found duplicate assignment of '${printPlace(instr.lvalue)}'`,
                details: [
                    {
                        kind: 'error',
                        loc: instr.lvalue.loc,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            assignments.add(instr.lvalue.identifier.id);
            for (const operand of eachInstructionLValue(instr)) {
                validate(identifiers, operand.identifier, operand.loc);
            }
            for (const operand of eachInstructionValueOperand(instr.value)) {
                validate(identifiers, operand.identifier, operand.loc);
            }
        }
        for (const operand of eachTerminalOperand(block.terminal)) {
            validate(identifiers, operand.identifier, operand.loc);
        }
    }
}
function validate(identifiers, identifier, loc = null) {
    const previous = identifiers.get(identifier.id);
    if (previous === undefined) {
        identifiers.set(identifier.id, identifier);
    }
    else {
        CompilerError.invariant(identifier === previous, {
            reason: `Duplicate identifier object`,
            description: `Found duplicate identifier object for id ${identifier.id}`,
            details: [
                {
                    kind: 'error',
                    loc: loc !== null && loc !== void 0 ? loc : GeneratedSource,
                    message: null,
                },
            ],
            suggestions: null,
        });
    }
}

function assertTerminalSuccessorsExist(fn) {
    for (const [, block] of fn.body.blocks) {
        mapTerminalSuccessors(block.terminal, successor => {
            var _a;
            CompilerError.invariant(fn.body.blocks.has(successor), {
                reason: `Terminal successor references unknown block`,
                description: `Block bb${successor} does not exist for terminal '${printTerminal(block.terminal)}'`,
                details: [
                    {
                        kind: 'error',
                        loc: (_a = block.terminal.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            return successor;
        });
    }
}
function assertTerminalPredsExist(fn) {
    for (const [, block] of fn.body.blocks) {
        for (const pred of block.preds) {
            const predBlock = fn.body.blocks.get(pred);
            CompilerError.invariant(predBlock != null, {
                reason: 'Expected predecessor block to exist',
                description: `Block ${block.id} references non-existent ${pred}`,
                details: [
                    {
                        kind: 'error',
                        loc: GeneratedSource,
                        message: null,
                    },
                ],
            });
            CompilerError.invariant([...eachTerminalSuccessor(predBlock.terminal)].includes(block.id), {
                reason: 'Terminal successor does not reference correct predecessor',
                description: `Block bb${block.id} has bb${predBlock.id} as a predecessor, but bb${predBlock.id}'s successors do not include bb${block.id}`,
                details: [
                    {
                        kind: 'error',
                        loc: GeneratedSource,
                        message: null,
                    },
                ],
            });
        }
    }
}

function getScopes(fn) {
    const scopes = new Set();
    function visitPlace(place) {
        const scope = place.identifier.scope;
        if (scope != null) {
            if (scope.range.start !== scope.range.end) {
                scopes.add(scope);
            }
        }
    }
    for (const [, block] of fn.body.blocks) {
        for (const instr of block.instructions) {
            for (const operand of eachInstructionLValue(instr)) {
                visitPlace(operand);
            }
            for (const operand of eachInstructionOperand(instr)) {
                visitPlace(operand);
            }
        }
        for (const operand of eachTerminalOperand(block.terminal)) {
            visitPlace(operand);
        }
    }
    return scopes;
}
function rangePreOrderComparator(a, b) {
    const startDiff = a.start - b.start;
    if (startDiff !== 0)
        return startDiff;
    return b.end - a.end;
}
function recursivelyTraverseItems(items, getRange, context, enter, exit) {
    items.sort((a, b) => rangePreOrderComparator(getRange(a), getRange(b)));
    let activeItems = [];
    const ranges = items.map(getRange);
    for (let i = 0; i < items.length; i++) {
        const curr = items[i];
        const currRange = ranges[i];
        for (let i = activeItems.length - 1; i >= 0; i--) {
            const maybeParent = activeItems[i];
            const maybeParentRange = getRange(maybeParent);
            const disjoint = currRange.start >= maybeParentRange.end;
            const nested = currRange.end <= maybeParentRange.end;
            CompilerError.invariant(disjoint || nested, {
                reason: 'Invalid nesting in program blocks or scopes',
                description: `Items overlap but are not nested: ${maybeParentRange.start}:${maybeParentRange.end}(${currRange.start}:${currRange.end})`,
                details: [
                    {
                        kind: 'error',
                        loc: GeneratedSource,
                        message: null,
                    },
                ],
            });
            if (disjoint) {
                exit(maybeParent, context);
                activeItems.length = i;
            }
            else {
                break;
            }
        }
        enter(curr, context);
        activeItems.push(curr);
    }
    let curr = activeItems.pop();
    while (curr != null) {
        exit(curr, context);
        curr = activeItems.pop();
    }
}
const no_op = () => { };
function assertValidBlockNesting(fn) {
    var _a, _b;
    const scopes = getScopes(fn);
    const blocks = [...scopes].map(scope => (Object.assign({ kind: 'Scope', id: scope.id }, scope.range)));
    for (const [, block] of fn.body.blocks) {
        const fallthroughId = terminalFallthrough(block.terminal);
        if (fallthroughId != null) {
            const fallthrough = fn.body.blocks.get(fallthroughId);
            const end = (_b = (_a = fallthrough.instructions[0]) === null || _a === void 0 ? void 0 : _a.id) !== null && _b !== void 0 ? _b : fallthrough.terminal.id;
            blocks.push({
                kind: 'ProgramBlockSubtree',
                id: block.id,
                start: block.terminal.id,
                end,
            });
        }
    }
    recursivelyTraverseItems(blocks, block => block, null, no_op, no_op);
}

function assertValidMutableRanges(fn) {
    for (const [, block] of fn.body.blocks) {
        for (const phi of block.phis) {
            visit$1(phi.place, `phi for block bb${block.id}`);
            for (const [pred, operand] of phi.operands) {
                visit$1(operand, `phi predecessor bb${pred} for block bb${block.id}`);
            }
        }
        for (const instr of block.instructions) {
            for (const operand of eachInstructionLValue(instr)) {
                visit$1(operand, `instruction [${instr.id}]`);
            }
            for (const operand of eachInstructionOperand(instr)) {
                visit$1(operand, `instruction [${instr.id}]`);
            }
        }
        for (const operand of eachTerminalOperand(block.terminal)) {
            visit$1(operand, `terminal [${block.terminal.id}]`);
        }
    }
}
function visit$1(place, description) {
    validateMutableRange(place, place.identifier.mutableRange, description);
    if (place.identifier.scope !== null) {
        validateMutableRange(place, place.identifier.scope.range, description);
    }
}
function validateMutableRange(place, range, description) {
    CompilerError.invariant((range.start === 0 && range.end === 0) || range.end > range.start, {
        reason: `Invalid mutable range: [${range.start}:${range.end}]`,
        description: `${printPlace(place)} in ${description}`,
        details: [
            {
                kind: 'error',
                loc: place.loc,
                message: null,
            },
        ],
    });
}

var _HIRBuilder_instances, _HIRBuilder_completed, _HIRBuilder_current, _HIRBuilder_entry, _HIRBuilder_scopes, _HIRBuilder_context, _HIRBuilder_bindings, _HIRBuilder_env, _HIRBuilder_exceptionHandlerStack, _HIRBuilder_resolveBabelBinding;
function newBlock(id, kind) {
    return { id, kind, instructions: [] };
}
class HIRBuilder {
    get nextIdentifierId() {
        return __classPrivateFieldGet(this, _HIRBuilder_env, "f").nextIdentifierId;
    }
    get context() {
        return __classPrivateFieldGet(this, _HIRBuilder_context, "f");
    }
    get bindings() {
        return __classPrivateFieldGet(this, _HIRBuilder_bindings, "f");
    }
    get environment() {
        return __classPrivateFieldGet(this, _HIRBuilder_env, "f");
    }
    constructor(env, options) {
        var _a, _b, _c;
        _HIRBuilder_instances.add(this);
        _HIRBuilder_completed.set(this, new Map());
        _HIRBuilder_current.set(this, void 0);
        _HIRBuilder_entry.set(this, void 0);
        _HIRBuilder_scopes.set(this, []);
        _HIRBuilder_context.set(this, void 0);
        _HIRBuilder_bindings.set(this, void 0);
        _HIRBuilder_env.set(this, void 0);
        _HIRBuilder_exceptionHandlerStack.set(this, []);
        this.errors = new CompilerError();
        this.fbtDepth = 0;
        __classPrivateFieldSet(this, _HIRBuilder_env, env, "f");
        __classPrivateFieldSet(this, _HIRBuilder_bindings, (_a = options === null || options === void 0 ? void 0 : options.bindings) !== null && _a !== void 0 ? _a : new Map(), "f");
        __classPrivateFieldSet(this, _HIRBuilder_context, (_b = options === null || options === void 0 ? void 0 : options.context) !== null && _b !== void 0 ? _b : new Map(), "f");
        __classPrivateFieldSet(this, _HIRBuilder_entry, makeBlockId(env.nextBlockId), "f");
        __classPrivateFieldSet(this, _HIRBuilder_current, newBlock(__classPrivateFieldGet(this, _HIRBuilder_entry, "f"), (_c = options === null || options === void 0 ? void 0 : options.entryBlockKind) !== null && _c !== void 0 ? _c : 'block'), "f");
    }
    currentBlockKind() {
        return __classPrivateFieldGet(this, _HIRBuilder_current, "f").kind;
    }
    push(instruction) {
        __classPrivateFieldGet(this, _HIRBuilder_current, "f").instructions.push(instruction);
        const exceptionHandler = __classPrivateFieldGet(this, _HIRBuilder_exceptionHandlerStack, "f").at(-1);
        if (exceptionHandler !== undefined) {
            const continuationBlock = this.reserve(this.currentBlockKind());
            this.terminateWithContinuation({
                kind: 'maybe-throw',
                continuation: continuationBlock.id,
                handler: exceptionHandler,
                id: makeInstructionId(0),
                loc: instruction.loc,
                effects: null,
            }, continuationBlock);
        }
    }
    enterTryCatch(handler, fn) {
        __classPrivateFieldGet(this, _HIRBuilder_exceptionHandlerStack, "f").push(handler);
        fn();
        __classPrivateFieldGet(this, _HIRBuilder_exceptionHandlerStack, "f").pop();
    }
    resolveThrowHandler() {
        const handler = __classPrivateFieldGet(this, _HIRBuilder_exceptionHandlerStack, "f").at(-1);
        return handler !== null && handler !== void 0 ? handler : null;
    }
    makeTemporary(loc) {
        const id = this.nextIdentifierId;
        return makeTemporaryIdentifier(id, loc);
    }
    resolveIdentifier(path) {
        const originalName = path.node.name;
        const babelBinding = __classPrivateFieldGet(this, _HIRBuilder_instances, "m", _HIRBuilder_resolveBabelBinding).call(this, path);
        if (babelBinding == null) {
            return { kind: 'Global', name: originalName };
        }
        const outerBinding = __classPrivateFieldGet(this, _HIRBuilder_env, "f").parentFunction.scope.parent.getBinding(originalName);
        if (babelBinding === outerBinding) {
            const path = babelBinding.path;
            if (path.isImportDefaultSpecifier()) {
                const importDeclaration = path.parentPath;
                return {
                    kind: 'ImportDefault',
                    name: originalName,
                    module: importDeclaration.node.source.value,
                };
            }
            else if (path.isImportSpecifier()) {
                const importDeclaration = path.parentPath;
                return {
                    kind: 'ImportSpecifier',
                    name: originalName,
                    module: importDeclaration.node.source.value,
                    imported: path.node.imported.type === 'Identifier'
                        ? path.node.imported.name
                        : path.node.imported.value,
                };
            }
            else if (path.isImportNamespaceSpecifier()) {
                const importDeclaration = path.parentPath;
                return {
                    kind: 'ImportNamespace',
                    name: originalName,
                    module: importDeclaration.node.source.value,
                };
            }
            else {
                return {
                    kind: 'ModuleLocal',
                    name: originalName,
                };
            }
        }
        const resolvedBinding = this.resolveBinding(babelBinding.identifier);
        if (resolvedBinding.name && resolvedBinding.name.value !== originalName) {
            babelBinding.scope.rename(originalName, resolvedBinding.name.value);
        }
        return {
            kind: 'Identifier',
            identifier: resolvedBinding,
            bindingKind: babelBinding.kind,
        };
    }
    isContextIdentifier(path) {
        const binding = __classPrivateFieldGet(this, _HIRBuilder_instances, "m", _HIRBuilder_resolveBabelBinding).call(this, path);
        if (binding) {
            const outerBinding = __classPrivateFieldGet(this, _HIRBuilder_env, "f").parentFunction.scope.parent.getBinding(path.node.name);
            if (binding === outerBinding) {
                return false;
            }
            return __classPrivateFieldGet(this, _HIRBuilder_env, "f").isContextIdentifier(binding.identifier);
        }
        else {
            return false;
        }
    }
    resolveBinding(node) {
        var _a, _b, _c;
        if (node.name === 'fbt') {
            CompilerError.throwDiagnostic({
                category: ErrorCategory.Todo,
                reason: 'Support local variables named `fbt`',
                description: 'Local variables named `fbt` may conflict with the fbt plugin and are not yet supported',
                details: [
                    {
                        kind: 'error',
                        message: 'Rename to avoid conflict with fbt plugin',
                        loc: (_a = node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    },
                ],
            });
        }
        if (node.name === 'this') {
            CompilerError.throwDiagnostic({
                category: ErrorCategory.UnsupportedSyntax,
                reason: '`this` is not supported syntax',
                description: 'React Compiler does not support compiling functions that use `this`',
                details: [
                    {
                        kind: 'error',
                        message: '`this` was used here',
                        loc: (_b = node.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
                    },
                ],
            });
        }
        const originalName = node.name;
        let name = originalName;
        let index = 0;
        while (true) {
            const mapping = __classPrivateFieldGet(this, _HIRBuilder_bindings, "f").get(name);
            if (mapping === undefined) {
                const id = this.nextIdentifierId;
                const identifier = {
                    id,
                    declarationId: makeDeclarationId(id),
                    name: makeIdentifierName(name),
                    mutableRange: {
                        start: makeInstructionId(0),
                        end: makeInstructionId(0),
                    },
                    scope: null,
                    type: makeType(),
                    loc: (_c = node.loc) !== null && _c !== void 0 ? _c : GeneratedSource,
                };
                __classPrivateFieldGet(this, _HIRBuilder_env, "f").programContext.addNewReference(name);
                __classPrivateFieldGet(this, _HIRBuilder_bindings, "f").set(name, { node, identifier });
                return identifier;
            }
            else if (mapping.node === node) {
                return mapping.identifier;
            }
            else {
                name = `${originalName}_${index++}`;
            }
        }
    }
    build() {
        var _a, _b;
        let ir = {
            blocks: __classPrivateFieldGet(this, _HIRBuilder_completed, "f"),
            entry: __classPrivateFieldGet(this, _HIRBuilder_entry, "f"),
        };
        const rpoBlocks = getReversePostorderedBlocks(ir);
        for (const [id, block] of ir.blocks) {
            if (!rpoBlocks.has(id) &&
                block.instructions.some(instr => instr.value.kind === 'FunctionExpression')) {
                CompilerError.throwTodo({
                    reason: `Support functions with unreachable code that may contain hoisted declarations`,
                    loc: (_b = (_a = block.instructions[0]) === null || _a === void 0 ? void 0 : _a.loc) !== null && _b !== void 0 ? _b : block.terminal.loc,
                    description: null,
                    suggestions: null,
                });
            }
        }
        ir.blocks = rpoBlocks;
        removeUnreachableForUpdates(ir);
        removeDeadDoWhileStatements(ir);
        removeUnnecessaryTryCatch(ir);
        markInstructionIds(ir);
        markPredecessors(ir);
        return ir;
    }
    terminate(terminal, nextBlockKind) {
        const { id: blockId, kind, instructions } = __classPrivateFieldGet(this, _HIRBuilder_current, "f");
        __classPrivateFieldGet(this, _HIRBuilder_completed, "f").set(blockId, {
            kind,
            id: blockId,
            instructions,
            terminal,
            preds: new Set(),
            phis: new Set(),
        });
        if (nextBlockKind) {
            const nextId = __classPrivateFieldGet(this, _HIRBuilder_env, "f").nextBlockId;
            __classPrivateFieldSet(this, _HIRBuilder_current, newBlock(nextId, nextBlockKind), "f");
        }
        return blockId;
    }
    terminateWithContinuation(terminal, continuation) {
        const { id: blockId, kind, instructions } = __classPrivateFieldGet(this, _HIRBuilder_current, "f");
        __classPrivateFieldGet(this, _HIRBuilder_completed, "f").set(blockId, {
            kind: kind,
            id: blockId,
            instructions,
            terminal: terminal,
            preds: new Set(),
            phis: new Set(),
        });
        __classPrivateFieldSet(this, _HIRBuilder_current, continuation, "f");
    }
    reserve(kind) {
        return newBlock(makeBlockId(__classPrivateFieldGet(this, _HIRBuilder_env, "f").nextBlockId), kind);
    }
    complete(block, terminal) {
        const { id: blockId, kind, instructions } = block;
        __classPrivateFieldGet(this, _HIRBuilder_completed, "f").set(blockId, {
            kind,
            id: blockId,
            instructions,
            terminal,
            preds: new Set(),
            phis: new Set(),
        });
    }
    enterReserved(wip, fn) {
        const current = __classPrivateFieldGet(this, _HIRBuilder_current, "f");
        __classPrivateFieldSet(this, _HIRBuilder_current, wip, "f");
        const terminal = fn();
        const { id: blockId, kind, instructions } = __classPrivateFieldGet(this, _HIRBuilder_current, "f");
        __classPrivateFieldGet(this, _HIRBuilder_completed, "f").set(blockId, {
            kind,
            id: blockId,
            instructions,
            terminal,
            preds: new Set(),
            phis: new Set(),
        });
        __classPrivateFieldSet(this, _HIRBuilder_current, current, "f");
    }
    enter(nextBlockKind, fn) {
        const wip = this.reserve(nextBlockKind);
        this.enterReserved(wip, () => {
            return fn(wip.id);
        });
        return wip.id;
    }
    label(label, breakBlock, fn) {
        __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").push({
            kind: 'label',
            breakBlock,
            label,
        });
        const value = fn();
        const last = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").pop();
        CompilerError.invariant(last != null &&
            last.kind === 'label' &&
            last.label === label &&
            last.breakBlock === breakBlock, {
            reason: 'Mismatched label',
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: null,
                    message: null,
                },
            ],
            suggestions: null,
        });
        return value;
    }
    switch(label, breakBlock, fn) {
        __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").push({
            kind: 'switch',
            breakBlock,
            label,
        });
        const value = fn();
        const last = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").pop();
        CompilerError.invariant(last != null &&
            last.kind === 'switch' &&
            last.label === label &&
            last.breakBlock === breakBlock, {
            reason: 'Mismatched label',
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: null,
                    message: null,
                },
            ],
            suggestions: null,
        });
        return value;
    }
    loop(label, continueBlock, breakBlock, fn) {
        __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").push({
            kind: 'loop',
            label,
            continueBlock,
            breakBlock,
        });
        const value = fn();
        const last = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").pop();
        CompilerError.invariant(last != null &&
            last.kind === 'loop' &&
            last.label === label &&
            last.continueBlock === continueBlock &&
            last.breakBlock === breakBlock, {
            reason: 'Mismatched loops',
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: null,
                    message: null,
                },
            ],
            suggestions: null,
        });
        return value;
    }
    lookupBreak(label) {
        for (let ii = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").length - 1; ii >= 0; ii--) {
            const scope = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f")[ii];
            if ((label === null &&
                (scope.kind === 'loop' || scope.kind === 'switch')) ||
                label === scope.label) {
                return scope.breakBlock;
            }
        }
        CompilerError.invariant(false, {
            reason: 'Expected a loop or switch to be in scope',
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: null,
                    message: null,
                },
            ],
            suggestions: null,
        });
    }
    lookupContinue(label) {
        for (let ii = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f").length - 1; ii >= 0; ii--) {
            const scope = __classPrivateFieldGet(this, _HIRBuilder_scopes, "f")[ii];
            if (scope.kind === 'loop') {
                if (label === null || label === scope.label) {
                    return scope.continueBlock;
                }
            }
            else if (label !== null && scope.label === label) {
                CompilerError.invariant(false, {
                    reason: 'Continue may only refer to a labeled loop',
                    description: null,
                    details: [
                        {
                            kind: 'error',
                            loc: null,
                            message: null,
                        },
                    ],
                    suggestions: null,
                });
            }
        }
        CompilerError.invariant(false, {
            reason: 'Expected a loop to be in scope',
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: null,
                    message: null,
                },
            ],
            suggestions: null,
        });
    }
}
_HIRBuilder_completed = new WeakMap(), _HIRBuilder_current = new WeakMap(), _HIRBuilder_entry = new WeakMap(), _HIRBuilder_scopes = new WeakMap(), _HIRBuilder_context = new WeakMap(), _HIRBuilder_bindings = new WeakMap(), _HIRBuilder_env = new WeakMap(), _HIRBuilder_exceptionHandlerStack = new WeakMap(), _HIRBuilder_instances = new WeakSet(), _HIRBuilder_resolveBabelBinding = function _HIRBuilder_resolveBabelBinding(path) {
    const originalName = path.node.name;
    const binding = path.scope.getBinding(originalName);
    if (binding == null) {
        return null;
    }
    return binding;
};
function removeUnreachableForUpdates(fn) {
    for (const [, block] of fn.blocks) {
        if (block.terminal.kind === 'for' &&
            block.terminal.update !== null &&
            !fn.blocks.has(block.terminal.update)) {
            block.terminal.update = null;
        }
    }
}
function removeDeadDoWhileStatements(func) {
    const visited = new Set();
    for (const [_, block] of func.blocks) {
        visited.add(block.id);
    }
    for (const [_, block] of func.blocks) {
        if (block.terminal.kind === 'do-while') {
            if (!visited.has(block.terminal.test)) {
                block.terminal = {
                    kind: 'goto',
                    block: block.terminal.loop,
                    variant: GotoVariant.Break,
                    id: block.terminal.id,
                    loc: block.terminal.loc,
                };
            }
        }
    }
}
function reversePostorderBlocks(func) {
    const rpoBlocks = getReversePostorderedBlocks(func);
    func.blocks = rpoBlocks;
}
function getReversePostorderedBlocks(func) {
    const visited = new Set();
    const used = new Set();
    const usedFallthroughs = new Set();
    const postorder = [];
    function visit(blockId, isUsed) {
        const wasUsed = used.has(blockId);
        const wasVisited = visited.has(blockId);
        visited.add(blockId);
        if (isUsed) {
            used.add(blockId);
        }
        if (wasVisited && (wasUsed || !isUsed)) {
            return;
        }
        const block = func.blocks.get(blockId);
        CompilerError.invariant(block != null, {
            reason: '[HIRBuilder] Unexpected null block',
            description: `expected block ${blockId} to exist`,
            details: [
                {
                    kind: 'error',
                    loc: GeneratedSource,
                    message: null,
                },
            ],
        });
        const successors = [...eachTerminalSuccessor(block.terminal)].reverse();
        const fallthrough = terminalFallthrough(block.terminal);
        if (fallthrough != null) {
            if (isUsed) {
                usedFallthroughs.add(fallthrough);
            }
            visit(fallthrough, false);
        }
        for (const successor of successors) {
            visit(successor, isUsed);
        }
        if (!wasVisited) {
            postorder.push(blockId);
        }
    }
    visit(func.entry, true);
    const blocks = new Map();
    for (const blockId of postorder.reverse()) {
        const block = func.blocks.get(blockId);
        if (used.has(blockId)) {
            blocks.set(blockId, func.blocks.get(blockId));
        }
        else if (usedFallthroughs.has(blockId)) {
            blocks.set(blockId, Object.assign(Object.assign({}, block), { instructions: [], terminal: {
                    kind: 'unreachable',
                    id: block.terminal.id,
                    loc: block.terminal.loc,
                } }));
        }
    }
    return blocks;
}
function markInstructionIds(func) {
    let id = 0;
    const visited = new Set();
    for (const [_, block] of func.blocks) {
        for (const instr of block.instructions) {
            CompilerError.invariant(!visited.has(instr), {
                reason: `${printInstruction(instr)} already visited!`,
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: instr.loc,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            visited.add(instr);
            instr.id = makeInstructionId(++id);
        }
        block.terminal.id = makeInstructionId(++id);
    }
}
function markPredecessors(func) {
    for (const [, block] of func.blocks) {
        block.preds.clear();
    }
    const visited = new Set();
    function visit(blockId, prevBlock) {
        const block = func.blocks.get(blockId);
        if (block == null) {
            return;
        }
        CompilerError.invariant(block != null, {
            reason: 'unexpected missing block',
            description: `block ${blockId}`,
            details: [
                {
                    kind: 'error',
                    loc: GeneratedSource,
                    message: null,
                },
            ],
        });
        if (prevBlock) {
            block.preds.add(prevBlock.id);
        }
        if (visited.has(blockId)) {
            return;
        }
        visited.add(blockId);
        const { terminal } = block;
        for (const successor of eachTerminalSuccessor(terminal)) {
            visit(successor, block);
        }
    }
    visit(func.entry, null);
}
function removeUnnecessaryTryCatch(fn) {
    for (const [, block] of fn.blocks) {
        if (block.terminal.kind === 'try' &&
            !fn.blocks.has(block.terminal.handler)) {
            const handlerId = block.terminal.handler;
            const fallthroughId = block.terminal.fallthrough;
            const fallthrough = fn.blocks.get(fallthroughId);
            block.terminal = {
                kind: 'goto',
                block: block.terminal.block,
                id: makeInstructionId(0),
                loc: block.terminal.loc,
                variant: GotoVariant.Break,
            };
            if (fallthrough != null) {
                if (fallthrough.preds.size === 1 && fallthrough.preds.has(handlerId)) {
                    fn.blocks.delete(fallthroughId);
                }
                else {
                    fallthrough.preds.delete(handlerId);
                }
            }
        }
    }
}
function createTemporaryPlace(env, loc) {
    return {
        kind: 'Identifier',
        identifier: makeTemporaryIdentifier(env.nextIdentifierId, loc),
        reactive: false,
        effect: Effect.Unknown,
        loc: GeneratedSource,
    };
}
function clonePlaceToTemporary(env, place) {
    const temp = createTemporaryPlace(env, place.loc);
    temp.effect = place.effect;
    temp.identifier.type = place.identifier.type;
    temp.reactive = place.reactive;
    return temp;
}
function fixScopeAndIdentifierRanges(func) {
    var _a, _b;
    for (const [, block] of func.blocks) {
        const terminal = block.terminal;
        if (terminal.kind === 'scope' || terminal.kind === 'pruned-scope') {
            const fallthroughBlock = func.blocks.get(terminal.fallthrough);
            const firstId = (_b = (_a = fallthroughBlock.instructions[0]) === null || _a === void 0 ? void 0 : _a.id) !== null && _b !== void 0 ? _b : fallthroughBlock.terminal.id;
            terminal.scope.range.start = terminal.id;
            terminal.scope.range.end = firstId;
        }
    }
}

const PRIMITIVE_TYPE = {
    kind: 'Primitive',
};
let nextAnonId = 0;
function createAnonId() {
    return `<generated_${nextAnonId++}>`;
}
function addFunction(registry, properties, fn, id = null, isConstructor = false) {
    const shapeId = id !== null && id !== void 0 ? id : createAnonId();
    const aliasing = fn.aliasing != null
        ? parseAliasingSignatureConfig(fn.aliasing, '<builtin>', GeneratedSource)
        : null;
    addShape(registry, shapeId, properties, Object.assign(Object.assign({}, fn), { aliasing, hookKind: null }));
    return {
        kind: 'Function',
        return: fn.returnType,
        shapeId,
        isConstructor,
    };
}
function addHook(registry, fn, id = null) {
    const shapeId = id !== null && id !== void 0 ? id : createAnonId();
    const aliasing = fn.aliasing != null
        ? parseAliasingSignatureConfig(fn.aliasing, '<builtin>', GeneratedSource)
        : null;
    addShape(registry, shapeId, [], Object.assign(Object.assign({}, fn), { aliasing }));
    return {
        kind: 'Function',
        return: fn.returnType,
        shapeId,
        isConstructor: false,
    };
}
function parseAliasingSignatureConfig(typeConfig, moduleName, loc) {
    const lifetimes = new Map();
    function define(temp) {
        CompilerError.invariant(!lifetimes.has(temp), {
            reason: `Invalid type configuration for module`,
            description: `Expected aliasing signature to have unique names for receiver, params, rest, returns, and temporaries in module '${moduleName}'`,
            details: [
                {
                    kind: 'error',
                    loc,
                    message: null,
                },
            ],
        });
        const place = signatureArgument(lifetimes.size);
        lifetimes.set(temp, place);
        return place;
    }
    function lookup(temp) {
        const place = lifetimes.get(temp);
        CompilerError.invariant(place != null, {
            reason: `Invalid type configuration for module`,
            description: `Expected aliasing signature effects to reference known names from receiver/params/rest/returns/temporaries, but '${temp}' is not a known name in '${moduleName}'`,
            details: [
                {
                    kind: 'error',
                    loc,
                    message: null,
                },
            ],
        });
        return place;
    }
    const receiver = define(typeConfig.receiver);
    const params = typeConfig.params.map(define);
    const rest = typeConfig.rest != null ? define(typeConfig.rest) : null;
    const returns = define(typeConfig.returns);
    const temporaries = typeConfig.temporaries.map(define);
    const effects = typeConfig.effects.map((effect) => {
        switch (effect.kind) {
            case 'ImmutableCapture':
            case 'CreateFrom':
            case 'Capture':
            case 'Alias':
            case 'Assign': {
                const from = lookup(effect.from);
                const into = lookup(effect.into);
                return {
                    kind: effect.kind,
                    from,
                    into,
                };
            }
            case 'Mutate':
            case 'MutateTransitiveConditionally': {
                const value = lookup(effect.value);
                return { kind: effect.kind, value };
            }
            case 'Create': {
                const into = lookup(effect.into);
                return {
                    kind: 'Create',
                    into,
                    reason: effect.reason,
                    value: effect.value,
                };
            }
            case 'Freeze': {
                const value = lookup(effect.value);
                return {
                    kind: 'Freeze',
                    value,
                    reason: effect.reason,
                };
            }
            case 'Impure': {
                const place = lookup(effect.place);
                return {
                    kind: 'Impure',
                    place,
                    error: CompilerError.throwTodo({
                        reason: 'Support impure effect declarations',
                        loc: GeneratedSource,
                    }),
                };
            }
            case 'Apply': {
                const receiver = lookup(effect.receiver);
                const fn = lookup(effect.function);
                const args = effect.args.map(arg => {
                    if (typeof arg === 'string') {
                        return lookup(arg);
                    }
                    else if (arg.kind === 'Spread') {
                        return { kind: 'Spread', place: lookup(arg.place) };
                    }
                    else {
                        return arg;
                    }
                });
                const into = lookup(effect.into);
                return {
                    kind: 'Apply',
                    receiver,
                    function: fn,
                    mutatesFunction: effect.mutatesFunction,
                    args,
                    into,
                    loc,
                    signature: null,
                };
            }
            default: {
                assertExhaustive$1(effect, `Unexpected effect kind '${effect.kind}'`);
            }
        }
    });
    return {
        receiver: receiver.identifier.id,
        params: params.map(p => p.identifier.id),
        rest: rest != null ? rest.identifier.id : null,
        returns: returns.identifier.id,
        temporaries,
        effects,
    };
}
function addObject(registry, id, properties) {
    const shapeId = id !== null && id !== void 0 ? id : createAnonId();
    addShape(registry, shapeId, properties, null);
    return {
        kind: 'Object',
        shapeId,
    };
}
function addShape(registry, id, properties, functionType) {
    const shape = {
        properties: new Map(properties),
        functionType,
    };
    CompilerError.invariant(!registry.has(id), {
        reason: `[ObjectShape] Could not add shape to registry: name ${id} already exists.`,
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    registry.set(id, shape);
    return shape;
}
const BuiltInPropsId = 'BuiltInProps';
const BuiltInArrayId = 'BuiltInArray';
const BuiltInSetId = 'BuiltInSet';
const BuiltInMapId = 'BuiltInMap';
const BuiltInWeakSetId = 'BuiltInWeakSet';
const BuiltInWeakMapId = 'BuiltInWeakMap';
const BuiltInFunctionId = 'BuiltInFunction';
const BuiltInJsxId = 'BuiltInJsx';
const BuiltInObjectId = 'BuiltInObject';
const BuiltInUseStateId = 'BuiltInUseState';
const BuiltInSetStateId = 'BuiltInSetState';
const BuiltInUseActionStateId = 'BuiltInUseActionState';
const BuiltInSetActionStateId = 'BuiltInSetActionState';
const BuiltInUseRefId = 'BuiltInUseRefId';
const BuiltInRefValueId = 'BuiltInRefValue';
const BuiltInMixedReadonlyId = 'BuiltInMixedReadonly';
const BuiltInUseEffectHookId = 'BuiltInUseEffectHook';
const BuiltInUseLayoutEffectHookId = 'BuiltInUseLayoutEffectHook';
const BuiltInUseInsertionEffectHookId = 'BuiltInUseInsertionEffectHook';
const BuiltInUseOperatorId = 'BuiltInUseOperator';
const BuiltInUseReducerId = 'BuiltInUseReducer';
const BuiltInDispatchId = 'BuiltInDispatch';
const BuiltInUseContextHookId = 'BuiltInUseContextHook';
const BuiltInUseTransitionId = 'BuiltInUseTransition';
const BuiltInStartTransitionId = 'BuiltInStartTransition';
const BuiltInFireId = 'BuiltInFire';
const BuiltInFireFunctionId = 'BuiltInFireFunction';
const BuiltInUseEffectEventId = 'BuiltInUseEffectEvent';
const BuiltinEffectEventId = 'BuiltInEffectEventFunction';
const BuiltInAutodepsId = 'BuiltInAutoDepsId';
const ReanimatedSharedValueId = 'ReanimatedSharedValueId';
const BUILTIN_SHAPES = new Map();
addObject(BUILTIN_SHAPES, BuiltInPropsId, [
    ['ref', { kind: 'Object', shapeId: BuiltInUseRefId }],
]);
addObject(BUILTIN_SHAPES, BuiltInArrayId, [
    [
        'indexOf',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'includes',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'pop',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'at',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'concat',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Capture,
            returnType: {
                kind: 'Object',
                shapeId: BuiltInArrayId,
            },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    ['length', PRIMITIVE_TYPE],
    [
        'push',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Capture,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
            aliasing: {
                receiver: '@receiver',
                params: [],
                rest: '@rest',
                returns: '@returns',
                temporaries: [],
                effects: [
                    { kind: 'Mutate', value: '@receiver' },
                    {
                        kind: 'Capture',
                        from: '@rest',
                        into: '@receiver',
                    },
                    {
                        kind: 'Create',
                        into: '@returns',
                        value: ValueKind.Primitive,
                        reason: ValueReason.KnownReturnSignature,
                    },
                ],
            },
        }),
    ],
    [
        'slice',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: {
                kind: 'Object',
                shapeId: BuiltInArrayId,
            },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'map',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
            aliasing: {
                receiver: '@receiver',
                params: ['@callback'],
                rest: null,
                returns: '@returns',
                temporaries: [
                    '@item',
                    '@callbackReturn',
                    '@thisArg',
                ],
                effects: [
                    {
                        kind: 'Create',
                        into: '@returns',
                        value: ValueKind.Mutable,
                        reason: ValueReason.KnownReturnSignature,
                    },
                    {
                        kind: 'CreateFrom',
                        from: '@receiver',
                        into: '@item',
                    },
                    {
                        kind: 'Create',
                        into: '@thisArg',
                        value: ValueKind.Primitive,
                        reason: ValueReason.KnownReturnSignature,
                    },
                    {
                        kind: 'Apply',
                        receiver: '@thisArg',
                        args: ['@item', { kind: 'Hole' }, '@receiver'],
                        function: '@callback',
                        into: '@callbackReturn',
                        mutatesFunction: false,
                    },
                    {
                        kind: 'Capture',
                        from: '@callbackReturn',
                        into: '@returns',
                    },
                ],
            },
        }),
    ],
    [
        'flatMap',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'filter',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'every',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'some',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'find',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'findIndex',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'join',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInObjectId, [
    [
        'toString',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInSetId, [
    [
        'add',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInSetId },
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Mutable,
            aliasing: {
                receiver: '@receiver',
                params: [],
                rest: '@rest',
                returns: '@returns',
                temporaries: [],
                effects: [
                    {
                        kind: 'Assign',
                        from: '@receiver',
                        into: '@returns',
                    },
                    {
                        kind: 'Mutate',
                        value: '@receiver',
                    },
                    {
                        kind: 'Capture',
                        from: '@rest',
                        into: '@receiver',
                    },
                ],
            },
        }),
    ],
    [
        'clear',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'delete',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'has',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    ['size', PRIMITIVE_TYPE],
    [
        'difference',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInSetId },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'union',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInSetId },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'symmetricalDifference',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInSetId },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'isSubsetOf',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'isSupersetOf',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'forEach',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'entries',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'keys',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'values',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInMapId, [
    [
        'clear',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'delete',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'get',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'has',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'set',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture, Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInMapId },
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    ['size', PRIMITIVE_TYPE],
    [
        'forEach',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'entries',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'keys',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'values',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInWeakSetId, [
    [
        'add',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInWeakSetId },
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'delete',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'has',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInWeakMapId, [
    [
        'delete',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'get',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: { kind: 'Poly' },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'has',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'set',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Capture, Effect.Capture],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInWeakMapId },
            calleeEffect: Effect.Store,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInUseStateId, [
    ['0', { kind: 'Poly' }],
    [
        '1',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Freeze,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }, BuiltInSetStateId),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInUseTransitionId, [
    ['0', { kind: 'Primitive' }],
    [
        '1',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: null,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }, BuiltInStartTransitionId),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInUseActionStateId, [
    ['0', { kind: 'Poly' }],
    [
        '1',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Freeze,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }, BuiltInSetActionStateId),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInUseReducerId, [
    ['0', { kind: 'Poly' }],
    [
        '1',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Freeze,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }, BuiltInDispatchId),
    ],
]);
addObject(BUILTIN_SHAPES, BuiltInUseRefId, [
    ['current', { kind: 'Object', shapeId: BuiltInRefValueId }],
]);
addObject(BUILTIN_SHAPES, BuiltInRefValueId, [
    ['*', { kind: 'Object', shapeId: BuiltInRefValueId }],
]);
addObject(BUILTIN_SHAPES, ReanimatedSharedValueId, []);
addFunction(BUILTIN_SHAPES, [], {
    positionalParams: [],
    restParam: Effect.ConditionallyMutate,
    returnType: { kind: 'Poly' },
    calleeEffect: Effect.ConditionallyMutate,
    returnValueKind: ValueKind.Mutable,
}, BuiltinEffectEventId);
addObject(BUILTIN_SHAPES, BuiltInMixedReadonlyId, [
    [
        'toString',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'indexOf',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'includes',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    [
        'at',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [Effect.Read],
            restParam: null,
            returnType: { kind: 'Object', shapeId: BuiltInMixedReadonlyId },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Frozen,
        }),
    ],
    [
        'map',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
        }),
    ],
    [
        'flatMap',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
        }),
    ],
    [
        'filter',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInArrayId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Mutable,
            noAlias: true,
        }),
    ],
    [
        'concat',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Capture,
            returnType: {
                kind: 'Object',
                shapeId: BuiltInArrayId,
            },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'slice',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: {
                kind: 'Object',
                shapeId: BuiltInArrayId,
            },
            calleeEffect: Effect.Capture,
            returnValueKind: ValueKind.Mutable,
        }),
    ],
    [
        'every',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'some',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'find',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Object', shapeId: BuiltInMixedReadonlyId },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Frozen,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'findIndex',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.ConditionallyMutate,
            returnType: { kind: 'Primitive' },
            calleeEffect: Effect.ConditionallyMutate,
            returnValueKind: ValueKind.Primitive,
            noAlias: true,
            mutableOnlyIfOperandsAreMutable: true,
        }),
    ],
    [
        'join',
        addFunction(BUILTIN_SHAPES, [], {
            positionalParams: [],
            restParam: Effect.Read,
            returnType: PRIMITIVE_TYPE,
            calleeEffect: Effect.Read,
            returnValueKind: ValueKind.Primitive,
        }),
    ],
    ['*', { kind: 'Object', shapeId: BuiltInMixedReadonlyId }],
]);
addObject(BUILTIN_SHAPES, BuiltInJsxId, []);
addObject(BUILTIN_SHAPES, BuiltInFunctionId, []);
const DefaultMutatingHook = addHook(BUILTIN_SHAPES, {
    positionalParams: [],
    restParam: Effect.ConditionallyMutate,
    returnType: { kind: 'Poly' },
    calleeEffect: Effect.Read,
    hookKind: 'Custom',
    returnValueKind: ValueKind.Mutable,
}, 'DefaultMutatingHook');
const DefaultNonmutatingHook = addHook(BUILTIN_SHAPES, {
    positionalParams: [],
    restParam: Effect.Freeze,
    returnType: { kind: 'Poly' },
    calleeEffect: Effect.Read,
    hookKind: 'Custom',
    returnValueKind: ValueKind.Frozen,
    aliasing: {
        receiver: '@receiver',
        params: [],
        rest: '@rest',
        returns: '@returns',
        temporaries: [],
        effects: [
            {
                kind: 'Freeze',
                value: '@rest',
                reason: ValueReason.HookCaptured,
            },
            {
                kind: 'Create',
                into: '@returns',
                value: ValueKind.Frozen,
                reason: ValueReason.HookReturn,
            },
            {
                kind: 'Alias',
                from: '@rest',
                into: '@returns',
            },
        ],
    },
}, 'DefaultNonmutatingHook');
function signatureArgument(id) {
    const place = {
        kind: 'Identifier',
        effect: Effect.Unknown,
        loc: GeneratedSource,
        reactive: false,
        identifier: {
            declarationId: makeDeclarationId(id),
            id: makeIdentifierId(id),
            loc: GeneratedSource,
            mutableRange: { start: makeInstructionId(0), end: makeInstructionId(0) },
            name: null,
            scope: null,
            type: makeType(),
        },
    };
    return place;
}

function lower(func, env, bindings = null, capturedRefs = new Map()) {
    var _a, _b, _c;
    const builder = new HIRBuilder(env, {
        bindings,
        context: capturedRefs,
    });
    const context = [];
    for (const [ref, loc] of capturedRefs !== null && capturedRefs !== void 0 ? capturedRefs : []) {
        context.push({
            kind: 'Identifier',
            identifier: builder.resolveBinding(ref),
            effect: Effect.Unknown,
            reactive: false,
            loc,
        });
    }
    let id = null;
    if (func.isFunctionDeclaration() || func.isFunctionExpression()) {
        const idNode = func.get('id');
        if (hasNode(idNode)) {
            id = idNode.node.name;
        }
    }
    const params = [];
    func.get('params').forEach(param => {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        if (param.isIdentifier()) {
            const binding = builder.resolveIdentifier(param);
            if (binding.kind !== 'Identifier') {
                builder.errors.pushDiagnostic(CompilerDiagnostic.create({
                    category: ErrorCategory.Invariant,
                    reason: 'Could not find binding',
                    description: `[BuildHIR] Could not find binding for param \`${param.node.name}\``,
                }).withDetails({
                    kind: 'error',
                    loc: (_a = param.node.loc) !== null && _a !== void 0 ? _a : null,
                    message: 'Could not find binding',
                }));
                return;
            }
            const place = {
                kind: 'Identifier',
                identifier: binding.identifier,
                effect: Effect.Unknown,
                reactive: false,
                loc: (_b = param.node.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
            };
            params.push(place);
        }
        else if (param.isObjectPattern() ||
            param.isArrayPattern() ||
            param.isAssignmentPattern()) {
            const place = {
                kind: 'Identifier',
                identifier: builder.makeTemporary((_c = param.node.loc) !== null && _c !== void 0 ? _c : GeneratedSource),
                effect: Effect.Unknown,
                reactive: false,
                loc: (_d = param.node.loc) !== null && _d !== void 0 ? _d : GeneratedSource,
            };
            promoteTemporary(place.identifier);
            params.push(place);
            lowerAssignment(builder, (_e = param.node.loc) !== null && _e !== void 0 ? _e : GeneratedSource, InstructionKind.Let, param, place, 'Assignment');
        }
        else if (param.isRestElement()) {
            const place = {
                kind: 'Identifier',
                identifier: builder.makeTemporary((_f = param.node.loc) !== null && _f !== void 0 ? _f : GeneratedSource),
                effect: Effect.Unknown,
                reactive: false,
                loc: (_g = param.node.loc) !== null && _g !== void 0 ? _g : GeneratedSource,
            };
            params.push({
                kind: 'Spread',
                place,
            });
            lowerAssignment(builder, (_h = param.node.loc) !== null && _h !== void 0 ? _h : GeneratedSource, InstructionKind.Let, param.get('argument'), place, 'Assignment');
        }
        else {
            builder.errors.pushDiagnostic(CompilerDiagnostic.create({
                category: ErrorCategory.Todo,
                reason: `Handle ${param.node.type} parameters`,
                description: `[BuildHIR] Add support for ${param.node.type} parameters`,
            }).withDetails({
                kind: 'error',
                loc: (_j = param.node.loc) !== null && _j !== void 0 ? _j : null,
                message: 'Unsupported parameter type',
            }));
        }
    });
    let directives = [];
    const body = func.get('body');
    if (body.isExpression()) {
        const fallthrough = builder.reserve('block');
        const terminal = {
            kind: 'return',
            returnVariant: 'Implicit',
            loc: GeneratedSource,
            value: lowerExpressionToTemporary(builder, body),
            id: makeInstructionId(0),
            effects: null,
        };
        builder.terminateWithContinuation(terminal, fallthrough);
    }
    else if (body.isBlockStatement()) {
        lowerStatement(builder, body);
        directives = body.get('directives').map(d => d.node.value.value);
    }
    else {
        builder.errors.pushDiagnostic(CompilerDiagnostic.create({
            category: ErrorCategory.Syntax,
            reason: `Unexpected function body kind`,
            description: `Expected function body to be an expression or a block statement, got \`${body.type}\``,
        }).withDetails({
            kind: 'error',
            loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : null,
            message: 'Expected a block statement or expression',
        }));
    }
    let validatedId = null;
    if (id != null) {
        const idResult = validateIdentifierName(id);
        if (idResult.isErr()) {
            builder.errors.merge(idResult.unwrapErr());
        }
        else {
            validatedId = idResult.unwrap().value;
        }
    }
    if (builder.errors.hasAnyErrors()) {
        return Err(builder.errors);
    }
    builder.terminate({
        kind: 'return',
        returnVariant: 'Void',
        loc: GeneratedSource,
        value: lowerValueToTemporary(builder, {
            kind: 'Primitive',
            value: undefined,
            loc: GeneratedSource,
        }),
        id: makeInstructionId(0),
        effects: null,
    }, null);
    return Ok({
        id: validatedId,
        nameHint: null,
        params,
        fnType: bindings == null ? env.fnType : 'Other',
        returnTypeAnnotation: null,
        returns: createTemporaryPlace(env, (_b = func.node.loc) !== null && _b !== void 0 ? _b : GeneratedSource),
        body: builder.build(),
        context,
        generator: func.node.generator === true,
        async: func.node.async === true,
        loc: (_c = func.node.loc) !== null && _c !== void 0 ? _c : GeneratedSource,
        env,
        effects: null,
        aliasingEffects: null,
        directives,
    });
}
function lowerStatement(builder, stmtPath, label = null) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32, _33, _34, _35, _36;
    const stmtNode = stmtPath.node;
    switch (stmtNode.type) {
        case 'ThrowStatement': {
            const stmt = stmtPath;
            const value = lowerExpressionToTemporary(builder, stmt.get('argument'));
            const handler = builder.resolveThrowHandler();
            if (handler != null) {
                builder.errors.push({
                    reason: '(BuildHIR::lowerStatement) Support ThrowStatement inside of try/catch',
                    category: ErrorCategory.Todo,
                    loc: (_a = stmt.node.loc) !== null && _a !== void 0 ? _a : null,
                    suggestions: null,
                });
            }
            const terminal = {
                kind: 'throw',
                value,
                id: makeInstructionId(0),
                loc: (_b = stmt.node.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
            };
            builder.terminate(terminal, 'block');
            return;
        }
        case 'ReturnStatement': {
            const stmt = stmtPath;
            const argument = stmt.get('argument');
            let value;
            if (argument.node === null) {
                value = lowerValueToTemporary(builder, {
                    kind: 'Primitive',
                    value: undefined,
                    loc: GeneratedSource,
                });
            }
            else {
                value = lowerExpressionToTemporary(builder, argument);
            }
            const terminal = {
                kind: 'return',
                returnVariant: 'Explicit',
                loc: (_c = stmt.node.loc) !== null && _c !== void 0 ? _c : GeneratedSource,
                value,
                id: makeInstructionId(0),
                effects: null,
            };
            builder.terminate(terminal, 'block');
            return;
        }
        case 'IfStatement': {
            const stmt = stmtPath;
            const continuationBlock = builder.reserve('block');
            const consequentBlock = builder.enter('block', _blockId => {
                var _a;
                const consequent = stmt.get('consequent');
                lowerStatement(builder, consequent);
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: (_a = consequent.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                };
            });
            let alternateBlock;
            const alternate = stmt.get('alternate');
            if (hasNode(alternate)) {
                alternateBlock = builder.enter('block', _blockId => {
                    var _a, _b;
                    lowerStatement(builder, alternate);
                    return {
                        kind: 'goto',
                        block: continuationBlock.id,
                        variant: GotoVariant.Break,
                        id: makeInstructionId(0),
                        loc: (_b = (_a = alternate.node) === null || _a === void 0 ? void 0 : _a.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
                    };
                });
            }
            else {
                alternateBlock = continuationBlock.id;
            }
            const test = lowerExpressionToTemporary(builder, stmt.get('test'));
            const terminal = {
                kind: 'if',
                test,
                consequent: consequentBlock,
                alternate: alternateBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: (_d = stmt.node.loc) !== null && _d !== void 0 ? _d : GeneratedSource,
            };
            builder.terminateWithContinuation(terminal, continuationBlock);
            return;
        }
        case 'BlockStatement': {
            const stmt = stmtPath;
            const statements = stmt.get('body');
            const hoistableIdentifiers = new Set();
            for (const [, binding] of Object.entries(stmt.scope.bindings)) {
                if (binding.kind !== 'param') {
                    hoistableIdentifiers.add(binding.identifier);
                }
            }
            for (const s of statements) {
                const willHoist = new Set();
                let fnDepth = s.isFunctionDeclaration() ? 1 : 0;
                const withFunctionContext = {
                    enter: () => {
                        fnDepth++;
                    },
                    exit: () => {
                        fnDepth--;
                    },
                };
                s.traverse({
                    FunctionExpression: withFunctionContext,
                    FunctionDeclaration: withFunctionContext,
                    ArrowFunctionExpression: withFunctionContext,
                    ObjectMethod: withFunctionContext,
                    Identifier(id) {
                        const id2 = id;
                        if (!id2.isReferencedIdentifier() &&
                            id.parent.type !== 'AssignmentExpression') {
                            return;
                        }
                        const binding = id.scope.getBinding(id.node.name);
                        if (binding != null &&
                            hoistableIdentifiers.has(binding.identifier) &&
                            (fnDepth > 0 || binding.kind === 'hoisted')) {
                            willHoist.add(id);
                        }
                    },
                });
                s.traverse({
                    Identifier(path) {
                        if (hoistableIdentifiers.has(path.node)) {
                            hoistableIdentifiers.delete(path.node);
                        }
                    },
                });
                for (const id of willHoist) {
                    const binding = stmt.scope.getBinding(id.node.name);
                    CompilerError.invariant(binding != null, {
                        reason: 'Expected to find binding for hoisted identifier',
                        description: `Could not find a binding for ${id.node.name}`,
                        suggestions: null,
                        details: [
                            {
                                kind: 'error',
                                loc: (_e = id.node.loc) !== null && _e !== void 0 ? _e : GeneratedSource,
                                message: null,
                            },
                        ],
                    });
                    if (builder.environment.isHoistedIdentifier(binding.identifier)) {
                        continue;
                    }
                    let kind;
                    if (binding.kind === 'const' || binding.kind === 'var') {
                        kind = InstructionKind.HoistedConst;
                    }
                    else if (binding.kind === 'let') {
                        kind = InstructionKind.HoistedLet;
                    }
                    else if (binding.path.isFunctionDeclaration()) {
                        kind = InstructionKind.HoistedFunction;
                    }
                    else if (!binding.path.isVariableDeclarator()) {
                        builder.errors.push({
                            category: ErrorCategory.Todo,
                            reason: 'Unsupported declaration type for hoisting',
                            description: `variable "${binding.identifier.name}" declared with ${binding.path.type}`,
                            suggestions: null,
                            loc: (_f = id.parentPath.node.loc) !== null && _f !== void 0 ? _f : GeneratedSource,
                        });
                        continue;
                    }
                    else {
                        builder.errors.push({
                            category: ErrorCategory.Todo,
                            reason: 'Handle non-const declarations for hoisting',
                            description: `variable "${binding.identifier.name}" declared with ${binding.kind}`,
                            suggestions: null,
                            loc: (_g = id.parentPath.node.loc) !== null && _g !== void 0 ? _g : GeneratedSource,
                        });
                        continue;
                    }
                    const identifier = builder.resolveIdentifier(id);
                    CompilerError.invariant(identifier.kind === 'Identifier', {
                        reason: 'Expected hoisted binding to be a local identifier, not a global',
                        description: null,
                        details: [
                            {
                                kind: 'error',
                                loc: (_h = id.node.loc) !== null && _h !== void 0 ? _h : GeneratedSource,
                                message: null,
                            },
                        ],
                    });
                    const place = {
                        effect: Effect.Unknown,
                        identifier: identifier.identifier,
                        kind: 'Identifier',
                        reactive: false,
                        loc: (_j = id.node.loc) !== null && _j !== void 0 ? _j : GeneratedSource,
                    };
                    lowerValueToTemporary(builder, {
                        kind: 'DeclareContext',
                        lvalue: {
                            kind,
                            place,
                        },
                        loc: (_k = id.node.loc) !== null && _k !== void 0 ? _k : GeneratedSource,
                    });
                    builder.environment.addHoistedIdentifier(binding.identifier);
                }
                lowerStatement(builder, s);
            }
            return;
        }
        case 'BreakStatement': {
            const stmt = stmtPath;
            const block = builder.lookupBreak((_m = (_l = stmt.node.label) === null || _l === void 0 ? void 0 : _l.name) !== null && _m !== void 0 ? _m : null);
            builder.terminate({
                kind: 'goto',
                block,
                variant: GotoVariant.Break,
                id: makeInstructionId(0),
                loc: (_o = stmt.node.loc) !== null && _o !== void 0 ? _o : GeneratedSource,
            }, 'block');
            return;
        }
        case 'ContinueStatement': {
            const stmt = stmtPath;
            const block = builder.lookupContinue((_q = (_p = stmt.node.label) === null || _p === void 0 ? void 0 : _p.name) !== null && _q !== void 0 ? _q : null);
            builder.terminate({
                kind: 'goto',
                block,
                variant: GotoVariant.Continue,
                id: makeInstructionId(0),
                loc: (_r = stmt.node.loc) !== null && _r !== void 0 ? _r : GeneratedSource,
            }, 'block');
            return;
        }
        case 'ForStatement': {
            const stmt = stmtPath;
            const testBlock = builder.reserve('loop');
            const continuationBlock = builder.reserve('block');
            const initBlock = builder.enter('loop', _blockId => {
                var _a, _b, _c, _d;
                const init = stmt.get('init');
                if (!init.isVariableDeclaration()) {
                    builder.errors.push({
                        reason: '(BuildHIR::lowerStatement) Handle non-variable initialization in ForStatement',
                        category: ErrorCategory.Todo,
                        loc: (_a = stmt.node.loc) !== null && _a !== void 0 ? _a : null,
                        suggestions: null,
                    });
                    return {
                        kind: 'unsupported',
                        id: makeInstructionId(0),
                        loc: (_c = (_b = init.node) === null || _b === void 0 ? void 0 : _b.loc) !== null && _c !== void 0 ? _c : GeneratedSource,
                    };
                }
                lowerStatement(builder, init);
                return {
                    kind: 'goto',
                    block: testBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: (_d = init.node.loc) !== null && _d !== void 0 ? _d : GeneratedSource,
                };
            });
            let updateBlock = null;
            const update = stmt.get('update');
            if (hasNode(update)) {
                updateBlock = builder.enter('loop', _blockId => {
                    var _a, _b;
                    lowerExpressionToTemporary(builder, update);
                    return {
                        kind: 'goto',
                        block: testBlock.id,
                        variant: GotoVariant.Break,
                        id: makeInstructionId(0),
                        loc: (_b = (_a = update.node) === null || _a === void 0 ? void 0 : _a.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
                    };
                });
            }
            const bodyBlock = builder.enter('block', _blockId => {
                return builder.loop(label, updateBlock !== null && updateBlock !== void 0 ? updateBlock : testBlock.id, continuationBlock.id, () => {
                    var _a;
                    const body = stmt.get('body');
                    lowerStatement(builder, body);
                    return {
                        kind: 'goto',
                        block: updateBlock !== null && updateBlock !== void 0 ? updateBlock : testBlock.id,
                        variant: GotoVariant.Continue,
                        id: makeInstructionId(0),
                        loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    };
                });
            });
            builder.terminateWithContinuation({
                kind: 'for',
                loc: (_s = stmtNode.loc) !== null && _s !== void 0 ? _s : GeneratedSource,
                init: initBlock,
                test: testBlock.id,
                update: updateBlock,
                loop: bodyBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
            }, testBlock);
            const test = stmt.get('test');
            if (test.node == null) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerStatement) Handle empty test in ForStatement`,
                    category: ErrorCategory.Todo,
                    loc: (_t = stmt.node.loc) !== null && _t !== void 0 ? _t : null,
                    suggestions: null,
                });
            }
            else {
                builder.terminateWithContinuation({
                    kind: 'branch',
                    test: lowerExpressionToTemporary(builder, test),
                    consequent: bodyBlock,
                    alternate: continuationBlock.id,
                    fallthrough: continuationBlock.id,
                    id: makeInstructionId(0),
                    loc: (_u = stmt.node.loc) !== null && _u !== void 0 ? _u : GeneratedSource,
                }, continuationBlock);
            }
            return;
        }
        case 'WhileStatement': {
            const stmt = stmtPath;
            const conditionalBlock = builder.reserve('loop');
            const continuationBlock = builder.reserve('block');
            const loopBlock = builder.enter('block', _blockId => {
                return builder.loop(label, conditionalBlock.id, continuationBlock.id, () => {
                    var _a;
                    const body = stmt.get('body');
                    lowerStatement(builder, body);
                    return {
                        kind: 'goto',
                        block: conditionalBlock.id,
                        variant: GotoVariant.Continue,
                        id: makeInstructionId(0),
                        loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    };
                });
            });
            const loc = (_v = stmt.node.loc) !== null && _v !== void 0 ? _v : GeneratedSource;
            builder.terminateWithContinuation({
                kind: 'while',
                loc,
                test: conditionalBlock.id,
                loop: loopBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
            }, conditionalBlock);
            const test = lowerExpressionToTemporary(builder, stmt.get('test'));
            const terminal = {
                kind: 'branch',
                test,
                consequent: loopBlock,
                alternate: continuationBlock.id,
                fallthrough: conditionalBlock.id,
                id: makeInstructionId(0),
                loc: (_w = stmt.node.loc) !== null && _w !== void 0 ? _w : GeneratedSource,
            };
            builder.terminateWithContinuation(terminal, continuationBlock);
            return;
        }
        case 'LabeledStatement': {
            const stmt = stmtPath;
            const label = stmt.node.label.name;
            const body = stmt.get('body');
            switch (body.node.type) {
                case 'ForInStatement':
                case 'ForOfStatement':
                case 'ForStatement':
                case 'WhileStatement':
                case 'DoWhileStatement': {
                    lowerStatement(builder, stmt.get('body'), label);
                    break;
                }
                default: {
                    const continuationBlock = builder.reserve('block');
                    const block = builder.enter('block', () => {
                        var _a;
                        const body = stmt.get('body');
                        builder.label(label, continuationBlock.id, () => {
                            lowerStatement(builder, body);
                        });
                        return {
                            kind: 'goto',
                            block: continuationBlock.id,
                            variant: GotoVariant.Break,
                            id: makeInstructionId(0),
                            loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                        };
                    });
                    builder.terminateWithContinuation({
                        kind: 'label',
                        block,
                        fallthrough: continuationBlock.id,
                        id: makeInstructionId(0),
                        loc: (_x = stmt.node.loc) !== null && _x !== void 0 ? _x : GeneratedSource,
                    }, continuationBlock);
                }
            }
            return;
        }
        case 'SwitchStatement': {
            const stmt = stmtPath;
            const continuationBlock = builder.reserve('block');
            let fallthrough = continuationBlock.id;
            const cases = [];
            let hasDefault = false;
            for (let ii = stmt.get('cases').length - 1; ii >= 0; ii--) {
                const case_ = stmt.get('cases')[ii];
                const testExpr = case_.get('test');
                if (testExpr.node == null) {
                    if (hasDefault) {
                        builder.errors.push({
                            reason: `Expected at most one \`default\` branch in a switch statement, this code should have failed to parse`,
                            category: ErrorCategory.Syntax,
                            loc: (_y = case_.node.loc) !== null && _y !== void 0 ? _y : null,
                            suggestions: null,
                        });
                        break;
                    }
                    hasDefault = true;
                }
                const block = builder.enter('block', _blockId => {
                    return builder.switch(label, continuationBlock.id, () => {
                        var _a;
                        case_
                            .get('consequent')
                            .forEach(consequent => lowerStatement(builder, consequent));
                        return {
                            kind: 'goto',
                            block: fallthrough,
                            variant: GotoVariant.Break,
                            id: makeInstructionId(0),
                            loc: (_a = case_.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                        };
                    });
                });
                let test = null;
                if (hasNode(testExpr)) {
                    test = lowerReorderableExpression(builder, testExpr);
                }
                cases.push({
                    test,
                    block,
                });
                fallthrough = block;
            }
            cases.reverse();
            if (!hasDefault) {
                cases.push({ test: null, block: continuationBlock.id });
            }
            const test = lowerExpressionToTemporary(builder, stmt.get('discriminant'));
            builder.terminateWithContinuation({
                kind: 'switch',
                test,
                cases,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: (_z = stmt.node.loc) !== null && _z !== void 0 ? _z : GeneratedSource,
            }, continuationBlock);
            return;
        }
        case 'VariableDeclaration': {
            const stmt = stmtPath;
            const nodeKind = stmt.node.kind;
            if (nodeKind === 'var') {
                builder.errors.push({
                    reason: `(BuildHIR::lowerStatement) Handle ${nodeKind} kinds in VariableDeclaration`,
                    category: ErrorCategory.Todo,
                    loc: (_0 = stmt.node.loc) !== null && _0 !== void 0 ? _0 : null,
                    suggestions: null,
                });
                return;
            }
            const kind = nodeKind === 'let' ? InstructionKind.Let : InstructionKind.Const;
            for (const declaration of stmt.get('declarations')) {
                const id = declaration.get('id');
                const init = declaration.get('init');
                if (hasNode(init)) {
                    const value = lowerExpressionToTemporary(builder, init);
                    lowerAssignment(builder, (_1 = stmt.node.loc) !== null && _1 !== void 0 ? _1 : GeneratedSource, kind, id, value, id.isObjectPattern() || id.isArrayPattern()
                        ? 'Destructure'
                        : 'Assignment');
                }
                else if (id.isIdentifier()) {
                    const binding = builder.resolveIdentifier(id);
                    if (binding.kind !== 'Identifier') {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerAssignment) Could not find binding for declaration.`,
                            category: ErrorCategory.Invariant,
                            loc: (_2 = id.node.loc) !== null && _2 !== void 0 ? _2 : null,
                            suggestions: null,
                        });
                    }
                    else {
                        const place = {
                            effect: Effect.Unknown,
                            identifier: binding.identifier,
                            kind: 'Identifier',
                            reactive: false,
                            loc: (_3 = id.node.loc) !== null && _3 !== void 0 ? _3 : GeneratedSource,
                        };
                        if (builder.isContextIdentifier(id)) {
                            if (kind === InstructionKind.Const) {
                                const declRangeStart = declaration.parentPath.node.start;
                                builder.errors.push({
                                    reason: `Expect \`const\` declaration not to be reassigned`,
                                    category: ErrorCategory.Syntax,
                                    loc: (_4 = id.node.loc) !== null && _4 !== void 0 ? _4 : null,
                                    suggestions: [
                                        {
                                            description: 'Change to a `let` declaration',
                                            op: CompilerSuggestionOperation.Replace,
                                            range: [declRangeStart, declRangeStart + 5],
                                            text: 'let',
                                        },
                                    ],
                                });
                            }
                            lowerValueToTemporary(builder, {
                                kind: 'DeclareContext',
                                lvalue: {
                                    kind: InstructionKind.Let,
                                    place,
                                },
                                loc: (_5 = id.node.loc) !== null && _5 !== void 0 ? _5 : GeneratedSource,
                            });
                        }
                        else {
                            const typeAnnotation = id.get('typeAnnotation');
                            let type;
                            if (typeAnnotation.isTSTypeAnnotation()) {
                                const typePath = typeAnnotation.get('typeAnnotation');
                                type = typePath.node;
                            }
                            else if (typeAnnotation.isTypeAnnotation()) {
                                const typePath = typeAnnotation.get('typeAnnotation');
                                type = typePath.node;
                            }
                            else {
                                type = null;
                            }
                            lowerValueToTemporary(builder, {
                                kind: 'DeclareLocal',
                                lvalue: {
                                    kind,
                                    place,
                                },
                                type,
                                loc: (_6 = id.node.loc) !== null && _6 !== void 0 ? _6 : GeneratedSource,
                            });
                        }
                    }
                }
                else {
                    builder.errors.push({
                        reason: `Expected variable declaration to be an identifier if no initializer was provided`,
                        description: `Got a \`${id.type}\``,
                        category: ErrorCategory.Syntax,
                        loc: (_7 = stmt.node.loc) !== null && _7 !== void 0 ? _7 : null,
                        suggestions: null,
                    });
                }
            }
            return;
        }
        case 'ExpressionStatement': {
            const stmt = stmtPath;
            const expression = stmt.get('expression');
            lowerExpressionToTemporary(builder, expression);
            return;
        }
        case 'DoWhileStatement': {
            const stmt = stmtPath;
            const conditionalBlock = builder.reserve('loop');
            const continuationBlock = builder.reserve('block');
            const loopBlock = builder.enter('block', _loopBlockId => {
                return builder.loop(label, conditionalBlock.id, continuationBlock.id, () => {
                    var _a;
                    const body = stmt.get('body');
                    lowerStatement(builder, body);
                    return {
                        kind: 'goto',
                        block: conditionalBlock.id,
                        variant: GotoVariant.Continue,
                        id: makeInstructionId(0),
                        loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    };
                });
            });
            const loc = (_8 = stmt.node.loc) !== null && _8 !== void 0 ? _8 : GeneratedSource;
            builder.terminateWithContinuation({
                kind: 'do-while',
                loc,
                test: conditionalBlock.id,
                loop: loopBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
            }, conditionalBlock);
            const test = lowerExpressionToTemporary(builder, stmt.get('test'));
            const terminal = {
                kind: 'branch',
                test,
                consequent: loopBlock,
                alternate: continuationBlock.id,
                fallthrough: conditionalBlock.id,
                id: makeInstructionId(0),
                loc,
            };
            builder.terminateWithContinuation(terminal, continuationBlock);
            return;
        }
        case 'FunctionDeclaration': {
            const stmt = stmtPath;
            stmt.skip();
            CompilerError.invariant(stmt.get('id').type === 'Identifier', {
                reason: 'function declarations must have a name',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: (_9 = stmt.node.loc) !== null && _9 !== void 0 ? _9 : null,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            const id = stmt.get('id');
            const fn = lowerValueToTemporary(builder, lowerFunctionToValue(builder, stmt));
            lowerAssignment(builder, (_10 = stmt.node.loc) !== null && _10 !== void 0 ? _10 : GeneratedSource, InstructionKind.Function, id, fn, 'Assignment');
            return;
        }
        case 'ForOfStatement': {
            const stmt = stmtPath;
            const continuationBlock = builder.reserve('block');
            const initBlock = builder.reserve('loop');
            const testBlock = builder.reserve('loop');
            if (stmt.node.await) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerStatement) Handle for-await loops`,
                    category: ErrorCategory.Todo,
                    loc: (_11 = stmt.node.loc) !== null && _11 !== void 0 ? _11 : null,
                    suggestions: null,
                });
                return;
            }
            const loopBlock = builder.enter('block', _blockId => {
                return builder.loop(label, initBlock.id, continuationBlock.id, () => {
                    var _a;
                    const body = stmt.get('body');
                    lowerStatement(builder, body);
                    return {
                        kind: 'goto',
                        block: initBlock.id,
                        variant: GotoVariant.Continue,
                        id: makeInstructionId(0),
                        loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    };
                });
            });
            const loc = (_12 = stmt.node.loc) !== null && _12 !== void 0 ? _12 : GeneratedSource;
            const value = lowerExpressionToTemporary(builder, stmt.get('right'));
            builder.terminateWithContinuation({
                kind: 'for-of',
                loc,
                init: initBlock.id,
                test: testBlock.id,
                loop: loopBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
            }, initBlock);
            const iterator = lowerValueToTemporary(builder, {
                kind: 'GetIterator',
                loc: value.loc,
                collection: Object.assign({}, value),
            });
            builder.terminateWithContinuation({
                id: makeInstructionId(0),
                kind: 'goto',
                block: testBlock.id,
                variant: GotoVariant.Break,
                loc: (_13 = stmt.node.loc) !== null && _13 !== void 0 ? _13 : GeneratedSource,
            }, testBlock);
            const left = stmt.get('left');
            const leftLoc = (_14 = left.node.loc) !== null && _14 !== void 0 ? _14 : GeneratedSource;
            let test;
            const advanceIterator = lowerValueToTemporary(builder, {
                kind: 'IteratorNext',
                loc: leftLoc,
                iterator: Object.assign({}, iterator),
                collection: Object.assign({}, value),
            });
            if (left.isVariableDeclaration()) {
                const declarations = left.get('declarations');
                CompilerError.invariant(declarations.length === 1, {
                    reason: `Expected only one declaration in the init of a ForOfStatement, got ${declarations.length}`,
                    description: null,
                    details: [
                        {
                            kind: 'error',
                            loc: (_15 = left.node.loc) !== null && _15 !== void 0 ? _15 : null,
                            message: null,
                        },
                    ],
                    suggestions: null,
                });
                const id = declarations[0].get('id');
                const assign = lowerAssignment(builder, leftLoc, InstructionKind.Let, id, advanceIterator, 'Assignment');
                test = lowerValueToTemporary(builder, assign);
            }
            else {
                CompilerError.invariant(left.isLVal(), {
                    reason: 'Expected ForOf init to be a variable declaration or lval',
                    description: null,
                    details: [
                        {
                            kind: 'error',
                            loc: leftLoc,
                            message: null,
                        },
                    ],
                });
                const assign = lowerAssignment(builder, leftLoc, InstructionKind.Reassign, left, advanceIterator, 'Assignment');
                test = lowerValueToTemporary(builder, assign);
            }
            builder.terminateWithContinuation({
                id: makeInstructionId(0),
                kind: 'branch',
                test,
                consequent: loopBlock,
                alternate: continuationBlock.id,
                loc: (_16 = stmt.node.loc) !== null && _16 !== void 0 ? _16 : GeneratedSource,
                fallthrough: continuationBlock.id,
            }, continuationBlock);
            return;
        }
        case 'ForInStatement': {
            const stmt = stmtPath;
            const continuationBlock = builder.reserve('block');
            const initBlock = builder.reserve('loop');
            const loopBlock = builder.enter('block', _blockId => {
                return builder.loop(label, initBlock.id, continuationBlock.id, () => {
                    var _a;
                    const body = stmt.get('body');
                    lowerStatement(builder, body);
                    return {
                        kind: 'goto',
                        block: initBlock.id,
                        variant: GotoVariant.Continue,
                        id: makeInstructionId(0),
                        loc: (_a = body.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    };
                });
            });
            const loc = (_17 = stmt.node.loc) !== null && _17 !== void 0 ? _17 : GeneratedSource;
            const value = lowerExpressionToTemporary(builder, stmt.get('right'));
            builder.terminateWithContinuation({
                kind: 'for-in',
                loc,
                init: initBlock.id,
                loop: loopBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
            }, initBlock);
            const left = stmt.get('left');
            const leftLoc = (_18 = left.node.loc) !== null && _18 !== void 0 ? _18 : GeneratedSource;
            let test;
            const nextPropertyTemp = lowerValueToTemporary(builder, {
                kind: 'NextPropertyOf',
                loc: leftLoc,
                value,
            });
            if (left.isVariableDeclaration()) {
                const declarations = left.get('declarations');
                CompilerError.invariant(declarations.length === 1, {
                    reason: `Expected only one declaration in the init of a ForInStatement, got ${declarations.length}`,
                    description: null,
                    details: [
                        {
                            kind: 'error',
                            loc: (_19 = left.node.loc) !== null && _19 !== void 0 ? _19 : null,
                            message: null,
                        },
                    ],
                    suggestions: null,
                });
                const id = declarations[0].get('id');
                const assign = lowerAssignment(builder, leftLoc, InstructionKind.Let, id, nextPropertyTemp, 'Assignment');
                test = lowerValueToTemporary(builder, assign);
            }
            else {
                CompilerError.invariant(left.isLVal(), {
                    reason: 'Expected ForIn init to be a variable declaration or lval',
                    description: null,
                    details: [
                        {
                            kind: 'error',
                            loc: leftLoc,
                            message: null,
                        },
                    ],
                });
                const assign = lowerAssignment(builder, leftLoc, InstructionKind.Reassign, left, nextPropertyTemp, 'Assignment');
                test = lowerValueToTemporary(builder, assign);
            }
            builder.terminateWithContinuation({
                id: makeInstructionId(0),
                kind: 'branch',
                test,
                consequent: loopBlock,
                alternate: continuationBlock.id,
                fallthrough: continuationBlock.id,
                loc: (_20 = stmt.node.loc) !== null && _20 !== void 0 ? _20 : GeneratedSource,
            }, continuationBlock);
            return;
        }
        case 'DebuggerStatement': {
            const stmt = stmtPath;
            const loc = (_21 = stmt.node.loc) !== null && _21 !== void 0 ? _21 : GeneratedSource;
            builder.push({
                id: makeInstructionId(0),
                lvalue: buildTemporaryPlace(builder, loc),
                value: {
                    kind: 'Debugger',
                    loc,
                },
                effects: null,
                loc,
            });
            return;
        }
        case 'EmptyStatement': {
            return;
        }
        case 'TryStatement': {
            const stmt = stmtPath;
            const continuationBlock = builder.reserve('block');
            const handlerPath = stmt.get('handler');
            if (!hasNode(handlerPath)) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerStatement) Handle TryStatement without a catch clause`,
                    category: ErrorCategory.Todo,
                    loc: (_22 = stmt.node.loc) !== null && _22 !== void 0 ? _22 : null,
                    suggestions: null,
                });
                return;
            }
            if (hasNode(stmt.get('finalizer'))) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerStatement) Handle TryStatement with a finalizer ('finally') clause`,
                    category: ErrorCategory.Todo,
                    loc: (_23 = stmt.node.loc) !== null && _23 !== void 0 ? _23 : null,
                    suggestions: null,
                });
            }
            const handlerBindingPath = handlerPath.get('param');
            let handlerBinding = null;
            if (hasNode(handlerBindingPath)) {
                const place = {
                    kind: 'Identifier',
                    identifier: builder.makeTemporary((_24 = handlerBindingPath.node.loc) !== null && _24 !== void 0 ? _24 : GeneratedSource),
                    effect: Effect.Unknown,
                    reactive: false,
                    loc: (_25 = handlerBindingPath.node.loc) !== null && _25 !== void 0 ? _25 : GeneratedSource,
                };
                promoteTemporary(place.identifier);
                lowerValueToTemporary(builder, {
                    kind: 'DeclareLocal',
                    lvalue: {
                        kind: InstructionKind.Catch,
                        place: Object.assign({}, place),
                    },
                    type: null,
                    loc: (_26 = handlerBindingPath.node.loc) !== null && _26 !== void 0 ? _26 : GeneratedSource,
                });
                handlerBinding = {
                    path: handlerBindingPath,
                    place,
                };
            }
            const handler = builder.enter('catch', _blockId => {
                var _a, _b;
                if (handlerBinding !== null) {
                    lowerAssignment(builder, (_a = handlerBinding.path.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource, InstructionKind.Catch, handlerBinding.path, Object.assign({}, handlerBinding.place), 'Assignment');
                }
                lowerStatement(builder, handlerPath.get('body'));
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: (_b = handlerPath.node.loc) !== null && _b !== void 0 ? _b : GeneratedSource,
                };
            });
            const block = builder.enter('block', _blockId => {
                var _a;
                const block = stmt.get('block');
                builder.enterTryCatch(handler, () => {
                    lowerStatement(builder, block);
                });
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Try,
                    id: makeInstructionId(0),
                    loc: (_a = block.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                };
            });
            builder.terminateWithContinuation({
                kind: 'try',
                block,
                handlerBinding: handlerBinding !== null ? Object.assign({}, handlerBinding.place) : null,
                handler,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: (_27 = stmt.node.loc) !== null && _27 !== void 0 ? _27 : GeneratedSource,
            }, continuationBlock);
            return;
        }
        case 'WithStatement': {
            builder.errors.push({
                reason: `JavaScript 'with' syntax is not supported`,
                description: `'with' syntax is considered deprecated and removed from JavaScript standards, consider alternatives`,
                category: ErrorCategory.UnsupportedSyntax,
                loc: (_28 = stmtPath.node.loc) !== null && _28 !== void 0 ? _28 : null,
                suggestions: null,
            });
            lowerValueToTemporary(builder, {
                kind: 'UnsupportedNode',
                loc: (_29 = stmtPath.node.loc) !== null && _29 !== void 0 ? _29 : GeneratedSource,
                node: stmtPath.node,
            });
            return;
        }
        case 'ClassDeclaration': {
            builder.errors.push({
                reason: 'Inline `class` declarations are not supported',
                description: `Move class declarations outside of components/hooks`,
                category: ErrorCategory.UnsupportedSyntax,
                loc: (_30 = stmtPath.node.loc) !== null && _30 !== void 0 ? _30 : null,
                suggestions: null,
            });
            lowerValueToTemporary(builder, {
                kind: 'UnsupportedNode',
                loc: (_31 = stmtPath.node.loc) !== null && _31 !== void 0 ? _31 : GeneratedSource,
                node: stmtPath.node,
            });
            return;
        }
        case 'EnumDeclaration':
        case 'TSEnumDeclaration': {
            lowerValueToTemporary(builder, {
                kind: 'UnsupportedNode',
                loc: (_32 = stmtPath.node.loc) !== null && _32 !== void 0 ? _32 : GeneratedSource,
                node: stmtPath.node,
            });
            return;
        }
        case 'ExportAllDeclaration':
        case 'ExportDefaultDeclaration':
        case 'ExportNamedDeclaration':
        case 'ImportDeclaration':
        case 'TSExportAssignment':
        case 'TSImportEqualsDeclaration': {
            builder.errors.push({
                reason: 'JavaScript `import` and `export` statements may only appear at the top level of a module',
                category: ErrorCategory.Syntax,
                loc: (_33 = stmtPath.node.loc) !== null && _33 !== void 0 ? _33 : null,
                suggestions: null,
            });
            lowerValueToTemporary(builder, {
                kind: 'UnsupportedNode',
                loc: (_34 = stmtPath.node.loc) !== null && _34 !== void 0 ? _34 : GeneratedSource,
                node: stmtPath.node,
            });
            return;
        }
        case 'TSNamespaceExportDeclaration': {
            builder.errors.push({
                reason: 'TypeScript `namespace` statements may only appear at the top level of a module',
                category: ErrorCategory.Syntax,
                loc: (_35 = stmtPath.node.loc) !== null && _35 !== void 0 ? _35 : null,
                suggestions: null,
            });
            lowerValueToTemporary(builder, {
                kind: 'UnsupportedNode',
                loc: (_36 = stmtPath.node.loc) !== null && _36 !== void 0 ? _36 : GeneratedSource,
                node: stmtPath.node,
            });
            return;
        }
        case 'DeclareClass':
        case 'DeclareExportAllDeclaration':
        case 'DeclareExportDeclaration':
        case 'DeclareFunction':
        case 'DeclareInterface':
        case 'DeclareModule':
        case 'DeclareModuleExports':
        case 'DeclareOpaqueType':
        case 'DeclareTypeAlias':
        case 'DeclareVariable':
        case 'InterfaceDeclaration':
        case 'OpaqueType':
        case 'TSDeclareFunction':
        case 'TSInterfaceDeclaration':
        case 'TSModuleDeclaration':
        case 'TSTypeAliasDeclaration':
        case 'TypeAlias': {
            return;
        }
        default: {
            return assertExhaustive$1(stmtNode, `Unsupported statement kind '${stmtNode.type}'`);
        }
    }
}
function lowerObjectMethod(builder, property) {
    var _a;
    const loc = (_a = property.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const loweredFunc = lowerFunction(builder, property);
    if (!loweredFunc) {
        return { kind: 'UnsupportedNode', node: property.node, loc: loc };
    }
    return {
        kind: 'ObjectMethod',
        loc,
        loweredFunc,
    };
}
function lowerObjectPropertyKey(builder, property) {
    var _a;
    const key = property.get('key');
    if (key.isStringLiteral()) {
        return {
            kind: 'string',
            name: key.node.value,
        };
    }
    else if (property.node.computed && key.isExpression()) {
        const place = lowerExpressionToTemporary(builder, key);
        return {
            kind: 'computed',
            name: place,
        };
    }
    else if (key.isIdentifier()) {
        return {
            kind: 'identifier',
            name: key.node.name,
        };
    }
    else if (key.isNumericLiteral()) {
        return {
            kind: 'identifier',
            name: String(key.node.value),
        };
    }
    builder.errors.push({
        reason: `(BuildHIR::lowerExpression) Expected Identifier, got ${key.type} key in ObjectExpression`,
        category: ErrorCategory.Todo,
        loc: (_a = key.node.loc) !== null && _a !== void 0 ? _a : null,
        suggestions: null,
    });
    return null;
}
function lowerExpression(builder, exprPath) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23;
    const exprNode = exprPath.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    switch (exprNode.type) {
        case 'Identifier': {
            const expr = exprPath;
            const place = lowerIdentifier(builder, expr);
            return {
                kind: getLoadKind(builder, expr),
                place,
                loc: exprLoc,
            };
        }
        case 'NullLiteral': {
            return {
                kind: 'Primitive',
                value: null,
                loc: exprLoc,
            };
        }
        case 'BooleanLiteral':
        case 'NumericLiteral':
        case 'StringLiteral': {
            const expr = exprPath;
            const value = expr.node.value;
            return {
                kind: 'Primitive',
                value,
                loc: exprLoc,
            };
        }
        case 'ObjectExpression': {
            const expr = exprPath;
            const propertyPaths = expr.get('properties');
            const properties = [];
            for (const propertyPath of propertyPaths) {
                if (propertyPath.isObjectProperty()) {
                    const loweredKey = lowerObjectPropertyKey(builder, propertyPath);
                    if (!loweredKey) {
                        continue;
                    }
                    const valuePath = propertyPath.get('value');
                    if (!valuePath.isExpression()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerExpression) Handle ${valuePath.type} values in ObjectExpression`,
                            category: ErrorCategory.Todo,
                            loc: (_b = valuePath.node.loc) !== null && _b !== void 0 ? _b : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    const value = lowerExpressionToTemporary(builder, valuePath);
                    properties.push({
                        kind: 'ObjectProperty',
                        type: 'property',
                        place: value,
                        key: loweredKey,
                    });
                }
                else if (propertyPath.isSpreadElement()) {
                    const place = lowerExpressionToTemporary(builder, propertyPath.get('argument'));
                    properties.push({
                        kind: 'Spread',
                        place,
                    });
                }
                else if (propertyPath.isObjectMethod()) {
                    if (propertyPath.node.kind !== 'method') {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerExpression) Handle ${propertyPath.node.kind} functions in ObjectExpression`,
                            category: ErrorCategory.Todo,
                            loc: (_c = propertyPath.node.loc) !== null && _c !== void 0 ? _c : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    const method = lowerObjectMethod(builder, propertyPath);
                    const place = lowerValueToTemporary(builder, method);
                    const loweredKey = lowerObjectPropertyKey(builder, propertyPath);
                    if (!loweredKey) {
                        continue;
                    }
                    properties.push({
                        kind: 'ObjectProperty',
                        type: 'method',
                        place,
                        key: loweredKey,
                    });
                }
                else {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Handle ${propertyPath.type} properties in ObjectExpression`,
                        category: ErrorCategory.Todo,
                        loc: (_d = propertyPath.node.loc) !== null && _d !== void 0 ? _d : null,
                        suggestions: null,
                    });
                    continue;
                }
            }
            return {
                kind: 'ObjectExpression',
                properties,
                loc: exprLoc,
            };
        }
        case 'ArrayExpression': {
            const expr = exprPath;
            let elements = [];
            for (const element of expr.get('elements')) {
                if (element.node == null) {
                    elements.push({
                        kind: 'Hole',
                    });
                    continue;
                }
                else if (element.isExpression()) {
                    elements.push(lowerExpressionToTemporary(builder, element));
                }
                else if (element.isSpreadElement()) {
                    const place = lowerExpressionToTemporary(builder, element.get('argument'));
                    elements.push({ kind: 'Spread', place });
                }
                else {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Handle ${element.type} elements in ArrayExpression`,
                        category: ErrorCategory.Todo,
                        loc: (_e = element.node.loc) !== null && _e !== void 0 ? _e : null,
                        suggestions: null,
                    });
                    continue;
                }
            }
            return {
                kind: 'ArrayExpression',
                elements,
                loc: exprLoc,
            };
        }
        case 'NewExpression': {
            const expr = exprPath;
            const calleePath = expr.get('callee');
            if (!calleePath.isExpression()) {
                builder.errors.push({
                    reason: `Expected an expression as the \`new\` expression receiver (v8 intrinsics are not supported)`,
                    description: `Got a \`${calleePath.node.type}\``,
                    category: ErrorCategory.Syntax,
                    loc: (_f = calleePath.node.loc) !== null && _f !== void 0 ? _f : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const callee = lowerExpressionToTemporary(builder, calleePath);
            const args = lowerArguments(builder, expr.get('arguments'));
            return {
                kind: 'NewExpression',
                callee,
                args,
                loc: exprLoc,
            };
        }
        case 'OptionalCallExpression': {
            const expr = exprPath;
            return lowerOptionalCallExpression(builder, expr, null);
        }
        case 'CallExpression': {
            const expr = exprPath;
            const calleePath = expr.get('callee');
            if (!calleePath.isExpression()) {
                builder.errors.push({
                    reason: `Expected Expression, got ${calleePath.type} in CallExpression (v8 intrinsics not supported). This error is likely caused by a bug in React Compiler. Please file an issue`,
                    category: ErrorCategory.Todo,
                    loc: (_g = calleePath.node.loc) !== null && _g !== void 0 ? _g : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            if (calleePath.isMemberExpression()) {
                const memberExpr = lowerMemberExpression(builder, calleePath);
                const propertyPlace = lowerValueToTemporary(builder, memberExpr.value);
                const args = lowerArguments(builder, expr.get('arguments'));
                return {
                    kind: 'MethodCall',
                    receiver: memberExpr.object,
                    property: Object.assign({}, propertyPlace),
                    args,
                    loc: exprLoc,
                };
            }
            else {
                const callee = lowerExpressionToTemporary(builder, calleePath);
                const args = lowerArguments(builder, expr.get('arguments'));
                return {
                    kind: 'CallExpression',
                    callee,
                    args,
                    loc: exprLoc,
                };
            }
        }
        case 'BinaryExpression': {
            const expr = exprPath;
            const leftPath = expr.get('left');
            if (!leftPath.isExpression()) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Expected Expression, got ${leftPath.type} lval in BinaryExpression`,
                    category: ErrorCategory.Todo,
                    loc: (_h = leftPath.node.loc) !== null && _h !== void 0 ? _h : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const left = lowerExpressionToTemporary(builder, leftPath);
            const right = lowerExpressionToTemporary(builder, expr.get('right'));
            const operator = expr.node.operator;
            if (operator === '|>') {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Pipe operator not supported`,
                    category: ErrorCategory.Todo,
                    loc: (_j = leftPath.node.loc) !== null && _j !== void 0 ? _j : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            return {
                kind: 'BinaryExpression',
                operator,
                left,
                right,
                loc: exprLoc,
            };
        }
        case 'SequenceExpression': {
            const expr = exprPath;
            const exprLoc = (_k = expr.node.loc) !== null && _k !== void 0 ? _k : GeneratedSource;
            const continuationBlock = builder.reserve(builder.currentBlockKind());
            const place = buildTemporaryPlace(builder, exprLoc);
            const sequenceBlock = builder.enter('sequence', _ => {
                var _a;
                let last = null;
                for (const item of expr.get('expressions')) {
                    last = lowerExpressionToTemporary(builder, item);
                }
                if (last === null) {
                    builder.errors.push({
                        reason: `Expected sequence expression to have at least one expression`,
                        category: ErrorCategory.Syntax,
                        loc: (_a = expr.node.loc) !== null && _a !== void 0 ? _a : null,
                        suggestions: null,
                    });
                }
                else {
                    lowerValueToTemporary(builder, {
                        kind: 'StoreLocal',
                        lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                        value: last,
                        type: null,
                        loc: exprLoc,
                    });
                }
                return {
                    kind: 'goto',
                    id: makeInstructionId(0),
                    block: continuationBlock.id,
                    loc: exprLoc,
                    variant: GotoVariant.Break,
                };
            });
            builder.terminateWithContinuation({
                kind: 'sequence',
                block: sequenceBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: exprLoc,
            }, continuationBlock);
            return { kind: 'LoadLocal', place, loc: place.loc };
        }
        case 'ConditionalExpression': {
            const expr = exprPath;
            const exprLoc = (_l = expr.node.loc) !== null && _l !== void 0 ? _l : GeneratedSource;
            const continuationBlock = builder.reserve(builder.currentBlockKind());
            const testBlock = builder.reserve('value');
            const place = buildTemporaryPlace(builder, exprLoc);
            const consequentBlock = builder.enter('value', _blockId => {
                var _a;
                const consequentPath = expr.get('consequent');
                const consequent = lowerExpressionToTemporary(builder, consequentPath);
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                    value: consequent,
                    type: null,
                    loc: exprLoc,
                });
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: (_a = consequentPath.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                };
            });
            const alternateBlock = builder.enter('value', _blockId => {
                var _a;
                const alternatePath = expr.get('alternate');
                const alternate = lowerExpressionToTemporary(builder, alternatePath);
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                    value: alternate,
                    type: null,
                    loc: exprLoc,
                });
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: (_a = alternatePath.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                };
            });
            builder.terminateWithContinuation({
                kind: 'ternary',
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                test: testBlock.id,
                loc: exprLoc,
            }, testBlock);
            const testPlace = lowerExpressionToTemporary(builder, expr.get('test'));
            builder.terminateWithContinuation({
                kind: 'branch',
                test: Object.assign({}, testPlace),
                consequent: consequentBlock,
                alternate: alternateBlock,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: exprLoc,
            }, continuationBlock);
            return { kind: 'LoadLocal', place, loc: place.loc };
        }
        case 'LogicalExpression': {
            const expr = exprPath;
            const exprLoc = (_m = expr.node.loc) !== null && _m !== void 0 ? _m : GeneratedSource;
            const continuationBlock = builder.reserve(builder.currentBlockKind());
            const testBlock = builder.reserve('value');
            const place = buildTemporaryPlace(builder, exprLoc);
            const leftPlace = buildTemporaryPlace(builder, (_o = expr.get('left').node.loc) !== null && _o !== void 0 ? _o : GeneratedSource);
            const consequent = builder.enter('value', () => {
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                    value: Object.assign({}, leftPlace),
                    type: null,
                    loc: leftPlace.loc,
                });
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: leftPlace.loc,
                };
            });
            const alternate = builder.enter('value', () => {
                const right = lowerExpressionToTemporary(builder, expr.get('right'));
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                    value: Object.assign({}, right),
                    type: null,
                    loc: right.loc,
                });
                return {
                    kind: 'goto',
                    block: continuationBlock.id,
                    variant: GotoVariant.Break,
                    id: makeInstructionId(0),
                    loc: right.loc,
                };
            });
            builder.terminateWithContinuation({
                kind: 'logical',
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                test: testBlock.id,
                operator: expr.node.operator,
                loc: exprLoc,
            }, testBlock);
            const leftValue = lowerExpressionToTemporary(builder, expr.get('left'));
            builder.push({
                id: makeInstructionId(0),
                lvalue: Object.assign({}, leftPlace),
                value: {
                    kind: 'LoadLocal',
                    place: leftValue,
                    loc: exprLoc,
                },
                effects: null,
                loc: exprLoc,
            });
            builder.terminateWithContinuation({
                kind: 'branch',
                test: Object.assign({}, leftPlace),
                consequent,
                alternate,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc: exprLoc,
            }, continuationBlock);
            return { kind: 'LoadLocal', place, loc: place.loc };
        }
        case 'AssignmentExpression': {
            const expr = exprPath;
            const operator = expr.node.operator;
            if (operator === '=') {
                const left = expr.get('left');
                if (left.isLVal()) {
                    return lowerAssignment(builder, (_p = left.node.loc) !== null && _p !== void 0 ? _p : GeneratedSource, InstructionKind.Reassign, left, lowerExpressionToTemporary(builder, expr.get('right')), left.isArrayPattern() || left.isObjectPattern()
                        ? 'Destructure'
                        : 'Assignment');
                }
                else {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Unsupported syntax on the left side of an AssignmentExpression`,
                        description: `Expected an LVal, got: ${left.type}`,
                        category: ErrorCategory.Todo,
                        loc: (_q = left.node.loc) !== null && _q !== void 0 ? _q : null,
                        suggestions: null,
                    });
                    return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
                }
            }
            const operators = {
                '+=': '+',
                '-=': '-',
                '/=': '/',
                '%=': '%',
                '*=': '*',
                '**=': '**',
                '&=': '&',
                '|=': '|',
                '>>=': '>>',
                '>>>=': '>>>',
                '<<=': '<<',
                '^=': '^',
            };
            const binaryOperator = operators[operator];
            if (binaryOperator == null) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Handle ${operator} operators in AssignmentExpression`,
                    category: ErrorCategory.Todo,
                    loc: (_r = expr.node.loc) !== null && _r !== void 0 ? _r : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const left = expr.get('left');
            const leftNode = left.node;
            switch (leftNode.type) {
                case 'Identifier': {
                    const leftExpr = left;
                    const leftPlace = lowerExpressionToTemporary(builder, leftExpr);
                    const right = lowerExpressionToTemporary(builder, expr.get('right'));
                    const binaryPlace = lowerValueToTemporary(builder, {
                        kind: 'BinaryExpression',
                        operator: binaryOperator,
                        left: leftPlace,
                        right,
                        loc: exprLoc,
                    });
                    const binding = builder.resolveIdentifier(leftExpr);
                    if (binding.kind === 'Identifier') {
                        const identifier = lowerIdentifier(builder, leftExpr);
                        const kind = getStoreKind(builder, leftExpr);
                        if (kind === 'StoreLocal') {
                            lowerValueToTemporary(builder, {
                                kind: 'StoreLocal',
                                lvalue: {
                                    place: Object.assign({}, identifier),
                                    kind: InstructionKind.Reassign,
                                },
                                value: Object.assign({}, binaryPlace),
                                type: null,
                                loc: exprLoc,
                            });
                            return { kind: 'LoadLocal', place: identifier, loc: exprLoc };
                        }
                        else {
                            lowerValueToTemporary(builder, {
                                kind: 'StoreContext',
                                lvalue: {
                                    place: Object.assign({}, identifier),
                                    kind: InstructionKind.Reassign,
                                },
                                value: Object.assign({}, binaryPlace),
                                loc: exprLoc,
                            });
                            return { kind: 'LoadContext', place: identifier, loc: exprLoc };
                        }
                    }
                    else {
                        const temporary = lowerValueToTemporary(builder, {
                            kind: 'StoreGlobal',
                            name: leftExpr.node.name,
                            value: Object.assign({}, binaryPlace),
                            loc: exprLoc,
                        });
                        return { kind: 'LoadLocal', place: temporary, loc: temporary.loc };
                    }
                }
                case 'MemberExpression': {
                    const leftExpr = left;
                    const { object, property, value } = lowerMemberExpression(builder, leftExpr);
                    const previousValuePlace = lowerValueToTemporary(builder, value);
                    const newValuePlace = lowerValueToTemporary(builder, {
                        kind: 'BinaryExpression',
                        operator: binaryOperator,
                        left: Object.assign({}, previousValuePlace),
                        right: lowerExpressionToTemporary(builder, expr.get('right')),
                        loc: (_s = leftExpr.node.loc) !== null && _s !== void 0 ? _s : GeneratedSource,
                    });
                    if (typeof property === 'string' || typeof property === 'number') {
                        return {
                            kind: 'PropertyStore',
                            object: Object.assign({}, object),
                            property: makePropertyLiteral(property),
                            value: Object.assign({}, newValuePlace),
                            loc: (_t = leftExpr.node.loc) !== null && _t !== void 0 ? _t : GeneratedSource,
                        };
                    }
                    else {
                        return {
                            kind: 'ComputedStore',
                            object: Object.assign({}, object),
                            property: Object.assign({}, property),
                            value: Object.assign({}, newValuePlace),
                            loc: (_u = leftExpr.node.loc) !== null && _u !== void 0 ? _u : GeneratedSource,
                        };
                    }
                }
                default: {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Expected Identifier or MemberExpression, got ${expr.type} lval in AssignmentExpression`,
                        category: ErrorCategory.Todo,
                        loc: (_v = expr.node.loc) !== null && _v !== void 0 ? _v : null,
                        suggestions: null,
                    });
                    return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
                }
            }
        }
        case 'OptionalMemberExpression': {
            const expr = exprPath;
            const { value } = lowerOptionalMemberExpression(builder, expr, null);
            return { kind: 'LoadLocal', place: value, loc: value.loc };
        }
        case 'MemberExpression': {
            const expr = exprPath;
            const { value } = lowerMemberExpression(builder, expr);
            const place = lowerValueToTemporary(builder, value);
            return { kind: 'LoadLocal', place, loc: place.loc };
        }
        case 'JSXElement': {
            const expr = exprPath;
            const opening = expr.get('openingElement');
            const openingLoc = (_w = opening.node.loc) !== null && _w !== void 0 ? _w : GeneratedSource;
            const tag = lowerJsxElementName(builder, opening.get('name'));
            const props = [];
            for (const attribute of opening.get('attributes')) {
                if (attribute.isJSXSpreadAttribute()) {
                    const argument = lowerExpressionToTemporary(builder, attribute.get('argument'));
                    props.push({ kind: 'JsxSpreadAttribute', argument });
                    continue;
                }
                if (!attribute.isJSXAttribute()) {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Handle ${attribute.type} attributes in JSXElement`,
                        category: ErrorCategory.Todo,
                        loc: (_x = attribute.node.loc) !== null && _x !== void 0 ? _x : null,
                        suggestions: null,
                    });
                    continue;
                }
                const namePath = attribute.get('name');
                let propName;
                if (namePath.isJSXIdentifier()) {
                    propName = namePath.node.name;
                    if (propName.indexOf(':') !== -1) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerExpression) Unexpected colon in attribute name \`${propName}\``,
                            category: ErrorCategory.Todo,
                            loc: (_y = namePath.node.loc) !== null && _y !== void 0 ? _y : null,
                            suggestions: null,
                        });
                    }
                }
                else {
                    CompilerError.invariant(namePath.isJSXNamespacedName(), {
                        reason: 'Refinement',
                        description: null,
                        details: [
                            {
                                kind: 'error',
                                loc: (_z = namePath.node.loc) !== null && _z !== void 0 ? _z : null,
                                message: null,
                            },
                        ],
                        suggestions: null,
                    });
                    const namespace = namePath.node.namespace.name;
                    const name = namePath.node.name.name;
                    propName = `${namespace}:${name}`;
                }
                const valueExpr = attribute.get('value');
                let value;
                if (valueExpr.isJSXElement() || valueExpr.isStringLiteral()) {
                    value = lowerExpressionToTemporary(builder, valueExpr);
                }
                else if (valueExpr.type == null) {
                    value = lowerValueToTemporary(builder, {
                        kind: 'Primitive',
                        value: true,
                        loc: (_0 = attribute.node.loc) !== null && _0 !== void 0 ? _0 : GeneratedSource,
                    });
                }
                else {
                    if (!valueExpr.isJSXExpressionContainer()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerExpression) Handle ${valueExpr.type} attribute values in JSXElement`,
                            category: ErrorCategory.Todo,
                            loc: (_2 = (_1 = valueExpr.node) === null || _1 === void 0 ? void 0 : _1.loc) !== null && _2 !== void 0 ? _2 : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    const expression = valueExpr.get('expression');
                    if (!expression.isExpression()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerExpression) Handle ${expression.type} expressions in JSXExpressionContainer within JSXElement`,
                            category: ErrorCategory.Todo,
                            loc: (_3 = valueExpr.node.loc) !== null && _3 !== void 0 ? _3 : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    value = lowerExpressionToTemporary(builder, expression);
                }
                props.push({ kind: 'JsxAttribute', name: propName, place: value });
            }
            const isFbt = tag.kind === 'BuiltinTag' && (tag.name === 'fbt' || tag.name === 'fbs');
            if (isFbt) {
                const tagName = tag.name;
                const openingIdentifier = opening.get('name');
                const tagIdentifier = openingIdentifier.isJSXIdentifier()
                    ? builder.resolveIdentifier(openingIdentifier)
                    : null;
                if (tagIdentifier != null) {
                    CompilerError.invariant(tagIdentifier.kind !== 'Identifier', {
                        reason: `<${tagName}> tags should be module-level imports`,
                        description: null,
                        details: [
                            {
                                kind: 'error',
                                loc: (_4 = openingIdentifier.node.loc) !== null && _4 !== void 0 ? _4 : GeneratedSource,
                                message: null,
                            },
                        ],
                        suggestions: null,
                    });
                }
                const fbtLocations = {
                    enum: new Array(),
                    plural: new Array(),
                    pronoun: new Array(),
                };
                expr.traverse({
                    JSXClosingElement(path) {
                        path.skip();
                    },
                    JSXNamespacedName(path) {
                        var _a, _b, _c;
                        if (path.node.namespace.name === tagName) {
                            switch (path.node.name.name) {
                                case 'enum':
                                    fbtLocations.enum.push((_a = path.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource);
                                    break;
                                case 'plural':
                                    fbtLocations.plural.push((_b = path.node.loc) !== null && _b !== void 0 ? _b : GeneratedSource);
                                    break;
                                case 'pronoun':
                                    fbtLocations.pronoun.push((_c = path.node.loc) !== null && _c !== void 0 ? _c : GeneratedSource);
                                    break;
                            }
                        }
                    },
                });
                for (const [name, locations] of Object.entries(fbtLocations)) {
                    if (locations.length > 1) {
                        CompilerError.throwDiagnostic({
                            category: ErrorCategory.Todo,
                            reason: 'Support duplicate fbt tags',
                            description: `Support \`<${tagName}>\` tags with multiple \`<${tagName}:${name}>\` values`,
                            details: locations.map(loc => {
                                return {
                                    kind: 'error',
                                    message: `Multiple \`<${tagName}:${name}>\` tags found`,
                                    loc,
                                };
                            }),
                        });
                    }
                }
            }
            isFbt && builder.fbtDepth++;
            const children = expr
                .get('children')
                .map(child => lowerJsxElement(builder, child))
                .filter(notNull);
            isFbt && builder.fbtDepth--;
            return {
                kind: 'JsxExpression',
                tag,
                props,
                children: children.length === 0 ? null : children,
                loc: exprLoc,
                openingLoc: openingLoc,
                closingLoc: (_6 = (_5 = expr.get('closingElement').node) === null || _5 === void 0 ? void 0 : _5.loc) !== null && _6 !== void 0 ? _6 : GeneratedSource,
            };
        }
        case 'JSXFragment': {
            const expr = exprPath;
            const children = expr
                .get('children')
                .map(child => lowerJsxElement(builder, child))
                .filter(notNull);
            return {
                kind: 'JsxFragment',
                children,
                loc: exprLoc,
            };
        }
        case 'ArrowFunctionExpression':
        case 'FunctionExpression': {
            const expr = exprPath;
            return lowerFunctionToValue(builder, expr);
        }
        case 'TaggedTemplateExpression': {
            const expr = exprPath;
            if (expr.get('quasi').get('expressions').length !== 0) {
                builder.errors.push({
                    reason: '(BuildHIR::lowerExpression) Handle tagged template with interpolations',
                    category: ErrorCategory.Todo,
                    loc: (_7 = exprPath.node.loc) !== null && _7 !== void 0 ? _7 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            CompilerError.invariant(expr.get('quasi').get('quasis').length == 1, {
                reason: "there should be only one quasi as we don't support interpolations yet",
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: (_8 = expr.node.loc) !== null && _8 !== void 0 ? _8 : null,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            const value = expr.get('quasi').get('quasis').at(0).node.value;
            if (value.raw !== value.cooked) {
                builder.errors.push({
                    reason: '(BuildHIR::lowerExpression) Handle tagged template where cooked value is different from raw value',
                    category: ErrorCategory.Todo,
                    loc: (_9 = exprPath.node.loc) !== null && _9 !== void 0 ? _9 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            return {
                kind: 'TaggedTemplateExpression',
                tag: lowerExpressionToTemporary(builder, expr.get('tag')),
                value,
                loc: exprLoc,
            };
        }
        case 'TemplateLiteral': {
            const expr = exprPath;
            const subexprs = expr.get('expressions');
            const quasis = expr.get('quasis');
            if (subexprs.length !== quasis.length - 1) {
                builder.errors.push({
                    reason: `Unexpected quasi and subexpression lengths in template literal`,
                    category: ErrorCategory.Syntax,
                    loc: (_10 = exprPath.node.loc) !== null && _10 !== void 0 ? _10 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            if (subexprs.some(e => !e.isExpression())) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerAssignment) Handle TSType in TemplateLiteral.`,
                    category: ErrorCategory.Todo,
                    loc: (_11 = exprPath.node.loc) !== null && _11 !== void 0 ? _11 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const subexprPlaces = subexprs.map(e => lowerExpressionToTemporary(builder, e));
            return {
                kind: 'TemplateLiteral',
                subexprs: subexprPlaces,
                quasis: expr.get('quasis').map(q => q.node.value),
                loc: exprLoc,
            };
        }
        case 'UnaryExpression': {
            let expr = exprPath;
            if (expr.node.operator === 'delete') {
                const argument = expr.get('argument');
                if (argument.isMemberExpression()) {
                    const { object, property } = lowerMemberExpression(builder, argument);
                    if (typeof property === 'string' || typeof property === 'number') {
                        return {
                            kind: 'PropertyDelete',
                            object,
                            property: makePropertyLiteral(property),
                            loc: exprLoc,
                        };
                    }
                    else {
                        return {
                            kind: 'ComputedDelete',
                            object,
                            property,
                            loc: exprLoc,
                        };
                    }
                }
                else {
                    builder.errors.push({
                        reason: `Only object properties can be deleted`,
                        category: ErrorCategory.Syntax,
                        loc: (_12 = expr.node.loc) !== null && _12 !== void 0 ? _12 : null,
                        suggestions: [
                            {
                                description: 'Remove this line',
                                range: [expr.node.start, expr.node.end],
                                op: CompilerSuggestionOperation.Remove,
                            },
                        ],
                    });
                    return { kind: 'UnsupportedNode', node: expr.node, loc: exprLoc };
                }
            }
            else if (expr.node.operator === 'throw') {
                builder.errors.push({
                    reason: `Throw expressions are not supported`,
                    category: ErrorCategory.Syntax,
                    loc: (_13 = expr.node.loc) !== null && _13 !== void 0 ? _13 : null,
                    suggestions: [
                        {
                            description: 'Remove this line',
                            range: [expr.node.start, expr.node.end],
                            op: CompilerSuggestionOperation.Remove,
                        },
                    ],
                });
                return { kind: 'UnsupportedNode', node: expr.node, loc: exprLoc };
            }
            else {
                return {
                    kind: 'UnaryExpression',
                    operator: expr.node.operator,
                    value: lowerExpressionToTemporary(builder, expr.get('argument')),
                    loc: exprLoc,
                };
            }
        }
        case 'AwaitExpression': {
            let expr = exprPath;
            return {
                kind: 'Await',
                value: lowerExpressionToTemporary(builder, expr.get('argument')),
                loc: exprLoc,
            };
        }
        case 'TypeCastExpression': {
            let expr = exprPath;
            const typeAnnotation = expr.get('typeAnnotation').get('typeAnnotation');
            return {
                kind: 'TypeCastExpression',
                value: lowerExpressionToTemporary(builder, expr.get('expression')),
                typeAnnotation: typeAnnotation.node,
                typeAnnotationKind: 'cast',
                type: lowerType(typeAnnotation.node),
                loc: exprLoc,
            };
        }
        case 'TSSatisfiesExpression': {
            let expr = exprPath;
            const typeAnnotation = expr.get('typeAnnotation');
            return {
                kind: 'TypeCastExpression',
                value: lowerExpressionToTemporary(builder, expr.get('expression')),
                typeAnnotation: typeAnnotation.node,
                typeAnnotationKind: 'satisfies',
                type: lowerType(typeAnnotation.node),
                loc: exprLoc,
            };
        }
        case 'TSAsExpression': {
            let expr = exprPath;
            const typeAnnotation = expr.get('typeAnnotation');
            return {
                kind: 'TypeCastExpression',
                value: lowerExpressionToTemporary(builder, expr.get('expression')),
                typeAnnotation: typeAnnotation.node,
                typeAnnotationKind: 'as',
                type: lowerType(typeAnnotation.node),
                loc: exprLoc,
            };
        }
        case 'UpdateExpression': {
            let expr = exprPath;
            const argument = expr.get('argument');
            if (argument.isMemberExpression()) {
                const binaryOperator = expr.node.operator === '++' ? '+' : '-';
                const leftExpr = argument;
                const { object, property, value } = lowerMemberExpression(builder, leftExpr);
                const previousValuePlace = lowerValueToTemporary(builder, value);
                const updatedValue = lowerValueToTemporary(builder, {
                    kind: 'BinaryExpression',
                    operator: binaryOperator,
                    left: Object.assign({}, previousValuePlace),
                    right: lowerValueToTemporary(builder, {
                        kind: 'Primitive',
                        value: 1,
                        loc: GeneratedSource,
                    }),
                    loc: (_14 = leftExpr.node.loc) !== null && _14 !== void 0 ? _14 : GeneratedSource,
                });
                let newValuePlace;
                if (typeof property === 'string' || typeof property === 'number') {
                    newValuePlace = lowerValueToTemporary(builder, {
                        kind: 'PropertyStore',
                        object: Object.assign({}, object),
                        property: makePropertyLiteral(property),
                        value: Object.assign({}, updatedValue),
                        loc: (_15 = leftExpr.node.loc) !== null && _15 !== void 0 ? _15 : GeneratedSource,
                    });
                }
                else {
                    newValuePlace = lowerValueToTemporary(builder, {
                        kind: 'ComputedStore',
                        object: Object.assign({}, object),
                        property: Object.assign({}, property),
                        value: Object.assign({}, updatedValue),
                        loc: (_16 = leftExpr.node.loc) !== null && _16 !== void 0 ? _16 : GeneratedSource,
                    });
                }
                return {
                    kind: 'LoadLocal',
                    place: expr.node.prefix
                        ? Object.assign({}, newValuePlace) : Object.assign({}, previousValuePlace),
                    loc: exprLoc,
                };
            }
            if (!argument.isIdentifier()) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Handle UpdateExpression with ${argument.type} argument`,
                    category: ErrorCategory.Todo,
                    loc: (_17 = exprPath.node.loc) !== null && _17 !== void 0 ? _17 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            else if (builder.isContextIdentifier(argument)) {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Handle UpdateExpression to variables captured within lambdas.`,
                    category: ErrorCategory.Todo,
                    loc: (_18 = exprPath.node.loc) !== null && _18 !== void 0 ? _18 : null,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const lvalue = lowerIdentifierForAssignment(builder, (_19 = argument.node.loc) !== null && _19 !== void 0 ? _19 : GeneratedSource, InstructionKind.Reassign, argument);
            if (lvalue === null) {
                if (!builder.errors.hasAnyErrors()) {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerExpression) Found an invalid UpdateExpression without a previously reported error`,
                        category: ErrorCategory.Invariant,
                        loc: exprLoc,
                        suggestions: null,
                    });
                }
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            else if (lvalue.kind === 'Global') {
                builder.errors.push({
                    reason: `(BuildHIR::lowerExpression) Support UpdateExpression where argument is a global`,
                    category: ErrorCategory.Todo,
                    loc: exprLoc,
                    suggestions: null,
                });
                return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
            }
            const value = lowerIdentifier(builder, argument);
            if (expr.node.prefix) {
                return {
                    kind: 'PrefixUpdate',
                    lvalue,
                    operation: expr.node.operator,
                    value,
                    loc: exprLoc,
                };
            }
            else {
                return {
                    kind: 'PostfixUpdate',
                    lvalue,
                    operation: expr.node.operator,
                    value,
                    loc: exprLoc,
                };
            }
        }
        case 'RegExpLiteral': {
            let expr = exprPath;
            return {
                kind: 'RegExpLiteral',
                pattern: expr.node.pattern,
                flags: expr.node.flags,
                loc: (_20 = expr.node.loc) !== null && _20 !== void 0 ? _20 : GeneratedSource,
            };
        }
        case 'TSInstantiationExpression':
        case 'TSNonNullExpression': {
            let expr = exprPath;
            return lowerExpression(builder, expr.get('expression'));
        }
        case 'MetaProperty': {
            let expr = exprPath;
            if (expr.node.meta.name === 'import' &&
                expr.node.property.name === 'meta') {
                return {
                    kind: 'MetaProperty',
                    meta: expr.node.meta.name,
                    property: expr.node.property.name,
                    loc: (_21 = expr.node.loc) !== null && _21 !== void 0 ? _21 : GeneratedSource,
                };
            }
            builder.errors.push({
                reason: `(BuildHIR::lowerExpression) Handle MetaProperty expressions other than import.meta`,
                category: ErrorCategory.Todo,
                loc: (_22 = exprPath.node.loc) !== null && _22 !== void 0 ? _22 : null,
                suggestions: null,
            });
            return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
        }
        default: {
            builder.errors.push({
                reason: `(BuildHIR::lowerExpression) Handle ${exprPath.type} expressions`,
                category: ErrorCategory.Todo,
                loc: (_23 = exprPath.node.loc) !== null && _23 !== void 0 ? _23 : null,
                suggestions: null,
            });
            return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
        }
    }
}
function lowerOptionalMemberExpression(builder, expr, parentAlternate) {
    var _a;
    const optional = expr.node.optional;
    const loc = (_a = expr.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const place = buildTemporaryPlace(builder, loc);
    const continuationBlock = builder.reserve(builder.currentBlockKind());
    const consequent = builder.reserve('value');
    const alternate = parentAlternate !== null
        ? parentAlternate
        : builder.enter('value', () => {
            const temp = lowerValueToTemporary(builder, {
                kind: 'Primitive',
                value: undefined,
                loc,
            });
            lowerValueToTemporary(builder, {
                kind: 'StoreLocal',
                lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                value: Object.assign({}, temp),
                type: null,
                loc,
            });
            return {
                kind: 'goto',
                variant: GotoVariant.Break,
                block: continuationBlock.id,
                id: makeInstructionId(0),
                loc,
            };
        });
    let object = null;
    const testBlock = builder.enter('value', () => {
        const objectPath = expr.get('object');
        if (objectPath.isOptionalMemberExpression()) {
            const { value } = lowerOptionalMemberExpression(builder, objectPath, alternate);
            object = value;
        }
        else if (objectPath.isOptionalCallExpression()) {
            const value = lowerOptionalCallExpression(builder, objectPath, alternate);
            object = lowerValueToTemporary(builder, value);
        }
        else {
            object = lowerExpressionToTemporary(builder, objectPath);
        }
        return {
            kind: 'branch',
            test: Object.assign({}, object),
            consequent: consequent.id,
            alternate,
            fallthrough: continuationBlock.id,
            id: makeInstructionId(0),
            loc,
        };
    });
    CompilerError.invariant(object !== null, {
        reason: 'Satisfy type checker',
        description: null,
        details: [
            {
                kind: 'error',
                loc: null,
                message: null,
            },
        ],
        suggestions: null,
    });
    builder.enterReserved(consequent, () => {
        const { value } = lowerMemberExpression(builder, expr, object);
        const temp = lowerValueToTemporary(builder, value);
        lowerValueToTemporary(builder, {
            kind: 'StoreLocal',
            lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
            value: Object.assign({}, temp),
            type: null,
            loc,
        });
        return {
            kind: 'goto',
            variant: GotoVariant.Break,
            block: continuationBlock.id,
            id: makeInstructionId(0),
            loc,
        };
    });
    builder.terminateWithContinuation({
        kind: 'optional',
        optional,
        test: testBlock,
        fallthrough: continuationBlock.id,
        id: makeInstructionId(0),
        loc,
    }, continuationBlock);
    return { object, value: place };
}
function lowerOptionalCallExpression(builder, expr, parentAlternate) {
    var _a;
    const optional = expr.node.optional;
    const calleePath = expr.get('callee');
    const loc = (_a = expr.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const place = buildTemporaryPlace(builder, loc);
    const continuationBlock = builder.reserve(builder.currentBlockKind());
    const consequent = builder.reserve('value');
    const alternate = parentAlternate !== null
        ? parentAlternate
        : builder.enter('value', () => {
            const temp = lowerValueToTemporary(builder, {
                kind: 'Primitive',
                value: undefined,
                loc,
            });
            lowerValueToTemporary(builder, {
                kind: 'StoreLocal',
                lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
                value: Object.assign({}, temp),
                type: null,
                loc,
            });
            return {
                kind: 'goto',
                variant: GotoVariant.Break,
                block: continuationBlock.id,
                id: makeInstructionId(0),
                loc,
            };
        });
    let callee;
    const testBlock = builder.enter('value', () => {
        if (calleePath.isOptionalCallExpression()) {
            const value = lowerOptionalCallExpression(builder, calleePath, alternate);
            const valuePlace = lowerValueToTemporary(builder, value);
            callee = {
                kind: 'CallExpression',
                callee: valuePlace,
            };
        }
        else if (calleePath.isOptionalMemberExpression()) {
            const { object, value } = lowerOptionalMemberExpression(builder, calleePath, alternate);
            callee = {
                kind: 'MethodCall',
                receiver: object,
                property: value,
            };
        }
        else if (calleePath.isMemberExpression()) {
            const memberExpr = lowerMemberExpression(builder, calleePath);
            const propertyPlace = lowerValueToTemporary(builder, memberExpr.value);
            callee = {
                kind: 'MethodCall',
                receiver: memberExpr.object,
                property: propertyPlace,
            };
        }
        else {
            callee = {
                kind: 'CallExpression',
                callee: lowerExpressionToTemporary(builder, calleePath),
            };
        }
        const testPlace = callee.kind === 'CallExpression' ? callee.callee : callee.property;
        return {
            kind: 'branch',
            test: Object.assign({}, testPlace),
            consequent: consequent.id,
            alternate,
            fallthrough: continuationBlock.id,
            id: makeInstructionId(0),
            loc,
        };
    });
    builder.enterReserved(consequent, () => {
        const args = lowerArguments(builder, expr.get('arguments'));
        const temp = buildTemporaryPlace(builder, loc);
        if (callee.kind === 'CallExpression') {
            builder.push({
                id: makeInstructionId(0),
                lvalue: Object.assign({}, temp),
                value: {
                    kind: 'CallExpression',
                    callee: Object.assign({}, callee.callee),
                    args,
                    loc,
                },
                effects: null,
                loc,
            });
        }
        else {
            builder.push({
                id: makeInstructionId(0),
                lvalue: Object.assign({}, temp),
                value: {
                    kind: 'MethodCall',
                    receiver: Object.assign({}, callee.receiver),
                    property: Object.assign({}, callee.property),
                    args,
                    loc,
                },
                effects: null,
                loc,
            });
        }
        lowerValueToTemporary(builder, {
            kind: 'StoreLocal',
            lvalue: { kind: InstructionKind.Const, place: Object.assign({}, place) },
            value: Object.assign({}, temp),
            type: null,
            loc,
        });
        return {
            kind: 'goto',
            variant: GotoVariant.Break,
            block: continuationBlock.id,
            id: makeInstructionId(0),
            loc,
        };
    });
    builder.terminateWithContinuation({
        kind: 'optional',
        optional,
        test: testBlock,
        fallthrough: continuationBlock.id,
        id: makeInstructionId(0),
        loc,
    }, continuationBlock);
    return { kind: 'LoadLocal', place, loc: place.loc };
}
function lowerReorderableExpression(builder, expr) {
    var _a;
    if (!isReorderableExpression(builder, expr, true)) {
        builder.errors.push({
            reason: `(BuildHIR::node.lowerReorderableExpression) Expression type \`${expr.type}\` cannot be safely reordered`,
            category: ErrorCategory.Todo,
            loc: (_a = expr.node.loc) !== null && _a !== void 0 ? _a : null,
            suggestions: null,
        });
    }
    return lowerExpressionToTemporary(builder, expr);
}
function isReorderableExpression(builder, expr, allowLocalIdentifiers) {
    switch (expr.node.type) {
        case 'Identifier': {
            const binding = builder.resolveIdentifier(expr);
            if (binding.kind === 'Identifier') {
                return allowLocalIdentifiers;
            }
            else {
                return true;
            }
        }
        case 'TSInstantiationExpression': {
            const innerExpr = expr.get('expression');
            return isReorderableExpression(builder, innerExpr, allowLocalIdentifiers);
        }
        case 'RegExpLiteral':
        case 'StringLiteral':
        case 'NumericLiteral':
        case 'NullLiteral':
        case 'BooleanLiteral':
        case 'BigIntLiteral': {
            return true;
        }
        case 'UnaryExpression': {
            const unary = expr;
            switch (expr.node.operator) {
                case '!':
                case '+':
                case '-': {
                    return isReorderableExpression(builder, unary.get('argument'), allowLocalIdentifiers);
                }
                default: {
                    return false;
                }
            }
        }
        case 'TSAsExpression':
        case 'TSNonNullExpression':
        case 'TypeCastExpression': {
            return isReorderableExpression(builder, expr.get('expression'), allowLocalIdentifiers);
        }
        case 'LogicalExpression': {
            const logical = expr;
            return (isReorderableExpression(builder, logical.get('left'), allowLocalIdentifiers) &&
                isReorderableExpression(builder, logical.get('right'), allowLocalIdentifiers));
        }
        case 'ConditionalExpression': {
            const conditional = expr;
            return (isReorderableExpression(builder, conditional.get('test'), allowLocalIdentifiers) &&
                isReorderableExpression(builder, conditional.get('consequent'), allowLocalIdentifiers) &&
                isReorderableExpression(builder, conditional.get('alternate'), allowLocalIdentifiers));
        }
        case 'ArrayExpression': {
            return expr
                .get('elements')
                .every(element => element.isExpression() &&
                isReorderableExpression(builder, element, allowLocalIdentifiers));
        }
        case 'ObjectExpression': {
            return expr
                .get('properties')
                .every(property => {
                if (!property.isObjectProperty() || property.node.computed) {
                    return false;
                }
                const value = property.get('value');
                return (value.isExpression() &&
                    isReorderableExpression(builder, value, allowLocalIdentifiers));
            });
        }
        case 'MemberExpression': {
            const test = expr;
            let innerObject = test;
            while (innerObject.isMemberExpression()) {
                innerObject = innerObject.get('object');
            }
            if (innerObject.isIdentifier() &&
                builder.resolveIdentifier(innerObject).kind !== 'Identifier') {
                return true;
            }
            else {
                return false;
            }
        }
        case 'ArrowFunctionExpression': {
            const fn = expr;
            const body = fn.get('body');
            if (body.node.type === 'BlockStatement') {
                return body.node.body.length === 0;
            }
            else {
                invariant(body.isExpression(), 'Expected an expression');
                return isReorderableExpression(builder, body, false);
            }
        }
        case 'CallExpression': {
            const call = expr;
            const callee = call.get('callee');
            return (callee.isExpression() &&
                isReorderableExpression(builder, callee, allowLocalIdentifiers) &&
                call
                    .get('arguments')
                    .every(arg => arg.isExpression() &&
                    isReorderableExpression(builder, arg, allowLocalIdentifiers)));
        }
        default: {
            return false;
        }
    }
}
function lowerArguments(builder, expr) {
    var _a;
    let args = [];
    for (const argPath of expr) {
        if (argPath.isSpreadElement()) {
            args.push({
                kind: 'Spread',
                place: lowerExpressionToTemporary(builder, argPath.get('argument')),
            });
        }
        else if (argPath.isExpression()) {
            args.push(lowerExpressionToTemporary(builder, argPath));
        }
        else {
            builder.errors.push({
                reason: `(BuildHIR::lowerExpression) Handle ${argPath.type} arguments in CallExpression`,
                category: ErrorCategory.Todo,
                loc: (_a = argPath.node.loc) !== null && _a !== void 0 ? _a : null,
                suggestions: null,
            });
        }
    }
    return args;
}
function lowerMemberExpression(builder, expr, loweredObject = null) {
    var _a, _b, _c;
    const exprNode = expr.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const objectNode = expr.get('object');
    const propertyNode = expr.get('property');
    const object = loweredObject !== null && loweredObject !== void 0 ? loweredObject : lowerExpressionToTemporary(builder, objectNode);
    if (!expr.node.computed || expr.node.property.type === 'NumericLiteral') {
        let property;
        if (propertyNode.isIdentifier()) {
            property = makePropertyLiteral(propertyNode.node.name);
        }
        else if (propertyNode.isNumericLiteral()) {
            property = makePropertyLiteral(propertyNode.node.value);
        }
        else {
            builder.errors.push({
                reason: `(BuildHIR::lowerMemberExpression) Handle ${propertyNode.type} property`,
                category: ErrorCategory.Todo,
                loc: (_b = propertyNode.node.loc) !== null && _b !== void 0 ? _b : null,
                suggestions: null,
            });
            return {
                object,
                property: propertyNode.toString(),
                value: { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc },
            };
        }
        const value = {
            kind: 'PropertyLoad',
            object: Object.assign({}, object),
            property,
            loc: exprLoc,
        };
        return { object, property, value };
    }
    else {
        if (!propertyNode.isExpression()) {
            builder.errors.push({
                reason: `(BuildHIR::lowerMemberExpression) Expected Expression, got ${propertyNode.type} property`,
                category: ErrorCategory.Todo,
                loc: (_c = propertyNode.node.loc) !== null && _c !== void 0 ? _c : null,
                suggestions: null,
            });
            return {
                object,
                property: propertyNode.toString(),
                value: {
                    kind: 'UnsupportedNode',
                    node: exprNode,
                    loc: exprLoc,
                },
            };
        }
        const property = lowerExpressionToTemporary(builder, propertyNode);
        const value = {
            kind: 'ComputedLoad',
            object: Object.assign({}, object),
            property: Object.assign({}, property),
            loc: exprLoc,
        };
        return { object, property, value };
    }
}
function lowerJsxElementName(builder, exprPath) {
    var _a, _b, _c;
    const exprNode = exprPath.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    if (exprPath.isJSXIdentifier()) {
        const tag = exprPath.node.name;
        if (tag.match(/^[A-Z]/)) {
            const kind = getLoadKind(builder, exprPath);
            return lowerValueToTemporary(builder, {
                kind: kind,
                place: lowerIdentifier(builder, exprPath),
                loc: exprLoc,
            });
        }
        else {
            return {
                kind: 'BuiltinTag',
                name: tag,
                loc: exprLoc,
            };
        }
    }
    else if (exprPath.isJSXMemberExpression()) {
        return lowerJsxMemberExpression(builder, exprPath);
    }
    else if (exprPath.isJSXNamespacedName()) {
        const namespace = exprPath.node.namespace.name;
        const name = exprPath.node.name.name;
        const tag = `${namespace}:${name}`;
        if (namespace.indexOf(':') !== -1 || name.indexOf(':') !== -1) {
            builder.errors.push({
                reason: `Expected JSXNamespacedName to have no colons in the namespace or name`,
                description: `Got \`${namespace}\` : \`${name}\``,
                category: ErrorCategory.Syntax,
                loc: (_b = exprPath.node.loc) !== null && _b !== void 0 ? _b : null,
                suggestions: null,
            });
        }
        const place = lowerValueToTemporary(builder, {
            kind: 'Primitive',
            value: tag,
            loc: exprLoc,
        });
        return place;
    }
    else {
        builder.errors.push({
            reason: `(BuildHIR::lowerJsxElementName) Handle ${exprPath.type} tags`,
            category: ErrorCategory.Todo,
            loc: (_c = exprPath.node.loc) !== null && _c !== void 0 ? _c : null,
            suggestions: null,
        });
        return lowerValueToTemporary(builder, {
            kind: 'UnsupportedNode',
            node: exprNode,
            loc: exprLoc,
        });
    }
}
function lowerJsxMemberExpression(builder, exprPath) {
    var _a, _b, _c;
    const loc = (_a = exprPath.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const object = exprPath.get('object');
    let objectPlace;
    if (object.isJSXMemberExpression()) {
        objectPlace = lowerJsxMemberExpression(builder, object);
    }
    else {
        CompilerError.invariant(object.isJSXIdentifier(), {
            reason: `TypeScript refinement fail: expected 'JsxIdentifier', got \`${object.node.type}\``,
            description: null,
            details: [
                {
                    kind: 'error',
                    loc: (_b = object.node.loc) !== null && _b !== void 0 ? _b : null,
                    message: null,
                },
            ],
            suggestions: null,
        });
        const kind = getLoadKind(builder, object);
        objectPlace = lowerValueToTemporary(builder, {
            kind: kind,
            place: lowerIdentifier(builder, object),
            loc: (_c = exprPath.node.loc) !== null && _c !== void 0 ? _c : GeneratedSource,
        });
    }
    const property = exprPath.get('property').node.name;
    return lowerValueToTemporary(builder, {
        kind: 'PropertyLoad',
        object: objectPlace,
        property: makePropertyLiteral(property),
        loc,
    });
}
function lowerJsxElement(builder, exprPath) {
    var _a, _b, _c;
    const exprNode = exprPath.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    if (exprPath.isJSXElement() || exprPath.isJSXFragment()) {
        return lowerExpressionToTemporary(builder, exprPath);
    }
    else if (exprPath.isJSXExpressionContainer()) {
        const expression = exprPath.get('expression');
        if (expression.isJSXEmptyExpression()) {
            return null;
        }
        else {
            CompilerError.invariant(expression.isExpression(), {
                reason: `(BuildHIR::lowerJsxElement) Expected Expression but found ${expression.type}!`,
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: (_b = expression.node.loc) !== null && _b !== void 0 ? _b : null,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            return lowerExpressionToTemporary(builder, expression);
        }
    }
    else if (exprPath.isJSXText()) {
        let text;
        if (builder.fbtDepth > 0) {
            text = exprPath.node.value;
        }
        else {
            text = trimJsxText(exprPath.node.value);
        }
        if (text === null) {
            return null;
        }
        const place = lowerValueToTemporary(builder, {
            kind: 'JSXText',
            value: text,
            loc: exprLoc,
        });
        return place;
    }
    else {
        builder.errors.push({
            reason: `(BuildHIR::lowerJsxElement) Unhandled JsxElement, got: ${exprPath.type}`,
            category: ErrorCategory.Todo,
            loc: (_c = exprPath.node.loc) !== null && _c !== void 0 ? _c : null,
            suggestions: null,
        });
        const place = lowerValueToTemporary(builder, {
            kind: 'UnsupportedNode',
            node: exprNode,
            loc: exprLoc,
        });
        return place;
    }
}
function trimJsxText(original) {
    const lines = original.split(/\r\n|\n|\r/);
    let lastNonEmptyLine = 0;
    for (let i = 0; i < lines.length; i++) {
        if (lines[i].match(/[^ \t]/)) {
            lastNonEmptyLine = i;
        }
    }
    let str = '';
    for (let i = 0; i < lines.length; i++) {
        const line = lines[i];
        const isFirstLine = i === 0;
        const isLastLine = i === lines.length - 1;
        const isLastNonEmptyLine = i === lastNonEmptyLine;
        let trimmedLine = line.replace(/\t/g, ' ');
        if (!isFirstLine) {
            trimmedLine = trimmedLine.replace(/^[ ]+/, '');
        }
        if (!isLastLine) {
            trimmedLine = trimmedLine.replace(/[ ]+$/, '');
        }
        if (trimmedLine) {
            if (!isLastNonEmptyLine) {
                trimmedLine += ' ';
            }
            str += trimmedLine;
        }
    }
    if (str.length !== 0) {
        return str;
    }
    else {
        return null;
    }
}
function lowerFunctionToValue(builder, expr) {
    var _a;
    const exprNode = expr.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const loweredFunc = lowerFunction(builder, expr);
    if (!loweredFunc) {
        return { kind: 'UnsupportedNode', node: exprNode, loc: exprLoc };
    }
    return {
        kind: 'FunctionExpression',
        name: loweredFunc.func.id,
        nameHint: null,
        type: expr.node.type,
        loc: exprLoc,
        loweredFunc,
    };
}
function lowerFunction(builder, expr) {
    const componentScope = builder.environment.parentFunction.scope;
    const capturedContext = gatherCapturedContext(expr, componentScope);
    const lowering = lower(expr, builder.environment, builder.bindings, new Map([...builder.context, ...capturedContext]));
    let loweredFunc;
    if (lowering.isErr()) {
        const functionErrors = lowering.unwrapErr();
        builder.errors.merge(functionErrors);
        return null;
    }
    loweredFunc = lowering.unwrap();
    return {
        func: loweredFunc,
    };
}
function lowerExpressionToTemporary(builder, exprPath) {
    const value = lowerExpression(builder, exprPath);
    return lowerValueToTemporary(builder, value);
}
function lowerValueToTemporary(builder, value) {
    if (value.kind === 'LoadLocal' && value.place.identifier.name === null) {
        return value.place;
    }
    const place = buildTemporaryPlace(builder, value.loc);
    builder.push({
        id: makeInstructionId(0),
        lvalue: Object.assign({}, place),
        value: value,
        effects: null,
        loc: value.loc,
    });
    return place;
}
function lowerIdentifier(builder, exprPath) {
    var _a, _b;
    const exprNode = exprPath.node;
    const exprLoc = (_a = exprNode.loc) !== null && _a !== void 0 ? _a : GeneratedSource;
    const binding = builder.resolveIdentifier(exprPath);
    switch (binding.kind) {
        case 'Identifier': {
            const place = {
                kind: 'Identifier',
                identifier: binding.identifier,
                effect: Effect.Unknown,
                reactive: false,
                loc: exprLoc,
            };
            return place;
        }
        default: {
            if (binding.kind === 'Global' && binding.name === 'eval') {
                builder.errors.push({
                    reason: `The 'eval' function is not supported`,
                    description: 'Eval is an anti-pattern in JavaScript, and the code executed cannot be evaluated by React Compiler',
                    category: ErrorCategory.UnsupportedSyntax,
                    loc: (_b = exprPath.node.loc) !== null && _b !== void 0 ? _b : null,
                    suggestions: null,
                });
            }
            return lowerValueToTemporary(builder, {
                kind: 'LoadGlobal',
                binding,
                loc: exprLoc,
            });
        }
    }
}
function buildTemporaryPlace(builder, loc) {
    const place = {
        kind: 'Identifier',
        identifier: builder.makeTemporary(loc),
        effect: Effect.Unknown,
        reactive: false,
        loc,
    };
    return place;
}
function getStoreKind(builder, identifier) {
    const isContext = builder.isContextIdentifier(identifier);
    return isContext ? 'StoreContext' : 'StoreLocal';
}
function getLoadKind(builder, identifier) {
    const isContext = builder.isContextIdentifier(identifier);
    return isContext ? 'LoadContext' : 'LoadLocal';
}
function lowerIdentifierForAssignment(builder, loc, kind, path) {
    var _a, _b;
    const binding = builder.resolveIdentifier(path);
    if (binding.kind !== 'Identifier') {
        if (kind === InstructionKind.Reassign) {
            return { kind: 'Global', name: path.node.name };
        }
        else {
            builder.errors.push({
                reason: `(BuildHIR::lowerAssignment) Could not find binding for declaration.`,
                category: ErrorCategory.Invariant,
                loc: (_a = path.node.loc) !== null && _a !== void 0 ? _a : null,
                suggestions: null,
            });
            return null;
        }
    }
    else if (binding.bindingKind === 'const' &&
        kind === InstructionKind.Reassign) {
        builder.errors.push({
            reason: `Cannot reassign a \`const\` variable`,
            category: ErrorCategory.Syntax,
            loc: (_b = path.node.loc) !== null && _b !== void 0 ? _b : null,
            description: binding.identifier.name != null
                ? `\`${binding.identifier.name.value}\` is declared as const`
                : null,
        });
        return null;
    }
    const place = {
        kind: 'Identifier',
        identifier: binding.identifier,
        effect: Effect.Unknown,
        reactive: false,
        loc,
    };
    return place;
}
function lowerAssignment(builder, loc, kind, lvaluePath, value, assignmentKind) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2;
    const lvalueNode = lvaluePath.node;
    switch (lvalueNode.type) {
        case 'Identifier': {
            const lvalue = lvaluePath;
            const place = lowerIdentifierForAssignment(builder, loc, kind, lvalue);
            if (place === null) {
                return {
                    kind: 'UnsupportedNode',
                    loc: (_a = lvalue.node.loc) !== null && _a !== void 0 ? _a : GeneratedSource,
                    node: lvalue.node,
                };
            }
            else if (place.kind === 'Global') {
                const temporary = lowerValueToTemporary(builder, {
                    kind: 'StoreGlobal',
                    name: place.name,
                    value,
                    loc,
                });
                return { kind: 'LoadLocal', place: temporary, loc: temporary.loc };
            }
            const isHoistedIdentifier = builder.environment.isHoistedIdentifier(lvalue.node);
            let temporary;
            if (builder.isContextIdentifier(lvalue)) {
                if (kind === InstructionKind.Const && !isHoistedIdentifier) {
                    builder.errors.push({
                        reason: `Expected \`const\` declaration not to be reassigned`,
                        category: ErrorCategory.Syntax,
                        loc: (_b = lvalue.node.loc) !== null && _b !== void 0 ? _b : null,
                        suggestions: null,
                    });
                }
                if (kind !== InstructionKind.Const &&
                    kind !== InstructionKind.Reassign &&
                    kind !== InstructionKind.Let &&
                    kind !== InstructionKind.Function) {
                    builder.errors.push({
                        reason: `Unexpected context variable kind`,
                        category: ErrorCategory.Syntax,
                        loc: (_c = lvalue.node.loc) !== null && _c !== void 0 ? _c : null,
                        suggestions: null,
                    });
                    temporary = lowerValueToTemporary(builder, {
                        kind: 'UnsupportedNode',
                        node: lvalueNode,
                        loc: (_d = lvalueNode.loc) !== null && _d !== void 0 ? _d : GeneratedSource,
                    });
                }
                else {
                    temporary = lowerValueToTemporary(builder, {
                        kind: 'StoreContext',
                        lvalue: { place: Object.assign({}, place), kind },
                        value,
                        loc,
                    });
                }
            }
            else {
                const typeAnnotation = lvalue.get('typeAnnotation');
                let type;
                if (typeAnnotation.isTSTypeAnnotation()) {
                    const typePath = typeAnnotation.get('typeAnnotation');
                    type = typePath.node;
                }
                else if (typeAnnotation.isTypeAnnotation()) {
                    const typePath = typeAnnotation.get('typeAnnotation');
                    type = typePath.node;
                }
                else {
                    type = null;
                }
                temporary = lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { place: Object.assign({}, place), kind },
                    value,
                    type,
                    loc,
                });
            }
            return { kind: 'LoadLocal', place: temporary, loc: temporary.loc };
        }
        case 'MemberExpression': {
            CompilerError.invariant(kind === InstructionKind.Reassign, {
                reason: 'MemberExpression may only appear in an assignment expression',
                description: null,
                details: [
                    {
                        kind: 'error',
                        loc: (_e = lvaluePath.node.loc) !== null && _e !== void 0 ? _e : null,
                        message: null,
                    },
                ],
                suggestions: null,
            });
            const lvalue = lvaluePath;
            const property = lvalue.get('property');
            const object = lowerExpressionToTemporary(builder, lvalue.get('object'));
            if (!lvalue.node.computed || lvalue.get('property').isNumericLiteral()) {
                let temporary;
                if (property.isIdentifier()) {
                    temporary = lowerValueToTemporary(builder, {
                        kind: 'PropertyStore',
                        object,
                        property: makePropertyLiteral(property.node.name),
                        value,
                        loc,
                    });
                }
                else if (property.isNumericLiteral()) {
                    temporary = lowerValueToTemporary(builder, {
                        kind: 'PropertyStore',
                        object,
                        property: makePropertyLiteral(property.node.value),
                        value,
                        loc,
                    });
                }
                else {
                    builder.errors.push({
                        reason: `(BuildHIR::lowerAssignment) Handle ${property.type} properties in MemberExpression`,
                        category: ErrorCategory.Todo,
                        loc: (_f = property.node.loc) !== null && _f !== void 0 ? _f : null,
                        suggestions: null,
                    });
                    return { kind: 'UnsupportedNode', node: lvalueNode, loc };
                }
                return { kind: 'LoadLocal', place: temporary, loc: temporary.loc };
            }
            else {
                if (!property.isExpression()) {
                    builder.errors.push({
                        reason: '(BuildHIR::lowerAssignment) Expected private name to appear as a non-computed property',
                        category: ErrorCategory.Todo,
                        loc: (_g = property.node.loc) !== null && _g !== void 0 ? _g : null,
                        suggestions: null,
                    });
                    return { kind: 'UnsupportedNode', node: lvalueNode, loc };
                }
                const propertyPlace = lowerExpressionToTemporary(builder, property);
                const temporary = lowerValueToTemporary(builder, {
                    kind: 'ComputedStore',
                    object,
                    property: propertyPlace,
                    value,
                    loc,
                });
                return { kind: 'LoadLocal', place: temporary, loc: temporary.loc };
            }
        }
        case 'ArrayPattern': {
            const lvalue = lvaluePath;
            const elements = lvalue.get('elements');
            const items = [];
            const followups = [];
            const forceTemporaries = kind === InstructionKind.Reassign &&
                (elements.some(element => !element.isIdentifier()) ||
                    elements.some(element => element.isIdentifier() &&
                        (getStoreKind(builder, element) !== 'StoreLocal' ||
                            builder.resolveIdentifier(element).kind !== 'Identifier')));
            for (let i = 0; i < elements.length; i++) {
                const element = elements[i];
                if (element.node == null) {
                    items.push({
                        kind: 'Hole',
                    });
                    continue;
                }
                if (element.isRestElement()) {
                    const argument = element.get('argument');
                    if (argument.isIdentifier() &&
                        !forceTemporaries &&
                        (assignmentKind === 'Assignment' ||
                            getStoreKind(builder, argument) === 'StoreLocal')) {
                        const identifier = lowerIdentifierForAssignment(builder, (_h = element.node.loc) !== null && _h !== void 0 ? _h : GeneratedSource, kind, argument);
                        if (identifier === null) {
                            continue;
                        }
                        else if (identifier.kind === 'Global') {
                            builder.errors.push({
                                category: ErrorCategory.Todo,
                                reason: 'Expected reassignment of globals to enable forceTemporaries',
                                loc: (_j = element.node.loc) !== null && _j !== void 0 ? _j : GeneratedSource,
                            });
                            continue;
                        }
                        items.push({
                            kind: 'Spread',
                            place: identifier,
                        });
                    }
                    else {
                        const temp = buildTemporaryPlace(builder, (_k = element.node.loc) !== null && _k !== void 0 ? _k : GeneratedSource);
                        promoteTemporary(temp.identifier);
                        items.push({
                            kind: 'Spread',
                            place: Object.assign({}, temp),
                        });
                        followups.push({ place: temp, path: argument });
                    }
                }
                else if (element.isIdentifier() &&
                    !forceTemporaries &&
                    (assignmentKind === 'Assignment' ||
                        getStoreKind(builder, element) === 'StoreLocal')) {
                    const identifier = lowerIdentifierForAssignment(builder, (_l = element.node.loc) !== null && _l !== void 0 ? _l : GeneratedSource, kind, element);
                    if (identifier === null) {
                        continue;
                    }
                    else if (identifier.kind === 'Global') {
                        builder.errors.push({
                            category: ErrorCategory.Todo,
                            reason: 'Expected reassignment of globals to enable forceTemporaries',
                            loc: (_m = element.node.loc) !== null && _m !== void 0 ? _m : GeneratedSource,
                        });
                        continue;
                    }
                    items.push(identifier);
                }
                else {
                    const temp = buildTemporaryPlace(builder, (_o = element.node.loc) !== null && _o !== void 0 ? _o : GeneratedSource);
                    promoteTemporary(temp.identifier);
                    items.push(Object.assign({}, temp));
                    followups.push({ place: temp, path: element });
                }
            }
            const temporary = lowerValueToTemporary(builder, {
                kind: 'Destructure',
                lvalue: {
                    kind,
                    pattern: {
                        kind: 'ArrayPattern',
                        items,
                    },
                },
                value,
                loc,
            });
            for (const { place, path } of followups) {
                lowerAssignment(builder, (_p = path.node.loc) !== null && _p !== void 0 ? _p : loc, kind, path, place, assignmentKind);
            }
            return { kind: 'LoadLocal', place: temporary, loc: value.loc };
        }
        case 'ObjectPattern': {
            const lvalue = lvaluePath;
            const propertiesPaths = lvalue.get('properties');
            const properties = [];
            const followups = [];
            const forceTemporaries = kind === InstructionKind.Reassign &&
                propertiesPaths.some(property => property.isRestElement() ||
                    (property.isObjectProperty() &&
                        (!property.get('value').isIdentifier() ||
                            builder.resolveIdentifier(property.get('value')).kind !== 'Identifier')));
            for (let i = 0; i < propertiesPaths.length; i++) {
                const property = propertiesPaths[i];
                if (property.isRestElement()) {
                    const argument = property.get('argument');
                    if (!argument.isIdentifier()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerAssignment) Handle ${argument.node.type} rest element in ObjectPattern`,
                            category: ErrorCategory.Todo,
                            loc: (_q = argument.node.loc) !== null && _q !== void 0 ? _q : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    if (forceTemporaries ||
                        getStoreKind(builder, argument) === 'StoreContext') {
                        const temp = buildTemporaryPlace(builder, (_r = property.node.loc) !== null && _r !== void 0 ? _r : GeneratedSource);
                        promoteTemporary(temp.identifier);
                        properties.push({
                            kind: 'Spread',
                            place: Object.assign({}, temp),
                        });
                        followups.push({ place: temp, path: argument });
                    }
                    else {
                        const identifier = lowerIdentifierForAssignment(builder, (_s = property.node.loc) !== null && _s !== void 0 ? _s : GeneratedSource, kind, argument);
                        if (identifier === null) {
                            continue;
                        }
                        else if (identifier.kind === 'Global') {
                            builder.errors.push({
                                category: ErrorCategory.Todo,
                                reason: 'Expected reassignment of globals to enable forceTemporaries',
                                loc: (_t = property.node.loc) !== null && _t !== void 0 ? _t : GeneratedSource,
                            });
                            continue;
                        }
                        properties.push({
                            kind: 'Spread',
                            place: identifier,
                        });
                    }
                }
                else {
                    if (!property.isObjectProperty()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerAssignment) Handle ${property.type} properties in ObjectPattern`,
                            category: ErrorCategory.Todo,
                            loc: (_u = property.node.loc) !== null && _u !== void 0 ? _u : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    if (property.node.computed) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerAssignment) Handle computed properties in ObjectPattern`,
                            category: ErrorCategory.Todo,
                            loc: (_v = property.node.loc) !== null && _v !== void 0 ? _v : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    const loweredKey = lowerObjectPropertyKey(builder, property);
                    if (!loweredKey) {
                        continue;
                    }
                    const element = property.get('value');
                    if (!element.isLVal()) {
                        builder.errors.push({
                            reason: `(BuildHIR::lowerAssignment) Expected object property value to be an LVal, got: ${element.type}`,
                            category: ErrorCategory.Todo,
                            loc: (_w = element.node.loc) !== null && _w !== void 0 ? _w : null,
                            suggestions: null,
                        });
                        continue;
                    }
                    if (element.isIdentifier() &&
                        !forceTemporaries &&
                        (assignmentKind === 'Assignment' ||
                            getStoreKind(builder, element) === 'StoreLocal')) {
                        const identifier = lowerIdentifierForAssignment(builder, (_x = element.node.loc) !== null && _x !== void 0 ? _x : GeneratedSource, kind, element);
                        if (identifier === null) {
                            continue;
                        }
                        else if (identifier.kind === 'Global') {
                            builder.errors.push({
                                category: ErrorCategory.Todo,
                                reason: 'Expected reassignment of globals to enable forceTemporaries',
                                loc: (_y = element.node.loc) !== null && _y !== void 0 ? _y : GeneratedSource,
                            });
                            continue;
                        }
                        properties.push({
                            kind: 'ObjectProperty',
                            type: 'property',
                            place: identifier,
                            key: loweredKey,
                        });
                    }
                    else {
                        const temp = buildTemporaryPlace(builder, (_z = element.node.loc) !== null && _z !== void 0 ? _z : GeneratedSource);
                        promoteTemporary(temp.identifier);
                        properties.push({
                            kind: 'ObjectProperty',
                            type: 'property',
                            place: Object.assign({}, temp),
                            key: loweredKey,
                        });
                        followups.push({ place: temp, path: element });
                    }
                }
            }
            const temporary = lowerValueToTemporary(builder, {
                kind: 'Destructure',
                lvalue: {
                    kind,
                    pattern: {
                        kind: 'ObjectPattern',
                        properties,
                    },
                },
                value,
                loc,
            });
            for (const { place, path } of followups) {
                lowerAssignment(builder, (_0 = path.node.loc) !== null && _0 !== void 0 ? _0 : loc, kind, path, place, assignmentKind);
            }
            return { kind: 'LoadLocal', place: temporary, loc: value.loc };
        }
        case 'AssignmentPattern': {
            const lvalue = lvaluePath;
            const loc = (_1 = lvalue.node.loc) !== null && _1 !== void 0 ? _1 : GeneratedSource;
            const temp = buildTemporaryPlace(builder, loc);
            const testBlock = builder.reserve('value');
            const continuationBlock = builder.reserve(builder.currentBlockKind());
            const consequent = builder.enter('value', () => {
                const defaultValue = lowerReorderableExpression(builder, lvalue.get('right'));
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, temp) },
                    value: Object.assign({}, defaultValue),
                    type: null,
                    loc,
                });
                return {
                    kind: 'goto',
                    variant: GotoVariant.Break,
                    block: continuationBlock.id,
                    id: makeInstructionId(0),
                    loc,
                };
            });
            const alternate = builder.enter('value', () => {
                lowerValueToTemporary(builder, {
                    kind: 'StoreLocal',
                    lvalue: { kind: InstructionKind.Const, place: Object.assign({}, temp) },
                    value: Object.assign({}, value),
                    type: null,
                    loc,
                });
                return {
                    kind: 'goto',
                    variant: GotoVariant.Break,
                    block: continuationBlock.id,
                    id: makeInstructionId(0),
                    loc,
                };
            });
            builder.terminateWithContinuation({
                kind: 'ternary',
                test: testBlock.id,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc,
            }, testBlock);
            const undef = lowerValueToTemporary(builder, {
                kind: 'Primitive',
                value: undefined,
                loc,
            });
            const test = lowerValueToTemporary(builder, {
                kind: 'BinaryExpression',
                left: Object.assign({}, value),
                operator: '===',
                right: Object.assign({}, undef),
                loc,
            });
            builder.terminateWithContinuation({
                kind: 'branch',
                test: Object.assign({}, test),
                consequent,
                alternate,
                fallthrough: continuationBlock.id,
                id: makeInstructionId(0),
                loc,
            }, continuationBlock);
            return lowerAssignment(builder, loc, kind, lvalue.get('left'), temp, assignmentKind);
        }
        default: {
            builder.errors.push({
                reason: `(BuildHIR::lowerAssignment) Handle ${lvaluePath.type} assignments`,
                category: ErrorCategory.Todo,
                loc: (_2 = lvaluePath.node.loc) !== null && _2 !== void 0 ? _2 : null,
                suggestions: null,
            });
            return { kind: 'UnsupportedNode', node: lvalueNode, loc };
        }
    }
}
function captureScopes({ from, to }) {
    let scopes = new Set();
    while (from) {
        scopes.add(from);
        if (from === to) {
            break;
        }
        from = from.parent;
    }
    return scopes;
}
function gatherCapturedContext(fn, componentScope) {
    const capturedIds = new Map();
    const pureScopes = captureScopes({
        from: fn.scope.parent,
        to: componentScope,
    });
    function handleMaybeDependency(path) {
        var _a, _b;
        let baseIdentifier;
        if (path.isJSXOpeningElement()) {
            const name = path.get('name');
            if (!(name.isJSXMemberExpression() || name.isJSXIdentifier())) {
                return;
            }
            let current = name;
            while (current.isJSXMemberExpression()) {
                current = current.get('object');
            }
            invariant(current.isJSXIdentifier(), 'Invalid logic in gatherCapturedDeps');
            baseIdentifier = current;
        }
        else {
            baseIdentifier = path;
        }
        path.skip();
        const binding = baseIdentifier.scope.getBinding(baseIdentifier.node.name);
        if (binding !== undefined &&
            pureScopes.has(binding.scope) &&
            !capturedIds.has(binding.identifier)) {
            capturedIds.set(binding.identifier, (_b = (_a = path.node.loc) !== null && _a !== void 0 ? _a : binding.identifier.loc) !== null && _b !== void 0 ? _b : GeneratedSource);
        }
    }
    fn.traverse({
        TypeAnnotation(path) {
            path.skip();
        },
        TSTypeAnnotation(path) {
            path.skip();
        },
        TypeAlias(path) {
            path.skip();
        },
        TSTypeAliasDeclaration(path) {
            path.skip();
        },
        Expression(path) {
            if (path.isAssignmentExpression()) {
                const left = path.get('left');
                if (left.isIdentifier()) {
                    handleMaybeDependency(left);
                }
                return;
            }
            else if (path.isJSXElement()) {
                handleMaybeDependency(path.get('openingElement'));
            }
            else if (path.isIdentifier()) {
                handleMaybeDependency(path);
            }
        },
    });
    return capturedIds;
}
function notNull(value) {
    return value !== null;
}
function lowerType(node) {
    switch (node.type) {
        case 'GenericTypeAnnotation': {
            const id = node.id;
            if (id.type === 'Identifier' && id.name === 'Array') {
                return { kind: 'Object', shapeId: BuiltInArrayId };
            }
            return makeType();
        }
        case 'TSTypeReference': {
            const typeName = node.typeName;
            if (typeName.type === 'Identifier' && typeName.name === 'Array') {
                return { kind: 'Object', shapeId: BuiltInArrayId };
            }
            return makeType();
        }
        case 'ArrayTypeAnnotation':
        case 'TSArrayType': {
            return { kind: 'Object', shapeId: BuiltInArrayId };
        }
        case 'BooleanLiteralTypeAnnotation':
        case 'BooleanTypeAnnotation':
        case 'NullLiteralTypeAnnotation':
        case 'NumberLiteralTypeAnnotation':
        case 'NumberTypeAnnotation':
        case 'StringLiteralTypeAnnotation':
        case 'StringTypeAnnotation':
        case 'TSBooleanKeyword':
        case 'TSNullKeyword':
        case 'TSNumberKeyword':
        case 'TSStringKeyword':
        case 'TSSymbolKeyword':
        case 'TSUndefinedKeyword':
        case 'TSVoidKeyword':
        case 'VoidTypeAnnotation': {
            return { kind: 'Primitive' };
        }
        default: {
            return makeType();
        }
    }
}

function buildReactiveScopeTerminalsHIR(fn) {
    const queuedRewrites = [];
    recursivelyTraverseItems([...getScopes(fn)], scope => scope.range, {
        fallthroughs: new Map(),
        rewrites: queuedRewrites,
        env: fn.env,
    }, pushStartScopeTerminal, pushEndScopeTerminal);
    const rewrittenFinalBlocks = new Map();
    const nextBlocks = new Map();
    queuedRewrites.reverse();
    for (const [, block] of fn.body.blocks) {
        const context = {
            nextBlockId: block.id,
            rewrites: [],
            nextPreds: block.preds,
            instrSliceIdx: 0,
            source: block,
        };
        for (let i = 0; i < block.instructions.length + 1; i++) {
            const instrId = i < block.instructions.length
                ? block.instructions[i].id
                : block.terminal.id;
            let rewrite = queuedRewrites.at(-1);
            while (rewrite != null && rewrite.instrId <= instrId) {
                handleRewrite(rewrite, i, context);
                queuedRewrites.pop();
                rewrite = queuedRewrites.at(-1);
            }
        }
        if (context.rewrites.length > 0) {
            const finalBlock = {
                id: context.nextBlockId,
                kind: block.kind,
                preds: context.nextPreds,
                terminal: block.terminal,
                instructions: block.instructions.slice(context.instrSliceIdx),
                phis: new Set(),
            };
            context.rewrites.push(finalBlock);
            for (const b of context.rewrites) {
                nextBlocks.set(b.id, b);
            }
            rewrittenFinalBlocks.set(block.id, finalBlock.id);
        }
        else {
            nextBlocks.set(block.id, block);
        }
    }
    const originalBlocks = fn.body.blocks;
    fn.body.blocks = nextBlocks;
    for (const [, block] of originalBlocks) {
        for (const phi of block.phis) {
            for (const [originalId, value] of phi.operands) {
                const newId = rewrittenFinalBlocks.get(originalId);
                if (newId != null) {
                    phi.operands.delete(originalId);
                    phi.operands.set(newId, value);
                }
            }
        }
    }
    reversePostorderBlocks(fn.body);
    markPredecessors(fn.body);
    markInstructionIds(fn.body);
    fixScopeAndIdentifierRanges(fn.body);
}
function pushStartScopeTerminal(scope, context) {
    const blockId = context.env.nextBlockId;
    const fallthroughId = context.env.nextBlockId;
    context.rewrites.push({
        kind: 'StartScope',
        blockId,
        fallthroughId,
        instrId: scope.range.start,
        scope,
    });
    context.fallthroughs.set(scope.id, fallthroughId);
}
function pushEndScopeTerminal(scope, context) {
    const fallthroughId = context.fallthroughs.get(scope.id);
    CompilerError.invariant(fallthroughId != null, {
        reason: 'Expected scope to exist',
        description: null,
        details: [
            {
                kind: 'error',
                loc: GeneratedSource,
                message: null,
            },
        ],
    });
    context.rewrites.push({
        kind: 'EndScope',
        fallthroughId,
        instrId: scope.range.end,
    });
}
function handleRewrite(terminalInfo, idx, context) {
    const terminal = terminalInfo.kind === 'StartScope'
        ? {
            kind: 'scope',
            fallthrough: terminalInfo.fallthroughId,
            block: terminalInfo.blockId,
            scope: terminalInfo.scope,
            id: terminalInfo.instrId,
            loc: GeneratedSource,
        }
        : {
            kind: 'goto',
            variant: GotoVariant.Break,
            block: terminalInfo.fallthroughId,
            id: terminalInfo.instrId,
            loc: GeneratedSource,
        };
    const currBlockId = context.nextBlockId;
    context.rewrites.push({
        kind: context.source.kind,
        id: currBlockId,
        instructions: context.source.instructions.slice(context.instrSliceIdx, idx),
        preds: context.nextPreds,
        phis: context.rewrites.length === 0 ? context.source.phis : new Set(),
        terminal,
    });
    context.nextPreds = new Set([currBlockId]);
    context.nextBlockId =
        terminalInfo.kind === 'StartScope'
            ? terminalInfo.blockId
            : terminalInfo.fallthroughId;
    context.instrSliceIdx = idx;
}

var ansiStylesExports = {};
var ansiStyles = {
  get exports(){ return ansiStylesExports; },
  set exports(v){ ansiStylesExports = v; },
};

var conversionsExports = {};
var conversions = {
  get exports(){ return conversionsExports; },
  set exports(v){ conversionsExports = v; },
};

var colorName;
var hasRequiredColorName;

function requireColorName () {
	if (hasRequiredColorName) return colorName;
	hasRequiredColorName = 1;

	colorName = {
		"aliceblue": [240, 248, 255],
		"antiquewhite": [250, 235, 215],
		"aqua": [0, 255, 255],
		"aquamarine": [127, 255, 212],
		"azure": [240, 255, 255],
		"beige": [245, 245, 220],
		"bisque": [255, 228, 196],
		"black": [0, 0, 0],
		"blanchedalmond": [255, 235, 205],
		"blue": [0, 0, 255],
		"blueviolet": [138, 43, 226],
		"brown": [165, 42, 42],
		"burlywood": [222, 184, 135],
		"cadetblue": [95, 158, 160],
		"chartreuse": [127, 255, 0],
		"chocolate": [210, 105, 30],
		"coral": [255, 127, 80],
		"cornflowerblue": [100, 149, 237],
		"cornsilk": [255, 248, 220],
		"crimson": [220, 20, 60],
		"cyan": [0, 255, 255],
		"darkblue": [0, 0, 139],
		"darkcyan": [0, 139, 139],
		"darkgoldenrod": [184, 134, 11],
		"darkgray": [169, 169, 169],
		"darkgreen": [0, 100, 0],
		"darkgrey": [169, 169, 169],
		"darkkhaki": [189, 183, 107],
		"darkmagenta": [139, 0, 139],
		"darkolivegreen": [85, 107, 47],
		"darkorange": [255, 140, 0],
		"darkorchid": [153, 50, 204],
		"darkred": [139, 0, 0],
		"darksalmon": [233, 150, 122],
		"darkseagreen": [143, 188, 143],
		"darkslateblue": [72, 61, 139],
		"darkslategray": [47, 79, 79],
		"darkslategrey": [47, 79, 79],
		"darkturquoise": [0, 206, 209],
		"darkviolet": [148, 0, 211],
		"deeppink": [255, 20, 147],
		"deepskyblue": [0, 191, 255],
		"dimgray": [105, 105, 105],
		"dimgrey": [105, 105, 105],
		"dodgerblue": [30, 144, 255],
		"firebrick": [178, 34, 34],
		"floralwhite": [255, 250, 240],
		"forestgreen": [34, 139, 34],
		"fuchsia": [255, 0, 255],
		"gainsboro": [220, 220, 220],
		"ghostwhite": [248, 248, 255],
		"gold": [255, 215, 0],
		"goldenrod": [218, 165, 32],
		"gray": [128, 128, 128],
		"green": [0, 128, 0],
		"greenyellow": [173, 255, 47],
		"grey": [128, 128, 128],
		"honeydew": [240, 255, 240],
		"hotpink": [255, 105, 180],
		"indianred": [205, 92, 92],
		"indigo": [75, 0, 130],
		"ivory": [255, 255, 240],
		"khaki": [240, 230, 140],
		"lavender": [230, 230, 250],
		"lavenderblush": [255, 240, 245],
		"lawngreen": [124, 252, 0],
		"lemonchiffon": [255, 250, 205],
		"lightblue": [173, 216, 230],
		"lightcoral": [240, 128, 128],
		"lightcyan": [224, 255, 255],
		"lightgoldenrodyellow": [250, 250, 210],
		"lightgray": [211, 211, 211],
		"lightgreen": [144, 238, 144],
		"lightgrey": [211, 211, 211],
		"lightpink": [255, 182, 193],
		"lightsalmon": [255, 160, 122],
		"lightseagreen": [32, 178, 170],
		"lightskyblue": [135, 206, 250],
		"lightslategray": [119, 136, 153],
		"lightslategrey": [119, 136, 153],
		"lightsteelblue": [176, 196, 222],
		"lightyellow": [255, 255, 224],
		"lime": [0, 255, 0],
		"limegreen": [50, 205, 50],
		"linen": [250, 240, 230],
		"magenta": [255, 0, 255],
		"maroon": [128, 0, 0],
		"mediumaquamarine": [102, 205, 170],
		"mediumblue": [0, 0, 205],
		"mediumorchid": [186, 85, 211],
		"mediumpurple": [147, 112, 219],
		"mediumseagreen": [60, 179, 113],
		"mediumslateblue": [123, 104, 238],
		"mediumspringgreen": [0, 250, 154],
		"mediumturquoise": [72, 209, 204],
		"mediumvioletred": [199, 21, 133],
		"midnightblue": [25, 25, 112],
		"mintcream": [245, 255, 250],
		"mistyrose": [255, 228, 225],
		"moccasin": [255, 228, 181],
		"navajowhite": [255, 222, 173],
		"navy": [0, 0, 128],
		"oldlace": [253, 245, 230],
		"olive": [128, 128, 0],
		"olivedrab": [107, 142, 35],
		"orange": [255, 165, 0],
		"orangered": [255, 69, 0],
		"orchid": [218, 112, 214],
		"palegoldenrod": [238, 232, 170],
		"palegreen": [152, 251, 152],
		"paleturquoise": [175, 238, 238],
		"palevioletred": [219, 112, 147],
		"papayawhip": [255, 239, 213],
		"peachpuff": [255, 218, 185],
		"peru": [205, 133, 63],
		"pink": [255, 192, 203],
		"plum": [221, 160, 221],
		"powderblue": [176, 224, 230],
		"purple": [128, 0, 128],
		"rebeccapurple": [102, 51, 153],
		"red": [255, 0, 0],
		"rosybrown": [188, 143, 143],
		"royalblue": [65, 105, 225],
		"saddlebrown": [139, 69, 19],
		"salmon": [250, 128, 114],
		"sandybrown": [244, 164, 96],
		"seagreen": [46, 139, 87],
		"seashell": [255, 245, 238],
		"sienna": [160, 82, 45],
		"silver": [192, 192, 192],
		"skyblue": [135, 206, 235],
		"slateblue": [106, 90, 205],
		"slategray": [112, 128, 144],
		"slategrey": [112, 128, 144],
		"snow": [255, 250, 250],
		"springgreen": [0, 255, 127],
		"steelblue": [70, 130, 180],
		"tan": [210, 180, 140],
		"teal": [0, 128, 128],
		"thistle": [216, 191, 216],
		"tomato": [255, 99, 71],
		"turquoise": [64, 224, 208],
		"violet": [238, 130, 238],
		"wheat": [245, 222, 179],
		"white": [255, 255, 255],
		"whitesmoke": [245, 245, 245],
		"yellow": [255, 255, 0],
		"yellowgreen": [154, 205, 50]
	};
	return colorName;
}

var hasRequiredConversions;

function requireConversions () {
	if (hasRequiredConversions) return conversionsExports;
	hasRequiredConversions = 1;
	var cssKeywords = requireColorName();

	// NOTE: conversions should only return primitive values (i.e. arrays, or
	//       values that give correct `typeof` results).
	//       do not use box values types (i.e. Number(), String(), etc.)

	var reverseKeywords = {};
	for (var key in cssKeywords) {
		if (cssKeywords.hasOwnProperty(key)) {
			reverseKeywords[cssKeywords[key]] = key;
		}
	}

	var convert = conversions.exports = {
		rgb: {channels: 3, labels: 'rgb'},
		hsl: {channels: 3, labels: 'hsl'},
		hsv: {channels: 3, labels: 'hsv'},
		hwb: {channels: 3, labels: 'hwb'},
		cmyk: {channels: 4, labels: 'cmyk'},
		xyz: {channels: 3, labels: 'xyz'},
		lab: {channels: 3, labels: 'lab'},
		lch: {channels: 3, labels: 'lch'},
		hex: {channels: 1, labels: ['hex']},
		keyword: {channels: 1, labels: ['keyword']},
		ansi16: {channels: 1, labels: ['ansi16']},
		ansi256: {channels: 1, labels: ['ansi256']},
		hcg: {channels: 3, labels: ['h', 'c', 'g']},
		apple: {channels: 3, labels: ['r16', 'g16', 'b16']},
		gray: {channels: 1, labels: ['gray']}
	};

	// hide .channels and .labels properties
	for (var model in convert) {
		if (convert.hasOwnProperty(model)) {
			if (!('channels' in convert[model])) {
				throw new Error('missing channels property: ' + model);
			}

			if (!('labels' in convert[model])) {
				throw new Error('missing channel labels property: ' + model);
			}

			if (convert[model].labels.length !== convert[model].channels) {
				throw new Error('channel and label counts mismatch: ' + model);
			}

			var channels = convert[model].channels;
			var labels = convert[model].labels;
			delete convert[model].channels;
			delete convert[model].labels;
			Object.defineProperty(convert[model], 'channels', {value: channels});
			Object.defineProperty(convert[model], 'labels', {value: labels});
		}
	}

	convert.rgb.hsl = function (rgb) {
		var r = rgb[0] / 255;
		var g = rgb[1] / 255;
		var b = rgb[2] / 255;
		var min = Math.min(r, g, b);
		var max = Math.max(r, g, b);
		var delta = max - min;
		var h;
		var s;
		var l;

		if (max === min) {
			h = 0;
		} else if (r === max) {
			h = (g - b) / delta;
		} else if (g === max) {
			h = 2 + (b - r) / delta;
		} else if (b === max) {
			h = 4 + (r - g) / delta;
		}

		h = Math.min(h * 60, 360);

		if (h < 0) {
			h += 360;
		}

		l = (min + max) / 2;

		if (max === min) {
			s = 0;
		} else if (l <= 0.5) {
			s = delta / (max + min);
		} else {
			s = delta / (2 - max - min);
		}

		return [h, s * 100, l * 100];
	};

	convert.rgb.hsv = function (rgb) {
		var rdif;
		var gdif;
		var bdif;
		var h;
		var s;

		var r = rgb[0] / 255;
		var g = rgb[1] / 255;
		var b = rgb[2] / 255;
		var v = Math.max(r, g, b);
		var diff = v - Math.min(r, g, b);
		var diffc = function (c) {
			return (v - c) / 6 / diff + 1 / 2;
		};

		if (diff === 0) {
			h = s = 0;
		} else {
			s = diff / v;
			rdif = diffc(r);
			gdif = diffc(g);
			bdif = diffc(b);

			if (r === v) {
				h = bdif - gdif;
			} else if (g === v) {
				h = (1 / 3) + rdif - bdif;
			} else if (b === v) {
				h = (2 / 3) + gdif - rdif;
			}
			if (h < 0) {
				h += 1;
			} else if (h > 1) {
				h -= 1;
			}
		}

		return [
			h * 360,
			s * 100,
			v * 100
		];
	};

	convert.rgb.hwb = function (rgb) {
		var r = rgb[0];
		var g = rgb[1];
		var b = rgb[2];
		var h = convert.rgb.hsl(rgb)[0];
		var w = 1 / 255 * Math.min(r, Math.min(g, b));

		b = 1 - 1 / 255 * Math.max(r, Math.max(g, b));

		return [h, w * 100, b * 100];
	};

	convert.rgb.cmyk = function (rgb) {
		var r = rgb[0] / 255;
		var g = rgb[1] / 255;
		var b = rgb[2] / 255;
		var c;
		var m;
		var y;
		var k;

		k = Math.min(1 - r, 1 - g, 1 - b);
		c = (1 - r - k) / (1 - k) || 0;
		m = (1 - g - k) / (1 - k) || 0;
		y = (1 - b - k) / (1 - k) || 0;

		return [c * 100, m * 100, y * 100, k * 100];
	};

	/**
	 * See https://en.m.wikipedia.org/wiki/Euclidean_distance#Squared_Euclidean_distance
	 * */
	function comparativeDistance(x, y) {
		return (
			Math.pow(x[0] - y[0], 2) +
			Math.pow(x[1] - y[1], 2) +
			Math.pow(x[2] - y[2], 2)
		);
	}

	convert.rgb.keyword = function (rgb) {
		var reversed = reverseKeywords[rgb];
		if (reversed) {
			return reversed;
		}

		var currentClosestDistance = Infinity;
		var currentClosestKeyword;

		for (var keyword in cssKeywords) {
			if (cssKeywords.hasOwnProperty(keyword)) {
				var value = cssKeywords[keyword];

				// Compute comparative distance
				var distance = comparativeDistance(rgb, value);

				// Check if its less, if so set as closest
				if (distance < currentClosestDistance) {
					currentClosestDistance = distance;
					currentClosestKeyword = keyword;
				}
			}
		}

		return currentClosestKeyword;
	};

	convert.keyword.rgb = function (keyword) {
		return cssKeywords[keyword];
	};

	convert.rgb.xyz = function (rgb) {
		var r = rgb[0] / 255;
		var g = rgb[1] / 255;
		var b = rgb[2] / 255;

		// assume sRGB
		r = r > 0.04045 ? Math.pow(((r + 0.055) / 1.055), 2.4) : (r / 12.92);
		g = g > 0.04045 ? Math.pow(((g + 0.055) / 1.055), 2.4) : (g / 12.92);
		b = b > 0.04045 ? Math.pow(((b + 0.055) / 1.055), 2.4) : (b / 12.92);

		var x = (r * 0.4124) + (g * 0.3576) + (b * 0.1805);
		var y = (r * 0.2126) + (g * 0.7152) + (b * 0.0722);
		var z = (r * 0.0193) + (g * 0.1192) + (b * 0.9505);

		return [x * 100, y * 100, z * 100];
	};

	convert.rgb.lab = function (rgb) {
		var xyz = convert.rgb.xyz(rgb);
		var x = xyz[0];
		var y = xyz[1];
		var z = xyz[2];
		var l;
		var a;
		var b;

		x /= 95.047;
		y /= 100;
		z /= 108.883;

		x = x > 0.008856 ? Math.pow(x, 1 / 3) : (7.787 * x) + (16 / 116);
		y = y > 0.008856 ? Math.pow(y, 1 / 3) : (7.787 * y) + (16 / 116);
		z = z > 0.008856 ? Math.pow(z, 1 / 3) : (7.787 * z) + (16 / 116);

		l = (116 * y) - 16;
		a = 500 * (x - y);
		b = 200 * (y - z);

		return [l, a, b];
	};

	convert.hsl.rgb = function (hsl) {
		var h = hsl[0] / 360;
		var s = hsl[1] / 100;
		var l = hsl[2] / 100;
		var t1;
		var t2;
		var t3;
		var rgb;
		var val;

		if (s === 0) {
			val = l * 255;
			return [val, val, val];
		}

		if (l < 0.5) {
			t2 = l * (1 + s);
		} else {
			t2 = l + s - l * s;
		}

		t1 = 2 * l - t2;

		rgb = [0, 0, 0];
		for (var i = 0; i < 3; i++) {
			t3 = h + 1 / 3 * -(i - 1);
			if (t3 < 0) {
				t3++;
			}
			if (t3 > 1) {
				t3--;
			}

			if (6 * t3 < 1) {
				val = t1 + (t2 - t1) * 6 * t3;
			} else if (2 * t3 < 1) {
				val = t2;
			} else if (3 * t3 < 2) {
				val = t1 + (t2 - t1) * (2 / 3 - t3) * 6;
			} else {
				val = t1;
			}

			rgb[i] = val * 255;
		}

		return rgb;
	};

	convert.hsl.hsv = function (hsl) {
		var h = hsl[0];
		var s = hsl[1] / 100;
		var l = hsl[2] / 100;
		var smin = s;
		var lmin = Math.max(l, 0.01);
		var sv;
		var v;

		l *= 2;
		s *= (l <= 1) ? l : 2 - l;
		smin *= lmin <= 1 ? lmin : 2 - lmin;
		v = (l + s) / 2;
		sv = l === 0 ? (2 * smin) / (lmin + smin) : (2 * s) / (l + s);

		return [h, sv * 100, v * 100];
	};

	convert.hsv.rgb = function (hsv) {
		var h = hsv[0] / 60;
		var s = hsv[1] / 100;
		var v = hsv[2] / 100;
		var hi = Math.floor(h) % 6;

		var f = h - Math.floor(h);
		var p = 255 * v * (1 - s);
		var q = 255 * v * (1 - (s * f));
		var t = 255 * v * (1 - (s * (1 - f)));
		v *= 255;

		switch (hi) {
			case 0:
				return [v, t, p];
			case 1:
				return [q, v, p];
			case 2:
				return [p, v, t];
			case 3:
				return [p, q, v];
			case 4:
				return [t, p, v];
			case 5:
				return [v, p, q];
		}
	};

	convert.hsv.hsl = function (hsv) {
		var h = hsv[0];
		var s = hsv[1] / 100;
		var v = hsv[2] / 100;
		var vmin = Math.max(v, 0.01);
		var lmin;
		var sl;
		var l;

		l = (2 - s) * v;
		lmin = (2 - s) * vmin;
		sl = s * vmin;
		sl /= (lmin <= 1) ? lmin : 2 - lmin;
		sl = sl || 0;
		l /= 2;

		return [h, sl * 100, l * 100];
	};

	// http://dev.w3.org/csswg/css-color/#hwb-to-rgb
	convert.hwb.rgb = function (hwb) {
		var h = hwb[0] / 360;
		var wh = hwb[1] / 100;
		var bl = hwb[2] / 100;
		var ratio = wh + bl;
		var i;
		var v;
		var f;
		var n;

		// wh + bl cant be > 1
		if (ratio > 1) {
			wh /= ratio;
			bl /= ratio;
		}

		i = Math.floor(6 * h);
		v = 1 - bl;
		f = 6 * h - i;

		if ((i & 0x01) !== 0) {
			f = 1 - f;
		}

		n = wh + f * (v - wh); // linear interpolation

		var r;
		var g;
		var b;
		switch (i) {
			default:
			case 6:
			case 0: r = v; g = n; b = wh; break;
			case 1: r = n; g = v; b = wh; break;
			case 2: r = wh; g = v; b = n; break;
			case 3: r = wh; g = n; b = v; break;
			case 4: r = n; g = wh; b = v; break;
			case 5: r = v; g = wh; b = n; break;
		}

		return [r * 255, g * 255, b * 255];
	};

	convert.cmyk.rgb = function (cmyk) {
		var c = cmyk[0] / 100;
		var m = cmyk[1] / 100;
		var y = cmyk[2] / 100;
		var k = cmyk[3] / 100;
		var r;
		var g;
		var b;

		r = 1 - Math.min(1, c * (1 - k) + k);
		g = 1 - Math.min(1, m * (1 - k) + k);
		b = 1 - Math.min(1, y * (1 - k) + k);

		return [r * 255, g * 255, b * 255];
	};

	convert.xyz.rgb = function (xyz) {
		var x = xyz[0] / 100;
		var y = xyz[1] / 100;
		var z = xyz[2] / 100;
		var r;
		var g;
		var b;

		r = (x * 3.2406) + (y * -1.5372) + (z * -0.4986);
		g = (x * -0.9689) + (y * 1.8758) + (z * 0.0415);
		b = (x * 0.0557) + (y * -0.2040) + (z * 1.0570);

		// assume sRGB
		r = r > 0.0031308
			? ((1.055 * Math.pow(r, 1.0 / 2.4)) - 0.055)
			: r * 12.92;

		g = g > 0.0031308
			? ((1.055 * Math.pow(g, 1.0 / 2.4)) - 0.055)
			: g * 12.92;

		b = b > 0.0031308
			? ((1.055 * Math.pow(b, 1.0 / 2.4)) - 0.055)
			: b * 12.92;

		r = Math.min(Math.max(0, r), 1);
		g = Math.min(Math.max(0, g), 1);
		b = Math.min(Math.max(0, b), 1);

		return [r * 255, g * 255, b * 255];
	};

	convert.xyz.lab = function (xyz) {
		var x = xyz[0];
		var y = xyz[1];
		var z = xyz[2];
		var l;
		var a;
		var b;

		x /= 95.047;
		y /= 100;
		z /= 108.883;

		x = x > 0.008856 ? Math.pow(x, 1 / 3) : (7.787 * x) + (16 / 116);
		y = y > 0.008856 ? Math.pow(y, 1 / 3) : (7.787 * y) + (16 / 116);
		z = z > 0.008856 ? Math.pow(z, 1 / 3) : (7.787 * z) + (16 / 116);

		l = (116 * y) - 16;
		a = 500 * (x - y);
		b = 200 * (y - z);

		return [l, a, b];
	};

	convert.lab.xyz = function (lab) {
		var l = lab[0];
		var a = lab[1];
		var b = lab[2];
		var x;
		var y;
		var z;

		y = (l + 16) / 116;
		x = a / 500 + y;
		z = y - b / 200;

		var y2 = Math.pow(y, 3);
		var x2 = Math.pow(x, 3);
		var z2 = Math.pow(z, 3);
		y = y2 > 0.008856 ? y2 : (y - 16 / 116) / 7.787;
		x = x2 > 0.008856 ? x2 : (x - 16 / 116) / 7.787;
		z = z2 > 0.008856 ? z2 : (z - 16 / 116) / 7.787;

		x *= 95.047;
		y *= 100;
		z *= 108.883;

		return [x, y, z];
	};

	convert.lab.lch = function (lab) {
		var l = lab[0];
		var a = lab[1];
		var b = lab[2];
		var hr;
		var h;
		var c;

		hr = Math.atan2(b, a);
		h = hr * 360 / 2 / Math.PI;

		if (h < 0) {
			h += 360;
		}

		c = Math.sqrt(a * a + b * b);

		return [l, c, h];
	};

	convert.lch.lab = function (lch) {
		var l = lch[0];
		var c = lch[1];
		var h = lch[2];
		var a;
		var b;
		var hr;

		hr = h / 360 * 2 * Math.PI;
		a = c * Math.cos(hr);
		b = c * Math.sin(hr);

		return [l, a, b];
	};

	convert.rgb.ansi16 = function (args) {
		var r = args[0];
		var g = args[1];
		var b = args[2];
		var value = 1 in arguments ? arguments[1] : convert.rgb.hsv(args)[2]; // hsv -> ansi16 optimization

		value = Math.round(value / 50);

		if (value === 0) {
			return 30;
		}

		var ansi = 30
			+ ((Math.round(b / 255) << 2)
			| (Math.round(g / 255) << 1)
			| Math.round(r / 255));

		if (value === 2) {
			ansi += 60;
		}

		return ansi;
	};

	convert.hsv.ansi16 = function (args) {
		// optimization here; we already know the value and don't need to get
		// it converted for us.
		return convert.rgb.ansi16(convert.hsv.rgb(args), args[2]);
	};

	convert.rgb.ansi256 = function (args) {
		var r = args[0];
		var g = args[1];
		var b = args[2];

		// we use the extended greyscale palette here, with the exception of
		// black and white. normal palette only has 4 greyscale shades.
		if (r === g && g === b) {
			if (r < 8) {
				return 16;
			}

			if (r > 248) {
				return 231;
			}

			return Math.round(((r - 8) / 247) * 24) + 232;
		}

		var ansi = 16
			+ (36 * Math.round(r / 255 * 5))
			+ (6 * Math.round(g / 255 * 5))
			+ Math.round(b / 255 * 5);

		return ansi;
	};

	convert.ansi16.rgb = function (args) {
		var color = args % 10;

		// handle greyscale
		if (color === 0 || color === 7) {
			if (args > 50) {
				color += 3.5;
			}

			color = color / 10.5 * 255;

			return [color, color, color];
		}

		var mult = (~~(args > 50) + 1) * 0.5;
		var r = ((color & 1) * mult) * 255;
		var g = (((color >> 1) & 1) * mult) * 255;
		var b = (((color >> 2) & 1) * mult) * 255;

		return [r, g, b];
	};

	convert.ansi256.rgb = function (args) {
		// handle greyscale
		if (args >= 232) {
			var c = (args - 232) * 10 + 8;
			return [c, c, c];
		}

		args -= 16;

		var rem;
		var r = Math.floor(args / 36) / 5 * 255;
		var g = Math.floor((rem = args % 36) / 6) / 5 * 255;
		var b = (rem % 6) / 5 * 255;

		return [r, g, b];
	};

	convert.rgb.hex = function (args) {
		var integer = ((Math.round(args[0]) & 0xFF) << 16)
			+ ((Math.round(args[1]) & 0xFF) << 8)
			+ (Math.round(args[2]) & 0xFF);

		var string = integer.toString(16).toUpperCase();
		return '000000'.substring(string.length) + string;
	};

	convert.hex.rgb = function (args) {
		var match = args.toString(16).match(/[a-f0-9]{6}|[a-f0-9]{3}/i);
		if (!match) {
			return [0, 0, 0];
		}

		var colorString = match[0];

		if (match[0].length === 3) {
			colorString = colorString.split('').map(function (char) {
				return char + char;
			}).join('');
		}

		var integer = parseInt(colorString, 16);
		var r = (integer >> 16) & 0xFF;
		var g = (integer >> 8) & 0xFF;
		var b = integer & 0xFF;

		return [r, g, b];
	};

	convert.rgb.hcg = function (rgb) {
		var r = rgb[0] / 255;
		var g = rgb[1] / 255;
		var b = rgb[2] / 255;
		var max = Math.max(Math.max(r, g), b);
		var min = Math.min(Math.min(r, g), b);
		var chroma = (max - min);
		var grayscale;
		var hue;

		if (chroma < 1) {
			grayscale = min / (1 - chroma);
		} else {
			grayscale = 0;
		}

		if (chroma <= 0) {
			hue = 0;
		} else
		if (max === r) {
			hue = ((g - b) / chroma) % 6;
		} else
		if (max === g) {
			hue = 2 + (b - r) / chroma;
		} else {
			hue = 4 + (r - g) / chroma + 4;
		}

		hue /= 6;
		hue %= 1;

		return [hue * 360, chroma * 100, grayscale * 100];
	};

	convert.hsl.hcg = function (hsl) {
		var s = hsl[1] / 100;
		var l = hsl[2] / 100;
		var c = 1;
		var f = 0;

		if (l < 0.5) {
			c = 2.0 * s * l;
		} else {
			c = 2.0 * s * (1.0 - l);
		}

		if (c < 1.0) {
			f = (l - 0.5 * c) / (1.0 - c);
		}

		return [hsl[0], c * 100, f * 100];
	};

	convert.hsv.hcg = function (hsv) {
		var s = hsv[1] / 100;
		var v = hsv[2] / 100;

		var c = s * v;
		var f = 0;

		if (c < 1.0) {
			f = (v - c) / (1 - c);
		}

		return [hsv[0], c * 100, f * 100];
	};

	convert.hcg.rgb = function (hcg) {
		var h = hcg[0] / 360;
		var c = hcg[1] / 100;
		var g = hcg[2] / 100;

		if (c === 0.0) {
			return [g * 255, g * 255, g * 255];
		}

		var pure = [0, 0, 0];
		var hi = (h % 1) * 6;
		var v = hi % 1;
		var w = 1 - v;
		var mg = 0;

		switch (Math.floor(hi)) {
			case 0:
				pure[0] = 1; pure[1] = v; pure[2] = 0; break;
			case 1:
				pure[0] = w; pure[1] = 1; pure[2] = 0; break;
			case 2:
				pure[0] = 0; pure[1] = 1; pure[2] = v; break;
			case 3:
				pure[0] = 0; pure[1] = w; pure[2] = 1; break;
			case 4:
				pure[0] = v; pure[1] = 0; pure[2] = 1; break;
			default:
				pure[0] = 1; pure[1] = 0; pure[2] = w;
		}

		mg = (1.0 - c) * g;

		return [
			(c * pure[0] + mg) * 255,
			(c * pure[1] + mg) * 255,
			(c * pure[2] + mg) * 255
		];
	};

	convert.hcg.hsv = function (hcg) {
		var c = hcg[1] / 100;
		var g = hcg[2] / 100;

		var v = c + g * (1.0 - c);
		var f = 0;

		if (v > 0.0) {
			f = c / v;
		}

		return [hcg[0], f * 100, v * 100];
	};

	convert.hcg.hsl = function (hcg) {
		var c = hcg[1] / 100;
		var g = hcg[2] / 100;

		var l = g * (1.0 - c) + 0.5 * c;
		var s = 0;

		if (l > 0.0 && l < 0.5) {
			s = c / (2 * l);
		} else
		if (l >= 0.5 && l < 1.0) {
			s = c / (2 * (1 - l));
		}

		return [hcg[0], s * 100, l * 100];
	};

	convert.hcg.hwb = function (hcg) {
		var c = hcg[1] / 100;
		var g = hcg[2] / 100;
		var v = c + g * (1.0 - c);
		return [hcg[0], (v - c) * 100, (1 - v) * 100];
	};

	convert.hwb.hcg = function (hwb) {
		var w = hwb[1] / 100;
		var b = hwb[2] / 100;
		var v = 1 - b;
		var c = v - w;
		var g = 0;

		if (c < 1) {
			g = (v - c) / (1 - c);
		}

		return [hwb[0], c * 100, g * 100];
	};

	convert.apple.rgb = function (apple) {
		return [(apple[0] / 65535) * 255, (apple[1] / 65535) * 255, (apple[2] / 65535) * 255];
	};

	convert.rgb.apple = function (rgb) {
		return [(rgb[0] / 255) * 65535, (rgb[1] / 255) * 65535, (rgb[2] / 255) * 65535];
	};

	convert.gray.rgb = function (args) {
		return [args[0] / 100 * 255, args[0] / 100 * 255, args[0] / 100 * 255];
	};

	convert.gray.hsl = convert.gray.hsv = function (args) {
		return [0, 0, args[0]];
	};

	convert.gray.hwb = function (gray) {
		return [0, 100, gray[0]];
	};

	convert.gray.cmyk = function (gray) {
		return [0, 0, 0, gray[0]];
	};

	convert.gray.lab = function (gray) {
		return [gray[0], 0, 0];
	};

	convert.gray.hex = function (gray) {
		var val = Math.round(gray[0] / 100 * 255) & 0xFF;
		var integer = (val << 16) + (val << 8) + val;

		var string = integer.toString(16).toUpperCase();
		return '000000'.substring(string.length) + string;
	};

	convert.rgb.gray = function (rgb) {
		var val = (rgb[0] + rgb[1] + rgb[2]) / 3;
		return [val / 255 * 100];
	};
	return conversionsExports;
}

var route;
var hasRequiredRoute;

function requireRoute () {
	if (hasRequiredRoute) return route;
	hasRequiredRoute = 1;
	var conversions = requireConversions();

	/*
		this function routes a model to all other models.

		all functions that are routed have a property `.conversion` attached
		to the returned synthetic function. This property is an array
		of strings, each with the steps in between the 'from' and 'to'
		color models (inclusive).

		conversions that are not possible simply are not included.
	*/

	function buildGraph() {
		var graph = {};
		// https://jsperf.com/object-keys-vs-for-in-with-closure/3
		var models = Object.keys(conversions);

		for (var len = models.length, i = 0; i < len; i++) {
			graph[models[i]] = {
				// http://jsperf.com/1-vs-infinity
				// micro-opt, but this is simple.
				distance: -1,
				parent: null
			};
		}

		return graph;
	}

	// https://en.wikipedia.org/wiki/Breadth-first_search
	function deriveBFS(fromModel) {
		var graph = buildGraph();
		var queue = [fromModel]; // unshift -> queue -> pop

		graph[fromModel].distance = 0;

		while (queue.length) {
			var current = queue.pop();
			var adjacents = Object.keys(conversions[current]);

			for (var len = adjacents.length, i = 0; i < len; i++) {
				var adjacent = adjacents[i];
				var node = graph[adjacent];

				if (node.distance === -1) {
					node.distance = graph[current].distance + 1;
					node.parent = current;
					queue.unshift(adjacent);
				}
			}
		}

		return graph;
	}

	function link(from, to) {
		return function (args) {
			return to(from(args));
		};
	}

	function wrapConversion(toModel, graph) {
		var path = [graph[toModel].parent, toModel];
		var fn = conversions[graph[toModel].parent][toModel];

		var cur = graph[toModel].parent;
		while (graph[cur].parent) {
			path.unshift(graph[cur].parent);
			fn = link(conversions[graph[cur].parent][cur], fn);
			cur = graph[cur].parent;
		}

		fn.conversion = path;
		return fn;
	}

	route = function (fromModel) {
		var graph = deriveBFS(fromModel);
		var conversion = {};

		var models = Object.keys(graph);
		for (var len = models.length, i = 0; i < len; i++) {
			var toModel = models[i];
			var node = graph[toModel];

			if (node.parent === null) {
				// no possible conversion, or this node is the source model.
				continue;
			}

			conversion[toModel] = wrapConversion(toModel, graph);
		}

		return conversion;
	};
	return route;
}

var colorConvert;
var hasRequiredColorConvert;

function requireColorConvert () {
	if (hasRequiredColorConvert) return colorConvert;
	hasRequiredColorConvert = 1;
	var conversions = requireConversions();
	var route = requireRoute();

	var convert = {};

	var models = Object.keys(conversions);

	function wrapRaw(fn) {
		var wrappedFn = function (args) {
			if (args === undefined || args === null) {
				return args;
			}

			if (arguments.length > 1) {
				args = Array.prototype.slice.call(arguments);
			}

			return fn(args);
		};

		// preserve .conversion property if there is one
		if ('conversion' in fn) {
			wrappedFn.conversion = fn.conversion;
		}

		return wrappedFn;
	}

	function wrapRounded(fn) {
		var wrappedFn = function (args) {
			if (args === undefined || args === null) {
				return args;
			}

			if (arguments.length > 1) {
				args = Array.prototype.slice.call(arguments);
			}

			var result = fn(args);

			// we're assuming the result is an array here.
			// see notice in conversions.js; don't use box types
			// in conversion functions.
			if (typeof result === 'object') {
				for (var len = result.length, i = 0; i < len; i++) {
					result[i] = Math.round(result[i]);
				}
			}

			return result;
		};

		// preserve .conversion property if there is one
		if ('conversion' in fn) {
			wrappedFn.conversion = fn.conversion;
		}

		return wrappedFn;
	}

	models.forEach(function (fromModel) {
		convert[fromModel] = {};

		Object.defineProperty(convert[fromModel], 'channels', {value: conversions[fromModel].channels});
		Object.defineProperty(convert[fromModel], 'labels', {value: conversions[fromModel].labels});

		var routes = route(fromModel);
		var routeModels = Object.keys(routes);

		routeModels.forEach(function (toModel) {
			var fn = routes[toModel];

			convert[fromModel][toModel] = wrapRounded(fn);
			convert[fromModel][toModel].raw = wrapRaw(fn);
		});
	});

	colorConvert = convert;
	return colorConvert;
}

var hasRequiredAnsiStyles;

function requireAnsiStyles () {
	if (hasRequiredAnsiStyles) return ansiStylesExports;
	hasRequiredAnsiStyles = 1;
	(function (module) {
		const colorConvert = requireColorConvert();

		const wrapAnsi16 = (fn, offset) => function () {
			const code = fn.apply(colorConvert, arguments);
			return `\u001B[${code + offset}m`;
		};

		const wrapAnsi256 = (fn, offset) => function () {
			const code = fn.apply(colorConvert, arguments);
			return `\u001B[${38 + offset};5;${code}m`;
		};

		const wrapAnsi16m = (fn, offset) => function () {
			const rgb = fn.apply(colorConvert, arguments);
			return `\u001B[${38 + offset};2;${rgb[0]};${rgb[1]};${rgb[2]}m`;
		};

		function assembleStyles() {
			const codes = new Map();
			const styles = {
				modifier: {
					reset: [0, 0],
					// 21 isn't widely supported and 22 does the same thing
					bold: [1, 22],
					dim: [2, 22],
					italic: [3, 23],
					underline: [4, 24],
					inverse: [7, 27],
					hidden: [8, 28],
					strikethrough: [9, 29]
				},
				color: {
					black: [30, 39],
					red: [31, 39],
					green: [32, 39],
					yellow: [33, 39],
					blue: [34, 39],
					magenta: [35, 39],
					cyan: [36, 39],
					white: [37, 39],
					gray: [90, 39],

					// Bright color
					redBright: [91, 39],
					greenBright: [92, 39],
					yellowBright: [93, 39],
					blueBright: [94, 39],
					magentaBright: [95, 39],
					cyanBright: [96, 39],
					whiteBright: [97, 39]
				},
				bgColor: {
					bgBlack: [40, 49],
					bgRed: [41, 49],
					bgGreen: [42, 49],
					bgYellow: [43, 49],
					bgBlue: [44, 49],
					bgMagenta: [45, 49],
					bgCyan: [46, 49],
					bgWhite: [47, 49],

					// Bright color
					bgBlackBright: [100, 49],
					bgRedBright: [101, 49],
					bgGreenBright: [102, 49],
					bgYellowBright: [103, 49],
					bgBlueBright: [104, 49],
					bgMagentaBright: [105, 49],
					bgCyanBright: [106, 49],
					bgWhiteBright: [107, 49]
				}
			};

			// Fix humans
			styles.color.grey = styles.color.gray;

			for (const groupName of Object.keys(styles)) {
				const group = styles[groupName];

				for (const styleName of Object.keys(group)) {
					const style = group[styleName];

					styles[styleName] = {
						open: `\u001B[${style[0]}m`,
						close: `\u001B[${style[1]}m`
					};

					group[styleName] = styles[styleName];

					codes.set(style[0], style[1]);
				}

				Object.defineProperty(styles, groupName, {
					value: group,
					enumerable: false
				});

				Object.defineProperty(styles, 'codes', {
					value: codes,
					enumerable: false
				});
			}

			const ansi2ansi = n => n;
			const rgb2rgb = (r, g, b) => [r, g, b];

			styles.color.close = '\u001B[39m';
			styles.bgColor.close = '\u001B[49m';

			styles.color.ansi = {
				ansi: wrapAnsi16(ansi2ansi, 0)
			};
			styles.color.ansi256 = {
				ansi256: wrapAnsi256(ansi2ansi, 0)
			};
			styles.color.ansi16m = {
				rgb: wrapAnsi16m(rgb2rgb, 0)
			};

			styles.bgColor.ansi = {
				ansi: wrapAnsi16(ansi2ansi, 10)
			};
			styles.bgColor.ansi256 = {
				ansi256: wrapAnsi256(ansi2ansi, 10)
			};
			styles.bgColor.ansi16m = {
				rgb: wrapAnsi16m(rgb2rgb, 10)
			};

			for (let key of Object.keys(colorConvert)) {
				if (typeof colorConvert[key] !== 'object') {
					continue;
				}

				const suite = colorConvert[key];

				if (key === 'ansi16') {
					key = 'ansi';
				}

				if ('ansi16' in suite) {
					styles.color.ansi[key] = wrapAnsi16(suite.ansi16, 0);
					styles.bgColor.ansi[key] = wrapAnsi16(suite.ansi16, 10);
				}

				if ('ansi256' in suite) {
					styles.color.ansi256[key] = wrapAnsi256(suite.ansi256, 0);
					styles.bgColor.ansi256[key] = wrapAnsi256(suite.ansi256, 10);
				}

				if ('rgb' in suite) {
					styles.color.ansi16m[key] = wrapAnsi16m(suite.rgb, 0);
					styles.bgColor.ansi16m[key] = wrapAnsi16m(suite.rgb, 10);
				}
			}

			return styles;
		}

		// Make the export immutable
		Object.defineProperty(module, 'exports', {
			enumerable: true,
			get: assembleStyles
		});
} (ansiStyles));
	return ansiStylesExports;
}

var collections = {};

var hasRequiredCollections;

function requireCollections () {
	if (hasRequiredCollections) return collections;
	hasRequiredCollections = 1;

	Object.defineProperty(collections, '__esModule', {
	  value: true
	});
	collections.printIteratorEntries = printIteratorEntries;
	collections.printIteratorValues = printIteratorValues;
	collections.printListItems = printListItems;
	collections.printObjectProperties = printObjectProperties;

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 *
	 */
	const getKeysOfEnumerableProperties = object => {
	  const keys = Object.keys(object).sort();

	  if (Object.getOwnPropertySymbols) {
	    Object.getOwnPropertySymbols(object).forEach(symbol => {
	      if (Object.getOwnPropertyDescriptor(object, symbol).enumerable) {
	        keys.push(symbol);
	      }
	    });
	  }

	  return keys;
	};
	/**
	 * Return entries (for example, of a map)
	 * with spacing, indentation, and comma
	 * without surrounding punctuation (for example, braces)
	 */

	function printIteratorEntries( // Flow 0.51.0: property `@@iterator` of $Iterator not found in Object
	  // To allow simplistic getRecordIterator in immutable.js
	  iterator,
	  config,
	  indentation,
	  depth,
	  refs,
	  printer, // Too bad, so sad that separator for ECMAScript Map has been ' => '
	  // What a distracting diff if you change a data structure to/from
	  separator = ': '
	) {
	  let result = '';
	  let current = iterator.next();

	  if (!current.done) {
	    result += config.spacingOuter;
	    const indentationNext = indentation + config.indent;

	    while (!current.done) {
	      const name = printer(
	        current.value[0],
	        config,
	        indentationNext,
	        depth,
	        refs
	      );
	      const value = printer(
	        current.value[1],
	        config,
	        indentationNext,
	        depth,
	        refs
	      );
	      result += indentationNext + name + separator + value;
	      current = iterator.next();

	      if (!current.done) {
	        result += ',' + config.spacingInner;
	      } else if (!config.min) {
	        result += ',';
	      }
	    }

	    result += config.spacingOuter + indentation;
	  }

	  return result;
	}
	/**
	 * Return values (for example, of a set)
	 * with spacing, indentation, and comma
	 * without surrounding punctuation (braces or brackets)
	 */

	function printIteratorValues(
	  iterator,
	  config,
	  indentation,
	  depth,
	  refs,
	  printer
	) {
	  let result = '';
	  let current = iterator.next();

	  if (!current.done) {
	    result += config.spacingOuter;
	    const indentationNext = indentation + config.indent;

	    while (!current.done) {
	      result +=
	        indentationNext +
	        printer(current.value, config, indentationNext, depth, refs);
	      current = iterator.next();

	      if (!current.done) {
	        result += ',' + config.spacingInner;
	      } else if (!config.min) {
	        result += ',';
	      }
	    }

	    result += config.spacingOuter + indentation;
	  }

	  return result;
	}
	/**
	 * Return items (for example, of an array)
	 * with spacing, indentation, and comma
	 * without surrounding punctuation (for example, brackets)
	 **/

	function printListItems(list, config, indentation, depth, refs, printer) {
	  let result = '';

	  if (list.length) {
	    result += config.spacingOuter;
	    const indentationNext = indentation + config.indent;

	    for (let i = 0; i < list.length; i++) {
	      result +=
	        indentationNext +
	        printer(list[i], config, indentationNext, depth, refs);

	      if (i < list.length - 1) {
	        result += ',' + config.spacingInner;
	      } else if (!config.min) {
	        result += ',';
	      }
	    }

	    result += config.spacingOuter + indentation;
	  }

	  return result;
	}
	/**
	 * Return properties of an object
	 * with spacing, indentation, and comma
	 * without surrounding punctuation (for example, braces)
	 */

	function printObjectProperties(val, config, indentation, depth, refs, printer) {
	  let result = '';
	  const keys = getKeysOfEnumerableProperties(val);

	  if (keys.length) {
	    result += config.spacingOuter;
	    const indentationNext = indentation + config.indent;

	    for (let i = 0; i < keys.length; i++) {
	      const key = keys[i];
	      const name = printer(key, config, indentationNext, depth, refs);
	      const value = printer(val[key], config, indentationNext, depth, refs);
	      result += indentationNext + name + ': ' + value;

	      if (i < keys.length - 1) {
	        result += ',' + config.spacingInner;
	      } else if (!config.min) {
	        result += ',';
	      }
	    }

	    result += config.spacingOuter + indentation;
	  }

	  return result;
	}
	return collections;
}

var AsymmetricMatcher = {};

var hasRequiredAsymmetricMatcher;

function requireAsymmetricMatcher () {
	if (hasRequiredAsymmetricMatcher) return AsymmetricMatcher;
	hasRequiredAsymmetricMatcher = 1;

	Object.defineProperty(AsymmetricMatcher, '__esModule', {
	  value: true
	});
	AsymmetricMatcher.default = AsymmetricMatcher.test = AsymmetricMatcher.serialize = void 0;

	var _collections = requireCollections();

	var Symbol = commonjsGlobal['jest-symbol-do-not-touch'] || commonjsGlobal.Symbol;
	const asymmetricMatcher = Symbol.for('jest.asymmetricMatcher');
	const SPACE = ' ';

	const serialize = (val, config, indentation, depth, refs, printer) => {
	  const stringedValue = val.toString();

	  if (
	    stringedValue === 'ArrayContaining' ||
	    stringedValue === 'ArrayNotContaining'
	  ) {
	    if (++depth > config.maxDepth) {
	      return '[' + stringedValue + ']';
	    }

	    return (
	      stringedValue +
	      SPACE +
	      '[' +
	      (0, _collections.printListItems)(
	        val.sample,
	        config,
	        indentation,
	        depth,
	        refs,
	        printer
	      ) +
	      ']'
	    );
	  }

	  if (
	    stringedValue === 'ObjectContaining' ||
	    stringedValue === 'ObjectNotContaining'
	  ) {
	    if (++depth > config.maxDepth) {
	      return '[' + stringedValue + ']';
	    }

	    return (
	      stringedValue +
	      SPACE +
	      '{' +
	      (0, _collections.printObjectProperties)(
	        val.sample,
	        config,
	        indentation,
	        depth,
	        refs,
	        printer
	      ) +
	      '}'
	    );
	  }

	  if (
	    stringedValue === 'StringMatching' ||
	    stringedValue === 'StringNotMatching'
	  ) {
	    return (
	      stringedValue +
	      SPACE +
	      printer(val.sample, config, indentation, depth, refs)
	    );
	  }

	  if (
	    stringedValue === 'StringContaining' ||
	    stringedValue === 'StringNotContaining'
	  ) {
	    return (
	      stringedValue +
	      SPACE +
	      printer(val.sample, config, indentation, depth, refs)
	    );
	  }

	  return val.toAsymmetricMatcher();
	};

	AsymmetricMatcher.serialize = serialize;

	const test = val => val && val.$$typeof === asymmetricMatcher;

	AsymmetricMatcher.test = test;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	AsymmetricMatcher.default = _default;
	return AsymmetricMatcher;
}

var ConvertAnsi = {};

var ansiRegex;
var hasRequiredAnsiRegex;

function requireAnsiRegex () {
	if (hasRequiredAnsiRegex) return ansiRegex;
	hasRequiredAnsiRegex = 1;

	ansiRegex = options => {
		options = Object.assign({
			onlyFirst: false
		}, options);

		const pattern = [
			'[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)',
			'(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))'
		].join('|');

		return new RegExp(pattern, options.onlyFirst ? undefined : 'g');
	};
	return ansiRegex;
}

var hasRequiredConvertAnsi;

function requireConvertAnsi () {
	if (hasRequiredConvertAnsi) return ConvertAnsi;
	hasRequiredConvertAnsi = 1;

	Object.defineProperty(ConvertAnsi, '__esModule', {
	  value: true
	});
	ConvertAnsi.default = ConvertAnsi.serialize = ConvertAnsi.test = void 0;

	var _ansiRegex = _interopRequireDefault(requireAnsiRegex());

	var _ansiStyles = _interopRequireDefault(requireAnsiStyles());

	function _interopRequireDefault(obj) {
	  return obj && obj.__esModule ? obj : {default: obj};
	}

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	const toHumanReadableAnsi = text =>
	  text.replace((0, _ansiRegex.default)(), match => {
	    switch (match) {
	      case _ansiStyles.default.red.close:
	      case _ansiStyles.default.green.close:
	      case _ansiStyles.default.cyan.close:
	      case _ansiStyles.default.gray.close:
	      case _ansiStyles.default.white.close:
	      case _ansiStyles.default.yellow.close:
	      case _ansiStyles.default.bgRed.close:
	      case _ansiStyles.default.bgGreen.close:
	      case _ansiStyles.default.bgYellow.close:
	      case _ansiStyles.default.inverse.close:
	      case _ansiStyles.default.dim.close:
	      case _ansiStyles.default.bold.close:
	      case _ansiStyles.default.reset.open:
	      case _ansiStyles.default.reset.close:
	        return '</>';

	      case _ansiStyles.default.red.open:
	        return '<red>';

	      case _ansiStyles.default.green.open:
	        return '<green>';

	      case _ansiStyles.default.cyan.open:
	        return '<cyan>';

	      case _ansiStyles.default.gray.open:
	        return '<gray>';

	      case _ansiStyles.default.white.open:
	        return '<white>';

	      case _ansiStyles.default.yellow.open:
	        return '<yellow>';

	      case _ansiStyles.default.bgRed.open:
	        return '<bgRed>';

	      case _ansiStyles.default.bgGreen.open:
	        return '<bgGreen>';

	      case _ansiStyles.default.bgYellow.open:
	        return '<bgYellow>';

	      case _ansiStyles.default.inverse.open:
	        return '<inverse>';

	      case _ansiStyles.default.dim.open:
	        return '<dim>';

	      case _ansiStyles.default.bold.open:
	        return '<bold>';

	      default:
	        return '';
	    }
	  });

	const test = val =>
	  typeof val === 'string' && !!val.match((0, _ansiRegex.default)());

	ConvertAnsi.test = test;

	const serialize = (val, config, indentation, depth, refs, printer) =>
	  printer(toHumanReadableAnsi(val), config, indentation, depth, refs);

	ConvertAnsi.serialize = serialize;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	ConvertAnsi.default = _default;
	return ConvertAnsi;
}

var DOMCollection = {};

var hasRequiredDOMCollection;

function requireDOMCollection () {
	if (hasRequiredDOMCollection) return DOMCollection;
	hasRequiredDOMCollection = 1;

	Object.defineProperty(DOMCollection, '__esModule', {
	  value: true
	});
	DOMCollection.default = DOMCollection.serialize = DOMCollection.test = void 0;

	var _collections = requireCollections();

	function _objectSpread(target) {
	  for (var i = 1; i < arguments.length; i++) {
	    var source = arguments[i] != null ? arguments[i] : {};
	    var ownKeys = Object.keys(source);
	    if (typeof Object.getOwnPropertySymbols === 'function') {
	      ownKeys = ownKeys.concat(
	        Object.getOwnPropertySymbols(source).filter(function(sym) {
	          return Object.getOwnPropertyDescriptor(source, sym).enumerable;
	        })
	      );
	    }
	    ownKeys.forEach(function(key) {
	      _defineProperty(target, key, source[key]);
	    });
	  }
	  return target;
	}

	function _defineProperty(obj, key, value) {
	  if (key in obj) {
	    Object.defineProperty(obj, key, {
	      value: value,
	      enumerable: true,
	      configurable: true,
	      writable: true
	    });
	  } else {
	    obj[key] = value;
	  }
	  return obj;
	}

	const SPACE = ' ';
	const OBJECT_NAMES = ['DOMStringMap', 'NamedNodeMap'];
	const ARRAY_REGEXP = /^(HTML\w*Collection|NodeList)$/;

	const testName = name =>
	  OBJECT_NAMES.indexOf(name) !== -1 || ARRAY_REGEXP.test(name);

	const test = val =>
	  val &&
	  val.constructor &&
	  val.constructor.name &&
	  testName(val.constructor.name); // Convert array of attribute objects to props object.

	DOMCollection.test = test;

	const propsReducer = (props, attribute) => {
	  props[attribute.name] = attribute.value;
	  return props;
	};

	const serialize = (collection, config, indentation, depth, refs, printer) => {
	  const name = collection.constructor.name;

	  if (++depth > config.maxDepth) {
	    return '[' + name + ']';
	  }

	  return (
	    (config.min ? '' : name + SPACE) +
	    (OBJECT_NAMES.indexOf(name) !== -1
	      ? '{' +
	        (0, _collections.printObjectProperties)(
	          name === 'NamedNodeMap'
	            ? Array.prototype.reduce.call(collection, propsReducer, {})
	            : _objectSpread({}, collection),
	          config,
	          indentation,
	          depth,
	          refs,
	          printer
	        ) +
	        '}'
	      : '[' +
	        (0, _collections.printListItems)(
	          Array.from(collection),
	          config,
	          indentation,
	          depth,
	          refs,
	          printer
	        ) +
	        ']')
	  );
	};

	DOMCollection.serialize = serialize;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	DOMCollection.default = _default;
	return DOMCollection;
}

var DOMElement = {};

var markup = {};

var escapeHTML = {};

var hasRequiredEscapeHTML;

function requireEscapeHTML () {
	if (hasRequiredEscapeHTML) return escapeHTML;
	hasRequiredEscapeHTML = 1;

	Object.defineProperty(escapeHTML, '__esModule', {
	  value: true
	});
	escapeHTML.default = escapeHTML$1;

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	function escapeHTML$1(str) {
	  return str.replace(/</g, '&lt;').replace(/>/g, '&gt;');
	}
	return escapeHTML;
}

var hasRequiredMarkup;

function requireMarkup () {
	if (hasRequiredMarkup) return markup;
	hasRequiredMarkup = 1;

	Object.defineProperty(markup, '__esModule', {
	  value: true
	});
	markup.printElementAsLeaf = markup.printElement = markup.printComment = markup.printText = markup.printChildren = markup.printProps = void 0;

	var _escapeHTML = _interopRequireDefault(requireEscapeHTML());

	function _interopRequireDefault(obj) {
	  return obj && obj.__esModule ? obj : {default: obj};
	}

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	// Return empty string if keys is empty.
	const printProps = (keys, props, config, indentation, depth, refs, printer) => {
	  const indentationNext = indentation + config.indent;
	  const colors = config.colors;
	  return keys
	    .map(key => {
	      const value = props[key];
	      let printed = printer(value, config, indentationNext, depth, refs);

	      if (typeof value !== 'string') {
	        if (printed.indexOf('\n') !== -1) {
	          printed =
	            config.spacingOuter +
	            indentationNext +
	            printed +
	            config.spacingOuter +
	            indentation;
	        }

	        printed = '{' + printed + '}';
	      }

	      return (
	        config.spacingInner +
	        indentation +
	        colors.prop.open +
	        key +
	        colors.prop.close +
	        '=' +
	        colors.value.open +
	        printed +
	        colors.value.close
	      );
	    })
	    .join('');
	}; // Return empty string if children is empty.

	markup.printProps = printProps;

	const printChildren = (children, config, indentation, depth, refs, printer) =>
	  children
	    .map(
	      child =>
	        config.spacingOuter +
	        indentation +
	        (typeof child === 'string'
	          ? printText(child, config)
	          : printer(child, config, indentation, depth, refs))
	    )
	    .join('');

	markup.printChildren = printChildren;

	const printText = (text, config) => {
	  const contentColor = config.colors.content;
	  return (
	    contentColor.open + (0, _escapeHTML.default)(text) + contentColor.close
	  );
	};

	markup.printText = printText;

	const printComment = (comment, config) => {
	  const commentColor = config.colors.comment;
	  return (
	    commentColor.open +
	    '<!--' +
	    (0, _escapeHTML.default)(comment) +
	    '-->' +
	    commentColor.close
	  );
	}; // Separate the functions to format props, children, and element,
	// so a plugin could override a particular function, if needed.
	// Too bad, so sad: the traditional (but unnecessary) space
	// in a self-closing tagColor requires a second test of printedProps.

	markup.printComment = printComment;

	const printElement = (
	  type,
	  printedProps,
	  printedChildren,
	  config,
	  indentation
	) => {
	  const tagColor = config.colors.tag;
	  return (
	    tagColor.open +
	    '<' +
	    type +
	    (printedProps &&
	      tagColor.close +
	        printedProps +
	        config.spacingOuter +
	        indentation +
	        tagColor.open) +
	    (printedChildren
	      ? '>' +
	        tagColor.close +
	        printedChildren +
	        config.spacingOuter +
	        indentation +
	        tagColor.open +
	        '</' +
	        type
	      : (printedProps && !config.min ? '' : ' ') + '/') +
	    '>' +
	    tagColor.close
	  );
	};

	markup.printElement = printElement;

	const printElementAsLeaf = (type, config) => {
	  const tagColor = config.colors.tag;
	  return (
	    tagColor.open +
	    '<' +
	    type +
	    tagColor.close +
	    ' …' +
	    tagColor.open +
	    ' />' +
	    tagColor.close
	  );
	};

	markup.printElementAsLeaf = printElementAsLeaf;
	return markup;
}

var hasRequiredDOMElement;

function requireDOMElement () {
	if (hasRequiredDOMElement) return DOMElement;
	hasRequiredDOMElement = 1;

	Object.defineProperty(DOMElement, '__esModule', {
	  value: true
	});
	DOMElement.default = DOMElement.serialize = DOMElement.test = void 0;

	var _markup = requireMarkup();

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	const ELEMENT_NODE = 1;
	const TEXT_NODE = 3;
	const COMMENT_NODE = 8;
	const FRAGMENT_NODE = 11;
	const ELEMENT_REGEXP = /^((HTML|SVG)\w*)?Element$/;

	const testNode = (nodeType, name) =>
	  (nodeType === ELEMENT_NODE && ELEMENT_REGEXP.test(name)) ||
	  (nodeType === TEXT_NODE && name === 'Text') ||
	  (nodeType === COMMENT_NODE && name === 'Comment') ||
	  (nodeType === FRAGMENT_NODE && name === 'DocumentFragment');

	const test = val =>
	  val &&
	  val.constructor &&
	  val.constructor.name &&
	  testNode(val.nodeType, val.constructor.name);

	DOMElement.test = test;

	function nodeIsText(node) {
	  return node.nodeType === TEXT_NODE;
	}

	function nodeIsComment(node) {
	  return node.nodeType === COMMENT_NODE;
	}

	function nodeIsFragment(node) {
	  return node.nodeType === FRAGMENT_NODE;
	}

	const serialize = (node, config, indentation, depth, refs, printer) => {
	  if (nodeIsText(node)) {
	    return (0, _markup.printText)(node.data, config);
	  }

	  if (nodeIsComment(node)) {
	    return (0, _markup.printComment)(node.data, config);
	  }

	  const type = nodeIsFragment(node)
	    ? `DocumentFragment`
	    : node.tagName.toLowerCase();

	  if (++depth > config.maxDepth) {
	    return (0, _markup.printElementAsLeaf)(type, config);
	  }

	  return (0, _markup.printElement)(
	    type,
	    (0, _markup.printProps)(
	      nodeIsFragment(node)
	        ? []
	        : Array.from(node.attributes)
	            .map(attr => attr.name)
	            .sort(),
	      nodeIsFragment(node)
	        ? []
	        : Array.from(node.attributes).reduce((props, attribute) => {
	            props[attribute.name] = attribute.value;
	            return props;
	          }, {}),
	      config,
	      indentation + config.indent,
	      depth,
	      refs,
	      printer
	    ),
	    (0, _markup.printChildren)(
	      Array.prototype.slice.call(node.childNodes || node.children),
	      config,
	      indentation + config.indent,
	      depth,
	      refs,
	      printer
	    ),
	    config,
	    indentation
	  );
	};

	DOMElement.serialize = serialize;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	DOMElement.default = _default;
	return DOMElement;
}

var Immutable = {};

var hasRequiredImmutable;

function requireImmutable () {
	if (hasRequiredImmutable) return Immutable;
	hasRequiredImmutable = 1;

	Object.defineProperty(Immutable, '__esModule', {
	  value: true
	});
	Immutable.default = Immutable.test = Immutable.serialize = void 0;

	var _collections = requireCollections();

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	// SENTINEL constants are from https://github.com/facebook/immutable-js
	const IS_ITERABLE_SENTINEL = '@@__IMMUTABLE_ITERABLE__@@';
	const IS_LIST_SENTINEL = '@@__IMMUTABLE_LIST__@@';
	const IS_KEYED_SENTINEL = '@@__IMMUTABLE_KEYED__@@';
	const IS_MAP_SENTINEL = '@@__IMMUTABLE_MAP__@@';
	const IS_ORDERED_SENTINEL = '@@__IMMUTABLE_ORDERED__@@';
	const IS_RECORD_SENTINEL = '@@__IMMUTABLE_RECORD__@@'; // immutable v4

	const IS_SEQ_SENTINEL = '@@__IMMUTABLE_SEQ__@@';
	const IS_SET_SENTINEL = '@@__IMMUTABLE_SET__@@';
	const IS_STACK_SENTINEL = '@@__IMMUTABLE_STACK__@@';

	const getImmutableName = name => 'Immutable.' + name;

	const printAsLeaf = name => '[' + name + ']';

	const SPACE = ' ';
	const LAZY = '…'; // Seq is lazy if it calls a method like filter

	const printImmutableEntries = (
	  val,
	  config,
	  indentation,
	  depth,
	  refs,
	  printer,
	  type
	) =>
	  ++depth > config.maxDepth
	    ? printAsLeaf(getImmutableName(type))
	    : getImmutableName(type) +
	      SPACE +
	      '{' +
	      (0, _collections.printIteratorEntries)(
	        val.entries(),
	        config,
	        indentation,
	        depth,
	        refs,
	        printer
	      ) +
	      '}'; // Record has an entries method because it is a collection in immutable v3.
	// Return an iterator for Immutable Record from version v3 or v4.

	const getRecordEntries = val => {
	  let i = 0;
	  return {
	    next() {
	      if (i < val._keys.length) {
	        const key = val._keys[i++];
	        return {
	          done: false,
	          value: [key, val.get(key)]
	        };
	      }

	      return {
	        done: true
	      };
	    }
	  };
	};

	const printImmutableRecord = (
	  val,
	  config,
	  indentation,
	  depth,
	  refs,
	  printer
	) => {
	  // _name property is defined only for an Immutable Record instance
	  // which was constructed with a second optional descriptive name arg
	  const name = getImmutableName(val._name || 'Record');
	  return ++depth > config.maxDepth
	    ? printAsLeaf(name)
	    : name +
	        SPACE +
	        '{' +
	        (0, _collections.printIteratorEntries)(
	          getRecordEntries(val),
	          config,
	          indentation,
	          depth,
	          refs,
	          printer
	        ) +
	        '}';
	};

	const printImmutableSeq = (val, config, indentation, depth, refs, printer) => {
	  const name = getImmutableName('Seq');

	  if (++depth > config.maxDepth) {
	    return printAsLeaf(name);
	  }

	  if (val[IS_KEYED_SENTINEL]) {
	    return (
	      name +
	      SPACE +
	      '{' + // from Immutable collection of entries or from ECMAScript object
	      (val._iter || val._object
	        ? (0, _collections.printIteratorEntries)(
	            val.entries(),
	            config,
	            indentation,
	            depth,
	            refs,
	            printer
	          )
	        : LAZY) +
	      '}'
	    );
	  }

	  return (
	    name +
	    SPACE +
	    '[' +
	    (val._iter || // from Immutable collection of values
	    val._array || // from ECMAScript array
	    val._collection || // from ECMAScript collection in immutable v4
	    val._iterable // from ECMAScript collection in immutable v3
	      ? (0, _collections.printIteratorValues)(
	          val.values(),
	          config,
	          indentation,
	          depth,
	          refs,
	          printer
	        )
	      : LAZY) +
	    ']'
	  );
	};

	const printImmutableValues = (
	  val,
	  config,
	  indentation,
	  depth,
	  refs,
	  printer,
	  type
	) =>
	  ++depth > config.maxDepth
	    ? printAsLeaf(getImmutableName(type))
	    : getImmutableName(type) +
	      SPACE +
	      '[' +
	      (0, _collections.printIteratorValues)(
	        val.values(),
	        config,
	        indentation,
	        depth,
	        refs,
	        printer
	      ) +
	      ']';

	const serialize = (val, config, indentation, depth, refs, printer) => {
	  if (val[IS_MAP_SENTINEL]) {
	    return printImmutableEntries(
	      val,
	      config,
	      indentation,
	      depth,
	      refs,
	      printer,
	      val[IS_ORDERED_SENTINEL] ? 'OrderedMap' : 'Map'
	    );
	  }

	  if (val[IS_LIST_SENTINEL]) {
	    return printImmutableValues(
	      val,
	      config,
	      indentation,
	      depth,
	      refs,
	      printer,
	      'List'
	    );
	  }

	  if (val[IS_SET_SENTINEL]) {
	    return printImmutableValues(
	      val,
	      config,
	      indentation,
	      depth,
	      refs,
	      printer,
	      val[IS_ORDERED_SENTINEL] ? 'OrderedSet' : 'Set'
	    );
	  }

	  if (val[IS_STACK_SENTINEL]) {
	    return printImmutableValues(
	      val,
	      config,
	      indentation,
	      depth,
	      refs,
	      printer,
	      'Stack'
	    );
	  }

	  if (val[IS_SEQ_SENTINEL]) {
	    return printImmutableSeq(val, config, indentation, depth, refs, printer);
	  } // For compatibility with immutable v3 and v4, let record be the default.

	  return printImmutableRecord(val, config, indentation, depth, refs, printer);
	}; // Explicitly comparing sentinel properties to true avoids false positive
	// when mock identity-obj-proxy returns the key as the value for any key.

	Immutable.serialize = serialize;

	const test = val =>
	  val &&
	  (val[IS_ITERABLE_SENTINEL] === true || val[IS_RECORD_SENTINEL] === true);

	Immutable.test = test;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	Immutable.default = _default;
	return Immutable;
}

var ReactElement = {};

var reactIsExports = {};
var reactIs = {
  get exports(){ return reactIsExports; },
  set exports(v){ reactIsExports = v; },
};

var reactIs_production_min = {};

var hasRequiredReactIs_production_min;

function requireReactIs_production_min () {
	if (hasRequiredReactIs_production_min) return reactIs_production_min;
	hasRequiredReactIs_production_min = 1;
var b="function"===typeof Symbol&&Symbol.for,c=b?Symbol.for("react.element"):60103,d=b?Symbol.for("react.portal"):60106,e=b?Symbol.for("react.fragment"):60107,f=b?Symbol.for("react.strict_mode"):60108,g=b?Symbol.for("react.profiler"):60114,h=b?Symbol.for("react.provider"):60109,k=b?Symbol.for("react.context"):60110,l=b?Symbol.for("react.async_mode"):60111,m=b?Symbol.for("react.concurrent_mode"):60111,n=b?Symbol.for("react.forward_ref"):60112,p=b?Symbol.for("react.suspense"):60113,q=b?
	Symbol.for("react.suspense_list"):60120,r=b?Symbol.for("react.memo"):60115,t=b?Symbol.for("react.lazy"):60116,v=b?Symbol.for("react.block"):60121,w=b?Symbol.for("react.fundamental"):60117,x=b?Symbol.for("react.responder"):60118,y=b?Symbol.for("react.scope"):60119;
	function z(a){if("object"===typeof a&&null!==a){var u=a.$$typeof;switch(u){case c:switch(a=a.type,a){case l:case m:case e:case g:case f:case p:return a;default:switch(a=a&&a.$$typeof,a){case k:case n:case t:case r:case h:return a;default:return u}}case d:return u}}}function A(a){return z(a)===m}reactIs_production_min.AsyncMode=l;reactIs_production_min.ConcurrentMode=m;reactIs_production_min.ContextConsumer=k;reactIs_production_min.ContextProvider=h;reactIs_production_min.Element=c;reactIs_production_min.ForwardRef=n;reactIs_production_min.Fragment=e;reactIs_production_min.Lazy=t;reactIs_production_min.Memo=r;reactIs_production_min.Portal=d;
	reactIs_production_min.Profiler=g;reactIs_production_min.StrictMode=f;reactIs_production_min.Suspense=p;reactIs_production_min.isAsyncMode=function(a){return A(a)||z(a)===l};reactIs_production_min.isConcurrentMode=A;reactIs_production_min.isContextConsumer=function(a){return z(a)===k};reactIs_production_min.isContextProvider=function(a){return z(a)===h};reactIs_production_min.isElement=function(a){return "object"===typeof a&&null!==a&&a.$$typeof===c};reactIs_production_min.isForwardRef=function(a){return z(a)===n};reactIs_production_min.isFragment=function(a){return z(a)===e};reactIs_production_min.isLazy=function(a){return z(a)===t};
	reactIs_production_min.isMemo=function(a){return z(a)===r};reactIs_production_min.isPortal=function(a){return z(a)===d};reactIs_production_min.isProfiler=function(a){return z(a)===g};reactIs_production_min.isStrictMode=function(a){return z(a)===f};reactIs_production_min.isSuspense=function(a){return z(a)===p};
	reactIs_production_min.isValidElementType=function(a){return "string"===typeof a||"function"===typeof a||a===e||a===m||a===g||a===f||a===p||a===q||"object"===typeof a&&null!==a&&(a.$$typeof===t||a.$$typeof===r||a.$$typeof===h||a.$$typeof===k||a.$$typeof===n||a.$$typeof===w||a.$$typeof===x||a.$$typeof===y||a.$$typeof===v)};reactIs_production_min.typeOf=z;
	return reactIs_production_min;
}

var hasRequiredReactIs;

function requireReactIs () {
	if (hasRequiredReactIs) return reactIsExports;
	hasRequiredReactIs = 1;
	(function (module) {

		{
		  module.exports = requireReactIs_production_min();
		}
} (reactIs));
	return reactIsExports;
}

var hasRequiredReactElement;

function requireReactElement () {
	if (hasRequiredReactElement) return ReactElement;
	hasRequiredReactElement = 1;

	Object.defineProperty(ReactElement, '__esModule', {
	  value: true
	});
	ReactElement.default = ReactElement.test = ReactElement.serialize = void 0;

	var ReactIs = _interopRequireWildcard(requireReactIs());

	var _markup = requireMarkup();

	function _interopRequireWildcard(obj) {
	  if (obj && obj.__esModule) {
	    return obj;
	  } else {
	    var newObj = {};
	    if (obj != null) {
	      for (var key in obj) {
	        if (Object.prototype.hasOwnProperty.call(obj, key)) {
	          var desc =
	            Object.defineProperty && Object.getOwnPropertyDescriptor
	              ? Object.getOwnPropertyDescriptor(obj, key)
	              : {};
	          if (desc.get || desc.set) {
	            Object.defineProperty(newObj, key, desc);
	          } else {
	            newObj[key] = obj[key];
	          }
	        }
	      }
	    }
	    newObj.default = obj;
	    return newObj;
	  }
	}

	/**
	 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
	 *
	 * This source code is licensed under the MIT license found in the
	 * LICENSE file in the root directory of this source tree.
	 */
	// Given element.props.children, or subtree during recursive traversal,
	// return flattened array of children.
	const getChildren = (arg, children = []) => {
	  if (Array.isArray(arg)) {
	    arg.forEach(item => {
	      getChildren(item, children);
	    });
	  } else if (arg != null && arg !== false) {
	    children.push(arg);
	  }

	  return children;
	};

	const getType = element => {
	  const type = element.type;

	  if (typeof type === 'string') {
	    return type;
	  }

	  if (typeof type === 'function') {
	    return type.displayName || type.name || 'Unknown';
	  }

	  if (ReactIs.isFragment(element)) {
	    return 'React.Fragment';
	  }

	  if (ReactIs.isSuspense(element)) {
	    return 'React.Suspense';
	  }

	  if (typeof type === 'object' && type !== null) {
	    if (ReactIs.isContextProvider(element)) {
	      return 'Context.Provider';
	    }

	    if (ReactIs.isContextConsumer(element)) {
	      return 'Context.Consumer';
	    }

	    if (ReactIs.isForwardRef(element)) {
	      const functionName = type.render.displayName || type.render.name || '';
	      return functionName !== ''
	        ? 'ForwardRef(' + functionName + ')'
	        : 'ForwardRef';
	    }

	    if (ReactIs.isMemo(type)) {
	      const functionName =
	        type.displayName || type.type.displayName || type.type.name || '';
	      return functionName !== '' ? 'Memo(' + functionName + ')' : 'Memo';
	    }
	  }

	  return 'UNDEFINED';
	};

	const getPropKeys = element => {
	  const props = element.props;
	  return Object.keys(props)
	    .filter(key => key !== 'children' && props[key] !== undefined)
	    .sort();
	};

	const serialize = (element, config, indentation, depth, refs, printer) =>
	  ++depth > config.maxDepth
	    ? (0, _markup.printElementAsLeaf)(getType(element), config)
	    : (0, _markup.printElement)(
	        getType(element),
	        (0, _markup.printProps)(
	          getPropKeys(element),
	          element.props,
	          config,
	          indentation + config.indent,
	          depth,
	          refs,
	          printer
	        ),
	        (0, _markup.printChildren)(
	          getChildren(element.props.children),
	          config,
	          indentation + config.indent,
	          depth,
	          refs,
	          printer
	        ),
	        config,
	        indentation
	      );

	ReactElement.serialize = serialize;

	const test = val => val && ReactIs.isElement(val);

	ReactElement.test = test;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	ReactElement.default = _default;
	return ReactElement;
}

var ReactTestComponent = {};

var hasRequiredReactTestComponent;

function requireReactTestComponent () {
	if (hasRequiredReactTestComponent) return ReactTestComponent;
	hasRequiredReactTestComponent = 1;

	Object.defineProperty(ReactTestComponent, '__esModule', {
	  value: true
	});
	ReactTestComponent.default = ReactTestComponent.test = ReactTestComponent.serialize = void 0;

	var _markup = requireMarkup();

	var Symbol = commonjsGlobal['jest-symbol-do-not-touch'] || commonjsGlobal.Symbol;
	const testSymbol = Symbol.for('react.test.json');

	const getPropKeys = object => {
	  const props = object.props;
	  return props
	    ? Object.keys(props)
	        .filter(key => props[key] !== undefined)
	        .sort()
	    : [];
	};

	const serialize = (object, config, indentation, depth, refs, printer) =>
	  ++depth > config.maxDepth
	    ? (0, _markup.printElementAsLeaf)(object.type, config)
	    : (0, _markup.printElement)(
	        object.type,
	        object.props
	          ? (0, _markup.printProps)(
	              getPropKeys(object),
	              object.props,
	              config,
	              indentation + config.indent,
	              depth,
	              refs,
	              printer
	            )
	          : '',
	        object.children
	          ? (0, _markup.printChildren)(
	              object.children,
	              config,
	              indentation + config.indent,
	              depth,
	              refs,
	              printer
	            )
	          : '',
	        config,
	        indentation
	      );

	ReactTestComponent.serialize = serialize;

	const test = val => val && val.$$typeof === testSymbol;

	ReactTestComponent.test = test;
	const plugin = {
	  serialize,
	  test
	};
	var _default = plugin;
	ReactTestComponent.default = _default;
	return ReactTestComponent;
}

var build;
var hasRequiredBuild;

function requireBuild () {
	if (hasRequiredBuild) return build;
	hasRequiredBuild = 1;

	var _ansiStyles = _interopRequireDefault(requireAnsiStyles());

	var _collections = requireCollections();

	var _AsymmetricMatcher = _interopRequireDefault(
	  requireAsymmetricMatcher()
	);

	var _ConvertAnsi = _interopRequireDefault(requireConvertAnsi());

	var _DOMCollection = _interopRequireDefault(requireDOMCollection());

	var _DOMElement = _interopRequireDefault(requireDOMElement());

	var _Immutable = _interopRequireDefault(requireImmutable());

	var _ReactElement = _interopRequireDefault(requireReactElement());

	var _ReactTestComponent = _interopRequireDefault(
	  requireReactTestComponent()
	);

	function _interopRequireDefault(obj) {
	  return obj && obj.__esModule ? obj : {default: obj};
	}

	var Symbol = commonjsGlobal['jest-symbol-do-not-touch'] || commonjsGlobal.Symbol;
	const toString = Object.prototype.toString;
	const toISOString = Date.prototype.toISOString;
	const errorToString = Error.prototype.toString;
	const regExpToString = RegExp.prototype.toString;
	const symbolToString = Symbol.prototype.toString;
	/**
	 * Explicitly comparing typeof constructor to 