import { useState } from "react";
import { useNavigate, Link } from "react-router-dom";
import api from "../api/api";
import { useAuth } from "../AuthContext";

// helper: bikin huruf depan tiap kata uppercase
const capitalizeWords = (text = "") => {
  return text
    .toLowerCase()
    .split(" ")
    .filter(Boolean)
    .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
    .join(" ");
};

export default function SignupPage() {
  const navigate = useNavigate();
  const { login } = useAuth();
  const [name, setName] = useState("");
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [passwordConfirm, setPasswordConfirm] = useState("");
  const [department, setDepartment] = useState("");
  const [position, setPosition] = useState("");
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError("");
    setLoading(true);

    const formattedName = capitalizeWords(name);
    const formattedPosition = capitalizeWords(position);

    try {
      await api.post("/register", {
        name: formattedName,
        email: email.toLowerCase(),          // email dibikin lowercase, bukan title case
        password,
        password_confirmation: passwordConfirm,
        department,
        position: formattedPosition || null,
      });

      await login(email.toLowerCase(), password);
      navigate("/");
    } catch (err) {
      console.error(err);
      setError(
        err.response?.data?.message ||
          (err.response?.data?.errors &&
            Object.values(err.response.data.errors).flat().join("\n")) ||
          "Gagal mendaftar. Cek kembali data yang diisi."
      );
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="login-page">
      <div className="login-card">
        <h1>Daftar Akun Baru</h1>
        <p>
          Akun baru otomatis memiliki role <strong>user</strong>.
        </p>

        <form onSubmit={handleSubmit}>
          {/* NAMA */}
          <div className="login-field">
            <label>Nama Lengkap</label>
            <input
              value={name}
              onChange={(e) => setName(e.target.value)}          // <- TIDAK di-capitalize di sini
              onBlur={() => setName((prev) => capitalizeWords(prev))} // <- format saat blur
              required
            />
          </div>

          {/* EMAIL */}
          <div className="login-field">
            <label>Email</label>
            <input
              type="email"
              value={email}
              onChange={(e) => setEmail(e.target.value)}         // biarin apa adanya
              required
            />
          </div>

          {/* PASSWORD */}
          <div className="login-field">
            <label>Password</label>
            <input
              type="password"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              required
            />
          </div>

          <div className="login-field">
            <label>Konfirmasi Password</label>
            <input
              type="password"
              value={passwordConfirm}
              onChange={(e) => setPasswordConfirm(e.target.value)}
              required
            />
          </div>

          {/* DEPARTEMEN */}
          <div className="login-field login-field-select">
            <label>Departemen</label>
            <select
              value={department}
              onChange={(e) => setDepartment(e.target.value)}
              required
            >
              <option value="">-- Pilih Departemen --</option>
              <option value="Produksi">Produksi</option>
              <option value="IT">IT</option>
              <option value="HRGA">HRGA</option>
              <option value="Warehouse">Warehouse</option>
              <option value="Purchasing">Purchasing</option>
              <option value="QC">QC</option>
              <option value="QA">QA</option>
              <option value="Marketing">Marketing</option>
              <option value="Logistik">Logistik</option>
            </select>
          </div>

          {/* JABATAN */}
          <div className="login-field">
            <label>Jabatan</label>
            <input
              value={position}
              onChange={(e) => setPosition(e.target.value)}       // lagi-lagi: raw
              onBlur={() => setPosition((prev) => capitalizeWords(prev))}
              placeholder="contoh: Operator"
              required
            />
          </div>

          {error && (
            <div style={{ color: "#fca5a5", fontSize: 12, marginBottom: 8 }}>
              {error}
            </div>
          )}

          <button
            type="submit"
            className="btn btn-primary"
            style={{ width: "100%", justifyContent: "center" }}
            disabled={loading}
          >
            {loading ? "Mendaftarkan..." : "Daftar"}
          </button>
        </form>

        <p style={{ marginTop: 12, fontSize: 12 }}>
          Sudah punya akun? <Link to="/">Login di sini</Link>
        </p>
      </div>
    </div>
  );
}