import { useEffect, useState } from "react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../api/api";

export default function TrainingEditPage() {
  const { id } = useParams();
  const navigate = useNavigate();

  const [form, setForm] = useState({
    title: "",
    training_date: "",
    vendor: "",
    cost: 0,
    purposes: [],
    description: "",
    expected_result: "",
  });

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState("");

  // helper: convert "2025-12-15T00:00:00.000000Z" → "2025-12-15"
  const toInputDate = (value) => {
    if (!value) return "";
    const d = new Date(value);
    if (Number.isNaN(d.getTime())) return value;
    const yyyy = d.getFullYear();
    const mm = String(d.getMonth() + 1).padStart(2, "0");
    const dd = String(d.getDate()).padStart(2, "0");
    return `${yyyy}-${mm}-${dd}`;
  };

  // load data awal
  useEffect(() => {
    const load = async () => {
      try {
        const res = await api.get(`/training-requests/${id}`);
        const data = res.data;

        setForm({
          title: data.title || "",
          training_date: toInputDate(data.training_date),
          vendor: data.vendor || "",
          cost: data.cost || 0,
          purposes: Array.isArray(data.purposes) ? data.purposes : [],
          description: data.description || "",
          expected_result: data.expected_result || "",
        });
      } catch (err) {
        console.error(err);
        setError(
          err.response?.data?.message ||
            "Gagal memuat data pengajuan. Mungkin sudah tidak bisa diedit."
        );
      } finally {
        setLoading(false);
      }
    };

    load();
  }, [id]);

  const togglePurpose = (value) => {
    setForm((prev) => ({
      ...prev,
      purposes: prev.purposes.includes(value)
        ? prev.purposes.filter((v) => v !== value)
        : [...prev.purposes, value],
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setSaving(true);
    setError("");

    try {
      await api.put(`/training-requests/${id}`, form);
      navigate("/"); // kembali ke list
    } catch (err) {
      console.error(err);
      setError(
        err.response?.data?.message ||
          "Gagal menyimpan perubahan. Cek lagi datanya."
      );
    } finally {
      setSaving(false);
    }
  };

  return (
    <div className="form-wrapper">
      <div className="form-container">
        <div className="form-card">
          <div className="form-header">
            <h1 className="form-title">Edit Pengajuan Training</h1>
            <p className="form-subtitle">
              Perbarui data pengajuan. Hanya pengajuan dengan status{" "}
              <strong>submitted</strong> yang dapat diubah.
            </p>
          </div>

          {loading ? (
            <div style={{ fontSize: 12 }}>Memuat data...</div>
          ) : (
            <form onSubmit={handleSubmit}>
              <div className="form-grid">
                {/* NAMA TRAINING */}
                <div className="form-field full">
                  <label className="form-label">Nama Training</label>
                  <input
                    className="form-input"
                    type="text"
                    value={form.title}
                    onChange={(e) =>
                      setForm({ ...form, title: e.target.value })
                    }
                    required
                  />
                </div>

                {/* TANGGAL */}
                <div className="form-field">
                  <label className="form-label">Tanggal Training</label>
                  <input
                    className="form-input"
                    type="date"
                    value={form.training_date}
                    onChange={(e) =>
                      setForm({ ...form, training_date: e.target.value })
                    }
                    required
                  />
                </div>

                {/* PENYELENGGARA */}
                <div className="form-field">
                  <label className="form-label">Penyelenggara</label>
                  <input
                    className="form-input"
                    type="text"
                    value={form.vendor}
                    onChange={(e) =>
                      setForm({ ...form, vendor: e.target.value })
                    }
                  />
                </div>

                {/* BIAYA */}
                <div className="form-field">
                  <label className="form-label">Biaya (Rp)</label>
                  <input
                    className="form-input"
                    type="number"
                    min={0}
                    value={form.cost}
                    onChange={(e) =>
                      setForm({
                        ...form,
                        cost: Number(e.target.value || 0),
                      })
                    }
                  />
                </div>

                {/* TUJUAN TRAINING */}
                <div className="form-field full">
                  <label className="form-label">Tujuan Training</label>
                  <div className="form-checkbox-group">
                    <label className="form-checkbox-item">
                      <input
                        type="checkbox"
                        checked={form.purposes.includes("skill")}
                        onChange={() => togglePurpose("skill")}
                      />
                      <span>Meningkatkan keterampilan</span>
                    </label>
                    <label className="form-checkbox-item">
                      <input
                        type="checkbox"
                        checked={form.purposes.includes("managerial")}
                        onChange={() => togglePurpose("managerial")}
                      />
                      <span>Meningkatkan pengetahuan manajerial</span>
                    </label>
                    <label className="form-checkbox-item">
                      <input
                        type="checkbox"
                        checked={form.purposes.includes("jobdesc")}
                        onChange={() => togglePurpose("jobdesc")}
                      />
                      <span>Memenuhi kriteria Job Description</span>
                    </label>
                    <label className="form-checkbox-item">
                      <input
                        type="checkbox"
                        checked={form.purposes.includes("refreshment")}
                        onChange={() => togglePurpose("refreshment")}
                      />
                      <span>Penyegaran / Refreshment</span>
                    </label>
                  </div>
                </div>

                {/* DESKRIPSI TAMBAHAN */}
                <div className="form-field full">
                  <label className="form-label">Deskripsi Tambahan</label>
                  <textarea
                    className="form-textarea"
                    value={form.description}
                    onChange={(e) =>
                      setForm({ ...form, description: e.target.value })
                    }
                  />
                </div>

                {/* HASIL DIHARAPKAN */}
                <div className="form-field full">
                  <label className="form-label">Hasil yang Diharapkan</label>
                  <textarea
                    className="form-textarea"
                    value={form.expected_result}
                    onChange={(e) =>
                      setForm({
                        ...form,
                        expected_result: e.target.value,
                      })
                    }
                  />
                </div>
              </div>

              {error && (
                <div
                  style={{
                    marginTop: 10,
                    fontSize: 12,
                    color: "#fca5a5",
                  }}
                >
                  {error}
                </div>
              )}

              <div className="form-actions">
                <button
                  type="button"
                  className="btn btn-xs"
                  onClick={() => navigate("/")}
                  disabled={saving}
                >
                  Batal
                </button>
                <button
                  type="submit"
                  className="btn btn-primary btn-xs"
                  disabled={saving}
                >
                  {saving ? "Menyimpan..." : "Simpan Perubahan"}
                </button>
              </div>
            </form>
          )}
        </div>
      </div>
    </div>
  );
}