import { useState } from "react";
import { useNavigate } from "react-router-dom";
import api from "../api/api";

export default function TrainingFormPage() {
  const navigate = useNavigate();

  const [form, setForm] = useState({
    title: "",
    training_date: "",
    vendor: "",
    cost: "",
    purposes: [],
    description: "",
    expected_result: "",
  });

  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  const togglePurpose = (value) => {
    setForm((prev) => ({
      ...prev,
      purposes: prev.purposes.includes(value)
        ? prev.purposes.filter((v) => v !== value)
        : [...prev.purposes, value],
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError("");

    try {
      await api.post("/training-requests", form);
      navigate("/"); // kembali ke list setelah sukses
    } catch (err) {
      console.error(err);
      setError(
        err.response?.data?.message ||
          "Gagal menyimpan pengajuan. Cek kembali data yang diisi."
      );
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="form-wrapper">
      <div className="form-container">
        <div className="form-card">
          <div className="form-header">
            <h1 className="form-title">Form Pengajuan Training</h1>
            <p className="form-subtitle">
              Lengkapi data di bawah ini untuk mengajukan permohonan training.
            </p>
          </div>

          <form onSubmit={handleSubmit}>
            <div className="form-grid">
              {/* NAMA TRAINING */}
              <div className="form-field full">
                <label className="form-label">Nama Training</label>
                <input
                  className="form-input"
                  type="text"
                  value={form.title}
                  onChange={(e) =>
                    setForm({ ...form, title: e.target.value })
                  }
                  placeholder="cth: Training K3 Dasar"
                  required
                />
              </div>

              {/* TANGGAL */}
              <div className="form-field">
                <label className="form-label">Tanggal Training</label>
                <input
                  className="form-input"
                  type="date"
                  value={form.training_date}
                  onChange={(e) =>
                    setForm({ ...form, training_date: e.target.value })
                  }
                  required
                />
              </div>

              {/* PENYELENGGARA */}
              <div className="form-field">
                <label className="form-label">Penyelenggara</label>
                <input
                  className="form-input"
                  type="text"
                  value={form.vendor}
                  onChange={(e) =>
                    setForm({ ...form, vendor: e.target.value })
                  }
                  placeholder="cth: Lembaga K3 XYZ"
                />
              </div>

              {/* BIAYA */}
              <div className="form-field">
                <label className="form-label">Biaya (Rp)</label>
                <input
                  className="form-input"
                  type="number"
                  min={0}
                  value={form.cost}
                  onChange={(e) =>
                    setForm({
                      ...form,
                      cost: Number(e.target.value || 0),
                    })
                  }
                />
              </div>

              {/* TUJUAN TRAINING */}
              <div className="form-field full">
                <label className="form-label">Tujuan Training</label>
                <div className="form-checkbox-group">
                  <label className="form-checkbox-item">
                    <input
                      type="checkbox"
                      checked={form.purposes.includes("skill")}
                      onChange={() => togglePurpose("skill")}
                    />
                    <span>Meningkatkan keterampilan</span>
                  </label>
                  <label className="form-checkbox-item">
                    <input
                      type="checkbox"
                      checked={form.purposes.includes("managerial")}
                      onChange={() => togglePurpose("managerial")}
                    />
                    <span>Meningkatkan pengetahuan manajerial</span>
                  </label>
                  <label className="form-checkbox-item">
                    <input
                      type="checkbox"
                      checked={form.purposes.includes("jobdesc")}
                      onChange={() => togglePurpose("jobdesc")}
                    />
                    <span>Memenuhi kriteria Job Description</span>
                  </label>
                  <label className="form-checkbox-item">
                    <input
                      type="checkbox"
                      checked={form.purposes.includes("refreshment")}
                      onChange={() => togglePurpose("refreshment")}
                    />
                    <span>Penyegaran / Refreshment</span>
                  </label>
                </div>
              </div>

              {/* DESKRIPSI TAMBAHAN */}
              <div className="form-field full">
                <label className="form-label">Deskripsi Tambahan</label>
                <textarea
                  className="form-textarea"
                  value={form.description}
                  onChange={(e) =>
                    setForm({ ...form, description: e.target.value })
                  }
                  placeholder="cth: Karyawan baru, butuh pemahaman dasar K3 di area produksi."
                />
              </div>

              {/* HASIL DIHARAPKAN */}
              <div className="form-field full">
                <label className="form-label">Hasil yang Diharapkan</label>
                <textarea
                  className="form-textarea"
                  value={form.expected_result}
                  onChange={(e) =>
                    setForm({ ...form, expected_result: e.target.value })
                  }
                  placeholder="cth: Peserta mampu menerapkan prosedur K3 di line produksi."
                />
              </div>
            </div>

            {error && (
              <div
                style={{
                  marginTop: 10,
                  fontSize: 12,
                  color: "#fca5a5",
                }}
              >
                {error}
              </div>
            )}

            <div className="form-actions">
              <button
                type="button"
                className="btn btn-xs"
                onClick={() => navigate("/")}
                disabled={loading}
              >
                Batal
              </button>
              <button
                type="submit"
                className="btn btn-primary btn-xs"
                disabled={loading}
              >
                {loading ? "Menyimpan..." : "Simpan Pengajuan"}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}