import { useEffect, useState } from "react";
import { Link , useNavigate } from "react-router-dom";
import api from "../api/api";
import { useAuth } from "../AuthContext";
import { formatDateWithDay } from "../utils/date";


export default function TrainingListPage() {
  const navigate = useNavigate();
  const { user, logout, isHrd, isAdmin, isUser } = useAuth();
  const [items, setItems] = useState([]);
  const [loading, setLoading] = useState(true);

  // FILTER STATE
  const [statusFilter, setStatusFilter] = useState("");
  const [fromDate, setFromDate] = useState("");
  const [toDate, setToDate] = useState("");
  const [deptFilter, setDeptFilter] = useState("");
  const [search, setSearch] = useState("");

  // PAGINATION & SUMMARY
const [page, setPage] = useState(1);
const [perPage, setPerPage] = useState(10);
const [meta, setMeta] = useState({
  current_page: 1,
  last_page: 1,
  total: 0,
  per_page: 10,
});
const [summary, setSummary] = useState({
  total_requests: 0,
  total_cost: 0,
  total_submitted: 0,
  total_approved_hrd: 0,
  total_approved_dir: 0,
  total_rejected_hrd: 0,
  total_rejected_dir: 0,
});

 const loadData = async () => {
  setLoading(true);

  const params = {};

  if (statusFilter) params.status = statusFilter;
  if (fromDate) params.from = fromDate;
  if (toDate) params.to = toDate;
  if (deptFilter) params.department = deptFilter;
  if (search) params.search = search;

  params.page = page;
  params.per_page = perPage;

  const res = await api.get("/training-requests", { params });

  setItems(res.data.data || []);
  setMeta(res.data.meta || meta);
  setSummary(res.data.summary || summary);

  setLoading(false);
};

  useEffect(() => {
  loadData();
  // eslint-disable-next-line react-hooks/exhaustive-deps
}, [page, perPage]);
  //useEffect(() => {
   // loadData();
  //}, []);

  const handleApproveHrd = async (id) => {
    await api.post(`/training-requests/${id}/approve-hrd`, {
      hrd_notes: "",
    });
    await loadData();
  };

  const handleRejectHrd = async (id) => {
    const reason = prompt("Alasan reject HRD:");
    await api.post(`/training-requests/${id}/reject-hrd`, {
      hrd_notes: reason || "",
    });
    await loadData();
  };

  const handleApproveDirector = async (id) => {
    await api.post(`/training-requests/${id}/approve-director`);
    await loadData();
  };

  const handleRejectDirector = async (id) => {
    const reason = prompt("Catatan (opsional):");
    await api.post(`/training-requests/${id}/reject-director`, {
      hrd_notes: reason || "",
    });
    await loadData();
  };

  //untuk delete list
  const deleteRequest = async (id) => {
    if (!window.confirm("Yakin ingin menghapus pengajuan ini?")) return;

    try {
        await api.delete(`/training-requests/${id}`);
        await loadData(); // refresh list setelah delete
    } catch (err) {
        console.error(err);
        alert(
        err.response?.data?.message ||
            "Gagal menghapus pengajuan. Cek apakah status masih SUBMITTED."
        );
    }
  };

  //untuk export to excel
  
  const handleExport = async () => {
    try {
        const params = {};

        if (statusFilter) params.status = statusFilter;
        if (fromDate) params.from = fromDate;
        if (toDate) params.to = toDate;
        if (deptFilter) params.department = deptFilter;
        if (search) params.search = search;

        const response = await api.get("/training-requests/export", {
        params,
        responseType: "blob", // penting untuk file
        });

        // Ambil nama file dari header kalau ada
        const disposition = response.headers["content-disposition"];
        let filename = "training_requests.csv";
        if (disposition && disposition.includes("filename=")) {
        const match = disposition.match(/filename="?([^"]+)"?/);
        if (match && match[1]) {
            filename = match[1];
        }
        }

        // Buat URL sementara untuk blob
        const url = window.URL.createObjectURL(new Blob([response.data]));
        const link = document.createElement("a");
        link.href = url;
        link.setAttribute("download", filename);
        document.body.appendChild(link);
        link.click();
        link.remove();
        window.URL.revokeObjectURL(url);
    } catch (err) {
        console.error(err);
        alert(
        err.response?.data?.message ||
            "Gagal export data. Pastikan kamu login sebagai HRD/Admin."
        );
    }
  };

  return (
  <div className="app-shell">
    <div className="app-container">

      <header className="app-header">
        <div>
          <div className="app-title">Pengajuan Training</div>
          <div className="app-subtitle">
            Modul internal | Training Request Workflow
          </div>
          <div className="app-userinfo">
            Login sebagai: <strong>{user?.name}</strong>{" "}
            <span className="tag-role">{user?.role}</span>
            <br />
            Dept: {user?.department} | Jabatan: {user?.position}
          </div>
        </div>

        <div className="header-actions">
          {isUser && (
           <button
             type="button"
             className="btn btn-primary btn-xs"
             onClick={() => navigate("/training/new")}
           >
             + Pengajuan Baru
           </button>
          )}

          {isHrd || isAdmin ? (
           <button
            type="button"
            className="btn btn-outline btn-xs"
            onClick={handleExport}
           >
             Export CSV
           </button>
          ) : null}

           <button
              type="button"
              className="btn btn-xs"
              onClick={logout}
           >
             Logout
           </button>
        </div>

      </header>

      <main className="card">
        {/* SUMMARY CARDS */}
        <div
          style={{
            display: "grid",
            gridTemplateColumns: "repeat(4, minmax(0, 1fr))",
            gap: 10,
            marginBottom: 12,
            fontSize: 12,
          }}
        >
          <SummaryCard
            title="Total Pengajuan"
            value={summary.total_requests}
            subtitle={formatCurrency(summary.total_cost)}
          />

          <SummaryCard
            title="Submitted"
            value={summary.total_submitted}
            subtitle="Menunggu HRD"
            tone="info"
          />

          <SummaryCard
            title="Approved"
            value={summary.total_approved_hrd + summary.total_approved_dir}
            subtitle={
              "HRD: " +
              summary.total_approved_hrd +
              " | Direktur: " +
              summary.total_approved_dir
            }
            tone="success"
          />

          <SummaryCard
            title="Rejected"
            value={summary.total_rejected_hrd + summary.total_rejected_dir}
            subtitle={
              "HRD: " +
              summary.total_rejected_hrd +
              " | Direktur: " +
              summary.total_rejected_dir
            }
            tone="danger"
          />
        </div>

        {/* FILTER BAR */}
        <div className="filter-bar">
          <div className="filter-field">
            <label>Status</label>
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
            >
              <option value="">Semua</option>
              <option value="submitted">Submitted</option>
              <option value="approved_hrd">Approved HRD</option>
              <option value="rejected_hrd">Rejected HRD</option>
              <option value="approved_director">Approved Direktur</option>
              <option value="rejected_director">Rejected Direktur</option>
            </select>
          </div>

          <div className="filter-field">
            <label>Dari Tanggal</label>
            <input
              type="date"
              value={fromDate}
              onChange={(e) => setFromDate(e.target.value)}
            />
          </div>

          <div className="filter-field">
            <label>Sampai Tanggal</label>
            <input
              type="date"
              value={toDate}
              onChange={(e) => setToDate(e.target.value)}
            />
          </div>

          <div className="filter-field">
            <label>Departemen</label>
            <input
              type="text"
              placeholder="cth: Produksi"
              value={deptFilter}
              onChange={(e) => setDeptFilter(e.target.value)}
            />
          </div>

          <div className="filter-field" style={{ flexGrow: 1 }}>
            <label>Cari</label>
            <input
              type="text"
              placeholder="Nama training / vendor / karyawan / No. pengajuan"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
            />
          </div>

          <div className="filter-field" style={{ flexDirection: "row", gap: 6 }}>
            <button type="button" className="btn btn-primary" onClick={loadData}>
              Apply
            </button>
            <button
              type="button"
              className="btn"
              onClick={() => {
                setStatusFilter("");
                setFromDate("");
                setToDate("");
                setDeptFilter("");
                setSearch("");
                setPage(1);
                loadData();
              }}
            >
              Reset
            </button>
          </div>
        </div>

        <div
          style={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            marginBottom: 8,
            fontSize: 12,
            color: "#9ca3af",
          }}
        >
        <div>
          <div>
            Total pengajuan (sesuai filter):{" "}
            <strong>{summary.total_requests}</strong>
          </div>
          <div>
            Total biaya:{" "}
            <strong>{formatCurrency(summary.total_cost)}</strong>
          </div>
        </div>

        <div>
          <span style={{ marginRight: 6 }}>Tampil per halaman:</span>
          <select
            value={perPage}
            onChange={(e) => {
              setPerPage(Number(e.target.value));
              setPage(1);
            }}
            style={{
              background: "var(--bg-soft)",
              color: "var(--text-main)",
              borderRadius: "999px",
              border: "1px solid var(--border-subtle)",
              fontSize: 11,
              padding: "4px 8px",
            }}
          >
            <option value={5}>5</option>
            <option value={10}>10</option>
            <option value={25}>25</option>
            <option value={50}>50</option>
          </select>
        </div>
        </div>
        
        {/* TABLE */}
        {loading ? (
          <div style={{ padding: 12, fontSize: 13 }}>Loading...</div>
        ) : (
          <div className="table-wrapper">
            <table className="table">
              <thead>
                <tr>
                  <th>No. Pengajuan</th>
                  <th>Nama</th>
                  <th>Tanggal</th>
                  <th>Vendor</th>
                  <th className="col-num">Biaya</th>
                  <th>Status</th>
                  <th>Pemohon</th>
                  <th>Dept / Jabatan</th>
                  <th>Aksi</th>
                </tr>
              </thead>
              <tbody>
                {items.length === 0 && (
                  <tr>
                    <td colSpan={9} style={{ padding: 10, textAlign: "center" }}>
                      Tidak ada data.
                    </td>
                  </tr>
                )}

                {items.map((t) => (
                  <tr key={t.id} className={getRowClass(t.status)}>
                    <td>{t.request_code}</td>
                    <td>{t.title}</td>
                    <td>{formatDateWithDay(t.training_date)}</td>
                    <td>{t.vendor}</td>
                    <td className="col-num">{formatCurrency(t.cost)}
                    </td>
                    <td>
                      <StatusBadge status={t.status} />
                    </td>
                    <td>{t.user?.name}</td>
                    <td>
                      {t.user?.department} / {t.user?.position}
                    </td>
                    <td>
                    <div className="actions">
                      {/* DETAIL – semua role boleh */}
                      <button
                        type="button"
                        className="btn btn-ghost btn-xs"
                        onClick={() => navigate(`/training/${t.id}`)}
                      >
                        Detail
                      </button>

                      {/* GROUP USER (EDIT/HAPUS) */}
                      {isUser && t.user?.id === user.id && t.status === "submitted" && (
                        <div className="actions-group">
                          <button
                            type="button"
                            className="btn btn-xs"
                            onClick={() => navigate(`/training/edit/${t.id}`)}
                          >
                            Edit
                          </button>
                          <button
                            type="button"
                            className="btn btn-danger btn-xs"
                            onClick={() => deleteRequest(t.id)}
                          >
                            Hapus
                          </button>
                        </div>
                      )}

                      {/* GROUP HRD */}
                      {isHrd && t.status === "submitted" && (
                        <div className="actions-group">
                          <button
                            type="button"
                            className="btn btn-primary btn-xs"
                            onClick={() => handleApproveHrd(t.id)}
                          >
                            Approve HRD
                          </button>
                          <button
                            type="button"
                            className="btn btn-danger btn-xs"
                            onClick={() => handleRejectHrd(t.id)}
                          >
                            Reject HRD
                          </button>
                        </div>
                      )}

                      {/* GROUP DIREKTUR / ADMIN */}
                      {isAdmin && t.status === "approved_hrd" && t.cost > 20000000 && (
                        <div className="actions-group">
                          <button
                            type="button"
                            className="btn btn-primary btn-xs"
                            onClick={() => handleApproveDirector(t.id)}
                          >
                            Approve Dir
                          </button>
                          <button
                            type="button"
                            className="btn btn-danger btn-xs"
                            onClick={() => handleRejectDirector(t.id)}
                          >
                            Reject Dir
                          </button>
                        </div>
                      )}
                    </div>
</td>
                  </tr>
                ))}
              </tbody>
            </table>
            {!loading && items.length > 0 && (
        <div
          style={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            marginTop: 8,
            fontSize: 12,
            color: "#9ca3af",
          }}
        >
          <div>
            Menampilkan halaman{" "}
            <strong>{meta.current_page}</strong> dari{" "}
            <strong>{meta.last_page}</strong> | Total data:{" "}
            <strong>{meta.total}</strong>
          </div>

          <div style={{ display: "flex", gap: 4 }}>
            <button
              type="button"
              className="btn btn-xs"
              disabled={meta.current_page <= 1}
              onClick={() => setPage((p) => Math.max(1, p - 1))}
            >
              &laquo; Prev
            </button>

            <button
              type="button"
              className="btn btn-xs"
              disabled={meta.current_page >= meta.last_page}
              onClick={() =>
                setPage((p) => Math.min(meta.last_page, p + 1))
              }
            >
              Next &raquo;
            </button>
          </div>
        </div>
)}
          </div>
        )}
      </main>

    </div>
  </div>
  );
}

function StatusBadge({ status }) {
  let text = "";
  let className = "badge ";
  let color = "#60a5fa"; // default biru

  switch (status) {
    case "submitted":
      text = "Submitted";
      className += "badge-submitted";
      color = "#60a5fa"; // biru
      break;

    case "approved_hrd":
      text = "Approved HRD";
      className += "badge-approved";
      color = "#4ade80"; // hijau
      break;

    case "approved_director":
      text = "Approved Direktur";
      className += "badge-approved";
      color = "#4ade80"; // hijau
      break;

    case "rejected_hrd":
      text = "Rejected HRD";
      className += "badge-rejected";
      color = "#f97373"; // merah
      break;

    case "rejected_director":
      text = "Rejected Direktur";
      className += "badge-rejected";
      color = "#f97373"; // merah
      break;

    default:
      text = status || "-";
      className += "badge-submitted";
      color = "#9ca3af"; // abu2
  }

  const icon = (
    <svg
      width="10"
      height="10"
      viewBox="0 0 10 10"
      aria-hidden="true"
    >
      <circle cx="5" cy="5" r="4" fill={color} />
    </svg>
  );

  return (
    <span
      className={className}
      style={{ display: "inline-flex", alignItems: "center", gap: "6px" }}
    >
      {icon}
      <span>{text}</span>
    </span>
  );
}

function formatDate(dateStr) {
  if (!dateStr) return "-";
  const d = new Date(dateStr);
  if (Number.isNaN(d.getTime())) return dateStr; // fallback kalau format aneh
  return d.toLocaleDateString("id-ID", {
    day: "2-digit",
    month: "2-digit",
    year: "numeric",
  });
}

function formatDateLong(dateStr) {
  if (!dateStr) return "-";

  const d = new Date(dateStr);
  if (Number.isNaN(d.getTime())) return dateStr;

  const day = d.getDate(); // 1–31 → tanpa leading zero
  const monthNames = [
    "Januari", "Februari", "Maret", "April", "Mei", "Juni",
    "Juli", "Agustus", "September", "Oktober", "November", "Desember"
  ];
  const month = monthNames[d.getMonth()];
  const year = d.getFullYear();

  return `${day} ${month} ${year}`;
}

function formatCurrency(value) {
  if (value === null || value === undefined) return "-";
  const num = Number(value);
  if (Number.isNaN(num)) return String(value);
  return num.toLocaleString("id-ID", {
    style: "currency",
    currency: "IDR",
    maximumFractionDigits: 0,
  });
}

function getRowClass(status) {
  switch (status) {
    case "submitted":
      return "row-submitted";
    case "approved_hrd":
    case "approved_director":
      return "row-approved";
    case "rejected_hrd":
    case "rejected_director":
      return "row-rejected";
    default:
      return "";
  }
}

function SummaryCard({ title, value, subtitle, tone = "default" }) {
  let border, glow;

  switch (tone) {
    case "info":
      border = "rgba(59, 130, 246, 0.6)";
      glow = "0 0 24px rgba(59, 130, 246, 0.2)";
      break;
    case "success":
      border = "rgba(34, 197, 94, 0.6)";
      glow = "0 0 24px rgba(34, 197, 94, 0.2)";
      break;
    case "danger":
      border = "rgba(239, 68, 68, 0.6)";
      glow = "0 0 24px rgba(239, 68, 68, 0.2)";
      break;
    default:
      border = "rgba(148, 163, 184, 0.5)";
      glow = "0 0 20px rgba(15, 23, 42, 0.8)";
  }

  return (
    <div
      style={{
        borderRadius: 14,
        padding: "10px 12px",
        background:
          "radial-gradient(circle at top, rgba(15,23,42,0.9), rgba(15,23,42,0.7))",
        border: `1px solid ${border}`,
        boxShadow: glow,
        display: "flex",
        flexDirection: "column",
        gap: 4,
      }}
    >
      <div style={{ fontSize: 11, color: "#9ca3af" }}>{title}</div>
      <div style={{ fontSize: 18, fontWeight: 600 }}>{value}</div>
      {subtitle && (
        <div style={{ fontSize: 11, color: "#9ca3af" }}>{subtitle}</div>
      )}
    </div>
  );
}