<?php

namespace App\Http\Controllers;

use App\Models\TrainingRequest;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

class TrainingRequestController extends Controller
{
    // batas biaya untuk butuh Direktur
    private const DIRECTOR_THRESHOLD = 20000000;

    // LIST
    public function index(Request $request)
    {

        $user = $request->user();

       /* if ($user->isHrd() || $user->isAdmin()) {
            $query = TrainingRequest::with('user');
        } else {
            $query = TrainingRequest::with('user')
                ->where('user_id', $user->id);
        }*/
                if ($user->isHrd() || $user->isAdmin()) {
        $query = TrainingRequest::with(['user', 'hrdApprover', 'directorApprover']);
    } else {
        $query = TrainingRequest::with(['user', 'hrdApprover', 'directorApprover'])
            ->where('user_id', $user->id);
    }

        // FILTER: status
        $status = $request->query('status');
        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }

        // FILTER: tanggal
        $from = $request->query('from');
        $to   = $request->query('to');

        if ($from) {
            $query->whereDate('training_date', '>=', $from);
        }

        if ($to) {
            $query->whereDate('training_date', '<=', $to);
        }

        // FILTER: departemen user
        $dept = $request->query('department');
        if ($dept) {
            $query->whereHas('user', function ($q) use ($dept) {
                $q->where('department', 'like', '%' . $dept . '%');
            });
        }

        // FILTER: search
        $search = $request->query('search');
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                ->orWhere('vendor', 'like', '%' . $search . '%')
                ->orWhere('request_code', 'like', '%' . $search . '%')
                ->orWhereHas('user', function ($uq) use ($search) {
                    $uq->where('name', 'like', '%' . $search . '%');
                });
            });
        }

        // setelah semua filter (status, tanggal, dept, search) SELESAI

        // Penting: clone SEBELUM paginate
        $baseQuery = clone $query;

        // Pagination
        $perPage = (int) $request->query('per_page', 10);
        if ($perPage < 1) {
            $perPage = 10;
        }
        if ($perPage > 100) {
            $perPage = 100;
        }

        $query->orderByDesc('created_at');
        $paginator = $query->paginate($perPage);

        // Summary GLOBAL (berdasarkan filter, tanpa paginate)
        $summaryQuery = clone $baseQuery;

        $summary = [
            'total_requests'      => $summaryQuery->count(),
            'total_cost'          => (clone $baseQuery)->sum('cost'),
            'total_submitted'     => (clone $baseQuery)->where('status', 'submitted')->count(),
            'total_approved_hrd'  => (clone $baseQuery)->where('status', 'approved_hrd')->count(),
            'total_approved_dir'  => (clone $baseQuery)->where('status', 'approved_director')->count(),
            'total_rejected_hrd'  => (clone $baseQuery)->where('status', 'rejected_hrd')->count(),
            'total_rejected_dir'  => (clone $baseQuery)->where('status', 'rejected_director')->count(),
        ];

        return response()->json([
            'data'    => $paginator->items(),
            'meta'    => [
                'current_page' => $paginator->currentPage(),
                'last_page'    => $paginator->lastPage(),
                'per_page'     => $paginator->perPage(),
                'total'        => $paginator->total(),
            ],
            'summary' => $summary,
        ]);

       
    }

    // CREATE (user)
    public function store(Request $request)
    {
        $user = $request->user();

       // Hanya karyawan (user) yang boleh membuat pengajuan
    if (! $user->isUser()) {
        return response()->json([
            'message' => 'Hanya karyawan (user) yang boleh membuat pengajuan.',
        ], 403);
    }

        $data = $request->validate([
            'title'           => ['required', 'string', 'max:255'],
            'description'     => ['nullable', 'string'],
            'training_date'   => ['required', 'date'],
            'vendor'          => ['nullable', 'string', 'max:255'],
            'cost'            => ['required', 'integer', 'min:0'],
            'purposes'        => ['nullable', 'array'],
            'purposes.*'      => ['string'],
            'expected_result' => ['nullable', 'string'],
        ]);

        $data['user_id'] = $user->id;
        $data['status']  = 'submitted';

        $training = TrainingRequest::create($data);

        return response()->json($training->load('user'), 201);
    }

    // DETAIL
    public function show(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

       /* if ($user->isUser() && $trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }*/
            if ($user->isUser() && $trainingRequest->user_id !== $user->id) {
        return response()->json(['message' => 'Forbidden'], 403);
    }

        return $trainingRequest->load('user','hrdApprover', 'directorApprover');
    }

    // UPDATE (oleh pemilik, hanya submitted)
    public function update(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if ($trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if (! $user->isUser()) {
        return response()->json(['message' => 'Hanya karyawan (user) yang boleh membuat pengajuan.'], 403);
    }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Tidak bisa edit request yang sudah di-approve/reject',
            ], 422);
        }

        $data = $request->validate([
            'title'           => ['required', 'string', 'max:255'],
            'description'     => ['nullable', 'string'],
            'training_date'   => ['required', 'date'],
            'vendor'          => ['nullable', 'string', 'max:255'],
            'cost'            => ['required', 'integer', 'min:0'],
            'purposes'        => ['nullable', 'array'],
            'purposes.*'      => ['string'],
            'expected_result' => ['nullable', 'string'],
        ]);

        $trainingRequest->update($data);

        return $trainingRequest->refresh()->load('user');
    }

    // DELETE (oleh pemilik, hanya submitted)
    public function destroy(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if ($trainingRequest->user_id !== $user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Tidak bisa hapus request yang sudah di-approve/reject',
            ], 422);
        }

        $trainingRequest->delete();

        return response()->json(['message' => 'Deleted']);
    }

    // APPROVE HRD
    public function approveHrd(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isHrd()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Hanya request dengan status submitted yang bisa di-approve HRD',
            ], 422);
        }

        $data = $request->validate([
            'hrd_notes' => ['nullable', 'string'],
        ]);

        $trainingRequest->update([
            'status'             => 'approved_hrd',
            'hrd_notes'          => $data['hrd_notes'] ?? $trainingRequest->hrd_notes,
            'approved_by_hrd'    => $user->id,
            'approved_at_hrd'    => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // REJECT HRD
    public function rejectHrd(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isHrd()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'submitted') {
            return response()->json([
                'message' => 'Hanya request dengan status submitted yang bisa di-reject HRD',
            ], 422);
        }

        $data = $request->validate([
            'hrd_notes' => ['nullable', 'string'],
        ]);

        $trainingRequest->update([
            'status'             => 'rejected_hrd',
            'hrd_notes'          => $data['hrd_notes'] ?? $trainingRequest->hrd_notes,
            'approved_by_hrd'    => $user->id,
            'approved_at_hrd'    => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // APPROVE DIREKTUR (role: admin), hanya jika cost > 20jt & sudah approved_hrd
    public function approveDirector(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isAdmin()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'approved_hrd') {
            return response()->json([
                'message' => 'Direktur hanya bisa approve request yang sudah disetujui HRD',
            ], 422);
        }

        if ($trainingRequest->cost <= self::DIRECTOR_THRESHOLD) {
            return response()->json([
                'message' => 'Request dengan biaya ≤ 20.000.000 tidak memerlukan persetujuan Direktur',
            ], 422);
        }

        $trainingRequest->update([
            'status'                 => 'approved_director',
            'approved_by_director'   => $user->id,
            'approved_at_director'   => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    // REJECT DIREKTUR
    public function rejectDirector(Request $request, TrainingRequest $trainingRequest)
    {
        $user = $request->user();

        if (! $user->isAdmin()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        if ($trainingRequest->status !== 'approved_hrd') {
            return response()->json([
                'message' => 'Direktur hanya bisa reject request yang sudah disetujui HRD',
            ], 422);
        }

        if ($trainingRequest->cost <= self::DIRECTOR_THRESHOLD) {
            return response()->json([
                'message' => 'Request dengan biaya ≤ 20.000.000 tidak memerlukan persetujuan Direktur',
            ], 422);
        }

        $trainingRequest->update([
            'status'                 => 'rejected_director',
            'approved_by_director'   => $user->id,
            'approved_at_director'   => now(),
        ]);

        return $trainingRequest->refresh()->load('user');
    }

    public function export(Request $request): StreamedResponse
    {
        $user = $request->user();

        // Hanya HRD & Admin yang boleh export
        if (! $user->isHrd() && ! $user->isAdmin()) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        // --- Reuse filter dari index() ---

        if ($user->isHrd() || $user->isAdmin()) {
            $query = TrainingRequest::with('user');
        } else {
            $query = TrainingRequest::with('user')
                ->where('user_id', $user->id);
        }

        $status = $request->query('status');
        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }

        $from = $request->query('from');
        $to   = $request->query('to');

        if ($from) {
            $query->whereDate('training_date', '>=', $from);
        }
        if ($to) {
            $query->whereDate('training_date', '<=', $to);
        }

        $dept = $request->query('department');
        if ($dept) {
            $query->whereHas('user', function ($q) use ($dept) {
                $q->where('department', 'like', '%' . $dept . '%');
            });
        }

        $search = $request->query('search');
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                ->orWhere('vendor', 'like', '%' . $search . '%')
                ->orWhere('request_code', 'like', '%' . $search . '%')
                ->orWhereHas('user', function ($uq) use ($search) {
                    $uq->where('name', 'like', '%' . $search . '%');
                });
            });
        }

        $items = $query->orderByDesc('created_at')->get();

        // --- Stream CSV ---

        $fileName = 'training_requests_' . now()->format('Ymd_His') . '.csv';

        $headers = [
            'Content-Type'        => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileName}\"",
        ];

        return response()->streamDownload(function () use ($items) {
            $handle = fopen('php://output', 'w');

            // Header kolom
            fputcsv($handle, [
                'No Pengajuan',
                'Tanggal Pengajuan',
                'Nama Karyawan',
                'Departemen',
                'Jabatan',
                'Nama Training',
                'Tanggal Training',
                'Penyelenggara',
                'Biaya',
                'Status',
                'Catatan HRD',
            ]);

            foreach ($items as $t) {
                fputcsv($handle, [
                    $t->request_code,
                    optional($t->created_at)->format('Y-m-d'),
                    $t->user->name ?? '',
                    $t->user->department ?? '',
                    $t->user->position ?? '',
                    $t->title,
                    optional($t->training_date)->format('Y-m-d'),
                    $t->vendor,
                    $t->cost,
                    $t->status,
                    $t->hrd_notes,
                ]);
            }

            fclose($handle);
        }, $fileName, $headers);
    }
}